//
// schedulerRun.js (Scheduler - Run Now)
//
//
// Note, any action added in Run Now keeps its state as of the time it has been added.
// If the same action is edited in the Actions from right afterwards and the action
// has not yet been run it would not affect the action in Run Now!

var schedulerRun = {

	panel: null,
	isOpenPanel: false,
	idCount: 0, // Used for element baseIds
	schedules: [], // Keeps schedule objects to be run, i.e.
	/*
		{
		id: 's0', 			--> ID issued in schedulerRun
		name: 'schedule A', --> The name we got from the schedule form
		actions: [] 		--> A clone of the actions at the time the user clicked Run Now
		},
		...
		
		Actions rows and elements contain a combined id as follows:
		scheduleId:actionId:..., i.e.:
		s0:i0:...
		
	*/
	
	basketId: '', // Used to check sync for the current basket array between client and server
	basket: [], // A flat array with schedule and action references to schedules array,
						// it is populated upon start()
	isRunning: false,
	isCancel: false,
	isComplete: false,
	runId: '', // Used for responses to check if the response is in sync with server per action
	runningBasketIndex: 0, // The index of the running action in the basket index.
							// Used upon cancel when schedule is running to indicate Canceled state
	
	// runningScheduleId: '',
	// runningActionId: '',
	
	// TempTimeMeasurement
	// startTime: 0,
		
	init: function() {
		
		//
		// Init Run Now panel
		//
		
		var panelObj = {
			panelId: 'scheduler_run_now:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: 'Run Now',
			right: 30,
			top: 5,
			zIndex: 5,
			isCover: false,
			isSticky: true,
			closeEvent: schedulerRun.cancel
		};
		
		schedulerRun.panel = new util.Panel3(panelObj);
		
		// Add single event to list element for remove items
		
		yEvent.addListener('scheduler_run_now:list:table', 'click', schedulerRun.removeActionItem);

		yEvent.addListener('scheduler_run_now:start_btn', 'click', schedulerRun.start);
		yEvent.addListener('scheduler_run_now:cancel_btn', 'click', schedulerRun.cancel);
	},
	
	setPanelState: function(panelState) {
		
		// panelState is: empty | start | running | complete
		var listLabel = '';
		var cancelBtnLabel = '';
		var isEnabledStartBtn = false;
		
		switch (panelState) {
			
			case 'start':
			
				listLabel = langVar('lang_admin.scheduler.run.click_start_to_run_schedules');
				cancelBtnLabel = langVar('lang_stats.btn.cancel');
				isEnabledStartBtn = true;
				break;
				
			case 'running':
			
				listLabel = langVar('lang_admin.scheduler.run.running_schedules');
				cancelBtnLabel = langVar('lang_stats.btn.cancel');
				isEnabledStartBtn = false;
				break;
				
			case 'complete':
			
				listLabel = langVar('lang_admin.scheduler.run.run_schedule_completed');
				cancelBtnLabel = langVar('lang_stats.btn.close');
				isEnabledStartBtn = false;
				break;
				
			case 'cancel': 
			
				// Canceled while an action is running
				listLabel = langVar('lang_admin.scheduler.run.canceled_schedule_info');
				cancelBtnLabel = langVar('lang_stats.btn.close');
				isEnabledStartBtn = false;
				break;
		}
		
		util.updateT('scheduler_run_now:list:label', listLabel);
		util.enableE('scheduler_run_now:start_btn', isEnabledStartBtn);
		// var cancelBtn = util.getE('scheduler_run_now:cancel_btn');
		// cancelBtn.value = cancelBtnLabel;
		util.updateT('scheduler_run_now:cancel_btn', cancelBtnLabel);
	},
		
	reset: function() {
		
		// Reset items
		var sr = schedulerRun;
		sr.idCount = 0;
		sr.schedules = [];
		sr.basket = [];
		sr.isRunning = false;
		sr.isCancel = false;
		sr.isComplete = false;
		
		// Clear list items
		util.removeChildElements('scheduler_run_now:list:table');
		
	},
	
	cancel: function() {
		
		// Invoked upon cancel or close
		if (schedulerRun.isRunning) {
			
			// Cancel without to close the panel
			
			// alert('Schedule is still running. The current running action will be completed in the background.');
			schedulerRun.isCancel = true;
			schedulerRun.setPanelState('cancel');
			
			// Indicated remaining actions as canceled
			var runningBasketIndex = schedulerRun.runningBasketIndex;
			var basket = schedulerRun.basket;
			
			for (var i = (runningBasketIndex + 1); i < basket.length; i++) {
				var baseId = basket[i].baseId;
				var stateElement = util.getE(baseId + ':state');
				util.removeChildElements(stateElement);
				var stateText = util.createT(langVar('lang_admin.scheduler.run.canceled'));
				stateElement.appendChild(stateText);
			}
		}
		else {
			
			// Close the panel
			
			schedulerRun.reset();
			schedulerRun.setPanelState('start');
			schedulerRun.panel.close();
			enableRunNowButton();
			schedulerRun.isOpenPanel = false;
		}
	},
	
	runNow: function() {
		
		// Note, runNow adds items to be run but does not run them yet!
		
		// Check if we have any actions defined
		var numOfActions = schedulerActionsList.list.getNumberOfItems();
		
		if (numOfActions > 0) {
			
			if (!schedulerRun.panel) {
				schedulerRun.init();
			}
			
			// Check if we need to reset an open panel
			if (schedulerRun.isOpenPanel && (schedulerRun.isCancel || schedulerRun.isComplete)) {
			
				// Panel is open with a completed or cancled list and 
				// the user adds a new schedule. So we start over.
				schedulerRun.reset();
				schedulerRun.setPanelState('start');
			}
			
			// alert('runNow() - numOfActions: ' + numOfActions);
			// util.disableE('scheduler:run_now_btn');
			disableRunNowButton();
			
			
			//
			// Create schedule object
			//
			
			var scheduleId = schedulerRun.getNewBaseId();
			var scheduleName = util.getF('scheduler:label');
			if (scheduleName == '') {
				scheduleName = langVar('lang_admin.scheduler.run.schedule');
			}
			var actions = schedulerActionsList.list.getItemsClone();
			
			// Add an id to each scheduler action (We use the id for element creation and deletion)
			for (var i = 0; i < actions.length; i++) {
				actions[i].id = 'i' + i;
			}
			
			var obj = {};
			obj.id = scheduleId;
			obj.name = scheduleName;
			obj.actions = actions;
			
			// Add schedule object to schedules
			var schedules = schedulerRun.schedules;
			schedules[schedules.length] = obj;
			
			// Build schedule object in list
			schedulerRun.addScheduleToList(scheduleId, scheduleName, actions);
			
			// util.showObject(actions);
			
			if (!schedulerRun.isOpenPanel) {
				
				schedulerRun.setPanelState('start');
				schedulerRun.panel.open();
				schedulerRun.isOpenPanel = true;
			}
		}
		else {
			
			alert(langVar('lang_admin.scheduler.run.no_action_defined_msg'));
		}
	},

	start: function() {
		
		// Start to run the schedule(s)
		
		// Check if there are any actions in the list as all could have been removed!
		
		// Create basket array with schedule and action references for simple sequential processing
		// Hide the remove buttons in schedule now list
		
		var schedules = schedulerRun.schedules;
		
		
		
		schedulerRun.basket = [];
		var basket = schedulerRun.basket;
		
		for (var i = 0; i < schedules.length; i++) {
			var scheduleId = schedules[i].id;
			var actions = schedules[i].actions;
			for (var j = 0; j < actions.length; j++) {
				var actionId = actions[j].id;
				
				var baseId = scheduleId + ':' + actionId;
			
				basket[basket.length] = {baseId:baseId, action:actions[j]};
				util.hideEV(baseId + ':remove');
			}
		}
		
		if (basket.length > 0) {
			
			// Issue new basketId
			var theDate = new Date();
			var basketId = 'basket_' + theDate.getTime();
			schedulerRun.basketId = basketId;
		
			disableRunNowButton();
			
			schedulerRun.isRunning = true;
			// schedulerRun.runId = schedulerRun.runId + 1;
			
			// update panel
			schedulerRun.setPanelState('running');
			schedulerRun.runScheduleAction(0);
		}
		else {
			alert('lang_admin.scheduler.run.no_schedule_defined_msg');
		}
	},
	
	runScheduleAction: function(basketIndex) {
		
		schedulerRun.runningBasketIndex = basketIndex;
		
		// TempTimeMeasurement
		/*
		var startDate = new Date();
		schedulerRun.startTime = startDate.getTime();
		*/
		
		var basketItem = schedulerRun.basket[basketIndex];
		var baseId = basketItem.baseId;
		var action = basketItem.action;
		
		//
		// Handle state cell
		//
		
		var stateElement = util.getE(baseId + ':state');
		util.removeChildElements(stateElement);
		var stateText = util.createT(langVar('lang_admin.scheduler.run.running'));
		stateElement.appendChild(stateText);
		
		// Issue a new runId
		var theDate = new Date();
		var runId = 'run_' + theDate.getTime();
		schedulerRun.runId = runId;
		
		// util.showObject(action);
		
		//
		// Assemble URL dat
		//
		
		var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
		dat += 'v.fp.run_id=' + runId + '&';
		dat += 'v.fp.basket_id=' + schedulerRun.basketId + '&';
		dat += 'v.fp.basket_index=' + basketIndex + '&';
		
		var path = 'v.fp.schedule';
		dat += path + '.disabled=false&';
		dat += path + '.label=&';
		dat += path + '.month=*&';
		dat += path + '.day=*&';
		dat += path + '.hour=0&';
		dat += path + '.minute=0&';
		
		// Add the action
		var actionPath = path + '.actions.0';
		dat += getActionItemDat(actionPath, action); // called in scheduler.js
		
		dat = dat.replace(/&$/, '');
		
		var url = '?dp+templates.admin_pages.scheduler.run_schedule_start';
		util.serverPost(url, dat);
	},
	
	runScheduleActionResponse: function(dat) {
		
		/*
		alert('runScheduleActionResponse()');
		util.showObject(dat);
		return false;
		*/
		
		if (dat.runId == schedulerRun.runId && dat.basketId == schedulerRun.basketId) {
			schedulerRun.runScheduleProgress();
		}
	},
	
	runScheduleProgress: function() {
		
		var url = '?dp+templates.admin_pages.scheduler.run_schedule_progress';
		
		// Checks if the current action completed
		var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
		dat += 'v.fp.run_id=' + schedulerRun.runId;
		
		util.serverPost(url, dat);
	},
	
	runScheduleProgressResponse: function(dat) {
		
		// alert('runScheduleProgressResponse()');
		// util.showObject(dat);
		// return false;
		
		if (dat.runId == schedulerRun.runId && dat.basketId == schedulerRun.basketId) {
			
			// var isComplete = (dat.isComplete == true);
			
			if (dat.isComplete) {
				
				// TempTimeMeasurement
				/*
				var endDate = new Date();
				var endTime = endDate.getTime();
				var startTime = schedulerRun.startTime;
				var deltaTime = endTime - startTime;
				var seconds = deltaTime / 1000;
				var minutes = seconds / 60;
				var msg = 'The action took\nseconds: ' + seconds + '\nminutes: ' +  minutes;
				alert(msg);
				*/
								
				
				// processed item data
				var basketIndex = dat.basketIndex;
				var isError = dat.isError;
				
				var stateText = !isError ? langVar('lang_admin.scheduler.run.complete') : langVar('lang_admin.scheduler.run.error');
				
				var basket = schedulerRun.basket;
				var baseId = basket[basketIndex].baseId;
				
				util.updateT(baseId + ':state', stateText);
				
				if (isError) {
					schedulerRun.handleActionErrorDisplay(baseId, dat.alertId);
				}
				
				//
				// Handle next basket item
				//
				
				if (!schedulerRun.isCancel) {
				
					nextBasketIndex = basketIndex + 1;
					
					if (nextBasketIndex < basket.length) {
						schedulerRun.runScheduleAction(nextBasketIndex);
					}
					else {
						// All actions have been run. Set completed state
						schedulerRun.setPanelState('complete');
						schedulerRun.isRunning = false;
						schedulerRun.isComplete = true;
						enableRunNowButton();
					}
				}
				else {
					// This must be the running action upon cancel
					// We just set isRunning
					schedulerRun.isRunning = false;
				}
			}
			else {
				
				// Continue progress calls
				setTimeout('schedulerRun.runScheduleProgress()', 1500);
			}
		}
	},
	
	
	handleActionErrorDisplay: function(baseId, alertId) {
		
		// Change error state column to Red
		var tdState = util.getE(baseId + ':state');
		tdState.style.color = 'Red';
		
		// Add error message to label cell
		var tdLabel = util.getE(baseId + ':label');
		var br = util.createE('br');
		
		var span = util.createE('span', {color:'Red'});
		var spanText = util.createT(langVar('lang_admin.scheduler.run.error_occured_msg') + ' ');
		var href = '?dp=alert&ai=' + alertId;
		var a = util.createE('a', {href:href, target:'_blank'});
		var aText = util.createT(langVar('lang_admin.scheduler.run.click_here_to_view_alert'));
		
		util.chainE([tdLabel, br, [span, spanText], [a, aText]]);
	},
	
	getNewBaseId: function() {
		
		var i = schedulerRun.idCount;
		var id = 'srn_' + i;
		schedulerRun.idCount = i + 1;
		
		return id;
	},
	
	
	//
	//
	// List handling functions
	//
	//
	
	removeActionItem: function(evt) {
		
		// Invoked upon list click, we have to check if the user actually clicked a link
		var element = evt.target || evt.srcElement;
		var elementId = element.id;
		
		if (elementId != '' && (elementId.indexOf(':remove') != -1)) {
			
			var dat = elementId.split(':');
			var scheduleId = dat[0];
			var actionId = dat[1];
			
			var schedules = schedulerRun.schedules;
			
			for (var i = 0; i < schedules.length; i++) {
				
				var theSchedule = schedules[i];
				
				if (theSchedule.id == scheduleId) {
					
					var actions = theSchedule.actions;
					var tbody = util.getE(scheduleId + ':tbody');
					
					if (actions.length > 1) {
						
						// Remove action from actions array
						for (var j = 0; j < actions.length; j++) {
							if (actions[j].id == actionId) {
								actions.splice(j, 1);
								break;
							}
						}
						// Remove action row
						var tr = util.getE(scheduleId + ':' + actionId + ':tr');
						tbody.removeChild(tr);					
					}
					else {
						// There is only one action, so we remove the entire schedule
						
						// Remove schedule from schedules array
						schedules.splice(i, 1);
						
						// Remove schedule from list
						var table = util.getE('scheduler_run_now:list:table');
						table.removeChild(tbody);
						
					}
				}
			}
		}
	},

	addScheduleToList: function(scheduleId, scheduleName, actions) {
		
		var table = util.getE('scheduler_run_now:list:table');
		var tbody = util.createE('tbody', {id:scheduleId + ':tbody'});
		
		// Build header row
		var tr = util.createE('tr', {className:'header'});
		var tdLabel = util.createE('td', {colSpan:3, className:'header'});
		// var tdState = util.createE('td');
		var text = util.createT(scheduleName);
		
		util.chainE(tbody, [tr, [tdLabel, text]]);
		
		// Add the action rows
		for (var i = 0; i < actions.length; i++) {
			schedulerRun.addActionRow(tbody, scheduleId, i, actions[i]);
		}
		
		util.chainE(table, tbody);
	},
	
	addActionRow: function(tbody, scheduleId, actionIndex, action) {
		
		actionBaseId = scheduleId + ':' + action.id;
		
		var tr = util.createE('tr', {id:actionBaseId + ':tr'});
		
		var th = util.createE('th');
		var thText = util.createT(actionIndex + 1);
		
		var tdLabel = util.createE('td', {id:actionBaseId + ':label'});
		
		// Note, the label we get from labelBuilder is a string whcih may
		// contain HTML tags, so we use innerHTML to display the label.
		var label = schedulerActions.labelBuilder(action);
		
		tdLabel.innerHTML = label;
		
		// var tdLabelText = util.createT(labelObj.label);
		// util.chainE(tdLabel, tdLabelText);
		/*
		if (labelObj.subLabel != '') {
			var span = util.createE('span', {className:'label'});
			var spanText = util.createT(' ' + labelObj.subLabel);
			util.chainE(tdLabel, span, spanText);
		}
		*/
		
		var tdState = util.createE('td', {id:actionBaseId + ':state', className:'state'});
		var a = util.createE('a', {id:actionBaseId + ':remove', href:'javascript:;'});
		var aText = util.createT(langVar('lang_stats.btn.remove'));
		
		util.chainE(tbody, [tr, [th, thText], tdLabel, [tdState, [a, aText]]]);
	}
}

