//
//
// creControl.js 
//
// Main object to handle Customize Report Element HTML object.        
// This object requires
// creGraphs.js
// creTable.js
// creTableOptions.js
// crePivotTable.js
// creGraphOptions.js
// creUtil.js
//

var creControl = {
	
	tabs: null,
	tabIds: [],
	
	queryFieldsDb: [], // Object reference to queryFieldsDb
	isCustomizeInReports: false,
	isPivotTablePermission: true,
	isGraphOptionsPermission: true,
	
	reportElementDb: {}, // Object reference to the active reportElementDb, it is set by initReportElementDb
	
	// graphsDb contains the active graphs object (default graphs overriden by any report element graphs data)
	// graphsDb is used by creGraphs and creGraphOptions
	// graphsDb: {}, // Object reference to the active graphs object
	
	
	// Main Customize Report Element objects
	
	// general: null --> This tab panel object is currently handled in reportElement.js!
	
	general: null,
	graphs: null,
	table: null,
	tableOptions: null,
	pivotTable: null,
	graphOptions: null,
	
	//
	//
	// Initializes creControl object
	//
	//
	
	initCreObject: function(
			queryFieldsDb,
			defaultGraphsDb,
			isCustomizeInReports,
			isPivotTablePermission,
			isGraphOptionsPermission,
			hideLogDetailSortingMessage) {
		
		creControl.queryFieldsDb = queryFieldsDb;
		creControl.defaultGraphsDb = defaultGraphsDb;
		creControl.isCustomizeInReports = isCustomizeInReports;
		creControl.isPivotTablePermission = isPivotTablePermission;
		creControl.isGraphOptionsPermission = isGraphOptionsPermission;
		
		// util.showObject(defaultGraphsDb);
		
		//
		// Init tabs
		//
		
		if (isCustomizeInReports) {
			creControl.tabIds = ['graphs', 'table', 'table_options', 'pivot_table', 'graph_options'];
		}
		else {
			creControl.tabIds = ['general', 'filters', 'graphs', 'table', 'table_options', 'pivot_table', 'graph_options'];
		}
		
		creControl.tabs = new util.Tabs3('cre_obj:tabs', creControl.tabIds, creControl.tabActivated);
		
		//
		// Init Customize Report Element objects
		//
		
		creControl.graphs = new CreGraphs(queryFieldsDb);
		creControl.table = new CreTable(queryFieldsDb, isCustomizeInReports, hideLogDetailSortingMessage);
		creControl.tableOptions = new CreTableOptions(queryFieldsDb, isCustomizeInReports);
		creControl.pivotTable = new CrePivotTable(queryFieldsDb);
		creControl.graphOptions = new CreGraphOptions(queryFieldsDb, defaultGraphsDb);
		
		// Set graphOptions reference in graphs object!
		creControl.graphs.setGraphOptionsReference(creControl.graphOptions);
		
		//
		// Init General panel, Filters panel and Manage Fields Editor
		//
		
		if (!isCustomizeInReports) {
		
			creGeneral.init();
			creFilters.init();
			reportElementMF.init();
			yEvent.addListener('cre_obj:graphs:manage_fields_btn', 'click', creControl.manageFields);
			yEvent.addListener('cre_obj:table:manage_fields_btn', 'click', creControl.manageFields);
		}
	},
	
	//
	//
	// Init a reportElementDb object (Set all tab objects according the reportElementDb object)
	// init is called upon every report element meta type change!
	//
	//
	
	init: function(reportElementDb, obj) {
		
		// obj.defaultTabId and obj.totalRows are optional, respectively they are only specified
		// if the CRE object is used in dynamic reports or in Config when init() is called
		// from Manage Fields panel.
		
		// alert('creControl.init() - obj.defaultTabId: ' + obj.defaultTabId);
		
		var totalRows = obj.totalRows ? obj.totalRows : 0;
		var defaultTabId = obj.defaultTabId ? obj.defaultTabId : '';
		
		creControl.reportElementDb = reportElementDb;
		
		// util.showObject(reportElementDb);
		
		var reportElementType = reportElementDb.type;
		var constructionType = creUtil.getConstructionType(reportElementType);
		var isLogDetail = (reportElementType == 'log_detail');
		
		// alert('constructionType: ' + constructionType);
		
		// Handle the general and filters tab dependencies
		if (!creControl.isCustomizeInReports) {
			
			// Note, creGeneral and creFilters form update is not required here, it is only
			// accomplished upon openeing the CRE element in callee!
			// We only handle some display issues depending on the constructionType at this point
			
			creGeneral.setConstructionTypeDependencies(constructionType, isLogDetail);
			creFilters.setConstructionTypeDependencies(constructionType, isLogDetail);
		}
		
		if (constructionType == 'standard') {
			
			// We update all panels regardless if all panels
			// are displayed or not because the tab panels
			// could be changed any time via show_graphs and show_table
			
			creControl.table.init(reportElementDb, totalRows);
			creControl.tableOptions.init(reportElementDb, totalRows);
			
			if (!isLogDetail) {
		
				creControl.graphs.init(reportElementDb);
				creControl.graphOptions.init(reportElementDb);
				creControl.pivotTable.init(reportElementDb);
			}
		}
		else if (constructionType == 'session_paths' || constructionType == 'session_page_paths') {
			creControl.tableOptions.init(reportElementDb, totalRows);
		}
		
		//
		// Set the tab sequence
		//
	
		creControl.updateTabSequence(defaultTabId);
	},
	
	saveAndReInitGraphOptions: function() {
		
		// This saves and re-inits the graphs due a change in
		// reportElementDb.show_graphs and  reportElementDb.show_table property.
		// A change in this two properties (which are conrolled outside creControl)
		// most likely changes the graphs sorting parameters.
		
		creControl.graphOptions.saveChanges();
		creControl.graphOptions.init(creControl.reportElementDb);
	},

	//
	//
	// Save changes to reportElementDb
	//
	//

	saveChanges: function() {
		
		// Note, some CRE interactions are already saved in ReportElementDb,
		// so we only save the remaining data to the ReportElementDb
		
		var isCustomizeInReports = creControl.isCustomizeInReports;
		
		var reportElementDb = creControl.reportElementDb;
		var reportElementType = reportElementDb.type;
		var constructionType = creUtil.getConstructionType(reportElementType);
		var isLogDetail = (reportElementType == 'log_detail');
		
		// util.showObject(reportElementDb);
		
		if (!isCustomizeInReports) {
			
			// Save creGeneral and creFilters data
			creGeneral.saveChanges(reportElementDb, constructionType, isLogDetail);
			creFilters.saveChanges(reportElementDb, constructionType, isLogDetail);
		}
		
		if (constructionType == 'standard') {
			
			// Note, we save graphs and table, also if only
			// one of the two is actually active. This shouldn't
			// matter because default data alwyas exist for graphs
			// and table. The only exception is the log_detail report
			// element which only has table and tableOptions data.
			
			creControl.tableOptions.saveChanges();
			
			if (!isLogDetail) {
			
				creControl.pivotTable.saveChanges();
				creControl.graphOptions.saveChanges();
			}
		}
		else if (constructionType == 'session_paths' || constructionType == 'session_page_paths') {
			creControl.tableOptions.saveChanges();
		}
		
		// Nothing is to save for overview reports!
	},
	
	//
	//
	// Manage Fields
	//
	//
	
	manageFields: function() {
		
		// Track the tabId so that we show the right tab upon saving the fields in reportElementMF.js
		var defaultTabId = (this.id == 'cre_obj:graphs:manage_fields_btn') ? 'graphs' : 'table';
		reportElementMF.openPanel(defaultTabId);
	},


	//
	//
	// Tabs handling
	//
	//
	
	tabActivated: function(tabId) {
		
		// alert('customizeRE.tabActivated() - tabActivated: ' + tabId);
	
		// KHP-RC, IE6 invokes tabActivated() twice, the argument tabId is
		// first an object and the second time the actual tabId string.
		// The reason is not yet known, it could be something wrong with
		// yEvent or with the self reference object in the tab3 class.
		// This has to be fixed. In the meantime we work around this 
		// problem by checking the argument to be not an object, via !isObject()
	
		if (!util.isObject(tabId)) { // isObject has be addded due the above described IE problem
		
			creControl.setTabPanel(tabId);
			
			// if (tabId == 'graphs') {
				// KHP-RC - resolve this function!
				// customizeRE.graphs.checkGraphsFormIntegrity();
			// }
		}
	},
	
	setTabPanel: function(tabId) {
				
		var tabIds = creControl.tabIds;
		for (var i = 0; i < tabIds.length; i++) {
			util.hideE('cre_obj:' + tabIds[i] + ':panel');
		}
		
		util.showE('cre_obj:' + tabId + ':panel');
		
		creControl.tabs.setActiveTab(tabId);
	},
	
	updateTabSequence: function(optionalDefaultTabId) {
		
		// Note, defaultTabId is optional!
		
		var defaultTabId = optionalDefaultTabId ? optionalDefaultTabId : '';
		
		// setTabSequence is called upon init() but also from outside creControl
		// upon show_table or show_graphs changes. In this case we don't need
		// to update any tab panel data but only change the tab sequence
		// and set a defaultTabId if not yet given.
		
		var reportElementDb = creControl.reportElementDb;
		var reportElementType = reportElementDb.type;
		var constructionType = creUtil.getConstructionType(reportElementType);
		
		// var showCreObjectPanel;
		
		var tabSequence = [];
		
		if (!creControl.isCustomizeInReports) {
			// We always show the general and filters tab in Config!
			tabSequence[0] = 'general';
			tabSequence[1] = 'filters';
		}
		
		if (constructionType == 'standard') {
			
			if (reportElementType != 'log_detail') {
				
				var showGrahps = reportElementDb.show_graphs;
				var showTable = reportElementDb.show_table;
				
				if (showGrahps) {
					tabSequence[tabSequence.length] = 'graphs';
				}
				
				if (showTable) {
					
					tabSequence[tabSequence.length] = 'table';
					tabSequence[tabSequence.length] = 'table_options';
					
					if (creControl.isPivotTablePermission) {
						tabSequence[tabSequence.length] = 'pivot_table';
					}
				}
				
				if (showGrahps && creControl.isGraphOptionsPermission) {
					tabSequence[tabSequence.length] = 'graph_options';
				}
			}
			else {
				
				// log_detail is limited to table and table_options
				tabSequence[tabSequence.length] = 'table';
				tabSequence[tabSequence.length] = 'table_options';
			}
		}
		else if (constructionType == 'session_paths'  || constructionType == 'session_page_paths') {
			tabSequence[tabSequence.length] = 'table_options';
		}
		
		if (defaultTabId == '') {
			// Use first tab of tabSequence
			defaultTabId = tabSequence[0];
		}
		
		// Set tab sequence and tab panel
		creControl.tabs.setSequence(tabSequence, defaultTabId);
		creControl.setTabPanel(defaultTabId);
	}
};
