//
//
// emailUtil.js
//
// Used in Scheduler and Send Report By Email to handle recipient addresses
//
//

var emailUtil = {};


// AddressControl Class

emailUtil.AddressControl = function(containerElementId, addressElementWidth) {
	
	// alert('emailUtil.AddressControl');
	
	this.numberOfRows = 0; // The number of actuially build recipient rows
	this.addresses = []; // keeps the state upon validation and is also used to get the final email data
						// because they are written to addresses at validation time
		
	// Get a unique ID prefix
	var idPrefix = util.getUniqueElementId();
	this.idPrefix = idPrefix;
	
	this.addressDivId = idPrefix + ':addresses:div';
	this.addressTbodyId = idPrefix + ':addresses:tbody';
	
	this.isErrorIndication = false;
	
	
	this._build(containerElementId, idPrefix, addressElementWidth);
	
	yEvent.addListener(this.addressTbodyId, 'keyup', this._addressItemActicated, this);
}

emailUtil.AddressControl.prototype.init = function(addresses) {
	
	// addresses contains any default email addresses, 
	// i.e.: [{type:'to',  address:'abc@abc.com'}, {type:'cc', address:'first last <abc@abc.com>'}]
	
	var i;
	var idPrefix = this.idPrefix;
	
	if (addresses.length > 4) {
		
		// There are more than 4 default addresses, so we have to add additional rows
		var tbody = util.getE(this.addressTbodyId);
		var numberOfExistingRows = this.numberOfRows;
		var numberOfRequiredRows = addresses.length;
		
		for (i = numberOfExistingRows; i < numberOfRequiredRows; i++) {
			
			this._addRow(tbody, idPrefix, i);
			this._populateSelect(idPrefix, i);
		}
	}
	
	// Handle default addresses
		
	for (i = 0; i < addresses.length; i++) {
		
		var item =  addresses[i];
		
		util.setF(idPrefix + ':' + i + ':select', item.type);
		util.setF(idPrefix + ':' + i + ':input', item.address);
	}
}

emailUtil.AddressControl.prototype.validate = function() {
	
	// Validates the email addresses and handles error indication.
	// At least one valid email address must exist. If one of multiple
	// email addresses is not valid we show an error.
	
	// We validate and write email data to this.addresses.
	// this.addresses objects contains an isError property so that we know
	// if we must reset any color if an email address is indicated in Red
	
	// this.addresses is in the format: 
	// [{type:'to', name:'', email:'abc@abc.com', isError:false}, {type:'cc', name:'first last', email:'abc@abc.com',  isError:false}]
	
	var idPrefix = this.idPrefix;
	this.addresses = [];
	var isEmailAddress = false;
	var isAddressWithError = false;
	
	for (var i = 0; i < this.numberOfRows; i++) {
		
		var type = ''; // We only define a type if we find a valid email address
		var address = util.getF(idPrefix + ':' + i + ':input');
		
		var isError = false;
		
		if (address != '') {
			
			if (util.isEmailAddress(address)) {
				
				isEmailAddress = true;
				type = util.getF(idPrefix + ':' + i + ':select');
			}
			else {
				isAddressWithError = true;
				isError = true;
				var element = util.getE(idPrefix + ':' + i + ':input');
				element.style.color = 'Red';
			}
		}
		
		// Add object to this.addresses
		this.addresses[i] = {type:type, address:address, isError:isError};
	}
	
	if (isEmailAddress && !isAddressWithError) {
		return true;
	}
	else {
		
		// Handle error indication
		// alert('Error in recipient addresses');
		var msg = isAddressWithError ? langVar('lang_stats.general.invalid_email_address_in_recipients_msg') : langVar('lang_stats.general.no_recipient_address_message');
		
		util.updateT('email-address-grid:error', msg);
		util.showE('email-address-grid:error');
		
		this.isErrorIndication = true;
	}
	
	return false;
}

emailUtil.AddressControl.prototype._addressItemActicated = function(evt, self) {
	
	var element = evt.target || evt.srcElement;
	var elementId = element.id;
	
	// alert('_addressItemActicated() - id: ' + elementId);
	
	var dat = elementId.split(':');
	var rowIndex = parseInt(dat[1], 10);
	var elementType = dat[2];
	
	if (elementType == 'input') {
		
		// Handle error div on bottom
		if (self.isErrorIndication) {
			util.hideE('email-address-grid:error');
			self.isErrorIndication = false;
		}
		
		// Reset error of current row
		// Note, the row may not yet exist in this.addresses because it is only
		// added upon validation, so check for existence.
		var addresses = self.addresses;
		if (addresses[rowIndex]) {
			
			var address = addresses[rowIndex];
			if (address.isError) {
				element.style.color = 'Black';
				address.isError = false;
			}
		}
		
		
		// Add a new row if the current row is the last row
		var numberOfRows = self.numberOfRows;
		
		if (rowIndex == numberOfRows - 1) {
			var tbody = util.getE(self.addressTbodyId);
			self._addRow(tbody, self.idPrefix, numberOfRows);
			self._populateSelect( self.idPrefix, numberOfRows);
		}
	}
}

emailUtil.AddressControl.prototype.getAddresses = function() {
	
	// Returns all valid addresses
	// Valid addresses already exist in this.addresses since validation.
	// Valid addresses have a defined type, others not.
	var a = this.addresses;
	var b = [];
	
	for (var i = 0; i < a.length; i++) {
		
		var item = a[i];
		
		if (item.type != '') {
			
			b[b.length] = {type:item.type, address:item.address};
		}
	}

	return b;
}

emailUtil.AddressControl.prototype.reset = function() {
	
	// Resets the control by:
	// a.) setting the row numbers to 4
	// b.) removing any error indication
	// c.) resetting form fields to "to:" address and empty input field
	
	var i;
	
	if (this.isErrorIndication) {
		util.hideE('email-address-grid:error');
		this.isErrorIndication = false;
	}
	
	this.addresses = [];
	
	//
	// Reset rows to four rows
	//
	var numberOfRows = this.numberOfRows;
	
	if (this.numberOfRows > 4) {
		
		// Remove rows
		var tbody = util.getE(this.addressTbodyId);
		var rows = tbody.getElementsByTagName('tr');
		
		for (i = rows.length - 1; i > 3; i--) {
			var theRow = rows[i];
			tbody.removeChild(theRow);
		}
		
		this.numberOfRows = 4;
	}
	
	//
	// Reset select and input elements
	//
	
	var idPrefix = this.idPrefix;
	
	for (i = 0; i < 4; i++) {
		
		util.setF(idPrefix + ':' + i + ':select', 'to');
		var input = util.getE(idPrefix + ':' + i + ':input');
		input.value = '';
		input.style.color = 'Black';
	}
}

emailUtil.AddressControl.prototype.disable = function(isDisable) {
	
	// isDisable is optional
	var makeDisabled = (isDisable != null) ? isDisable : true;
	var numberOfRows = this.numberOfRows;
	var idPrefix = this.idPrefix;
	
	var a = [];
	
	for (var i = 0; i < numberOfRows; i++) {
		
		a[a.length] = idPrefix + ':' + i + ':select';
		a[a.length] = idPrefix + ':' + i + ':input';
	}
	
	util.disableE(a, makeDisabled);
}

emailUtil.AddressControl.prototype.freezeOverflow = function(isFreeze) {
	
	// Fixes a firefox bug, we require to set overflow to hidden when
	// a subform is displayed above the div element where overflow is set
	// to scroll
	
	var overflow = (isFreeze == null || isFreeze == true) ? 'hidden' : 'auto';
	var element = util.getE(this.addressDivId);
	element.style.overflow = overflow;
}

emailUtil.AddressControl.prototype._build = function(containerElementId, idPrefix, addressElementWidth) {
	
	// containerElementId is usually a td element
	
	// Builds the control
	// We create four address rows as default, starting from row 0, 1, 2, 3
	
	var container = util.getE(containerElementId);
	var addressDiv = util.createE('div', {id:this.addressDivId, className:'email-address-grid', width:addressElementWidth + 'px'});
	var table = util.createE('table', {className:'email-address-grid', cellSpacing:0});
	
	var tbody = util.createE('tbody', {id:this.addressTbodyId});
	
	var i;
	
	for (i = 0; i < 4; i++) {
		this._addRow(tbody, idPrefix, i);
	}
	
	// Add a div for error indication
	var errorDiv = util.createE('div', {id:'email-address-grid:error', className:'form-error'});
	
	util.chainE([container, [addressDiv, [table, tbody]], errorDiv]);
	
	// Handle select elements
	// Populate select only works after the select element is part of the dom
	
	for (i = 0; i < 4; i++) {
		this._populateSelect(idPrefix, i);
	}
}

emailUtil.AddressControl.prototype._addRow = function(tbody, idPrefix, rowIndex) {
	
	// Builds a row with select and input element
	// We create four address rows as default, starting from row 0
	
	var selectId = idPrefix + ':' + rowIndex + ':select';
	var inputId = idPrefix + ':' + rowIndex + ':input';
	
	var tr = util.createE('tr');
	var th = util.createE('th', {padding:'0px'});
	var td = util.createE('td', {padding:'0px'});
	var select = util.createE('select', {id:selectId});
	var input = util.createE('input', {id:inputId, type:'text', value:''});
	
	util.chainE(tbody, [tr, [th, select], [td, input]]);
	
	this.numberOfRows = rowIndex + 1;
}

emailUtil.AddressControl.prototype._populateSelect = function(idPrefix, rowIndex) {
	
	var list = [{name:'to', label:langVar('lang_stats.general.email_to') + ':'}, {name:'cc', label:langVar('lang_stats.general.email_cc') + ':'}, {name:'bcc', label:langVar('lang_stats.general.email_bcc') + ':'}];
	
	var selectId = idPrefix + ':' + rowIndex + ':select';
	util.populateSelect(selectId, list, 'name', 'label');
	util.setF(selectId, 'to');
}


