//
// progress.js
//

// Note, progress also displays action infos
// such as Loading document, etc., without
// the progress meter section.
// We don't make this a class due referencing problems upon server calls!

var progress = {
	
	// pageToken: '', pageToken is set in progress HTML
	
	//
	// Initialization variables set in callee
	//
	
	decimalDivider: '.', // KHP-RC, get decimalDivider when initializing prgress.
	isCancelTaskPermission: false,
	profileName: '',
	
	// calleeListener
	// calleeListener is a function in the callee which is called upon progress state change so that the callee
	// can take any further action, i.e. upon progress complete or when a task is canceled.
	// The states which are send to the callee are:
	// 		report_progress_complete
	//		database_progress_complete
	// 		database_task_not_active
	// 		task_canceled
	
	calleeListener: null,

	//
	// Variables set by methods
	//
	
	getDbTaskStateCount: 0,
	isReportProgress: false,
	taskId: '',
	reportJobId: '',
	channelId: '', // The channelId is send from the client and resend from the server to check if the response matches the request.
	progressIsVisible: false,
	baseInitDone: false,
	cancelTaskIsActive: false,
		 
	showDebugOutput: false, // used internally to show debug output in progress display
	
	//
	// Methods
	//

	reset: function() {
		
		// reset the labels and progress meter
		util.updateT('progress:step_number', '&nbsp;');
		util.updateT('progress:step_label', '&nbsp;');
		util.updateT('progress:step_description', '&nbsp;');
		util.hideE(['progress:header_buttons', 'progress:body', 'progress:details:body', 'progress:details:show_btn', 'progress:details:hide_btn', 'progress:details:container']);
		progress.updateProgressMeter(0);
		progress.progressIsVisible = false;
	},
	
	showActionInfo: function(actionType) {
		
		// showActionInfo is a pre-progress information display, it
		// shows only the main label and spep label, i.e. Loading report, Generating report.
		// It is called for specific actions, i.e. loading or
		// when updating or rebuilding the database to indicate
		// pre-progress state, i.e. erasing database, building initiated
		
		progress.reset();
		
		var mainLabel = '';
		var stepLabel = '';
		
		switch (actionType) {
			
			case 'loading':
				mainLabel = langVar('lang_stats.progress.loading_document');
				break;
				
			case 'update_database_init':
				mainLabel = langVar('lang_stats.progress.preparing_update_database');
				stepLabel = langVar('lang_stats.progress.init_database');
				break;
			
			case 'rebuild_database_init':
				mainLabel = langVar('lang_stats.progress.preparing_build_database');
				stepLabel = langVar('lang_stats.progress.init_database');
				break;
				
			case 'rebuild_database_erase':
				mainLabel = langVar('lang_stats.progress.preparing_build_database');
				stepLabel = langVar('lang_stats.progress.erasing_existing_database_data');
				break;
				
			case 'task_canceled':
				// We keep the current mainLabel in prgress display!
				stepLabel = langVar('lang_stats.progress.task_canceled_info');
				break;
		}
		
		if (mainLabel != '') {
			util.updateT('progress:main_label', mainLabel);
		}
		
		if (stepLabel != '') {
			util.updateT('progress:step_label', stepLabel);
		}
	},
	
	initReportProgress: function(taskId, reportJobId) {
		
		// invoked from standby.js only
		progress.isReportProgress = true;
		progress.taskId = taskId;
		progress.reportJobId = reportJobId;
		progress.init();
	},
	
	initDatabaseProgress: function(writingTaskId) {
		
		// invoked from standby.js or from update/rebuild database page in Reports or Config!
		// Note, the writingTaskId argument is optional, it is only available if we manually
		// and successfully invoke update or rebuild database in Reports or Config
		progress.isReportProgress = false;
		progress.taskId = (arguments.length == 1) ? writingTaskId : '';
		
		// Reset progress.cancelTaskIsActive. This is particular required
		// in Config/Database Info, i.e. Cancel a rebuild and then restart a rebuild without page reload,
		// this requires to reset cancelTaskIsActive
		progress.cancelTaskIsActive = false;
		progress.init();
	},
	
	init: function() {
		
		// util.showObject({cancelTaskIsActive:progress.cancelTaskIsActive}, 'progress.init()');
		// Note, init may be called muliple times, i.e. when canceling the task
		// or when changing from database progress to report progress
		
		var isReportProgress = progress.isReportProgress;
		
		progress.reset();
		
		// Temp, hide total_time_elapsed until we have accurate progress for reports
		util.showE('progress:total_time_elapsed:section', !isReportProgress);
		
		if (isReportProgress) {
			progress.getProgressState();
		}
		else {
			
			// Database progress
			
			if (progress.taskId != '') {
				progress.getProgressState();
			}
			else {
				// We need to get database state and a task_id first
				progress.getDbTaskState();
			}
		}
		
		if (!progress.baseInitDone) {
			
			if (progress.isCancelTaskPermission) {
				yEvent.addListener('progress:cancel_taks_btn', 'click', progress.cancelTask);
			}
			else {
				util.hideE('progress:cancel_taks_btn');
			}
			
			yEvent.addListener('progress:refresh_btn', 'click', progress.refreshPage);
			yEvent.addListener('progress:details:link', 'click', progress.toggleProcessingDetails);
			progress.baseInitDone = true;
		}
		
		util.showE('progress:header_buttons');
		util.updateT('progress:step_label', langVar('lang_stats.progress.receiving_progress_information'));
		
		progress.debug();
	},
	
	getDbTaskState: function() {
		
		// alert('TEMP-getDbTaskState()');
		
		// gets current progress state, such as task Id, respectively
		// if the database is yet building or if there is any error
		
		// alert('progress.getDbTaskState()');
		// util.showObject({}, 'progress.getDbTaskState()');
		
		// util.showObject({cancelTaskIsActive:progress.cancelTaskIsActive}, 'progress.getDbTaskState()');

		if (!progress.cancelTaskIsActive) {
		
			var url = '?dp+progress.get_db_task_state';
			url += '+p+' + progress.profileName;
			var dat = 'v.fp.page_token=' + progress.pageToken;
			util.serverPost(url, dat);
		}
			
		progress.debug();
	},
	
	getDbTaskStateResponse: function(dat) {
		
		// util.showObject(dat, 'progress.getDbTaskStateResponse()');
		// alert('TEMP-progress.getDbTaskStateResponse()');
		
		// setTimeout('progress.getDbTaskState()', 1000);
		
		// util.showObject({cancelTaskIsActive:progress.cancelTaskIsActive}, 'progress.getDbTaskStateResponse()');
		
		if (!progress.cancelTaskIsActive) {
		
			progress.taskId = dat.taskId;
			
			if (dat.isActiveTask) {
				
				// get progress state
				progress.getProgressState();
			}
			else {
				
				// Call callee with 'database_task_not_active'
				progress.calleeListener('database_task_not_active');
			}
		}
		
		progress.debug();
	},
	
	getProgressState: function() {
		
		// alert('getProgressState()');
		// util.showObject({}, 'progress.getProgressState()');
		
		// Issue a new channelId
		
		// util.showObject({cancelTaskIsActive:progress.cancelTaskIsActive}, 'progress.getProgressState()');
		
		if (!progress.cancelTaskIsActive) {
		
			var d = new Date();
			var channelId = 'c_' + d.getTime();
			progress.channelId = channelId;
			
			var url = '?dp+progress.get_progress_state';
			url += '+p+' + progress.profileName;
			
			var dat = 'v.sys.page_token=' + progress.pageToken;
			dat += '&v.sys.is_report_progress=' + progress.isReportProgress;
			dat += '&v.sys.task_id=' + progress.taskId;
			dat += '&v.sys.report_job_id=' + progress.reportJobId;
			dat += '&v.sys.channel_id=' + channelId;
			
			util.serverPost(url, dat);
		}
		
		progress.debug();
	},
	
	getProgressStateResponse: function(dat) {
		
		// Ignore response in case that the task has been canceled
		// util.showObject(dat, 'progress.getProgressStateResponse()');
		// alert('progress.getProgressStateResponse()');
		
		// util.showObject({cancelTaskIsActive:progress.cancelTaskIsActive}, 'progress.getProgressStateResponse()');

		if (!progress.cancelTaskIsActive) {
		
			var channelId = dat.channelId;
			
			if (channelId == progress.channelId) {
				
				var progressState = dat.progressState;
			
				switch (progressState) {
					
					case 'progress':
					
						// util.showE(['progress:step_label', 'progress:step_description']);
					
						var progressInfo = dat.progress;
						
						progress.updateDisplay(progressInfo);
						
						// get progress state data
						setTimeout('progress.getProgressState()', 1200);
					
						break;
						
					case 'progress_prediction':
					
						// alert('progress prediction');
						// TEMP
						// util.showE('progress:step_description');
						util.updateT('progress:step_description', langVar('lang_stats.progress.progress_prediction_label'));
						
						// get progress state data
						setTimeout('progress.getProgressState()', 1200);
					
						break;
					
					case 'complete':
					
						progress.updateProgressMeter(100);
						
						// Call callee with calleeInfo 'report_progress_complete' or 'database_progress_complete'
						
						var calleeInfo = progress.isReportProgress ? 'report_progress_complete' : 'database_progress_complete';
						
						progress.calleeListener(calleeInfo);
					
						break;
						
					case 'error':
						// Currently not active, the error is fired
						// by a Salang error() message right away in get_progress_state.cfv!
						// alert('ERROR: Database build is not ready and task is not anymore active!');
						alert('getProgressStateResponse() error');
						break;
				}
			}
			else {
				
				// We got the progress state from a different progress request.
				// Ignore it for now but show an alert!
				alert('The response channelId does not match the channelId issued at request time.\n Response is ignored.');
			}
		}
		
		progress.debug();
	},
	
	cancelTask: function() {
		
		// alert('cancelTask()');
		
		// We set cancelTaskIsActive to true so that there is no
		// progress action while we confirm cancelTask
		
		// Note that the task may complete while the user cancles the task,
		// though this will be indicated anyway by the cancelTaskResponse.
		
		// We freeze progress by setting cancelTaskIsActive
		progress.cancelTaskIsActive = true;
		
		if (confirm(langVar('lang_stats.progress.confirm_cancel_task_message'))) {
		
			util.hideE('progress:header_buttons');
			progress.reset();
			
			util.updateT('progress:step_label', langVar('lang_stats.progress.canceling_task_info'));
			
			var url = '?dp+progress.cancel_task';
			url += '+p+' + progress.profileName;
			
			var dat = 'v.sys.page_token=' + progress.pageToken;
			dat += '&v.sys.is_report_progress=' + progress.isReportProgress;
			dat += '&v.sys.task_id=' + progress.taskId;
			
			util.serverPost(url, dat);			
		}
		else {
			
			// The task is actually not canceled, so we need to resolve progress.
			
			// We simply try a second init()!
			progress.cancelTaskIsActive = false;
			progress.init();
			util.showE('progress:header_buttons');
		}
	},
	
	cancelTaskResponse: function(dat) {
		
		// Update - we ignore the cancelTaskResponse message
		// util.updateT('progress:step_label', dat.msg);
		
		var calleeInfo = '';
		
		if (!dat.isError) {
			// Inform the progress listener that the task has been successfully canceled
			calleeInfo = 'task_canceled';
		}
		else {
			
			// An error occured, respectively the task already completed.
			// We show an error message but then proceed as progress would
			// have been completed.
			
			alert(dat.errorMessage);
			
			if (progress.isReportProgress) {
				calleeInfo = 'report_progress_complete';
			}
			else {
				calleeInfo = 'database_progress_complete';
			}
		}
								
		progress.calleeListener(calleeInfo);
	},
	
	refreshPage: function() {
		
		util.hideE('progress:header_buttons');
		progress.init();
	},
	
	toggleProcessingDetails: function() {
		
		var tbody = util.getE('progress:details:container');
		var showDetails = (tbody.style.display == 'none');		
		util.showE('progress:details:show_btn', !showDetails);
		util.showE('progress:details:hide_btn', showDetails);
		util.showE('progress:details:container', showDetails);
	},
	
	createProcessingDetailRow: function(rowIndex) {
		
		var tr = document.createElement('tr');
		var th = document.createElement('th');
		th.id = 'progress:processing_detail:th:' + rowIndex;
		var td = document.createElement('td');
		td.id = 'progress:processing_detail:td:' + rowIndex;
		
		tr.appendChild(th);
		tr.appendChild(td);
		
		return tr;
	},
	
	updateProcessingDetails: function(processingDetails) {
		
		var tbody = util.getE('progress:details:container');
		var allRows = tbody.getElementsByTagName('tr');
		var numberOfExistingRows = allRows.length;
		var numberOfRequiredRows = processingDetails.length;
		var i;
		var tr;
		
		// Check number of rows
		if (numberOfExistingRows != numberOfRequiredRows) {
			
			if (numberOfRequiredRows > numberOfExistingRows) {
				// add rows
				for (i = numberOfExistingRows; i < numberOfRequiredRows; i++) {
					tr = progress.createProcessingDetailRow(i);
					tbody.appendChild(tr);
				}
			}
			else {
				// KHP-RC, delete rows
				
			}
		}
		
		// Update row values
		for (i = 0; i < processingDetails.length; i++) {
			var item = processingDetails[i];
			util.updateT('progress:processing_detail:th:' + i, item.label + ':');
			util.updateT('progress:processing_detail:td:' + i, item.value);
		}
	},
	
	updateProgressMeter: function(meterPosition) {
		
		var meterPositionDisplay = meterPosition.toFixed(2);
		
		var meterBar = util.getE('progress:meter_bar');
		var meterBarLength = meterPosition * 3 // progress meterPosition is 1-100, bar length = 300px	
		meterBar.style.width = meterBarLength + 'px';
		
		util.updateT('progress:meter_value', meterPositionDisplay + ' %');
	},
	
	updateDisplay: function(dat) {
		
		var stepNumberLabel = langVar('lang_stats.progress.step_number_info');
		stepNumberLabel = stepNumberLabel.replace(/__PARAM__1/, dat.currentStep);
		stepNumberLabel = stepNumberLabel.replace(/__PARAM__2/, dat.numberOfSteps);
		
		var stepLabel = (dat.stepLabel == '-' || dat.stepLabel == '') ? '&nbsp;' : dat.stepLabel;
		var stepDescription = (dat.stepDescription == '-' || dat.stepDescription == '') ? '&nbsp;' : dat.stepDescription;
		
		util.updateT('progress:main_label', dat.mainLabel);
		util.updateT('progress:step_number', stepNumberLabel);
		util.updateT('progress:step_label', stepLabel);
		util.updateT('progress:step_description', stepDescription);
		util.updateT('progress:total_time_elapsed', dat.totalTimeElapsed);
		
		progress.updateProgressMeter(dat.meterPosition);
		
		if (!progress.isReportProgress) {progress.updateProcessingDetails(dat.processingDetails);}
		
		if (!progress.progressIsVisible) {
			
			// Show progress meter, etc.
			
			if (!progress.isReportProgress) {
				util.showE(['progress:details:body', 'progress:details:show_btn']);
			}
			
			util.showE('progress:body');
			progress.progressIsVisible = true;
		}
	},
	
	addDebug: function(div, label) {
		
		var p = document.createElement('p');
		var text = document.createTextNode(label);
		p.appendChild(text);
		div.appendChild(p);
	},
	
	debug: function() {
		
		if (progress.showDebugOutput) {
			
			var div = util.getE('progress:debug');
			
			while (div.lastChild != null) {
				var n = div.lastChild;
				div.removeChild(n);
			}
			
			progress.addDebug(div, 'isReportProgress: ' + progress.isReportProgress);
			progress.addDebug(div, 'taskId: ' + progress.taskId);
			progress.addDebug(div, 'cancelTaskIsActive: ' + progress.cancelTaskIsActive);
		}
	}
};
