/*
	listcontroller
		

	LIST PROPERTIES provided at object creation
	
	containerElementId		Element Id of div for ul element
	itemEvent				The event which should fire upon item click
	items					The array with item objects
	
	
	LIST PROPERTIES set in class
	
	selectedItemId
	maxItemIdNumber
	isModified
	
	
	ITEM PROPERTIES which must exist in the provided items array objects
	
	label
	id						i0, i1, i2, ... (Note, the itemId must start with a character followed by a number, so that it is of type string!)
	type					schedule | schedule_action | report | report_element | report_group
	switch1 (optional)		First checkbox to enable/disable an item
	switch2 (optional)		Second checkbox to enable/disable an item
	
	subitems (optional)		Any subitems, i.e. Scheduler actions, report elements, reports of a report group
	

	ITEM PROPERTIES which are added in List class
	
	isNew
	isModified
	listElementId
	anchorElementId (optional)
	itemSwitchElementId (optional)
	itemSwitch2ElementId (optional)
	parentId
		
	LIST METHODS
	
		init
		initItems
		getNewItemId
		prepareList
		addListItem
		createSubItemsUlElement
		updateItemsLookup
		registerItemInLookup
		selectItem
		searchItemId
		getItem
		getItemDatValue
		setItemDatValue
		isItems
		getSelectedItem
		getSelectedItemId
		getSelectedItemType
		getSelectedItemIsNew
		getSelectedItemIsSwitch1
		getFirstItemId
		getNumberOfItems
		updateListLabel
		newItem
		updateNewItem
		deleteItem
		cloneItem
		updateClonedItem
		getIsModified
		resetIsModified
		
		moveItem
		moveItemInStandardMode
		moveItemFromSubitemsToRoot
		moveItemFromRootToSubitems
		
		saveItem
		checkItemIsModified
		itemAisDifferentFromItemB
		getLookupItems
		getLookupItemsByConstraint
		getActiveSwitchKey
		switchActivated
		toggleSwitch
		updateSwitchInItems
		
		setMoveControlState
		moveItemActor
	
	
*/

listcontroller = {};

listcontroller.List = function(obj) {
	
	// Creates the list object
	
	var YE = YAHOO.util.Event;
	
	this.isIE = util.userAgent.isIE;
	this.containerElementId = obj.containerElementId;
	this.itemEvent = obj.itemEvent;
	
	// switchEvent is used in reports editor where a checkbox click requires to call a function in the callee.
	this.switchEvent = obj.switchEvent ? obj.switchEvent : null;
	
	// Optional switch arguments (Handles the checkbox in item list)
	// If the list only contains a switch1 property then we only require
	// to set "isSwitch1". If the list contains a switch1 and switch2
	// property then we require all switch properties (isSwitch2, switchLabelContainerId, etc.)
	// so that we have the ability to toggle the checkbox state between switch1 and switch2.
	this.isSwitch1 = obj.isSwitch1 ? obj.isSwitch1 : false;
	this.isSwitch2 = obj.isSwitch2 ? obj.isSwitch2 : false;
	var isDualSwitch = (this.isSwitch1 && this.isSwitch2);
	this.isDualSwitch = isDualSwitch;
	this.switchButtonElementId = isDualSwitch ? obj.switchButtonElementId : '';
	this.switchLabelElementId = isDualSwitch ? obj.switchLabelElementId : '';
	this.switch1Label = isDualSwitch ? obj.switch1Label : '';
	this.switch2Label = isDualSwitch ? obj.switch2Label : '';
	
	// activeSwitchKey ==> default must be 'switch1'!
	// activeSwitchKey defines which switch the check box presents in case of 
	// dual switches (if switch1 and switch2 is used, such as in reports editor)
	this.activeSwitchKey = 'switch1'; 	// switch1 | switch2 ==> this is the property name as defined in items object!
	
	this.isMoveControl = obj.isMoveControl ? obj.isMoveControl : false;
	
	/* numberOfFixedItems is the number of list items which cannot be moved.
		This is used i.e. when the first list item is fixed because
		it contains a form which apply to all remaining items in the list. */
	this.numberOfFixedItems = obj.numberOfFixedItems > 0 ? obj.numberOfFixedItems : -1;
	
	
	this.ulContainerId = this.containerElementId + ':ul_listcontrol';
	this.ulContainer = ''; // the ul object
	
	this.items = null; // items are set upon init
	this.selectedItemId = null;
	this.maxItemIdNumber = -1;
	this.itemsLookup = [];
	this.isModified = false;
	
	
	// Handle dual switch
	if (isDualSwitch) {
		
		YE.addListener(this.switchButtonElementId, 'click', this.toggleSwitch, this);
		// Set default label
		util.updateT(this.switchLabelElementId, this.switch1Label);
	}
	
	// Add generic list event
	YE.addListener(this.containerElementId, 'click', this.selectItemActor, this);
	
	//
	// Handle move control
	//
	
	// Move control baseId is the element ID of the move control container div
	// Each image element ID in this container is a combination
	// of the baseID and move direction as follwos:
	// id="baseId:top"
	// id="baseId:up"
	// id="baseId:down"
	// id="baseId:bottom"
	
	var moveControlBaseId = 'item_list_move_control';
	
	this.moveControl = {};
	this.moveControl.baseId = moveControlBaseId;
	this.moveControl.buttonsState = {
		top: {id:moveControlBaseId + ':top', isDisabled:true, src:imgDb.moveTop.src, srcDis:imgDb.moveTopDis.src},
		up: {id:moveControlBaseId + ':up', isDisabled:true, src:imgDb.moveUp.src, srcDis:imgDb.moveUpDis.src},
		down: {id:moveControlBaseId + ':down', isDisabled:true, src:imgDb.moveDown.src, srcDis:imgDb.moveDownDis.src},
		bottom: {id:moveControlBaseId + ':bottom', isDisabled:true, src:imgDb.moveBottom.src, srcDis:imgDb.moveBottomDis.src}
	}
}

listcontroller.List.prototype.init = function(items) { // NEW
	
	// This function creates or re-creates the list items for the given items
	
	// alert('listcontroller items.length: ' + items.length);

	//
	// Set/reset object properties
	//
	
	this.items = items;
	this.selectedItemId = null;
	this.maxItemIdNumber = -1;
	this.itemsLookup = [];
	this.isModified = false;
	
	// Note, we don't reset this.activeSwitchKey!
	// The active switch, if any, remains active!
	
	
	//
	// prepare the list
	//
	
	this.prepareList();
	
	// add class specific item properties to each item and set the max itemId
	
	this.initItems(items);
	
	this.updateItemsLookup();
	
	// create the initial list
	var ul = this.ulContainer;
	
	var isSubItem = false;
	
	for (var i = 0; i < items.length; i++) {
		
		var li = document.createElement('li');
		ul.appendChild(li);
		
		this.addListItem(li, items[i].id, isSubItem);
	}

	// alert('this.maxItemIdNumber: ' + this.maxItemIdNumber);
}


listcontroller.List.prototype.initItems = function(items, parentId) {

	for (var i = 0; i < items.length; i++) {
	
		var item = items[i];
		
		var itemId = item.id;
		var itemIdNumber = parseInt(itemId.substring(1), 10);
		
		// alert('itemIdNumber: ' + itemIdNumber);
		
		if (itemIdNumber > this.maxItemIdNumber) {
			this.maxItemIdNumber = itemIdNumber;
		}
		
		item.listElementId = util.getUniqueElementId();
		item.anchorElementId = util.getUniqueElementId();
		item.isModified = false;
		item.isNew = false;
		
		if (this.isSwitch1) {
			item.switchElementId = util.getUniqueElementId();
		}
		
		if (parentId) {
			// add parentId to each subitem
			item.parentId = parentId;
		}

		if (item.subitems != null && item.subitems.length > 0) {
			// init the subitems
			this.initItems(item.subitems, itemId);
		}
	}
}


listcontroller.List.prototype.getNewItemId = function() {
	
	var i = this.maxItemIdNumber + 1;
	var newItemId = 'i' + i;
	this.maxItemIdNumber = i;
	// alert('newItemId: ' + newItemId);
	return newItemId;
}


listcontroller.List.prototype.prepareList = function() {
	
	// creates the ul or deletes any li item if ul already exists
	
	var div = document.getElementById(this.containerElementId);
	var ul = document.getElementById(this.ulContainerId);
	
	if (ul != null) {
		// delete ul
		div.removeChild(ul);
	}
	
	ul = document.createElement('ul');
	ul.id = this.ulContainerId;
	div.appendChild(ul);
	this.ulContainer = ul;
}


listcontroller.List.prototype.addListItem = function(itemLi, itemId, isSubItem) {
	
	// Note, itemLi already is an inserted DOM li element, so we only need
	// to add the properties such as id, add checkboxes and the anchor.
	
	// If the item contains subitems then we recursivly call this function
	// to create each sub li element.
	
	var item = this.itemsLookup[itemId];
	
	var isCheckbox = this.isSwitch1;
	
	// util.showObject(item, "listcontroller addListItem()");
	
	var li = itemLi;
	li.id = item.listElementId;
	
	if (isSubItem) {
		li.className = 'subitem';
	}
	
	// if (item.switchElementId) {
	if (isCheckbox) {
		
		// create the checkbox
		
		var div = document.createElement('div');
		div.className = 'checkbox';
		var input = document.createElement('input');
		input.type = 'checkbox';
		// alert(item.switchElementId);
		input.id = item.switchElementId;
		input.checked = true; //item.switch1;
		
		// Set the input margins
		if (!this.isIE) {
			input.style.margin = '6px';
			input.style.marginRight = '4px';
		}
		else {
			// IE has a default margin, so we only fix the top margin
			// by giving some extra pixel
			input.style.marginTop = '3px';
		}
		
		// alert('item.switch1: ' + item.switch1);
		
		li.appendChild(div);
		div.appendChild(input);
		
		// add the event
		// YAHOO.util.Event.addListener(input, 'click', this.switchActor, this); DISABLED nov-2008
	}
	
	var a = document.createElement('a');
	a.id = item.anchorElementId;
	a.href = 'javascript:;';
	
	// a.onclick = this.itemEvent;
	
	// If the list does not contain a checkbox
	// then give some less top and bottom padding
	
	if (!isCheckbox) {
		a.style.paddingTop = (!this.isIE) ? '4px' : '3px';
		a.style.paddingBottom = '4px';
	}
	
	var aText = document.createTextNode(item.label);
	a.appendChild(aText);
	li.appendChild(a);
	
	// make the report group list item bold
	var liFontWeight = item.type != 'report_group' ? 'normal' : 'bold';
	li.style.fontWeight = liFontWeight;
	
	if (item.subitems && item.subitems.length > 0) {
		
		// create subitems
		var subitems = item.subitems;
		
		var ul = this.createSubItemsUlElement();		
		li.appendChild(ul);
		
		for (var i = 0; i < subitems.length; i++) {
			
			var nextItemLi = document.createElement('li');
			ul.appendChild(nextItemLi);
			
			this.addListItem(nextItemLi, subitems[i].id, true);
		}
	}
	
	// Set the item switch checkboxes.
	// We do that now due a problem in IE where checkboxes are not set above.
		
	if (this.isSwitch1) {
		var switchKey = this.activeSwitchKey;
		util.setF(item.switchElementId, item[switchKey]);
		// var switchElement = util.getE()
		// var switch1 = document.getElementById(item.switchElementId);
		// switch1.checked = item.switch1;
	}
}

listcontroller.List.prototype.createSubItemsUlElement = function() {
	
	// Returns an ul element with appropriate style to display subitems
	var ul = util.createE('ul', {marginLeft:'21px'});	
	return ul;
}

listcontroller.List.prototype.updateItemsLookup = function() {
	
	// this creates a new itemsLookup hash
	
	this.itemsLookup = [];
	
	var items = this.items;
	
	for (var i = 0; i < items.length; i++) {
	
		this.registerItemInLookup(items[i]);
	}
}

listcontroller.List.prototype.registerItemInLookup = function(item) {
	
	var itemsLookup = this.itemsLookup;
	
	itemsLookup[item.id] = item;
	itemsLookup[item.anchorElementId] = item;
	
	if (this.isSwitch1) {
		itemsLookup[item.switchElementId] = item;
	}
	
	// if subitems iterate through the subitems
	
	if (item.subitems != null) {
		
		var subitems = item.subitems;
		
		for (var i = 0; i < subitems.length; i++) {
			
			this.registerItemInLookup(subitems[i]);
		}
	}
}

// nov-2008
listcontroller.List.prototype.selectItemActor = function(evt, self) {
	
	// Activated upon list click
	
	var element = evt.target || evt.srcElement;
	var elementId = element.id;
	var tagName = element.nodeName;
	
	// alert('selectItemActor() \ntagName: ' + tagName + '\nelementId: ' + elementId);
	
	if ((tagName == 'A' || tagName == 'INPUT') && elementId != '') {
		
		var itemId = self.searchItemId(elementId);
		
		if (tagName == 'A') {
			
			// If the item is not yet selected
			
			if (itemId != self.selectedItemId) {
				
				// Invoke the itemEvent
				self.itemEvent(itemId);
			}
		}
		else {
			
			// toggle checkbox
			self.switchActivated(itemId, element.checked);
		}
	}
}

listcontroller.List.prototype.selectItem = function(itemId) {
	
	// Select the given (itemId) list item
	// alert('this.selectedItemId: ' + this.selectedItemId);
	
	if (this.selectedItemId != null) {
		
		// alert('deselct itemId: ' + this.selectedItemId);
		
		var itemObj = this.itemsLookup[this.selectedItemId];
		// util.showObject(itemObj);
		var a = document.getElementById(itemObj.anchorElementId);
		// var li = document.getElementById(itemObj.listElementId);
		// a.onclick = this.itemEvent;
		a.className = '';
		// li.className = '';
	}
	
	// select the item
	
	var itemObj =  this.itemsLookup[itemId];
	
	var a = document.getElementById(itemObj.anchorElementId);
	// var li = document.getElementById(itemObj.listElementId);
	
	// a.onclick = null;
	a.className = 'selected';
	// li.className = 'selected';
	
	// util.showObject(itemObj);
	
	this.selectedItemId = itemObj.id;
	
	// alert('this.selectedItemId: ' + this.selectedItemId);
	// alert('select itemId: ' + itemObj.itemId);
	
	// Update MoveControlState
	if (this.isMoveControl) {
		this.setMoveControlState();
	}
}

listcontroller.List.prototype.searchItemId = function(itemLookupValue) {
	
	// itemLookupValue can be itemId or anchorElementId
	
	var items = this.items;
	
	var item = this.itemsLookup[itemLookupValue];
	return item.id;
}

listcontroller.List.prototype.getItem = function(itemId) {
	
	return this.itemsLookup[itemId];
}

listcontroller.List.prototype.getItemDatValue = function(itemId, key) {
	// Returns the value of item.dat[key]
	var item = this.itemsLookup[itemId];
	return item.dat[key];
}

listcontroller.List.prototype.setItemDatValue = function(itemId, key, value) {
	// Sets the value of of item.dat[key]
	var item = this.itemsLookup[itemId];
	item.dat[key] = value;
}

listcontroller.List.prototype.isItems = function() {
	// Returns true if the list contains at least one item
	var isItems = (this.items.length > 0) ? true : false;
	return isItems;
}

listcontroller.List.prototype.getSelectedItem = function() {
	
	return this.itemsLookup[this.selectedItemId];
}

listcontroller.List.prototype.getSelectedItemId = function() {
	
	return this.selectedItemId;
}

listcontroller.List.prototype.getSelectedItemType = function() {
	
	if (this.selectedItemId != null) {
	
		var item = this.itemsLookup[this.selectedItemId];
		return item.type;
	}
	else {
		return '';
	}
}

listcontroller.List.prototype.getSelectedItemIsNew = function() {
	var item = this.getSelectedItem();
	return item.isNew;
}

listcontroller.List.prototype.getSelectedItemIsSwitch1 = function() {
	// Returns true if switch1 is true. If this is a subitem then the parent switch1 must be true as well,
	// else we return false.
	var item = this.getSelectedItem();
	var isSwitch1 = item.switch1;
	
	// Check if this item has a parent item, we only need to check the parent if isSwitch1 is true
	if (isSwitch1 && item.parentId) {
		var parentItem = this.itemsLookup[item.parentId];
		isSwitch1 = parentItem.switch1;
	}
	
	// alert('getSelectedItemIsSwitch1: ' + isSwitch1);
	
	return isSwitch1;
}

listcontroller.List.prototype.getFirstItemId = function() {
	
	// returns the itemId of the first item or null if no item exists.
	
	var items = this.items;
	
	if (items.length > 0) {
		return items[0].id;
	}
	
	return null;
}

listcontroller.List.prototype.getNumberOfItems = function() {
	// Returns the numberOfItems inside the active hierarchy
	var item = this.itemsLookup[this.selectedItemId];
	
	if (item.parentId) {
		var parentItem = this.itemsLookup[item.parentId];
		return parentItem.subitems.length;
	}
	else {
		return this.items.length;
	}
}

listcontroller.List.prototype.updateListLabel = function(label) {
	// Updates the list label of selected item and saves the label to the item.
	var item = this.itemsLookup[this.selectedItemId];
	item.label = label; // Note, this label is only relevant to the list, it is not relevant to any form data saved in the profile.
	util.updateT(item.anchorElementId, label);
}

listcontroller.List.prototype.newItem = function(obj, insertDirective) {
	
	// insertDirective is an optional object which is used for hierarchical lists
	// such as the reports editor. "insertDirective" contains only one property
	// which is "insertAtRootLevel", "insertAsItemOrSubItem", "insertAsFirstSubItem"
	
	var newItem = util.cloneObject(obj);

	var insertAtItemId;
	var itemsScope;
	var ulContainer;
	var li;
	
	// specify where the new newItem has to be inserted
	
	// var newItemIndex;
	var insertAtItemIndex;
	
	var isSubItem = false;
	
	this.updateNewItem(newItem);
	
	if (this.selectedItemId != null) {
		
		var selectedItem = this.itemsLookup[this.selectedItemId];
		var selectedItemType = selectedItem.type;
		var newItemType = newItem.type;
		
		// alert('selectedItemType: ' + selectedItemType + '\nnewItemType: ' + newItemType);
		
		//
		// Check insertMode
		//
		
		var insertMode = 'standard';
		
		if (insertDirective) {
			
			if (insertDirective.insertAtRootLevel) {
				
				// The new item must be inserted at the root level. If we are
				// already at the root level we can use standard, else we set 
				// insertMode to inser_at_root_level
				
				// alert('selectedItem.parentId: ' + selectedItem.parentId);
				insertMode = selectedItem.parentId ? 'inser_at_root_level' : 'standard';
				
			}
			else if (insertDirective.insertAsItemOrSubItem) {
				
				// The new item is inserted as next item at root items
				// level if the selected item is at root items level.
				// If the selected item is at subitems level then we insert
				// the item next to the subitem.
				
				// alert('selectedItem.parentId: ' + selectedItem.parentId);
				insertMode = selectedItem.parentId ? 'insert_subitem' : 'standard';
				
			}
			else if (insertDirective.insertAsFirstSubItem) {
				
				insertMode = 'insert_as_first_subitem';
			}
		}
		
		
		//
		// Insert new item according insertMode
		//
		
		// alert('insertMode: ' + insertMode);
		
		switch (insertMode) {
			
			case 'standard': 
			
				//
				// Standard inserts the new item next to the selected item
				//
				insertAtItemId = selectedItem.id;
				itemsScope = this.items;
				insertAtItemIndex = util.getArrayObjectIndex(itemsScope, 'id', insertAtItemId);
				break;
			
			case 'inser_at_root_level':
				
				//
				// The new item must be inserted at root level but the selected item
				// is in subitems
				//
				
				// alert('case: inser_at_root_level');
				
				insertAtItemId = selectedItem.parentId;
				itemsScope = this.items;
				insertAtItemIndex = util.getArrayObjectIndex(itemsScope, 'id', insertAtItemId);
			
				break;
			
			case 'insert_subitem':
				
				//
				// The selected item is a subitem, we insert the new item next to the selected subitem
				//
				
				isSubItem = true;
				
				insertAtItemId = selectedItem.id;
				
				var parentId = selectedItem.parentId;
				var parentItem = this.itemsLookup[parentId];
				itemsScope = parentItem.subitems;
				insertAtItemIndex = util.getArrayObjectIndex(itemsScope, 'id', insertAtItemId);
				
				// Set parentId in newItem!
				newItem.parentId = parentId;
			
				break;
				
			case 'insert_as_first_subitem': // report
				
				//
				// The selected item is a parentItem, we insert the new item as first subitem of the parentItem
				//
				
				isSubItem = true;
				
				var itemsScope = selectedItem.subitems;
				insertAtItemIndex = -1; // becomes 0 later, so it will be the first item
				
				// Set parentId in newItem!
				newItem.parentId = selectedItem.id;
				
				// Note, it is possible that no subitem exists, in this case we need
				// to create the UL container at the selected item
				if (itemsScope.length > 0) {
					// in this case the insertAtItemId is not the actual insertion point,
					// however, we use insertAtItemId only to get the parent ul element
					insertAtItemId = itemsScope[0].id;
				}
				else {
					// We need to create the ul element at the selected li item
					insertAtItemId = null;
				}
			
				break;
		}
		
		
		var newItemIndex = insertAtItemIndex + 1;
		
		// util.showObject(itemsScope);
		
		// Insert the new item into itemsScope
		itemsScope.splice(newItemIndex, 0, newItem);
		
		// util.showObject(itemsScope);
		// util.showObject(itemsScope[newItemIndex]);
		
		// get the ul container
		if (insertAtItemId != null) {
			var insertAtItemListElementId = this.itemsLookup[insertAtItemId].listElementId;
			var insertAtItemListElement = document.getElementById(insertAtItemListElementId);
			ulContainer = insertAtItemListElement.parentNode;
		}
		else {
			// We need to create the ul element at the selected li item
			var selectedLiElement = util.getE(selectedItem.listElementId);
			var ulContainer = this.createSubItemsUlElement();
			selectedLiElement.appendChild(ulContainer);
		}
		
		li = document.createElement('li');
		
		if (newItemIndex == itemsScope.length - 1) {
			
			// alert('append item as last item');
			
			// item is last item in ul scope, so simply append it
			ulContainer.appendChild(li);
		}
		else {
			
			// insert item before next item
			
			// alert('insert item before');
			
			var nextItemListElementId = itemsScope[newItemIndex + 1].listElementId;
			
			// alert('nextItemListElementId: ' + nextItemListElementId);
			
			var nextItemListElement =  document.getElementById(nextItemListElementId);
			
			ulContainer.insertBefore(li, nextItemListElement);
		}
	}
	else {
		
		// no item exists, make newItem the first list item
		
		newItemIndex = 0;
		itemsScope = this.items;
		ulContainer = this.ulContainer;
		
		li = document.createElement('li');
		ulContainer.appendChild(li);
		
		// Insert the new item into itemsScope
		itemsScope.splice(newItemIndex, 0, newItem);
	}
	
	
	// Update itemsLookup
	this.updateItemsLookup();

	// Add the new item to the list
	this.addListItem(li, newItem.id, isSubItem);
	
	// util.showObject(newItem);
	
	this.isModified = true;
	
	return newItem.id;
}

listcontroller.List.prototype.updateNewItem = function(item) {
	
	// Note, itemId's are already given in the new object!
		
	item.listElementId = util.getUniqueElementId();
	item.anchorElementId = util.getUniqueElementId();
	item.isModified = true;
	item.isNew = true;
	
	if (this.isSwitch1) {
		item.switchElementId = util.getUniqueElementId();
	}
}

listcontroller.List.prototype.deleteItem = function() {
	
	var item = this.itemsLookup[this.selectedItemId];
	var nextItemId = null;
	// alert('Delete itemObj type: ' + itemObj.type);
	
	// util.showObject(item);
	
	var itemsScope; // the items level/hierarchy
	
	var nextSelectedIsParentItem = false;
	
	if (item.parentId == null) {
		
		// We are in the first level of the list and can savely delete
		// the item. Subitems are part of the items object and a nested
		// ul of the li item, so they become automatically deleted.
		itemsScope = this.items;
	}
	else {
		
		// We are in a subitems list
		var parentItem = this.itemsLookup[item.parentId];
		var itemsScope = parentItem.subitems;
		
		// if there is only one subitem then the next selected item is the parent item
		if (itemsScope.length == 1) {
			nextSelectedIsParentItem = true;
		}		
	}	
	
	// get next item to be selected
	if (!nextSelectedIsParentItem) {
		
		var nextItem = util.getNextArrayObject(itemsScope, 'id', this.selectedItemId);
		if (nextItem != null) {
			nextItemId = nextItem.id;
		}
	}
	else {
		// we select the parent
		nextItemId = parentItem.id;
	}
	
	// delete the item in items
	util.deleteArrayObject(itemsScope,'id', this.selectedItemId);
	
	// delete the li element
	var li = document.getElementById(item.listElementId);
	var ul = li.parentNode;
	ul.removeChild(li);
	
	
	this.selectedItemId = null;
	
	this.isModified = true;
	
	return nextItemId;
}


listcontroller.List.prototype.cloneItem = function() {
	
	var item = this.itemsLookup[this.selectedItemId];

	var itemsScope; // the items level/hierarchy
	
	var isSubItem = false;
	
	if (item.parentId == null) {
		// We are in the first level of the list.
		itemsScope = this.items;
	}
	else {
		// We are in a subitems list
		var parentItem = this.itemsLookup[item.parentId];
		var itemsScope = parentItem.subitems;
		
		isSubItem = true;
	}
	
	var itemIndex = util.getArrayObjectIndex(itemsScope, 'id', item.id);
	var clonedItemIndex = itemIndex + 1;

	// Clone the item and modify all cloned class specific properties
	var clonedItem = util.cloneObject(item);
	
	if (clonedItem.type != 'schedule_action') {
		// var newLabel = clonedItem.label + ' copy';
		var newLabel = langVar('lang_stats.general.item_copy');
		newLabel = newLabel.replace(/__PARAM__1/, clonedItem.label);
		clonedItem.label = newLabel;
		
		// we also need to upate the item dat, label or whatever property is used as label (i.e. username for users)
		switch (clonedItem.type) {
			
			case 'users':
				clonedItem.dat.username = newLabel;
				break;
			
			default:
				clonedItem.dat.label = newLabel;
				break;
		}
	}
		
	this.updateClonedItem(clonedItem);

	// Insert the cloned item into itemsScope
	itemsScope.splice(clonedItemIndex, 0, clonedItem);
	
	// Update itemsLookup
	this.updateItemsLookup();
	
	// We nee to create and insert a li element
	// from which list generation starts
	var itemLi = document.getElementById(item.listElementId);
	var itemUl = itemLi.parentNode;
	var clonedItemLi = document.createElement('li');
	
	if (clonedItemIndex == itemsScope.length - 1) {
		
		// append the clonedItemLi
		itemUl.appendChild(clonedItemLi);
	}
	else {
		// insert clonedItemLi before next element
		var nextItemListElementId = itemsScope[clonedItemIndex + 1].listElementId;
		var nextItemLi = document.getElementById(nextItemListElementId);
		itemUl.insertBefore(clonedItemLi, nextItemLi);
	}

	// Add the cloned item to the list
	this.addListItem(clonedItemLi, clonedItem.id, isSubItem);
	
	this.isModified = true;
	
	return clonedItem.id;
}

listcontroller.List.prototype.updateClonedItem = function(item, parentId) {
	
	// This updates a cloned item, giving it new itemId's etc.
	
	// The parentId argument is only given when recursively calling
	// this function for any subitem. If the initial item already is a subitem
	// then the parentId doesn't change, it is the same as the source item. 
	
	var itemId = this.getNewItemId();
	
	item.id = itemId;
	
	if (parentId != null) {
		// alert('parentId: ' + parentId);
		item.parentId = parentId;
	}
	
	item.listElementId = util.getUniqueElementId();
	item.anchorElementId = util.getUniqueElementId();
	item.isModified = true;
	item.isNew = true;
	
	if (this.isSwitch1) {
		item.switchElementId = util.getUniqueElementId();
	}
	
	// update any subitems
	if (item.subitems != null) {
		
		var subitems = item.subitems;
		
		for (var i = 0; i < subitems.length; i++) {
			this.updateClonedItem(subitems[i], itemId);
		}
	}
}

listcontroller.List.prototype.getIsModified = function() {
	
	// alert('listcontroller.List.prototype.getIsModified - this.isModified :' + this.isModified);
	return this.isModified;
}

listcontroller.List.prototype.resetIsModified = function() {
	// Reset isModified
	this.isModified = false;
}

listcontroller.List.prototype.moveItem = function(direction) {
	
	// direction is: top | up | down | bottom
	// util.showObject({'moveItem() - direction: ':direction});
	
		function getTargetItemIndex(numberOfItems, selectedItemIndex, direction) {
			
			var targetItemIndex = -1; // return -1 if the item can't be moved
			
			switch (direction) {
			
				case 'down' :
					targetItemIndex = (selectedItemIndex + 1) < numberOfItems ? (selectedItemIndex + 1) : -1;
					break;
				
				case 'up' :
					targetItemIndex = (selectedItemIndex - 1) >= 0 ? (selectedItemIndex - 1) : -1;
					break;
				
				case 'top' :
					targetItemIndex = (selectedItemIndex != 0) ? 0 : -1;
					break;
				
				case 'bottom' :
					targetItemIndex = (numberOfItems - 1) != selectedItemIndex ? numberOfItems - 1 : -1;
					break;
			}
			
			return targetItemIndex;
		}
	
	var item = this.itemsLookup[this.selectedItemId];
	var itemType = item.type;
	var isSubItem = false;
	
	// util.showObject({'moveItem() - itemType: ':itemType});
	
	// get the itemsScope
	
	var itemsScope; // the items level/hierarchy
	
	if (item.parentId == null) {
		// We are in the first level of the list.
		itemsScope = this.items;
	}
	else {
		// We are in a subitems list
		var parentItem = this.itemsLookup[item.parentId];
		var itemsScope = parentItem.subitems;
		isSubItem = true;
	}
	
	var numberOfItemsInItemsScope = itemsScope.length;
	var selectedItemIndex = util.getArrayObjectIndex(itemsScope, 'id', this.selectedItemId);
	var targetItemIndex = getTargetItemIndex(numberOfItemsInItemsScope, selectedItemIndex, direction);
	
	// alert('selectedItemIndex: ' + selectedItemIndex + '\ntargetItemIndex: ' + targetItemIndex);
	
	var isMoveItemInStandardMode = false;
	
	if (itemType != 'report') {
		
		if (targetItemIndex != -1) {
			isMoveItemInStandardMode = true;
		}
	}
	else {
		
		// This is a report item.
		// Check of how we move this item
	
		if (isSubItem) {
			
			if (targetItemIndex != -1) {
				
				// Item is in sub group but it can be moved, so we move
				// it like a standard item.
				isMoveItemInStandardMode = true;
			}
			else {
				
				// Item is in subitems, move item out of subitems to root
				this.moveItemFromSubitemsToRoot(itemsScope, item, selectedItemIndex, direction);
			}
		}
		else if (targetItemIndex != -1) {
			
			if (direction == 'top' || direction == 'bottom') {
				
				// Item is in root, move item to top or bottom like a standard item
				isMoveItemInStandardMode = true;
			}
			else {
				
				// Before we move the item we need to know if the next item is a group or report
				// If the next item is a report then we move the current report like a standard item,
				// else we move the report into the report group
				
				var nextItem = itemsScope[targetItemIndex];
				var nextItemType = nextItem.type;
				
				// util.showObject(nextItem, 'moveItem(), check if next item is a report_group, here is the next item object');
				
				if (nextItemType == 'report') {
					
					// We can move the selected report item as a standard item
					isMoveItemInStandardMode = true;
				}
				else {
					
					// The next item is a report group, so we move the selected report into the report group
					// as first or last item (depending on direction)
					this.moveItemFromRootToSubitems(nextItem, item, selectedItemIndex, direction);
				}
			}
		}
	}
	
	if (isMoveItemInStandardMode) {
		
		this.moveItemInStandardMode(itemsScope, item, selectedItemIndex, targetItemIndex, direction);
	}
}

listcontroller.List.prototype.moveItemInStandardMode = function(itemsScope, item, selectedItemIndex, targetItemIndex, direction) {
	
	// This moves an item inside the current scope, it does not move items to subitems or vice versa.
	
	//
	// Move item in array
	//
	
	// remove object at selectedItemIndex
	var a = itemsScope.splice(selectedItemIndex, 1); // Note, obj is returned in array a!
	// insert object at targetItemIndex
	itemsScope.splice(targetItemIndex, 0, a[0]);
	
	//
	// Update the list
	//
	
	// Move the item in list
	var li = document.getElementById(item.listElementId);
	var parentUl = li.parentNode;	
	var liClone = parentUl.removeChild(li);
	
	// alert('itemsScope.length: ' + itemsScope.length + '\ntargetItemIndex: ' + targetItemIndex);
	
	if (direction == 'bottom' || targetItemIndex == itemsScope.length - 1) {
		
		// append the item as last element
		// alert('append as last element');
		parentUl.appendChild(liClone);
	}
	else {
		
		// insert the li element before the next li element
		
		// alert('insert before');
		var nextLiElementId = itemsScope[targetItemIndex + 1].listElementId;
		var nextLi = util.getE(nextLiElementId);
		
		parentUl.insertBefore(liClone, nextLi);
	}
	
	// update checkbox state which is lost in IE
	if (this.isSwitch1) {
		var switchKey = this.activeSwitchKey;
		util.setF(item.switchElementId, item[switchKey]);
	}
	
	this.isModified = true;
}

listcontroller.List.prototype.moveItemFromSubitemsToRoot = function(subitems, selectedItem, selectedItemIndex, direction) {
	
	// This moves a subitem to the root
	
	// alert('moveItemFromSubitemsToRoot()');
	
	var rootItems = this.items;
	var parentItemId = selectedItem.parentId;
	var rootItem = this.itemsLookup[parentItemId];
	var rootItemIndex = util.getArrayObjectIndex(rootItems, 'id', parentItemId);
	
	var targetItemIndex = (direction == 'down' || direction == 'bottom') ? rootItemIndex + 1 : rootItemIndex;
	
	
	//
	// Move selectedItem in array
	//

	// remove object at selectedItemIndex
	var a = subitems.splice(selectedItemIndex, 1); // Note, obj is returned in array a!
	var itemObj = a[0];
	// remove the parentId in itemObj
	delete itemObj.parentId;
	
	// util.showObject(itemObj);
	
	// insert object at targetItemIndex
	rootItems.splice(targetItemIndex, 0, itemObj);
	
	
	//
	// Move selectedItem in list
	//
	
	// Move the item in list
	var subitemLi = util.getE(selectedItem.listElementId);
	var subitemUl = subitemLi.parentNode;
	var liClone = subitemUl.removeChild(subitemLi);
	
	var rootUl = util.getE(this.ulContainerId);
	
	if (targetItemIndex == (rootItems.length - 1)) {
		
		// Append li as last item
		rootUl.appendChild(liClone);
	}
	else {
		
		// Insert li before the next item
		var nextLiElementId = rootItems[targetItemIndex + 1].listElementId;
		var nextLi = util.getE(nextLiElementId);
		rootUl.insertBefore(liClone, nextLi);
	}
	
	
	// update checkbox state which is lost in IE
	if (this.isSwitch1) {
		var switchKey = this.activeSwitchKey;
		util.setF(selectedItem.switchElementId, selectedItem[switchKey]);
	}
	
	
	this.isModified = true;
}

listcontroller.List.prototype.moveItemFromRootToSubitems = function(targetItem, selectedItem, selectedItemIndex, direction) {
	
	// alert('moveItemFromRootToSubitems()');

	// This moves the selcted item to the subitems in targetItem (targetItem and selectedItem are always root items!).
	var rootItems = this.items;
	
	var subitems = targetItem.subitems;
	var numberOfSubitems = subitems.length;
	
	// Check new position in subitems, it is the opposite of direction!
	var isMoveToBottomOfSubitems = (direction == 'up' || direction == 'top');
	
	var targetIndexInSubitems = isMoveToBottomOfSubitems ? numberOfSubitems : 0;
	
	//
	// Move selectedItem in array
	//

	// remove object at selectedItemIndex
	var a = rootItems.splice(selectedItemIndex, 1); // Note, obj is returned in array a!
	var itemObj = a[0];
	// add the parentId in itemObj
	itemObj.parentId = targetItem.id;
	
	// util.showObject(itemObj);
	
	// insert object in subitems of targetItem
	subitems.splice(targetIndexInSubitems, 0, itemObj);
	
	
	//
	// Move selectedItem in list
	//
	
	var rootUl = util.getE(this.ulContainerId);
	var li = util.getE(selectedItem.listElementId);
	var liClone = rootUl.removeChild(li);
	
	// alert(liClone);
	// util.showObject(targetItem);
	
	//
	// Handle subitem ul and li
	//
	
	// util.showObject(targetItem);
	
	// Check for ul subitems element in targetItemLi
	var targetItemLi = util.getE(targetItem.listElementId);
	var ulElements = targetItemLi.getElementsByTagName('ul');
	var subitemsUl;
	
	if (ulElements.length > 0) {
		subitemsUl = ulElements[0];
	}
	else {
		
		// No ul element exists (empty report group), create it now
		subitemsUl = this.createSubItemsUlElement();
		targetItemLi.appendChild(subitemsUl);
	}
	
	// alert(subitemsUl);
	
	
	if (targetIndexInSubitems == numberOfSubitems) {
		
		// Append li as last item
		// alert('Append liClone to subitemsUl');
		subitemsUl.appendChild(liClone);
	}
	else {
		
		// Insert li before the next item
		var nextLiElementId = subitems[targetIndexInSubitems + 1].listElementId;
		var nextLi = util.getE(nextLiElementId);
		
		subitemsUl.insertBefore(liClone, nextLi);
	}
	
	// update checkbox state which is lost in IE
	if (this.isSwitch1) {
		var switchKey = this.activeSwitchKey;
		util.setF(selectedItem.switchElementId, selectedItem[switchKey]);
	}
	
	this.isModified = true;
}

listcontroller.List.prototype.saveItem = function(formItemDat) {

	// Note, we first check if the form item data are different to the object data.
	// If the form item has not been modified we don't need to save it.
	
	// NOTE, due the existence of subitems we do not replace the existing object but
	// we update it!
	
	var selectedItem = this.itemsLookup[this.selectedItemId];
	
	// Get the original item data
	
	var itemDat = selectedItem.dat;
	
	// util.showObject(formItemDat, 'formItemDat upon saveItem');
	// util.showObject(itemDat, 'itemDat upon saveItem');
	
	// Check if the formItemDat is different from the existing item in items
	var itemIsModified = this.checkItemIsModified(formItemDat, itemDat);	
	
	// alert('itemIsModified: ' + itemIsModified);
	
	// util.showObject(formItemDat);
	// util.showObject(itemDat);
	
	// If the item has been modified update the item in items
	
	if (itemIsModified) {
		
		// save formItemDat in item.dat
		selectedItem.dat = util.cloneObject(formItemDat);
		
		selectedItem.isModified = true;
		this.isModified = true;
	
		// util.showObject(selectedItem, 'selectedItem after adding formItemDat properties');
	}
}

listcontroller.List.prototype.checkItemIsModified = function(itemA, itemB) {
	
	// Returns true if itemA is different from itemB.
	// Note, this function compares objects and sub-objects but not arrays.

	if (this.itemAisDifferentFromItemB(itemA, itemB)) {
		return true;
	}
	else {
		// check items in reverse order	
		if (this.itemAisDifferentFromItemB(itemB, itemA)) {
			return true;
		}
	}
	
	return false;
}

listcontroller.List.prototype.itemAisDifferentFromItemB = function(itemA, itemB) {
	
	// returns true if itemA is different from itemB
	
	for (var prop in itemA) {
		
		if(itemB[prop] != null) {
			
			// alert('we are checking prop "' + prop + '":\n' + itemA[prop] + '\n' + itemB[prop]);
			
			var itemAValue = itemA[prop];
			var itemBValue = itemB[prop];
			
			if (util.isObject(itemAValue)) {
				
				// recursively call the function												
				if (this.itemAisDifferentFromItemB(itemAValue, itemBValue)) {
					return true;
				}
			}
			else {
		
				if (itemAValue != itemBValue) {
					// alert('prop "' + prop + '" has different values:\nitemAValue: ' + itemAValue + '\nitemBValue: ' + itemBValue);
					return true;
				}
			}
		}
		else {
			// alert('prop "' + prop + '" does not exist in formItemDat or itemsItem');
			return true;
		}
	}
		
	return false;
}

listcontroller.List.prototype.getLookupItems = function(theKey) {
	
	// Returns an array with all items of the given "theKey" property (key of dat), except for the selected item.
	// Character case (uppercase, lowercase) must be kep intact, so we don't convert any item value to lowercase or uppercase!
	// I.e.: ['Item value 1', 'Item value 2', ...]
	
	var items = this.items;
	var a = [];
	
	for (var i = 0; i < items.length; i++) {

		if (items[i]['id'] != this.selectedItemId) {
			
			var s = items[i].dat[theKey];
			a[a.length] = s;
		}
	}
	
	// util.showObject(a);

	return a;
}

listcontroller.List.prototype.getLookupItemsByConstraint = function(theKey, constraintKey, constraintValue) {
	
	// Returns an array with all items of the given "theKey" property (key of dat) it the 
	// item matches the constraintValue of the constraintKey and is not the selected item.
	// Character case (uppercase, lowercase) must be kep intact, so we don't convert any item value to lowercase or uppercase!
	// I.e.: ['Item value 1', 'Item value 2', ...]
	
	var items = this.items;
	var a = [];
	
	var selectedItemId = this.selectedItemId;
	
	for (var i = 0; i < items.length; i++) {
		
		var item = items[i];
		var itemDat = item.dat;

		if ((itemDat[constraintKey] == constraintValue) && (item.id != selectedItemId)) {
			
			var s = items[i].dat[theKey];
			a[a.length] = s;
		}
	}
	
	// util.showObject(a);

	return a;
}

/*
	listcontroller.List helper utilities
*/

//
//
//	switch (list checkboxes) handling
//
//

listcontroller.List.prototype.getActiveSwitchKey = function() {
	return this.activeSwitchKey;
}

// listcontroller.List.prototype.switchActor = function(evt, self) {
listcontroller.List.prototype.switchActivated = function(itemId, isChecked) {
	
	// Initiated upon click on a list checkbox, though the function is called 
	// from the single selectItemActor() event listener
	var item = this.getItem(itemId);
	
	// util.showObject(item);
	
	// Update switch state (switch1 or switch2)
	var switchKey = this.activeSwitchKey;
	item[switchKey] = isChecked;
	
	this.isModified = true;
	
	// Call external event if a switch event is required
	if (this.switchEvent) {
		this.switchEvent(itemId);
	}
	
	// util.showObject(item);
}

listcontroller.List.prototype.toggleSwitch = function(evt, self) {
	
	// Initiated upon click on top of the items list to display
	// the checkbox state for switch1 or switch2.
	// I.e., clicking on "Show in dynmic reports" will change
	// state to switch2 and the label on top of the list to
	// "Show in static reports" 
	
	// alert('toggleSwitch()');
	// Set new switchKey
	var switchKey = (self.activeSwitchKey == 'switch1') ? 'switch2' : 'switch1';
	self.activeSwitchKey = switchKey;
	
	// Update the switch button label
	var label = (switchKey == 'switch2') ? self.switch2Label : self.switch1Label;
	util.updateT(self.switchLabelElementId, label);
	
	// Update the list items to show the checkbox state for the given switchKey
	self.updateSwitchInItems(switchKey, self.items);
}

listcontroller.List.prototype.updateSwitchInItems = function(switchKey, items) {
	
	// Updates the list checkboxes state according the given switchKey (switch1 or switch2)
	for (var i = 0; i < items.length; i++) {
		
		var item = items[i];
		
		util.setF(item.switchElementId, item[switchKey]);
		
		if (item.subitems) {
			this.updateSwitchInItems(switchKey, item.subitems);
		}
	}
}

/*
	Move buttons handling (Note that they are part of the listcontoller Class!)
*/

listcontroller.List.prototype.setMoveControlState = function() {
	
	// alert('setMoveControlState()');
	
	// This sets the button state of the moveControl
	
	//
	// Set default state
	//
	
	var moveDisabledObj = {
		top: true,
		up: true,
		down: true,
		bottom: true
	};
	
	//
	// Set active state
	//
	
	if (this.selectedItemId != null) {
		
		var selectedItem = this.getSelectedItem();
		
		if (selectedItem.parentId == null) {
			
			var items = this.items;
			var selectedItemIndex = util.getArrayObjectIndex(items, 'id', this.selectedItemId);
			var numberOfItems = items.length;
			
			var numberOfFixedItems = this.numberOfFixedItems;
			
			if (numberOfFixedItems < 1) {
			
				// All list items are movable
				if (selectedItemIndex >= 0 && numberOfItems > 1) {
				
					if (selectedItemIndex > 0) {
						moveDisabledObj.top = false;
						moveDisabledObj.up = false;
					}
					
					if (selectedItemIndex < (numberOfItems - 1)) {
						moveDisabledObj.down = false;
						moveDisabledObj.bottom = false;
					}
				}
			}
			else {
				
				// There are fixed items which cannot be moved
				var firstMovableItemIndex = numberOfFixedItems - 1;
				var numberOfMovableItems = numberOfItems - numberOfFixedItems;
				
				if (selectedItemIndex > firstMovableItemIndex && numberOfMovableItems > 1) {
					
					if (selectedItemIndex > firstMovableItemIndex + 1) {
						moveDisabledObj.up = false;
					}
					
					if (selectedItemIndex < (numberOfItems - 1)) {
						moveDisabledObj.down = false;
						moveDisabledObj.bottom = false;
					}
				}
			}
		}
		else {
			
			// This is a subitem, so all buttons become enabled because we allow subitems to be moved to the parent
			moveDisabledObj.top = false;
			moveDisabledObj.up = false;
			moveDisabledObj.down = false;
			moveDisabledObj.bottom = false;
		}
	}
	
	// util.showObject(moveDisabledObj);
	
	//
	// Set actual button state
	//
	
	this.setMoveControlButtons(moveDisabledObj);
}

listcontroller.List.prototype.setMoveControlButtons = function(moveDisabledObj) {

	var YE = YAHOO.util.Event;
	var buttonsState = this.moveControl.buttonsState;
	
	for (var prop in moveDisabledObj) {
		
		var isDisabled = moveDisabledObj[prop];
		
		var theButton = buttonsState[prop];
		
		if (isDisabled != theButton.isDisabled) {
			
			// Change the button state
			
			var img = util.getE(theButton.id);
			img.src = isDisabled ? theButton.srcDis : theButton.src;
			theButton.isDisabled = isDisabled;
			
			if (isDisabled) {
				YE.removeListener(img, 'click', this.moveItemActor);
			}
			else {
				YE.addListener(img, 'click', this.moveItemActor, this);
			}
		}
	}
}

/*
listcontroller.List.prototype.disableMoveControl = function() {
	
	// This disables all move buttons. This is required for lists where i.e. the
	// first list item can't be moved but all others can.
	
	var moveDisabledObj = {
		top: true,
		up: true,
		down: true,
		bottom: true
	};

	this.setMoveControlButtons(moveDisabledObj);
}
*/

listcontroller.List.prototype.moveItemActor = function(evt, self) {
	
	// Fired upon move button click
	
	var element = evt.target || evt.srcElement;
	var elementId = element.id;
	
	// Get the direction
	var dat = elementId.split(':');
	var lastItemIndex = dat.length - 1;
	var direction = dat[lastItemIndex];
	
	// Move the item in listcontroller
	self.moveItem(direction);
	
	// Update MoveControlState
	self.setMoveControlState();
}
