//
// databaseUtil - a collection of database server validation utilities
//
var databaseUtil = function() {
	function getIsValidMysqlDatabaseName(databaseName) {
		var isValid = false,
			errorMessage = '';
		if (databaseName.length > 64) {
			errorMessage = langVar('lang_admin.database.mysql_max_length_msg');
		}
		else if (/[\.\/\\]/.test(databaseName)) {
			errorMessage = langVar('lang_admin.database.mysql_invalid_char_msg');
		}
		else {
			isValid = true;
		}
		return {isValid:isValid, errorMessage:errorMessage};
	}
	function getIsValidOdbcMssqlDatabaseName(databaseName) {
		// odbc_mssql (MS SQL Server)
		// must be <= 128 characters
		// and must match the following rule (added on 09/Jan/2013 by KHP):
		// the name must start with a Unicode letter, _, @, or #; followed by one
		// or more letters, numbers, @, $, #, or _. The documentation clearly
		// states that embedded spaces or special characters are not allowed.
		var isValid = false,
			errorMessage = '',
			pattern = unicode_hack(/^(\p{L}|[_@#]){1}(\p{L}|[0-9_@$#])+$/); // \p{L} matches any unicode letter, this requires the unicode_hack utility
		if (databaseName.length > 128) {
			errorMessage = langVar('lang_admin.database.mssql_max_length_msg');
		}
		else if (!pattern.test(databaseName)) {
			errorMessage = langVar('lang_admin.database.mssql_name_definition_msg');
		}
		isValid = (errorMessage == '');
		return {isValid:isValid, errorMessage:errorMessage};
	}
	function getIsValidDatabaseName(serverType, databaseName) {
		// This validates the databaseName. We assume that the server side
		// uses quoted database names, which means almost any string is allowed.
		// So this covers basic validation required for quoted and unquoted
		// database names.
		if (serverType == 'mysql') {
			return getIsValidMysqlDatabaseName(databaseName);
		}
		else {
			// serverType is odbc_mssql
			return getIsValidOdbcMssqlDatabaseName(databaseName);
		}
	}
	// Return global properties and methods
	return {
		getIsValidDatabaseName: getIsValidDatabaseName
	}
}();
//
// database
//
var database = {
    isWindows: false,
    dirDivider: '',
	haveMYSQL: true,
	haveODBC: true,
	profilesDb: [], // all profiles, except the active profile. It is used to check for shared database directories and names
	itemsDb: [], // items work array
	itemsDbBackup: [], // items array in the state when first loaded or last saved, it is used upon Undo Changes
    theList: null,
    validator: null,
    saveChangesBtn: null,
    undoAllChangesBtn: null,
    activeServerType: '',
    keyupCount: 0,
	init: function() {
		var isEdit = pageInfo.permissions.isEdit;
		var isServerTypeSelectOptions = (pageInfo.licensingTier == 'enterprise' && (database.haveMYSQL || database.haveODBC));
		var YE = YAHOO.util.Event;
		database.validator = new util.Validator();
		//
		// init toolbar buttons
		//
		database.saveChangesBtn = new util.ToolbarButton('save_changes', database.saveChanges, toolbarButtonsDb);
		database.undoAllChangesBtn = new util.ToolbarButton('undo_all_changes', database.undoAllChanges, toolbarButtonsDb);
		var isEdit = pageInfo.permissions.isEdit;
		if (isEdit) {
			// Register isModifiedPageHandler in adminConfig.js
			// (We don't check for modifications if there is no edit
			// permission because there is no Save button anyway!)
			adminConfig.getIsModifiedPageHandler = database.getIsModifiedPage;
		}
		else {
			database.saveChangesBtn.disableAndIgnore();
			database.undoAllChangesBtn.disableAndIgnore();
		}
		//
		// Create theList object
		//
		database.theList = new listcontroller.List({
			containerElementId: 'item_list_body',
			itemEvent: database.itemActivated
		});
		// Add events
		// YE.addListener('automatically_update_when_older_than:active', 'click', database.toggleAutomaticallyUpdate);
		YE.addListener('type', 'change', database.toggleServerType);
		YE.addListener('db_tuning_split_queries:method', 'change', database.splitQueriesMethodActor);
		YE.addListener('mysql_server_bulk_import_method', 'change', database.toggleMysqlServerBulkImportMethod);
		YE.addListener('mssql_server_bulk_import_method', 'change', database.toggleMssqlServerBulkImportMethod);
        YE.addListener(['database_directory', 'database_name'], 'keyup', database.updateSharedDatabaseInfo);
		// Handle serverType
		util.showE('type', isServerTypeSelectOptions);
		util.showE('database:internal_text', !isServerTypeSelectOptions);
		// init OptionInfo
		optionInfo.init();
	},
	initItems: function() {
		// Sets all input controls to initial itemsDb values
		// Also used upon Undo Changes
		var itemsDb = database.itemsDb;
		// util.showObject(itemsDb);
		// Init item list
		database.theList.init(itemsDb);
		database.setItem('i1');
		// Init form data
		for (var i = 0; i < itemsDb.length; i++) {
			var item = itemsDb[i];
			var itemDat = item.dat;
			var prop;
			// util.showObject(itemDat);
			if (item.type != 'database:server') {
				for (prop in itemDat) {
					switch (prop) {
						/*
						// KHP 26/April/2010 - Disabled automatically_update_db in GUI but option is still valid in profile.
						case 'automatically_update_when_older_than':
							// handle automatically_update_when_older_than
							if (itemDat[prop].isActive) {
								util.setF('automatically_update_when_older_than', itemDat[prop].value);
							}
							util.setF('automatically_update_when_older_than:active', itemDat[prop].isActive);
							util.setF('automatically_update_when_older_than:unit', itemDat[prop].timeUnit);
							break;
						*/
						case 'split_queries':
							// handle database.tuning.split_queries.method and number_of_threads
							var splitQueriesMethod = itemDat.split_queries.method;
							var splitQueriesMethodIsCustom = (splitQueriesMethod == 'custom');
							var numberOfThreads = itemDat.split_queries.number_of_threads;
							// alert('numberOfThreads: ' + numberOfThreads);
							util.setF('db_tuning_split_queries:method', splitQueriesMethod);
							util.setF('db_tuning_split_queries:number_of_threads', numberOfThreads);
							util.showE('db_tuning_split_queries:number_of_threads:section', splitQueriesMethodIsCustom);
							break;
						default:
							var value = itemDat[prop].value;
							var type = itemDat[prop].type;
							util.setF(prop, value);
							// temp
							/*
							if (prop == 'type') {
								alert('prop: ' + prop + '\nvalue: ' + value + '\ntype: ' + type);
							}
							*/
							if (type == 'bytes') {
								util.setF(prop + ':unit', itemDat[prop].bytesUnit);
							}
					}
				}
			}
			else {
				// database:server
				for (prop in itemDat) {
					util.setF(prop, itemDat[prop]);
				}
			}
		}
		// database.toggleAutomaticallyUpdate();
		database.toggleServerType();
	},
	getDatabaseData: function() {
		// invoked upon editor load
		if (!pageInfo.exitActive) {
			// Init help
			util.helpWindow.init('');
			var url = '?dp=config_pages.database.get_db_data';
			url += '&p=' + pageInfo.profileName;
			var dat = 'v.fp.page_token=' + pageInfo.pageToken;
			util.serverPost(url, dat);
		}
	},
	getDatabaseDataResponse: function(dat) {
		if (!pageInfo.exitActive) {
//            util.showObject(dat);
            database.isWindows = dat.isWindows;
            database.dirDivider = dat.dirDivider;
			database.haveMYSQL = dat.haveMYSQL;
			database.haveODBC = dat.haveODBC;
			database.profilesDb = dat.profilesDb;
			// get itemsDb
			var itemsDb = dat.itemsDb;
			database.itemsDb = itemsDb;
			database.itemsDbBackup = util.cloneObject(itemsDb);
			// Init
			database.init();
			database.initItems();
			database.saveChangesBtn.enable();
			database.undoAllChangesBtn.enable();
			util.showE('form_section');
			adminConfig.setItemListSize();
			YAHOO.util.Event.addListener(window, 'resize', adminConfig.setItemListSize);
			pageInfo.initComplete = true;
		}
	},
	/*
	toggleAutomaticallyUpdate: function() {
		var updateOn = util.getF('automatically_update_when_older_than:active');
		util.enableE(['automatically_update_when_older_than', 'automatically_update_when_older_than:unit'], updateOn);
	},
	*/
	toggleServerType: function() {
        database.validator.reset();
		var serverType = util.getF('type');
        database.activeServerType = serverType;
		util.hideE([
			'internal_server_section',
			'server_hostname_section',
			'server_dsn_section',
			'server_common_section',
            'server_database_name_section',
			'server_socket_section',
			'mysql_server_bulk_import_method_section',
			'mssql_server_bulk_import_method_section',
			'oracle_server_bulk_import_method_section',
			'server_load_data_directory_section',
			'server_load_data_directory_on_server_section'
		]);
		switch (serverType) {
			case 'internal_sql':
				util.showE('internal_server_section');
				break;
			case 'mysql':
				util.showE('server_hostname_section');
				util.showE('server_socket_section');
				util.showE('server_common_section');
                util.showE('server_database_name_section');
				util.showE('mysql_server_bulk_import_method_section');
				database.toggleMysqlServerBulkImportMethod();
				break;
			case 'odbc_mssql':
				util.showE('server_dsn_section');
				util.showE('server_common_section');
                util.showE('server_database_name_section');
				util.showE('mssql_server_bulk_import_method_section');
				database.toggleMssqlServerBulkImportMethod();
				break;
			case 'odbc_oracle':
				util.showE('server_dsn_section');
				util.showE('server_common_section');
				util.showE('oracle_server_bulk_import_method_section');
				break;
		}
        database.setSharedDatabaseInfo();
	},
	splitQueriesMethodActor: function() {
		var method = util.getF('db_tuning_split_queries:method');
		var isCustom = (method == 'custom');
		util.showE('db_tuning_split_queries:number_of_threads:section', isCustom);
	},
	toggleMysqlServerBulkImportMethod: function() {
		var mysqlServerBulkImportMethod = util.getF('mysql_server_bulk_import_method');
		var isLoadDataServerInfile = (mysqlServerBulkImportMethod == 'load_data_server_infile');
		// alert('isLoadDataServerInfile: ' + isLoadDataServerInfile);
		util.showE('server_load_data_directory_section', isLoadDataServerInfile);
	},
	toggleMssqlServerBulkImportMethod: function() {
		var mssqlServerBulkImportMethod = util.getF('mssql_server_bulk_import_method');
		var isBulkInsert = (mssqlServerBulkImportMethod == 'bulk_insert');
		util.showE('server_load_data_directory_section', isBulkInsert);
		util.showE('server_load_data_directory_on_server_section', isBulkInsert);
	},
	itemActivated: function(itemId) {
		if (database.validateActiveItem()) {
			database.setItem(itemId);
		}
	},
	setItem: function(itemId) {
		database.theList.selectItem(itemId);
		database.displayForm(itemId);
	},
	displayForm: function(itemId) {
		var item = database.theList.getSelectedItem();
		var itemType = item.type;
		// util.showObject(item);
		util.hideE([/*'database:general:form', */'database:server:form', 'database:tuning1:form', 'database:tuning2:form']);
		util.updateT('item_form_label', item.label);
		util.showE(itemType + ':form');
	},
	validateActiveItem: function() {
		if (pageInfo.permissions.isEdit) {
			var validator = database.validator;
			validator.reset();
			var theList = database.theList;
			var activeItem = theList.getSelectedItem();
			// Clone the active item dat and set its values. (We clone it to keep
			// item dat properties such as type and bytesUnit)
			var o = util.cloneObject(activeItem.dat);
			var prop;
			var i;
			switch (activeItem.type) {
				/*
				case 'database:general':
					var autoUpdateActive = util.getF('automatically_update_when_older_than:active');
					o.automatically_update_when_older_than.isActive = autoUpdateActive;
					if (autoUpdateActive) {
						// validate auto update, else keep the existing values!
						o.automatically_update_when_older_than.value =  validator.isInteger('automatically_update_when_older_than', 1);
						o.automatically_update_when_older_than.timeUnit = util.getF('automatically_update_when_older_than:unit');
					}
					// o.prompt_before_erasing_database.value = util.getF('prompt_before_erasing_database');
					break;
				*/
				case 'database:server':
					// reset all option values
					for (prop in o) {
						o[prop] = '';
					}
					var profilesDb = database.profilesDb;
					var profileItem;
					var serverType = util.getF('type');
					o.type = serverType;
					// Set dafault values
					o.hostname = '';
					o.server_socket = '';
					o.dsn = '';
					o.username = '';
					o.password = '';
					o.database_name = '';
					o.database_directory = '';
					o.mysql_server_bulk_import_method = 'load_data_local_infile';
					o.mssql_server_bulk_import_method = 'ODBC';
					o.oracle_server_bulk_import_method = 'ODBC';
					o.load_data_directory = '';
					o.load_data_directory_on_server = '';
					if (serverType == 'internal_sql') {
						var databaseDirectory = util.getF('database_directory');
						o.database_directory = databaseDirectory;
						if (databaseDirectory != '') {
							// Check for unique databaseDirectory
							for (i = 0; i < profilesDb.length; i++) {
								profileItem = profilesDb[i];
								if (profileItem.serverType == serverType && profileItem.databaseDirectory == databaseDirectory) {
									// Throw custom error message
									validator.isCustom('database_directory', langVar('lang_admin.database.database_directory_already_exists'));
									break;
								}
							}
						}
					}
					else {
						// validate external server
						if (serverType == 'mysql' || serverType == 'odbc_mssql') {
							var databaseName = validator.isValue('database_name');
							if (databaseName != '') {
								// validate database name
								var databaseNameVal = databaseUtil.getIsValidDatabaseName(serverType, databaseName);
								if (!databaseNameVal.isValid) {
									// Throw custom error
									validator.isCustom('database_name', databaseNameVal.errorMessage);
								}
							}
							o.database_name = databaseName;
							if (serverType == 'mysql') {
								o.hostname = validator.isValue('hostname');
								o.server_socket = util.getF('server_socket');
								o.mysql_server_bulk_import_method = util.getF('mysql_server_bulk_import_method');
								if (o.mysql_server_bulk_import_method == 'load_data_server_infile') {
									o.load_data_directory = validator.isValue('load_data_directory');
								}
							}
							else {
								o.dsn = validator.isValue('dsn');
								o.mssql_server_bulk_import_method = util.getF('mssql_server_bulk_import_method');
								if (o.mssql_server_bulk_import_method == 'bulk_insert') {
									o.load_data_directory = validator.isValue('load_data_directory');
									o.load_data_directory_on_server = validator.isValue('load_data_directory_on_server');
								}
							}
						}
						else {
							// odbc_oracle
							o.dsn = validator.isValue('dsn');
							o.oracle_server_bulk_import_method = util.getF('oracle_server_bulk_import_method');
						}
						// The username and password is optional
						o.username = util.getF('username');
						o.password = util.getF('password');
					}
					o.sql_table_name_prefix = util.trim(util.getF('sql_table_name_prefix'));
					o.sql_table_name_suffix = util.trim(util.getF('sql_table_name_suffix'));
					break;
				case 'database:tuning1':
					for (prop in o) {
						if (prop != 'split_queries') {
							o[prop].value = util.getF(prop);
						}
						else {
							// handle
							// database.tuning.split_queries.method &
							// database.tuning.split_queries.number_of_threads
							var splitQueriesMethod = util.getF('db_tuning_split_queries:method');
							var splitQueriesNumberOfThreads = '';
							if (splitQueriesMethod == 'custom') {
								splitQueriesNumberOfThreads = validator.isInteger('db_tuning_split_queries:number_of_threads', 1);
							}
							else {
								// auto or none
								// Save splitQueriesNumberOfThreads if the value is empty or a valid integer > 1,
								// in general the value is not required when method is 'auto' or 'none'.
								splitQueriesNumberOfThreads = util.getF('db_tuning_split_queries:number_of_threads');
								if (splitQueriesNumberOfThreads != '' && !util.isInteger(splitQueriesNumberOfThreads, 1)) {
									splitQueriesNumberOfThreads = '';
								}
							}
							o[prop].method = splitQueriesMethod;
							o[prop].number_of_threads = splitQueriesNumberOfThreads;
						}
					}
					break;
				case 'database:tuning2':
					for (prop in o) {
						o[prop].value = validator.isInteger(prop, 0);
						if (o[prop].type == 'bytes') {
							o[prop].bytesUnit = util.getF(prop + ':unit');
						}
					}
					break;
			}
			if (validator.allValid()) {
				// util.showObject(o, 'new object to save');
				theList.saveItem(o);
				return true;
			}
			return false;
		}
		else {
			// No isEdit permission, don't validate
			return true;
		}
	},
	saveChanges: function() {
		if (database.validateActiveItem()) {
			var theList = database.theList;
			var isModified = theList.getIsModified();
			// alert('saveChanges() - isModified: ' + isModified);
			if (isModified) {
				util.hideE('form_section');
				util.showE('saving_info');
				// If external database server we need to check for a valid database connection
				if (util.getF('type') == 'internal_sql') {
					database.saveDatabase();
				}
				else {
					// check database server
					var url = '?dp=config_pages.database.validate_database_options';
					url += '&p=' + pageInfo.profileName;
					var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
					// add server options
					var server = theList.getItem('i1');
					var serverDat = server.dat;
					for (var prop in serverDat) {
						var itemValue = serverDat[prop];
						if (itemValue != '') {
							itemValue = encodeURIComponent(itemValue);
						}
						dat += 'v.fp.database_server.' + prop + '=' + itemValue + '&';
					}
					dat = dat.replace(/&$/, '');
					util.serverPost(url, dat);
				}
			}
			else {
				alert(langVar('lang_stats.general.no_changes_to_save'));
			}
		}
	},
	saveDatabaseCheckDbServerResponse: function(dat) {
		var saveDb = true;
		if (!dat.isValid) {
			var msg = langVar('lang_admin.database.no_database_server_connection') + '\n\n';
			msg += dat.errorMessage + '\n\n';
			msg += langVar('lang_admin.database.save_anyway');
			saveDb = confirm(msg);
		}
		if (saveDb) {
			database.saveDatabase();
		}
		else {
			util.hideE('saving_info');
			util.showE('form_section');
		}
	},
	getDatabaseItems: function(optionsGroupName, itemDat) {
		var dat = '';
		var itemValue;
		for (var prop in itemDat) {
				switch (prop) {
					case 'split_queries':
						// split_queries
						dat += 'v.fp.tuning.split_queries.method=' + itemDat[prop].method + '&';
						dat += 'v.fp.tuning.split_queries.number_of_threads=' + itemDat[prop].number_of_threads + '&';
						break;
					/*
					case 'automatically_update_when_older_than':
						itemValue = itemDat[prop].value;
						// handle automatically_update_when_older_than
						if (!itemDat[prop].isActive) {
							itemValue = 0;
						}
						else {
							// if timeUnit is not seconds then convert the value to seconds!
							switch (itemDat[prop].timeUnit) {
								case 'minutes':
									itemValue = itemValue * 60;
									break;
								case 'hours':
									itemValue = itemValue * 60 * 60;
									break;
								case 'days':
									itemValue = itemValue * 60 * 60 * 24;
									break;
							}
						}
						dat += 'v.fp.options.automatically_update_when_older_than=' + itemValue + '&';
						break;
					*/
					default:
						itemValue = itemDat[prop].value;
						var type = itemDat[prop].type;
						if (type == 'bytes' && itemDat[prop].bytesUnit != 'bytes') {
							itemValue = itemValue + itemDat[prop].bytesUnit;
						}
						dat += 'v.fp.' + optionsGroupName + '.' + prop + '=' + encodeURIComponent(itemValue) + '&';
				}
		}
		return dat;
	},
	getDatabaseServerItems: function(itemDat) {
		var dat = '';
		var serverType = itemDat.type;
		for (var prop in itemDat) {
			if (prop != 'mysql_server_bulk_import_method' &&
				prop != 'mssql_server_bulk_import_method' &&
				prop != 'oracle_server_bulk_import_method') {
				var itemValue = itemDat[prop];
				dat += 'v.fp.server.' + prop + '=' + encodeURIComponent(itemValue) + '&';
			}
		}
		// Add final bulk_import_method
		var bulkImportMethod = '';
		switch (serverType) {
			case 'mysql':
				bulkImportMethod = itemDat.mysql_server_bulk_import_method;
				break;
			case 'odbc_mssql':
				bulkImportMethod = itemDat.mssql_server_bulk_import_method;
				break
			case 'odbc_oracle':
				bulkImportMethod = itemDat.oracle_server_bulk_import_method;
				break;
		}
		dat += 'v.fp.server.bulk_import_method=' + bulkImportMethod + '&';
		return dat;
	},
	saveDatabase: function() {
		// If valid database
		var itemsDb = database.itemsDb;
		var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
		for (var i = 0; i < itemsDb.length; i++) {
			var itemType = itemsDb[i].type;
			var itemDat = itemsDb[i].dat;
			if (itemType != 'database:server') {
				var optionsGroupName = (itemType == 'database:general') ? 'options' : 'tuning';
				dat += database.getDatabaseItems(optionsGroupName, itemDat);
			}
			else {
				dat += database.getDatabaseServerItems(itemDat);
			}
		}
		dat = dat.replace(/&$/, '');
		// alert('dat:\n' + dat);
		var url = '?dp=config_pages.database.save_db_data';
		url += '&p=' + pageInfo.profileName;
		util.serverPost(url, dat);
	},
	saveDatabaseResponse: function() {
		// reset isModified
		database.theList.resetIsModified();
		// the saved work itemsDb becomes now the itemsDbBackup
		database.itemsDbBackup = util.cloneObject(database.itemsDb);
		util.hideE('saving_info');
		util.showE('form_section');
	},
	undoAllChanges: function() {
		database.itemsDb = util.cloneObject(database.itemsDbBackup);
		database.initItems();
	},
	getIsModifiedPage: function() {
		// Note, isModified will be false if only the active item has been edited
		// but has an error, hence we also check "if (!validateActiveItem() ..."
		if (!database.validateActiveItem() || database.theList.getIsModified()) {
			return true;
		}
		return false;
	},
    updateSharedDatabaseInfo: function() {
        // Increment keyupCount
        var latestKeyupCount = database.keyupCount + 1;
        database.keyupCount = latestKeyupCount;
        setTimeout(function() { database.delayedUpdateSharedDatabaseInfo(latestKeyupCount); }, 500);
    },
    delayedUpdateSharedDatabaseInfo: function(keyupCount) {
        // Call setSharedDatabaseInfo when there is no more key in action
        if (keyupCount == this.keyupCount) {
//            util.showObject({keyupCount:keyupCount});
            database.setSharedDatabaseInfo();
        }
    },
    setSharedDatabaseInfo: function() {
        // Checks if the database_directory (internal_sql) or
        // database_name (mysql or odbc_mssql) is shared with
        // another profile with the same database server type.
        var activeServerType = database.activeServerType;
        var isInternalDb = (activeServerType == 'internal_sql');
        // Get active database_directory or database_name
        var activeDbDirOrName = '';
        var isWindows = database.isWindows;
        var profileLabels = ''; // labels of profiles with shared database
        if (isInternalDb) {
            activeDbDirOrName = util.getF('database_directory');
            // Add directory divider for comparison only if ist missing
            if (activeDbDirOrName.substring(activeDbDirOrName.length -1) != database.dirDivider) {
                activeDbDirOrName += database.dirDivider;
            }
            if (isWindows) {
                activeDbDirOrName = activeDbDirOrName.toLowerCase();
            }
        }
        else {
            activeDbDirOrName = util.getF('database_name');
        }
        var showSharedDatabaseInfo = false;
        if (isInternalDb ||
            activeServerType == 'mysql' ||
            activeServerType == 'odbc_mssql') {
            var infoElementId = isInternalDb ? 'database_directory:warning' : 'database_name:warning';
            if (activeDbDirOrName != '') {
                var profilesDb = database.profilesDb;
                var max = profilesDb.length;
                for (var i = 0; i < max; i++) {
                    var theProfile = profilesDb[i];
                    if (theProfile.serverType == activeServerType) {
                        var existingDbDirOrName = '';
                        if (isInternalDb) {
                            existingDbDirOrName = theProfile.databaseDirectory;
                            if (isWindows) {
                                existingDbDirOrName = existingDbDirOrName.toLowerCase();
                            }
                        }
                        else {
                            existingDbDirOrName = theProfile.databaseName;
                        }
                        if (existingDbDirOrName == activeDbDirOrName) {
                            profileLabels += '"' + theProfile.label + '", ';
                            showSharedDatabaseInfo = true;
                        }
                    }
                }
            }
            // Show profiles which share the same database
            if (showSharedDatabaseInfo) {
                // Remove last comma and space from profileLabels
                profileLabels = profileLabels.substring(0, profileLabels.length - 2);
                var infoElementText = util.getE(infoElementId + ':text');
                var msg = '';
                if (isInternalDb) {
                    msg = langVar('lang_admin.database.shared_database_directory_warning');
                }
                else {
                    msg = langVar('lang_admin.database.shared_database_warning');
                }
                msg = msg.replace('__PARAM__1', profileLabels);
                msg += '<br />' + langVar('lang_admin.database.shared_database_warning_2');
                infoElementText.innerHTML = msg;
            }
            util.showE(infoElementId, showSharedDatabaseInfo);
        }
    }
};
