#!/usr/bin/perl
#
# This script converts an AMF file to a .cfg file for Sawmill's use in PDF generation.
#
# Usage: afm_to_cfg.pl filename.afm
#
# Calling this generates a file filename.cfg which contains the AFM information, but in Sawmill's CFG format.
#

use strict;

my $filename = $ARGV[0];

if ($filename eq "") {
    die("Usage: afm_to_cfg.pl {filename}\n");
}

# Build the CFG nodename from the AFM filename
my $nodename = $filename;
if ($nodename =~ /^([^\\\/]*)\.afm$/) {
    $nodename = $1;
}
else {
    die("Unrecognized filename format: $filename (must be filename.afm; no pathname dividers)");
}
$nodename =~ tr/A-Z/a-z/;
$nodename =~ tr/-/_/;

# Open the CFG file
my $cfgFilename = $nodename . ".cfg";
open(CFG, ">$cfgFilename") || die("Can't open $cfgFilename for writing: $!");
print CFG "$nodename = {\n";
#print "Writing $cfgFilename\n";

sub GENERATE {
    my ($line) = @_;
#    print "  -> $line";
    print CFG $line;
}

# Process all lines of the AFM file
#print "Reading $filename\n";
open(AFM, $filename) || die("Can't open $filename for reading: $!");
while (<AFM>) {

    chop;
    chop;
#    print "Processing: \"$_\"\n";

    if (0) {}

    # Convert Comment and Notice lines to CFG comments
    elsif (/^Comment (.*)$/) {
	GENERATE("  # Comment: $1\n");
    }
    elsif (/^Notice (.*)$/) {
	GENERATE("  # Notice: $1\n");
    }

    # Handle Char Metrics section
    elsif (/^StartCharMetrics ([0-9]+)/) {

	GENERATE("  CharMetrics = {\n");

	my $numCharMetricsLines = $1;
	for (my $i = 0; $i < $numCharMetricsLines; $i++) {

	    $_ = <AFM>;
	    chop;
	    chop;
#	    print "Processing metrics line: \"$_\"\n";
	    my $metrics = $_;
	    my $metricsNode = "";
	    my $charCode = "";
	    my $charName = "";
	    while(/^ *([A-Z]+) ([^;]+) ;(.*)$/) {
#		print "1 2: $1 $2\n";
		if ($1 eq "C") {
		    $charCode = $2;
		}
		elsif ($1 eq "B") {
		    my ($llx, $lly, $urx, $ury) = split(" ", $2);
		    $metricsNode .= "      B = {\n";
		    $metricsNode .= "        llx = \"$llx\"\n";
		    $metricsNode .= "        lly = \"$lly\"\n";
		    $metricsNode .= "        urx = \"$urx\"\n";
		    $metricsNode .= "        ury = \"$ury\"\n";
		    $metricsNode .= "      } # B\n";
		}
		else {
		    $metricsNode .= "      $1 = \"$2\"\n";
                    if ($1 eq 'N') {
			$charName = $2;
		    }
		}
		$_ = $3;
	    }
            if ($charCode eq "-1") {
		$metricsNode = "    $charName = {\n$metricsNode    } # $charCode\n";
	    }
            else {
		$metricsNode = "    $charCode = {\n$metricsNode    } # $charCode\n";
	    }
	    GENERATE($metricsNode);

	} # for lines

	GENERATE("  } # CharMetrics\n");

	# Get the EndCharMetrics line
	$_ = <AFM>;
	chop;
	chop;
	if (!/^EndCharMetrics$/) {
	    die("Error in CharMetrics; expected \"EndCharMetrics\"; found \"$_\"");
	}

    } # StartCharMetrics

    # Ignore StartKernData line
    elsif (/^StartKernData$/) {
#	print "Ignoring \"$_\"\n";
    }

    # Handle kerning pairs section
    elsif (/^StartKernPairs ([0-9]+)/) {

	GENERATE("  KernPairs = {\n");

	my $numKernPairsLines = $1;
	for (my $i = 0; $i < $numKernPairsLines; $i++) {

	    $_ = <AFM>;
	    chop;
	    chop;
#	    print "Processing KernPairs line: \"$_\"\n";
	    if (/^KPX ([^ ]+) ([^ ]+) ([0-9-]+)$/) {
		GENERATE("    $1" . "_$2 = $3\n");
	    }

	} # for lines

	GENERATE("  } # KernPairs\n");

	# Get the EndKernPairs line
	$_ = <AFM>;
	chop;
	chop;
#	print "end of ker pairs: $_\n";
	if (!/^EndKernPairs$/) {
	    die("Error in KernPairs; expected \"EndKernPairs\"; found \"$_\"");
	}

    } # StartKernPairs

    # Ignore EndKernData line
    elsif (/^EndKernData$/) {
#	print "Ignoring \"$_\"\n";
    }

    # Ignore EndFontMetrics line
    elsif (/^EndFontMetrics$/) {
#	print "Ignoring \"$_\"\n";
    }

    elsif ((/^([^ ]+) ([^ ]+)$/) ||
	   (/^(FullName) (.*)$/)) { 
	GENERATE("  $1 = \"$2\"\n");
    }
    elsif (/^FontBBox ([^ ]+) ([^ ]+) ([^ ]+) ([^ ]+) *$/) {
	GENERATE("  FontBBox = {\n");
	GENERATE("    llx = $1\n");
	GENERATE("    lly = $2\n");
	GENERATE("    urx = $3\n");
	GENERATE("    ury = $4\n");
	GENERATE("  } # FontBBox\n");
    }
    else {
	die("Unrecognized AFM line: $_");
    }

} # while AFM

close AFM;

print CFG "} # $nodename\n";
close CFG;
print "Converted $filename to $cfgFilename\n";

