//
//
// emailUtil.js
//
// Used in Scheduler and Send Report By Email to handle recipient addresses
//
//
var emailUtil = {};
// AddressControl Class
emailUtil.AddressControl = function(containerElementId, addressElementWidth) {
	// alert('emailUtil.AddressControl');
	this.numberOfRows = 0; // The number of actuially build recipient rows
	this.addresses = []; // keeps the state upon validation and is also used to get the final email data
						// because they are written to addresses at validation time
	// Get a unique ID prefix
	var idPrefix = util.getUniqueElementId();
	this.idPrefix = idPrefix;
	this.addressDivId = idPrefix + ':addresses:div';
	this.addressTbodyId = idPrefix + ':addresses:tbody';
	this.isErrorIndication = false;
	this._build(containerElementId, idPrefix, addressElementWidth);
	YAHOO.util.Event.addListener(this.addressTbodyId, 'keyup', this._addressItemActicated, this);
}
emailUtil.AddressControl.prototype.init = function(addresses) {
	// addresses contains any default email addresses, 
	// i.e.: [{type:'to',  address:'abc@abc.com'}, {type:'cc', address:'first last <abc@abc.com>'}]
	var i;
	var idPrefix = this.idPrefix;
	if (addresses.length > 4) {
		// There are more than 4 default addresses, so we have to add additional rows
		var tbody = util.getE(this.addressTbodyId);
		var numberOfExistingRows = this.numberOfRows;
		var numberOfRequiredRows = addresses.length;
		for (i = numberOfExistingRows; i < numberOfRequiredRows; i++) {
			this._addRow(tbody, idPrefix, i);
			this._populateSelect(idPrefix, i);
		}
	}
	// Handle default addresses
	for (i = 0; i < addresses.length; i++) {
		var item =  addresses[i];
		util.setF(idPrefix + ':' + i + ':select', item.type);
		util.setF(idPrefix + ':' + i + ':input', item.address);
	}
}
emailUtil.AddressControl.prototype.validate = function() {
	// Validates the email addresses and handles error indication.
	// At least one valid email address must exist. If one of multiple
	// email addresses is not valid we show an error.
	// We validate and write email data to this.addresses.
	// this.addresses objects contains an isError property so that we know
	// if we must reset any color if an email address is indicated in Red
	// this.addresses is in the format: 
	// [{type:'to', name:'', email:'abc@abc.com', isError:false}, {type:'cc', name:'first last', email:'abc@abc.com',  isError:false}]
	var idPrefix = this.idPrefix;
	this.addresses = [];
	var isEmailAddress = false;
	var isAddressWithError = false;
	for (var i = 0; i < this.numberOfRows; i++) {
		var type = ''; // We only define a type if we find a valid email address
		var address = util.getF(idPrefix + ':' + i + ':input');
		var isError = false;
		if (address != '') {
			if (util.isEmailAddress(address)) {
				isEmailAddress = true;
				type = util.getF(idPrefix + ':' + i + ':select');
			}
			else {
				isAddressWithError = true;
				isError = true;
				var element = util.getE(idPrefix + ':' + i + ':input');
				element.style.color = 'Red';
			}
		}
		// Add object to this.addresses
		this.addresses[i] = {type:type, address:address, isError:isError};
	}
	if (isEmailAddress && !isAddressWithError) {
		return true;
	}
	else {
		// Handle error indication
		// alert('Error in recipient addresses');
		var msg = isAddressWithError ? langVar('lang_stats.general.invalid_email_address_in_recipients_msg') : langVar('lang_stats.general.no_recipient_address_message');
		util.updateT('email-address-grid:error', msg);
		util.showE('email-address-grid:error');
		this.isErrorIndication = true;
	}
	return false;
}
emailUtil.AddressControl.prototype._addressItemActicated = function(evt, self) {
	var element = evt.target || evt.srcElement;
	var elementId = element.id;
	// alert('_addressItemActicated() - id: ' + elementId);
	var dat = elementId.split(':');
	var rowIndex = parseInt(dat[1], 10);
	var elementType = dat[2];
	if (elementType == 'input') {
		// Handle error div on bottom
		if (self.isErrorIndication) {
			util.hideE('email-address-grid:error');
			self.isErrorIndication = false;
		}
		// Reset error of current row
		// Note, the row may not yet exist in this.addresses because it is only
		// added upon validation, so check for existence.
		var addresses = self.addresses;
		if (addresses[rowIndex]) {
			var address = addresses[rowIndex];
			if (address.isError) {
				element.style.color = 'Black';
				address.isError = false;
			}
		}
		// Add a new row if the current row is the last row
		var numberOfRows = self.numberOfRows;
		if (rowIndex == numberOfRows - 1) {
			var tbody = util.getE(self.addressTbodyId);
			self._addRow(tbody, self.idPrefix, numberOfRows);
			self._populateSelect( self.idPrefix, numberOfRows);
		}
	}
}
emailUtil.AddressControl.prototype.getAddresses = function() {
	// Returns all valid addresses
	// Valid addresses already exist in this.addresses since validation.
	// Valid addresses have a defined type, others not.
	var a = this.addresses;
	var b = [];
	for (var i = 0; i < a.length; i++) {
		var item = a[i];
		if (item.type != '') {
			b[b.length] = {type:item.type, address:item.address};
		}
	}
	return b;
}
emailUtil.AddressControl.prototype.reset = function() {
	// Resets the control by:
	// a.) setting the row numbers to 4
	// b.) removing any error indication
	// c.) resetting form fields to "to:" address and empty input field
	var i;
	if (this.isErrorIndication) {
		util.hideE('email-address-grid:error');
		this.isErrorIndication = false;
	}
	this.addresses = [];
	//
	// Reset rows to four rows
	//
	var numberOfRows = this.numberOfRows;
	if (this.numberOfRows > 4) {
		// Remove rows
		var tbody = util.getE(this.addressTbodyId);
		var rows = tbody.getElementsByTagName('tr');
		for (i = rows.length - 1; i > 3; i--) {
			var theRow = rows[i];
			tbody.removeChild(theRow);
		}
		this.numberOfRows = 4;
	}
	//
	// Reset select and input elements
	//
	var idPrefix = this.idPrefix;
	for (i = 0; i < 4; i++) {
		util.setF(idPrefix + ':' + i + ':select', 'to');
		var input = util.getE(idPrefix + ':' + i + ':input');
		input.value = '';
		input.style.color = 'Black';
	}
}
emailUtil.AddressControl.prototype.disable = function(isDisable) {
	// isDisable is optional
	var makeDisabled = (isDisable != null) ? isDisable : true;
	var numberOfRows = this.numberOfRows;
	var idPrefix = this.idPrefix;
	var a = [];
	for (var i = 0; i < numberOfRows; i++) {
		a[a.length] = idPrefix + ':' + i + ':select';
		a[a.length] = idPrefix + ':' + i + ':input';
	}
	util.disableE(a, makeDisabled);
}
emailUtil.AddressControl.prototype.freezeOverflow = function(isFreeze) {
	// Fixes a firefox bug, we require to set overflow to hidden when
	// a subform is displayed above the div element where overflow is set
	// to scroll
	var overflow = (isFreeze == null || isFreeze == true) ? 'hidden' : 'auto';
	var element = util.getE(this.addressDivId);
	element.style.overflow = overflow;
}
emailUtil.AddressControl.prototype._build = function(containerElementId, idPrefix, addressElementWidth) {
	// containerElementId is usually a td element
	// Builds the control
	// We create four address rows as default, starting from row 0, 1, 2, 3
	var container = util.getE(containerElementId);
	var addressDiv = util.createE('div', {id:this.addressDivId, className:'email-address-grid', width:addressElementWidth + 'px'});
	var table = util.createE('table', {className:'email-address-grid', cellSpacing:0});
	var tbody = util.createE('tbody', {id:this.addressTbodyId});
	var i;
	for (i = 0; i < 4; i++) {
		this._addRow(tbody, idPrefix, i);
	}
	// Add a div for error indication
	var errorDiv = util.createE('div', {id:'email-address-grid:error', className:'form-error'});
	util.chainE([container, [addressDiv, [table, tbody]], errorDiv]);
	// Handle select elements
	// Populate select only works after the select element is part of the dom
	for (i = 0; i < 4; i++) {
		this._populateSelect(idPrefix, i);
	}
}
emailUtil.AddressControl.prototype._addRow = function(tbody, idPrefix, rowIndex) {
	// Builds a row with select and input element
	// We create four address rows as default, starting from row 0
	var selectId = idPrefix + ':' + rowIndex + ':select';
	var inputId = idPrefix + ':' + rowIndex + ':input';
	var tr = util.createE('tr');
	var th = util.createE('th', {padding:'0px'});
	var td = util.createE('td', {padding:'0px'});
	var select = util.createE('select', {id:selectId});
	var input = util.createE('input', {id:inputId, type:'text', value:''});
	util.chainE(tbody, [tr, [th, select], [td, input]]);
	this.numberOfRows = rowIndex + 1;
}
emailUtil.AddressControl.prototype._populateSelect = function(idPrefix, rowIndex) {
	var list = [{name:'to', label:langVar('lang_stats.general.email_to') + ':'}, {name:'cc', label:langVar('lang_stats.general.email_cc') + ':'}, {name:'bcc', label:langVar('lang_stats.general.email_bcc') + ':'}];
	var selectId = idPrefix + ':' + rowIndex + ':select';
	util.populateSelect(selectId, list, 'name', 'label');
	util.setF(selectId, 'to');
}
//
// schedulerUtil.js
//
var schedulerUtil = {
	actionsEmailToEmailAddressesObject: function(options) {
		// Converts the action email parameters in options (options.rca, options.ca and options.ba)
		// to an address array with objects required by emailUtil.js
		// i.e.: [{type:'to',  address:'abc@abc.com'}, {type:'cc', address:'first last <abc@abc.com>'}]
		var emailAddresses = [];
		var shortcuts = ['rca', 'ca', 'ba'];
		var recipientTypes = ['to', 'cc', 'bcc'];
		for (var i = 0; i < shortcuts.length; i++) {
			var shortcut = shortcuts[i];
			// Note, only options with a value exists!
			// If no cc_address has been defined then no cc_address exists in options!
			if (options[shortcut]) {
				var addressString = options[shortcut];
				// Remove white space
				addressString = addressString.replace(/\s/g, '');
				var type = recipientTypes[i];
				if (addressString.indexOf(',') == -1) {
					// Single email address
					emailAddresses[emailAddresses.length] = {type:type, address:addressString};
				}
				else {
					// Multiple email addresses
					var dat = addressString.split(',');
					for (var j = 0; j < dat.length; j++) {
						emailAddresses[emailAddresses.length] = {type:type, address:dat[j]};
					}
				}
			}
		}
		return emailAddresses;
	}
}
//
// schedulerActionsList.js
//
var schedulerActionsList = {
	moveControl: null,
	list: null,
	init: function() {
		//
		// Create the actionsList object
		//
		var obj = {
			name: 'actions',
			tbodyElementId: 'scheduler:actions_list:tbody',
			noItemText: langVar('lang_admin.scheduler.no_action_defined'),
			isDefaultItem: true,
			listChangeNotificationFunction: schedulerActionsList.listChangeNotification,
			editItemFunction: schedulerActions.editItem,
			duplicateItemFunction: schedulerActions.duplicateItem,
			labelBuilderFunction: schedulerActions.labelBuilder
		}
		schedulerActionsList.list = new listcontrollerB.List(obj);
		schedulerActionsList.moveControl = new util.MoveControl('sa_move_control', schedulerActionsList.moveItem);
	},
	listChangeNotification: function(listName, selectedItemIndex, numberOfItems) {
		// Called from wide item list controller upon list change
		// Set button state of move control
		schedulerActionsList.moveControl.setState(selectedItemIndex, numberOfItems);
	},
	moveItem: function(direction) {
		// Invoked from moveControl
		// var direction = schedulerActionsList.moveControl.getMoveDirection(this.id);
		// Move item in list
		schedulerActionsList.list.moveItem(direction);
	}
}
//
// schedulerActions.js
//
var schedulerActions = {
	panel: null,
	validator: null,
	addressControl: null, // email address control for To, Cc and Bcc address
	//
	// Active form parameters
	//
	mode: '', // the active mode of an item is: new | edit | duplicate (We need to know the mode when saving the item back to the listcontrollerB
	itemIndex: 0, // refers to the active item index in listcontrollerB items array, respectively to the active row index
	activeAction: '',
	isDatabaseAction: false,
	isProfileList: false, // no profile list has been populated yet
	isExtendedProfilesList: false, // specifies if the active profile list contains All profiles and Profile pattern list entry
	isExtraOptions: false, // specifies if the extra options textarea element row is displayed or not 
	isUnlimitedGrants: false,
	activeReportsDb: [], // the active reportsDb of the selected profile
	activeDF: { // the active date filter object per action
		type: '', // none, relative | custom
		custom: '', // custom expression if type is custom
		relativePrefix: '', // none | recent || last
		relativeDateUnit: '', // year | quarter, month, ...
		relativeDateUnitCount: 1
	},
	init: function() {
		var sa = schedulerActions;
		var YE = YAHOO.util.Event;
		sa.isUnlimitedGrants = pageInfo.isRootAdmin || pageInfo.permissions.isUnlimitedGrants;
		sa.validator = new util.Validator();
		//
		// Init the Action panel
		//
		var panelObj = {
			panelId: 'sa:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: '-',
			left: 316,
			top: 160,
			zIndex: 20,
			isCover: true,
			isSticky: true,
			closeEvent: sa.closePanel
		};
		sa.panel = new util.Panel3(panelObj);
		// Create email addressControl
		sa.addressControl = new emailUtil.AddressControl('sa:email_address_control:container', 360);
		//
		// Add events
		//
		YE.addListener('sa:action', 'change', sa.display.selectActionActor);
		YE.addListener('sa:profile', 'change', sa.display.selectProfileActor);
		YE.addListener('sa:report', 'change', sa.display.selectReportActor);
		YE.addListener('sa:generate_report_files:output_format_type', 'change', sa.display.setOutputFormatType);
		YE.addListener('sa:date_filter_text', 'click', schedulerDF.open);
		YE.addListener(['sa:remove_db_data_by_number_of_days', 'sa:remove_db_data_by_filter_expression'], 'click', sa.display.removeDatabaseDataEvent);
        YE.addListener('sa:export_csv_table:number_of_rows', 'keypress', sa.display.setCvsExportNumberOfRowsToCustom);
		YE.addListener('sa:extra_options:toggle_btn', 'click', sa.display.toggleExtraOptions);
		YE.addListener('sa:okay', 'click', sa.saveItem);
		YE.addListener('sa:cancel', 'click', sa.closePanel);
	},
	display: {
		selectActionActor: function() {
			var action = util.getF('sa:action');
			schedulerActions.display.setActionByName(action);
		},
		setActionByName: function(action) {
			// alert('setActionByName(): ' + action);
			var databaseTextPattern = /database/;
			var sa = schedulerActions;
			sa.activeAction = action;
			sa.isDatabaseAction = databaseTextPattern.test(action);
			// Reset the validator
			sa.validator.reset();
			// Reset the display
			var displayGroups = [
				'no_action_defined',
				'profile',
				'profile_pattern',
				'remove_database_data',
				'report',
				'export_csv_table',
				'generate_report_files',
				'generate_report_files_output_directory',
				'generate_report_files_output_file',
				'generate_report_files_pdf_not_supported',
				'email',
				'report_language',
				'extra_options',
                'execute_command_line',
				'vertical_spacer'
			];
			for (var i = 0; i < displayGroups.length; i++) {
				util.hideE('sa:' + displayGroups[i] + ':group');
			}
			var showExtendedProfileList = false;
			//
			// Handle the action
			//
			if (action != '') {
                if (action != 'execute_command_line') {
                    util.showE('sa:profile:group');
                    if (sa.isDatabaseAction) {
                        // database actions
                        showExtendedProfileList = true;
                        if (action == 'remove_database_data') {
                            util.showE('sa:remove_database_data:group');
                        }
                        util.showE('sa:vertical_spacer:group');
                    }
                    else {
                        // report actions
                        util.showE('sa:report:group');
                        if (action == 'generate_report_files') {
                            util.showE('sa:generate_report_files:group');
                            // sa.display.initOutputFormatType();
                            sa.display.setOutputFormatType();
                        }
                        else if (action == 'export_csv_table') {
                            util.showE('sa:export_csv_table:group');
                        }
                        else {
                            util.showE('sa:email:group');
                        }
                        util.showE('sa:report_language:group');
                    }
                    // sa.display.resetExtraOptions();
                    util.showE('sa:extra_options:group');
                }
                else {
                    util.showE('sa:execute_command_line:group');
                }
			}
			else {
				util.showE('sa:no_action_defined:group');
				util.showE('sa:vertical_spacer:group');
			}
			//
			//
			// Update the profile list
			//
			//
			var profilesDb = [];
			var baseDb = [];
			if (sa.isUnlimitedGrants) {
				// Update profiles list if required
				if (showExtendedProfileList != sa.isExtendedProfilesList || !sa.isProfileList) {
					profilesDb = scheduler.profilesDb;
					baseDb[0] = {name:'', label:langVar('lang_admin.scheduler.select_profile_label')};
					if (showExtendedProfileList) {
						baseDb[1] = {name:'*', label:langVar('lang_admin.scheduler.all_profiles_label')};
						baseDb[2] = {name:'__PROFILE__PATTERN__', label:langVar('lang_admin.scheduler.use_profile_pattern')};
					}
					util.populateSelect('sa:profile', baseDb, 'name', 'label');
					util.extendSelect('sa:profile', profilesDb, 'name', 'label');
					sa.isProfileList = true;
					sa.isExtendedProfilesList = showExtendedProfileList;
				}
			}
			else if (action != '') {
				// The user has limited grants and profiles may vary per action, so we always
				// update the profiles list upon action change!
				// We also don't allow a profile pattern for limited grants
				// var actionsDb = scheduler.actionsDb;
				// util.showObject(scheduler.actionsDb);
				// alert('action: ' + action);
				var superAction = scheduler.actionsDb[util.h(action)];
				// util.showObject(superAction);
				var isAllProfiles = superAction.isAllProfiles;
				profilesDb = isAllProfiles ? scheduler.profilesDb : superAction.profiles;
				baseDb[0] = {name:'', label:langVar('lang_admin.scheduler.select_profile_label')};
				if (showExtendedProfileList && isAllProfiles) {
					baseDb[1] = {name:'*', label:langVar('lang_admin.scheduler.all_profiles_label')};
				}
				util.populateSelect('sa:profile', baseDb, 'name', 'label');
				util.extendSelect('sa:profile', profilesDb, 'name', 'label');
			}
			// Reset the profile list to Select Profile
			sa.display.setProfileList('');
		},
		setTabPane: function(tabId) {
			util.hideE(['sa:general:pane', 'sa:email:pane', 'sa:options:pane']);
			util.showE('sa:' + tabId + ':pane');
		},
		selectProfileActor: function() {
			// activated upon onchange
			var sa = schedulerActions;
			var profileName = util.getF('sa:profile');
			sa.display.setProfileList(profileName);
		},
		setProfileList: function(profileName) {
			var sa = schedulerActions;
			if (profileName != '') {
				if (sa.isDatabaseAction) {
					var profileIsPattern = (profileName == '__PROFILE__PATTERN__');
					util.showE('sa:profile_pattern:group', profileIsPattern);
				}
				else {
					// Report action, we need to update the report list				
					sa.display.setReportList(profileName);
				}
			}
			else {
				sa.display.setReportList('');
			}
			// We re-set the form value because setProfileList() could be invoked other than by event
			util.setF('sa:profile', profileName);
		},
		selectReportActor: function() {
			if (schedulerActions.activeAction == 'generate_report_files') {
				// schedulerActions.display.initOutputFormatType();
				schedulerActions.display.setOutputFormatType();
			}
		},
		setReportList: function(profileName) {
			// util.disableE('sa:report');
			var disableSelect = false;
			var baseDb = [];
			baseDb[0] = {name:'', label:langVar('lang_admin.scheduler.select_report_label')};
			if (profileName != '') {
				schedulerActions.activeReportsDb = getReportsDb(profileName); // in scheduler.js
				// Create hash for activeReportsDb report name
				util.createHash(schedulerActions.activeReportsDb, 'name');
				if (schedulerActions.activeAction == 'generate_report_files') {
					baseDb[1] = {name:'*', label:langVar('lang_admin.scheduler.all_reports_label')};
				}
				// util.showObject(reportsDb);
				util.populateSelect('sa:report', baseDb, 'name', 'label');
				util.extendSelect('sa:report', schedulerActions.activeReportsDb, 'name', 'label');
			}
			else {
				// No profile selected
				util.populateSelect('sa:report', baseDb, 'name', 'label');
				disableSelect = true;
			}
			util.setF('sa:report', '');
			util.disableE('sa:report', disableSelect);
		},
		/*
		initOutputFormatType: function() {
			// alert('initOutputFormatType()');
			// PDF is only allowed if report is not "All Reports" and
			// if a report contains not more than one report element (due PDF performance)
			var reportName = util.getF('sa:report');
			// alert('reportName: ' + reportName);
			var isMultipleReportElements = false;
			var isAllReports = false;
			if (reportName != '') {
				if (reportName == '*') {
					isAllReports = true;
				}
				else {
					var reportItem = schedulerActions.activeReportsDb[util.h(reportName)];
					// util.showObject(reportItem);
					if (reportItem.numberOfReportElements > 1) {
						isMultipleReportElements = true;
					}
				}
			}
			var limitToHTMLOutput = (isAllReports || isMultipleReportElements);
			if (limitToHTMLOutput) {
				// Only HTML output format is allowed. Make sure that html is set!
				util.setF('sa:generate_report_files:output_format_type', 'html');
			}
			util.disableE('sa:generate_report_files:output_format_type', limitToHTMLOutput);
			schedulerActions.display.setOutputFormatType();
		},
		*/
		setOutputFormatType: function() {
			// Note, PDF is only allowed if report is not "All Reports" and
			// if a report contains not more than one report element (due PDF performance)
			var outputFormatType = util.getF('sa:generate_report_files:output_format_type');
			// alert('setOutputFormatType() - outputFormatType: ' + outputFormatType);
			var isHTML = (outputFormatType == 'html');
			util.hideE(['sa:generate_report_files_output_directory:group',
				'sa:generate_report_files_output_file:group',
				'sa:generate_report_files_pdf_not_supported:group'
			]);
			if (isHTML) {
				util.showE('sa:generate_report_files_output_directory:group');
			}
			else {
				var isPDFSupport = false;				
				var reportName = util.getF('sa:report');
				if (reportName == '') {
					// No report selected yet, assume we support it
					isPDFSupport = true;
				}
				else if (reportName == '*') {
					// 'All reports' selected, PDF output not supported
					isPDFSupport = false;
				}
				else {
					// Check number of report elements, only reports
					// with <= 1 report element are supported for PDF output.
					var reportItem = schedulerActions.activeReportsDb[util.h(reportName)];
					// util.showObject(reportItem);
					if (reportItem.numberOfReportElements <= 1) {
						isPDFSupport = true;
					}
				}
				util.showE('sa:generate_report_files_output_file:group', isPDFSupport);
				util.showE('sa:generate_report_files_pdf_not_supported:group', !isPDFSupport);
			}
		},
		updateDateFilterText: function() {
			// Updates the date filter text according activeDF
			var df = schedulerActions.activeDF;
			var type = df.type;
			var text = '';
			if (type == 'none') {
				text = langVar('lang_admin.scheduler.date_filter.entire_date_range');
			}
			else if (type == 'relative') {
				var prefix = df.relativePrefix;
				var dateUnit = df.relativeDateUnit;
				var dateUnitCount = df.relativeDateUnitCount;
				var param1Label = '';
				var param2Label = '';
				var dateUnitLabelsSingular = scheduler.dateUnitLabelsSingular;
				var dateUnitLabelsPlural = scheduler.dateUnitLabelsPlural;
				if (prefix != 'none') {
					/*
					text = 'Last ' + dateUnitCount + ' ' + dateUnit + 's ';
					if (prefix == 'recent') {
						text += 'including ' + dateUnit + ' of schedule execution date'; 
					}
					else {
						// prefix = last
						text += 'excluding ' + dateUnit + ' of schedule execution date';
					}
					*/
					if (prefix == 'recent') {
						// text = Last $param1 including $param2 of schedule execution date
						text = langVar('lang_admin.scheduler.date_filter.last_n_date_unit_including');
					}
					else {
						// text = Last $param1 excluding $param2 of schedule execution date
						text = langVar('lang_admin.scheduler.date_filter.last_n_date_unit_excluding');
					}
					param2Label = dateUnitLabelsSingular[dateUnit];
					text = text.replace(/__PARAM__2/, param2Label);
				}
				else {
					// text = Latest $param1 relative to end of log date
					text = langVar('lang_admin.scheduler.date_filter.latest_n_date_unit_relative');
				}
				// Handle param1Label
				if (dateUnitCount > 1) {
					param1Label = dateUnitCount + ' ' + dateUnitLabelsPlural[dateUnit];
				}
				else {
					param1Label = dateUnitCount + ' ' + dateUnitLabelsSingular[dateUnit];
				}
				text = text.replace(/__PARAM__1/, param1Label);
			}
			else {
				text = df.custom;
			}
			util.updateT('sa:date_filter_text', text);
		},
		removeDatabaseDataEvent: function() {
			// Invoked by radio button event
			var isCustomFilterExpression = util.getF('sa:remove_db_data_by_filter_expression');
			schedulerActions.display.setRemoveDatabaseData(isCustomFilterExpression);
		},
		setRemoveDatabaseData: function(isCustomFilterExpression) {
			util.disableE('sa:remove_db_data_number_of_days', isCustomFilterExpression);
			util.disableE('sa:remove_db_data_filter_expression', !isCustomFilterExpression);
		},
        setCvsExportNumberOfRowsToCustom: function() {
            // Makes custom radio button enabled when user types a row number
            util.setF('sa:export_csv_table:number_of_rows:custom', true);
        },
		toggleExtraOptions: function() {
			schedulerActions.display.setIsExtraOptions(!schedulerActions.isExtraOptions);
		},
		setIsExtraOptions: function(isExtraOptions) {
			schedulerActions.isExtraOptions = isExtraOptions;
			util.showE('sa:extra_options:row', isExtraOptions);
			var linkText = isExtraOptions ? langVar('lang_admin.scheduler.remove_extra_options') : langVar('lang_admin.scheduler.add_extra_options');
			util.updateT('sa:extra_options:toggle_btn', linkText);
		}
	},
	newItem: function() {
		// invoked from scheduler.js
		schedulerActions.mode = 'new';
		var panelLabel = langVar('lang_admin.scheduler.new_action');
		var newItemObj = {};
		schedulerActions.openPanel(panelLabel, newItemObj);
	},
	editItem: function(itemIndex, item) {
		// invoked from listcontrollerB
		schedulerActions.mode = 'edit';
		schedulerActions.itemIndex = itemIndex;
		var panelLabel = langVar('lang_admin.scheduler.edit_action');
		// alert('editItem with itemIndex: ' + itemIndex);
		schedulerActions.openPanel(panelLabel, item);
	},
	duplicateItem: function(itemIndex, item) {
		// invoked from listcontrollerB
		// itemIndex is the index of the item we duplicate!
		// alert('duplicateItem with itemIndex: ' + itemIndex);
		// util.showObject(item);
		schedulerActions.mode = 'duplicate';
		schedulerActions.itemIndex = itemIndex;
		var panelLabel = langVar('lang_admin.scheduler.edit_duplicated_action');
		schedulerActions.openPanel(panelLabel, item);
	},
	openPanel: function(panelLabel, item) {
		// Note, the item may be an empty object, in this case we simply
		// set the initial Action form state to "Select Action"
		var sa = schedulerActions;
		// util.showObject(item);
		var isEmpyItem = (item.type == null || item.type == '');
		// alert('openPanel: - isEmpyItem:' + isEmpyItem);
		//
		// Reset Action form
		//
		util.resetF('sa:form');
		sa.display.setActionByName('');
		sa.display.setRemoveDatabaseData(false);
		sa.display.setIsExtraOptions(false);
		//
		// Reset df
		//
		var dfObj = schedulerActions.activeDF;
		dfObj.type = 'none';
		dfObj.custom = '';
		dfObj.relativePrefix = 'recent'; // We set realtive default date to init the date picker panel
		dfObj.relativeDateUnit = 'year';
		dfObj.relativeDateUnitCount = 1;
		//
		// Reset email addresses
		//
		sa.addressControl.reset();
		//
		// Set data
		//
		if (!isEmpyItem) {
			sa.updateForm(item);
		}
		//
		// Update date filter text
		//
		sa.display.updateDateFilterText();
		//
		// Open the panel 
		//
		sa.fixPanel(true);
		sa.panel.open({label:panelLabel});
	},
	updateForm: function(item) {
		// alert('updateForm()');
		// util.showObject(item);
		var sa = schedulerActions;
		var action = item.type;
		sa.display.setActionByName(action);
		util.setF('sa:action', action);
        if (action != 'execute_command_line') {
            var profile = item.profile;
            var profileIsPattern = item.profile_is_pattern;
            // alert('updateForm() - action: ' + action);
            var isDatabaseAction = sa.isDatabaseAction;
            if (isDatabaseAction) {
                if (profileIsPattern) {
                    profile = '__PROFILE__PATTERN__';
                    util.showE('sa:profile_pattern:group', profileIsPattern);
                    util.setF('sa:profile_pattern', item.profile);
                }
                util.setF('sa:profile', profile);
                if (action == 'remove_database_data') {
                    var filterInfo = item.filter_info;
                    var isCustomFilterExpression = !filterInfo.number_of_days_active;
                    if (!isCustomFilterExpression) {
                        util.setF('sa:remove_db_data_by_number_of_days', true);
                        util.setF('sa:remove_db_data_number_of_days', filterInfo.number_of_days);
                    }
                    else {
                        util.setF('sa:remove_db_data_by_filter_expression', true);
                        util.setF('sa:remove_db_data_filter_expression', item.options.f);
                    }
                    sa.display.setRemoveDatabaseData(isCustomFilterExpression);
                }
            }
            else {
                //
                // Report action
                //
                util.setF('sa:profile', profile);
                var options = item.options;
                //
                // Set the report list
                //
                var reportName = options.rn;
                // Update the report list
                sa.display.setReportList(profile);
                util.setF('sa:report', reportName);
                //
                // Set Date filter
                //
                var dfObj = item.date_filter;
                // util.showObject(dfObj);
                var dateFilterType = dfObj.type;
                if (dateFilterType != 'none') {
                    // Set activeDF which is then manipulated by schedulerDF.js
                    var activeDfObj = schedulerActions.activeDF;
                    activeDfObj.type = dateFilterType;
                    if (dateFilterType == 'relative') {
                        activeDfObj.relativePrefix = dfObj.relative_prefix;
                        activeDfObj.relativeDateUnit = dfObj.relative_date_unit;
                        activeDfObj.relativeDateUnitCount = dfObj.relative_date_unit_count;
                    }
                    else {
                        activeDfObj.custom = dfObj.custom;
                    }
                }
                //
                // Set misc. report action options
                //
                if (action == 'export_csv_table') {
                    util.setF('sa:export_csv_table:output_file', options.of);
                    var endingRow = options.er;
                    var endingRowText = '';
                    if (endingRow == 0) {
                        util.setF('sa:export_csv_table:number_of_rows:default', true);
                    }
                    else if (endingRow == -1) {
                        // All rows
                        util.setF('sa:export_csv_table:number_of_rows:all', true);
                    }
                    else {
                        // Custom row number
                        endingRowText = endingRow;
                        util.setF('sa:export_csv_table:number_of_rows:custom', true);
                    }
                    util.setF('sa:export_csv_table:number_of_rows', endingRowText);
                }
                else if (action == 'generate_report_files') {
                    var outputFormatType = options.oft;
                    if (outputFormatType == 'html') {
                        // HTML
                        util.setF('sa:generate_report_files:output_directory', options.od);
                    }
                    else {
                        // PDF
                        util.setF('sa:generate_report_files:output_file', options.of);
                    }
                    util.setF('sa:generate_report_files:output_format_type', outputFormatType);
                    // sa.display.initOutputFormatType();
                    sa.display.setOutputFormatType();
                }
                else {
                    // send report_by_email
                    util.setF('sa:return_address', options.rna);
                    var emailAddresses = schedulerUtil.actionsEmailToEmailAddressesObject(options);
                    sa.addressControl.init(emailAddresses);
                    // util.setF('sa:recipient_address', options.rca);
                    // util.setF('sa:cc_address', options.ca);
                    // util.setF('sa:bcc_address', options.ba);
                    util.setF('sa:report_email_subject', options.res);
                }
                // Set language
                util.setF('sa:report_language', item.language);
            }
            if (item.extra_options != '') {
                sa.display.setIsExtraOptions(true);
                util.setF('sa:extra_options', item.extra_options);
            }
        }
        else {
            // execute_command_line
            util.setF('sa:execute_command_line:command_line', item.command_line);
        }
	},
	saveItem: function() {
		// We save the item object back to the items array in listcontrollerB
		// alert('saveItem()');
		var sa = schedulerActions;
		var validator = sa.validator;
		var action = sa.activeAction;
		var o = {};
        var isValidEmailAddresses = true;
		if (action != '') {
			o.type = action;
            if (action != 'execute_command_line') {
                var profileIsPattern = false;
                o.profile = validator.isValue('sa:profile');
                if (o.profile == '__PROFILE__PATTERN__') {
                    o.profile = validator.isValue('sa:profile_pattern');
                    profileIsPattern = true;
                }
                o.profile_is_pattern = profileIsPattern;
                if (sa.isDatabaseAction) {
                    if (action == 'remove_database_data') {
                        var number_of_days_active = util.getF('sa:remove_db_data_by_number_of_days');
                        var number_of_days = 0;
                        var f_option;
                        if (number_of_days_active) {
                            // number of days
                            number_of_days = validator.isInteger('sa:remove_db_data_number_of_days', 1);
                            f_option = '(date_time < now() - 60*60*24*' + number_of_days + ')';
                        }
                        else {
                            // custom filter expression
                            f_option = validator.isValue('sa:remove_db_data_filter_expression');
                        }
                        o.options = {f: f_option};
                        o.filter_info = {
                            number_of_days_active: number_of_days_active,
                            number_of_days: number_of_days
                        }
                    }
                }
                else {
                    //
                    // This is a report action
                    //
                    o.options = {};
                    var reportName = validator.isValue('sa:report');
                    o.options.rn = reportName;
                    //
                    // Get the date filter
                    //
                    var activeDF = schedulerActions.activeDF;
                    o.date_filter = {};
                    o.date_filter.type = activeDF.type;
                    o.date_filter.custom = activeDF.custom;
                    o.date_filter.relative_prefix = activeDF.relativePrefix;
                    o.date_filter.relative_date_unit = activeDF.relativeDateUnit;
                    o.date_filter.relative_date_unit_count = activeDF.relativeDateUnitCount;
                    //
                    // Handle misc. options
                    //
                    switch (action) {
                        case 'export_csv_table':
                            o.options.of = validator.isValue('sa:export_csv_table:output_file');
                            // Handle ending row
                            var endingRow;
                            if (util.getF('sa:export_csv_table:number_of_rows:default')) {
                                // default, use number of rows of report element
                                endingRow = 0;
                            }
                            else if (util.getF('sa:export_csv_table:number_of_rows:all')) {
                                // all rows
                                endingRow = -1;
                            }
                            else {
                                endingRow = validator.isInteger('sa:export_csv_table:number_of_rows', 1);
                            }
                            o.options.er = endingRow;
                            break;
                        case 'generate_report_files':
                            var ouputFormatType = util.getF('sa:generate_report_files:output_format_type');
                            o.options.oft = ouputFormatType;
                            if (ouputFormatType == 'html') {
                                o.options.od = validator.isValue('sa:generate_report_files:output_directory');
                            }
                            else {
                                // Output is PDF
                                // Make sure we throw a hidden error if output format is PDF but the selected report
                                // doesn't support it. Note, the error message is already displayed but we need to
                                // prevent the user from saving the form!
                                // We simply reset the 'sa:generate_report_files:output_file' field, which will fire an error anyway!
                                if (reportName != '') {
                                    isPDFOutputSupport = true;
                                    if (reportName == '*') {
                                        isPDFOutputSupport = false;
                                    }
                                    else {
                                        var reportItem = schedulerActions.activeReportsDb[util.h(reportName)];
                                        isPDFOutputSupport = (reportItem.numberOfReportElements <= 1);
                                    }
                                    if (!isPDFOutputSupport) {
                                        // Reset the 'sa:generate_report_files:output_file' so that it throws an error and avoids saving the action
                                        util.setF('sa:generate_report_files:output_file', '');
                                    }
                                }
                                o.options.of = validator.isValue('sa:generate_report_files:output_file');
                            }
                            break;
                        case 'send_report_by_email':
                            o.options.rna = validator.isValue('sa:return_address');
                            isValidEmailAddresses = sa.addressControl.validate();
                            if (isValidEmailAddresses) {
                                var addressControlAddresses = sa.addressControl.getAddresses();
                                var rca = ''; // To
                                var ca = ''; // Cc
                                var ba = ''; // Bcc
                                for (var i = 0; i < addressControlAddresses.length; i++) {
                                    var addressItem = addressControlAddresses[i];
                                    var address = addressItem.address;
                                    if (addressItem.type == 'to') {
                                        rca += address + ', ';
                                    }
                                    else if (addressItem.type == 'cc') {
                                        ca += address + ', ';
                                    }
                                    else {
                                        ba += address + ', ';
                                    }
                                }
                                // Add only defined addresses to options!
                                if (rca != '') {
                                    o.options.rca = rca.replace(/(,\s)$/, '');
                                }
                                if (ca != '') {
                                    o.options.ca = ca.replace(/(,\s)$/, '');
                                }
                                if (ba != '') {
                                    o.options.ba = ba.replace(/(,\s)$/, '');
                                }
                                // util.showObject(o.options);
                            }
                            o.options.res = validator.isValue('sa:report_email_subject');
                            break;
                    }
                    o.language = util.getF('sa:report_language');
                }
                // get extra options
                o.extra_options = sa.isExtraOptions ? util.getF('sa:extra_options') : '';
            }
            else {
                // validate execute_command_line action
                o.command_line = validator.isValue('sa:execute_command_line:command_line');
            }
		}
		else {
			o.action = validator.isValue('sa:action');
		}
		if (validator.allValid() && isValidEmailAddresses) {
			// alert('all valid, save the item');
			// Save the item to the actions list (listcontrollerB handles save of the object)
			var sa = schedulerActions;
			schedulerActionsList.list.saveItem(sa.mode, sa.itemIndex, o);
			sa.closePanel();
			// Check SMTP server warning
			handleSmtpServerWarning();
		}
	},
	closePanel: function() {
		var sa = schedulerActions;
		sa.validator.reset();
		sa.panel.close();
		sa.fixPanel(false);
	},
	fixPanel: function(isFix) {
		// IE 6 fix, show/hide select boxes in Scheduler form!
		util.showEV('schedule_date_time:tr', !isFix);
		// Firefox fix, change overlow property of underlying actions list to hidden (otherwise form field focus becomes lost!)
		var element = util.getE('scheduler:actions_list:div');
		element.style.overflow = isFix ? 'hidden' : 'scroll';
	},
	getReportLabel: function(profile, reportName) {
		var profilesDb = scheduler.profilesDb;
		var profileItem = profilesDb[util.h(profile)];
		var reportLabel = '';
		if (profileItem != null) {
			var reportNames = profileItem.rN;
			var reportLabels = profileItem.rL;
			var reportNamesDb = scheduler.reportNamesDb;
			var reportLabelsDb =  scheduler.reportLabelsDb;
			// We need the index of the reportName in rN of the profile
			for (var i = 0; i < reportNames.length; i++) {
				var reportNameIndex = reportNames[i];
				if (reportNamesDb[reportNameIndex] == reportName) {
					var reportLabelIndex = reportLabels[i];
					reportLabel = reportLabelsDb[reportLabelIndex];
				}
			}
		}
		if (reportLabel == '') {
			reportLabel = langVar('lang_admin.scheduler.unknown_report');
		}
		return reportLabel;
	},
	labelBuilder: function(actionItem) {
		// Invoked from listcontrollerB
		// Returns a text label with HTML markup
		// util.showObject(actionItem);
		var action = actionItem.type;
		var label = scheduler.actionsDb[util.h(action)].label;
        var finalLabel = '';
        var subLabel = '(';
		//
		// Create the label which is displayed within the span element
		//
        if (action != 'execute_command_line') {
            var profilesDb = scheduler.profilesDb;
            var profileItem = null;
            var profile = actionItem.profile;
            var profileLabel = '';
            if (profile != '*' && !actionItem.profile_is_pattern) {
                profileItem = profilesDb[util.h(profile)];
                profileLabel = (profileItem != null) ? profileItem.label : langVar('lang_admin.scheduler.unknown_profile');
            }
            else {
                if (!actionItem.profile_is_pattern) {
                    profileLabel = langVar('lang_admin.scheduler.all_profiles_label');
                }
                else {
                    profileLabel = profile;
                }
            }
            subLabel += profileLabel + ', ';
            if (action == 'export_csv_table' ||
                action == 'generate_report_files' ||
                action == 'send_report_by_email') {
                var reportName = actionItem.options.rn;
                var reportLabel = schedulerActions.getReportLabel(profile, reportName);
                subLabel += reportLabel + ', ';
                // KHP-RC, show date filter label if any
                if (action == 'send_report_by_email') {
                    var recipientAddress = actionItem.options.rca;
                    subLabel += recipientAddress + ', ';
                }
            }
            subLabel = subLabel.replace(/(,\s)$/, ')');
            finalLabel = label + ' <span>' + subLabel + '</span>';
        }
        else {
            // execute_command_line
            // TODO - check if we should add a part of the command_line as sublabel
            finalLabel = label;
        }
		return finalLabel;
	}
}
//
// schedulerDF.js (Scheduler date filter
//
var schedulerDF = {
	panel: null,
	validator: null,
	init: function() {
		var YE = YAHOO.util.Event;
		var panelObj = {
			panelId: 'sdf:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: langVar('lang_admin.scheduler.date_filter.edit_report_date'),
			left: 400,
			top: 190,
			zIndex: 40,
			isCover: true,
			isSticky: true,
			closeEvent: schedulerDF.closePanel
		};
		schedulerDF.panel = new util.Panel3(panelObj);
		schedulerDF.validator = new util.Validator();
		YE.addListener('sdf:meta_type', 'change', schedulerDF.setDateFilterTypeByEvent);
		YE.addListener('sdf:relative_date_unit', 'change', schedulerDF.setDateUnitByEvent);
		YE.addListener('sdf:okay_btn', 'click', schedulerDF.save);
		YE.addListener('sdf:cancel_btn', 'click', schedulerDF.closePanel);
	},
	open: function() {
		// schedulerActions.cover.setZIndex(30);
		// Reset form fields
		util.setF('sdf:relative_date_unit_count', 1);
		util.setF('sdf:relative_date_unit', 'year');
		util.setF('sdf:relative_prefix', 'recent');
		util.setF('sdf:custom_date_filter', '');
		schedulerDF.updateForm();
		schedulerDF.panel.open();
	},
	setDateFilterTypeByEvent: function() {
		schedulerDF.setDateFilterTypeByName(this.value);
	},
	setDateFilterTypeByName: function(metaType) {
		// dateType is:  none | relative_with_prefix | relative_without_prefix | custom
		schedulerDF.validator.reset();
		util.setF('sdf:meta_type', metaType);
		util.hideE(['sdf:group:none', 'sdf:group:relative', 'sdf:group:custom']);
		if (metaType == 'none') {
			util.showE('sdf:group:none');
		}
		else if (metaType == 'custom') {
			util.showE('sdf:group:custom');
		}
		else {
			var isWithPrefix = (metaType == 'relative_with_prefix');
			util.showE('sdf:relative_with_prefix:label', isWithPrefix);
			util.showE('sdf:relative_without_prefix:label', !isWithPrefix);
			util.showEV('sdf:relative_prefix', isWithPrefix);
			util.enableE('sdf:relative_prefix', isWithPrefix);
			if (isWithPrefix) {
				// update date unit in sdf:relative_prefix element
				schedulerDF.updateRelativePrefixDateUnit();
			}
			util.showE('sdf:group:relative');
		}
	},
	setDateUnitByEvent: function() {
		if (util.getF('sdf:meta_type') == 'relative_with_prefix') {
			schedulerDF.updateRelativePrefixDateUnit();
		}
	},
	updateRelativePrefixDateUnit: function() {
		// Updates the date unit in
		// 'including __DATE__UNIT__ of schedule date' and
		// 'excluding __DATE__UNIT__ of schedule date'
		var dateUnit = util.getF('sdf:relative_date_unit');
		var dateUnitLabelsSingular = scheduler.dateUnitLabelsSingular;
		var dateUnitLabel = dateUnitLabelsSingular[dateUnit];
		// 'including __DATE__UNIT__ of schedule execution date';
		var recentPrefixText = langVar('lang_admin.scheduler.date_filter.including_date_unit_of_exec');
		// 'excluding __DATE__UNIT__ of schedule execution date'
		var lastPrefixText = langVar('lang_admin.scheduler.date_filter.excluding_date_unit_of_exec');
		recentPrefixText = recentPrefixText.replace(/__PARAM__1/, dateUnitLabel);
		lastPrefixText = lastPrefixText.replace(/__PARAM__1/, dateUnitLabel);
		var list = [{name:'recent', label:recentPrefixText}, {name:'last', label:lastPrefixText}];
		// Get activePrefix which is set again after we re-populate the list 
		var activePrefix = util.getF('sdf:relative_prefix');
		util.populateSelect('sdf:relative_prefix', list, 'name', 'label');
		util.setF('sdf:relative_prefix', activePrefix);
	},
	updateForm: function() {
		var activeDF = schedulerActions.activeDF;
		// util.showObject(activeDF);
		var type = activeDF.type;
		var metaType = '';
		if (type == 'none') {
			metaType = type;
		}
		else if (type == 'relative') {
			var isPrefix = (activeDF.relativePrefix != 'none');
			var metaType = isPrefix ? 'relative_with_prefix' : 'relative_without_prefix';
			util.setF('sdf:relative_date_unit_count', activeDF.relativeDateUnitCount);
			util.setF('sdf:relative_date_unit', activeDF.relativeDateUnit);
			if (isPrefix) {
				util.setF('sdf:relative_prefix', activeDF.relativePrefix);
			}
		}
		else {
			// custom
			metaType = type;
			util.setF('sdf:custom_date_filter', activeDF.custom);
		}
		schedulerDF.setDateFilterTypeByName(metaType);
	},
	closePanel: function() {
		schedulerDF.panel.close();
		// schedulerActions.cover.setZIndex(10);
	},
	save: function() {
		var validator = schedulerDF.validator;
		validator.reset();
		var type = '';
		var custom = '';
		var relativePrefix = '';
		var relativeDateUnit = '';
		var relativeDateUnitCount = '1';
		var metaType = util.getF('sdf:meta_type');
		if (metaType == 'none') {
			type = metaType;
		}
		else if (metaType == 'custom') {
			type = metaType;
			var custom = validator.isValue('sdf:custom_date_filter');
		}
		else {
			// Relative
			var isPrefix = (metaType == 'relative_with_prefix');
			// alert('isPrefix: ' + isPrefix);
			type = 'relative';
			relativePrefix = isPrefix ? util.getF('sdf:relative_prefix') : 'none';
			relativeDateUnit = util.getF('sdf:relative_date_unit');
			relativeDateUnitCount = validator.isInteger('sdf:relative_date_unit_count', 1);
		}
		if (validator.allValid()) {
			var activeDF = schedulerActions.activeDF;
			activeDF.type = type;
			activeDF.custom = custom;
			activeDF.relativePrefix = relativePrefix;
			activeDF.relativeDateUnit = relativeDateUnit;
			activeDF.relativeDateUnitCount = relativeDateUnitCount;
			// util.showObject(activeDF);
			schedulerDF.closePanel();
			// Update date text in Actions form
			schedulerActions.display.updateDateFilterText();
		}
	}
}
//
// schedulerRun.js (Scheduler - Run Now)
//
//
// Note, any action added in Run Now keeps its state as of the time it has been added.
// If the same action is edited in the Actions from right afterwards and the action
// has not yet been run it would not affect the action in Run Now!
var schedulerRun = {
	panel: null,
	isOpenPanel: false,
	idCount: 0, // Used for element baseIds
	schedules: [], // Keeps schedule objects to be run, i.e.
	/*
		{
		id: 's0', 			--> ID issued in schedulerRun
		name: 'schedule A', --> The name we got from the schedule form
		actions: [] 		--> A clone of the actions at the time the user clicked Run Now
		},
		...
		Actions rows and elements contain a combined id as follows:
		scheduleId:actionId:..., i.e.:
		s0:i0:...
	*/
	basketId: '', // Used to check sync for the current basket array between client and server
	basket: [], // A flat array with schedule and action references to schedules array,
						// it is populated upon start()
	isRunning: false,
	isCancel: false,
	isComplete: false,
	runId: '', // Used for responses to check if the response is in sync with server per action
	runningBasketIndex: 0, // The index of the running action in the basket index.
							// Used upon cancel when schedule is running to indicate Canceled state
	// runningScheduleId: '',
	// runningActionId: '',
	// TempTimeMeasurement
	// startTime: 0,
	init: function() {
		var YE = YAHOO.util.Event;
		//
		// Init Run Now panel
		//
		var panelObj = {
			panelId: 'scheduler_run_now:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: 'Run Now',
			right: 30,
			top: 5,
			zIndex: 5,
			isCover: false,
			isSticky: true,
			closeEvent: schedulerRun.cancel
		};
		schedulerRun.panel = new util.Panel3(panelObj);
		// Add single event to list element for remove items
		YE.addListener('scheduler_run_now:list:table', 'click', schedulerRun.removeActionItem);
		YE.addListener('scheduler_run_now:start_btn', 'click', schedulerRun.start);
		YE.addListener('scheduler_run_now:cancel_btn', 'click', schedulerRun.cancel);
	},
	setPanelState: function(panelState) {
		// panelState is: empty | start | running | complete
		var listLabel = '';
		var cancelBtnLabel = '';
		var isEnabledStartBtn = false;
		switch (panelState) {
			case 'start':
				listLabel = langVar('lang_admin.scheduler.run.click_start_to_run_schedules');
				cancelBtnLabel = langVar('lang_stats.btn.cancel');
				isEnabledStartBtn = true;
				break;
			case 'running':
				listLabel = langVar('lang_admin.scheduler.run.running_schedules');
				cancelBtnLabel = langVar('lang_stats.btn.cancel');
				isEnabledStartBtn = false;
				break;
			case 'complete':
				listLabel = langVar('lang_admin.scheduler.run.run_schedule_completed');
				cancelBtnLabel = langVar('lang_stats.btn.close');
				isEnabledStartBtn = false;
				break;
			case 'cancel': 
				// Canceled while an action is running
				listLabel = langVar('lang_admin.scheduler.run.canceled_schedule_info');
				cancelBtnLabel = langVar('lang_stats.btn.close');
				isEnabledStartBtn = false;
				break;
		}
		util.updateT('scheduler_run_now:list:label', listLabel);
		util.enableE('scheduler_run_now:start_btn', isEnabledStartBtn);
		// var cancelBtn = util.getE('scheduler_run_now:cancel_btn');
		// cancelBtn.value = cancelBtnLabel;
		util.updateT('scheduler_run_now:cancel_btn', cancelBtnLabel);
	},
	reset: function() {
		// Reset items
		var sr = schedulerRun;
		sr.idCount = 0;
		sr.schedules = [];
		sr.basket = [];
		sr.isRunning = false;
		sr.isCancel = false;
		sr.isComplete = false;
		// Clear list items
		util.removeChildElements('scheduler_run_now:list:table');
	},
	cancel: function() {
		// Invoked upon cancel or close
		if (schedulerRun.isRunning) {
			// Cancel without to close the panel
			// alert('Schedule is still running. The current running action will be completed in the background.');
			schedulerRun.isCancel = true;
			schedulerRun.setPanelState('cancel');
			// Indicated remaining actions as canceled
			var runningBasketIndex = schedulerRun.runningBasketIndex;
			var basket = schedulerRun.basket;
			for (var i = (runningBasketIndex + 1); i < basket.length; i++) {
				var baseId = basket[i].baseId;
				var stateElement = util.getE(baseId + ':state');
				util.removeChildElements(stateElement);
				var stateText = util.createT(langVar('lang_admin.scheduler.run.canceled'));
				stateElement.appendChild(stateText);
			}
		}
		else {
			// Close the panel
			schedulerRun.reset();
			schedulerRun.setPanelState('start');
			schedulerRun.panel.close();
			enableRunNowButton();
			schedulerRun.isOpenPanel = false;
		}
	},
	runNow: function() {
		// Note, runNow adds items to be run but does not run them yet!
		// Check if we have any actions defined
		var numOfActions = schedulerActionsList.list.getNumberOfItems();
		if (numOfActions > 0) {
			if (!schedulerRun.panel) {
				schedulerRun.init();
			}
			// Check if we need to reset an open panel
			if (schedulerRun.isOpenPanel && (schedulerRun.isCancel || schedulerRun.isComplete)) {
				// Panel is open with a completed or cancled list and 
				// the user adds a new schedule. So we start over.
				schedulerRun.reset();
				schedulerRun.setPanelState('start');
			}
			// alert('runNow() - numOfActions: ' + numOfActions);
			// util.disableE('scheduler:run_now_btn');
			disableRunNowButton();
			//
			// Create schedule object
			//
			var scheduleId = schedulerRun.getNewBaseId();
			var scheduleName = util.getF('scheduler:label');
			if (scheduleName == '') {
				scheduleName = langVar('lang_admin.scheduler.run.schedule');
			}
			var actions = schedulerActionsList.list.getItemsClone();
			// Add an id to each scheduler action (We use the id for element creation and deletion)
			for (var i = 0; i < actions.length; i++) {
				actions[i].id = 'i' + i;
			}
			var obj = {};
			obj.id = scheduleId;
			obj.name = scheduleName;
			obj.actions = actions;
			// Add schedule object to schedules
			var schedules = schedulerRun.schedules;
			schedules[schedules.length] = obj;
			// Build schedule object in list
			schedulerRun.addScheduleToList(scheduleId, scheduleName, actions);
			// util.showObject(actions);
			if (!schedulerRun.isOpenPanel) {
				schedulerRun.setPanelState('start');
				schedulerRun.panel.open();
				schedulerRun.isOpenPanel = true;
			}
		}
		else {
			alert(langVar('lang_admin.scheduler.run.no_action_defined_msg'));
		}
	},
	start: function() {
		// Start to run the schedule(s)
		// Check if there are any actions in the list as all could have been removed!
		// Create basket array with schedule and action references for simple sequential processing
		// Hide the remove buttons in schedule now list
		var schedules = schedulerRun.schedules;
		schedulerRun.basket = [];
		var basket = schedulerRun.basket;
		for (var i = 0; i < schedules.length; i++) {
			var scheduleId = schedules[i].id;
			var actions = schedules[i].actions;
			for (var j = 0; j < actions.length; j++) {
				var actionId = actions[j].id;
				var baseId = scheduleId + ':' + actionId;
				basket[basket.length] = {baseId:baseId, action:actions[j]};
				util.hideEV(baseId + ':remove');
			}
		}
		if (basket.length > 0) {
			// Issue new basketId
			var theDate = new Date();
			var basketId = 'basket_' + theDate.getTime();
			schedulerRun.basketId = basketId;
			disableRunNowButton();
			schedulerRun.isRunning = true;
			// schedulerRun.runId = schedulerRun.runId + 1;
			// update panel
			schedulerRun.setPanelState('running');
			schedulerRun.runScheduleAction(0);
		}
		else {
			alert('lang_admin.scheduler.run.no_schedule_defined_msg');
		}
	},
	runScheduleAction: function(basketIndex) {
		schedulerRun.runningBasketIndex = basketIndex;
		// TempTimeMeasurement
		/*
		var startDate = new Date();
		schedulerRun.startTime = startDate.getTime();
		*/
		var basketItem = schedulerRun.basket[basketIndex];
		var baseId = basketItem.baseId;
		var action = basketItem.action;
		//
		// Handle state cell
		//
		var stateElement = util.getE(baseId + ':state');
		util.removeChildElements(stateElement);
		var stateText = util.createT(langVar('lang_admin.scheduler.run.running'));
		stateElement.appendChild(stateText);
		// Issue a new runId
		var theDate = new Date();
		var runId = 'run_' + theDate.getTime();
		schedulerRun.runId = runId;
		// util.showObject(action);
		var url = '?dp=admin_pages.scheduler.run_schedule_start';
		// Assemble URL dat
var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
		dat += 'v.fp.run_id=' + runId + '&';
		dat += 'v.fp.basket_id=' + schedulerRun.basketId + '&';
		dat += 'v.fp.basket_index=' + basketIndex + '&';
		var path = 'v.fp.schedule';
		dat += path + '.disabled=false&';
		dat += path + '.label=&';
		dat += path + '.month=*&';
		dat += path + '.day=*&';
		dat += path + '.hour=0&';
		dat += path + '.minute=0&';
		// Add the action
		var actionPath = path + '.actions.0';
		dat += getActionItemDat(actionPath, action); // called in scheduler.js
		dat = dat.replace(/&$/, '');
		util.serverPost(url, dat);
	},
	runScheduleActionResponse: function(dat) {
		/*
		alert('runScheduleActionResponse()');
		util.showObject(dat);
		return false;
		*/
		if (dat.runId == schedulerRun.runId && dat.basketId == schedulerRun.basketId) {
			schedulerRun.runScheduleProgress();
		}
	},
	runScheduleProgress: function() {
		var url = '?dp=admin_pages.scheduler.run_schedule_progress';
		// Checks if the current action completed
var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
		dat += 'v.fp.run_id=' + schedulerRun.runId;
		util.serverPost(url, dat);
	},
	runScheduleProgressResponse: function(dat) {
		// alert('runScheduleProgressResponse()');
		// util.showObject(dat);
		// return false;
		if (dat.runId == schedulerRun.runId && dat.basketId == schedulerRun.basketId) {
			// var isComplete = (dat.isComplete == true);
			if (dat.isComplete) {
				// TempTimeMeasurement
				/*
				var endDate = new Date();
				var endTime = endDate.getTime();
				var startTime = schedulerRun.startTime;
				var deltaTime = endTime - startTime;
				var seconds = deltaTime / 1000;
				var minutes = seconds / 60;
				var msg = 'The action took\nseconds: ' + seconds + '\nminutes: ' +  minutes;
				alert(msg);
				*/
				// processed item data
				var basketIndex = dat.basketIndex;
				var isError = dat.isError;
				var stateText = !isError ? langVar('lang_admin.scheduler.run.complete') : langVar('lang_admin.scheduler.run.error');
				var basket = schedulerRun.basket;
				var baseId = basket[basketIndex].baseId;
				util.updateT(baseId + ':state', stateText);
				if (isError) {
					schedulerRun.handleActionErrorDisplay(baseId, dat.errorId);
				}
				//
				// Handle next basket item
				//
				if (!schedulerRun.isCancel) {
					nextBasketIndex = basketIndex + 1;
					if (nextBasketIndex < basket.length) {
						schedulerRun.runScheduleAction(nextBasketIndex);
					}
					else {
						// All actions have been run. Set completed state
						schedulerRun.setPanelState('complete');
						schedulerRun.isRunning = false;
						schedulerRun.isComplete = true;
						enableRunNowButton();
					}
				}
				else {
					// This must be the running action upon cancel
					// We just set isRunning
					schedulerRun.isRunning = false;
				}
			}
			else {
				// Continue progress calls
				setTimeout('schedulerRun.runScheduleProgress()', 1500);
			}
		}
	},
	handleActionErrorDisplay: function(baseId, errorId) {
		// Change error state column to Red
		var tdState = util.getE(baseId + ':state');
		tdState.style.color = 'Red';
		// Add error message to label cell
		var tdLabel = util.getE(baseId + ':label');
		var br = util.createE('br');
		var span = util.createE('span', {color:'Red'});
		var spanText = util.createT(langVar('lang_admin.scheduler.run.error_occured_msg') + ' ');
		var href = '?dp=alert&ei=' + errorId;
		var a = util.createE('a', {href:href, target:'_blank'});
		var aText = util.createT(langVar('lang_admin.scheduler.run.click_here_to_view_alert'));
		util.chainE([tdLabel, br, [span, spanText], [a, aText]]);
	},
	getNewBaseId: function() {
		var i = schedulerRun.idCount;
		var id = 'srn_' + i;
		schedulerRun.idCount = i + 1;
		return id;
	},
	//
	//
	// List handling functions
	//
	//
	removeActionItem: function(evt) {
		// Invoked upon list click, we have to check if the user actually clicked a link
		var element = evt.target || evt.srcElement;
		var elementId = element.id;
		if (elementId != '' && (elementId.indexOf(':remove') != -1)) {
			var dat = elementId.split(':');
			var scheduleId = dat[0];
			var actionId = dat[1];
			var schedules = schedulerRun.schedules;
			for (var i = 0; i < schedules.length; i++) {
				var theSchedule = schedules[i];
				if (theSchedule.id == scheduleId) {
					var actions = theSchedule.actions;
					var tbody = util.getE(scheduleId + ':tbody');
					if (actions.length > 1) {
						// Remove action from actions array
						for (var j = 0; j < actions.length; j++) {
							if (actions[j].id == actionId) {
								actions.splice(j, 1);
								break;
							}
						}
						// Remove action row
						var tr = util.getE(scheduleId + ':' + actionId + ':tr');
						tbody.removeChild(tr);					
					}
					else {
						// There is only one action, so we remove the entire schedule
						// Remove schedule from schedules array
						schedules.splice(i, 1);
						// Remove schedule from list
						var table = util.getE('scheduler_run_now:list:table');
						table.removeChild(tbody);
					}
				}
			}
		}
	},
	addScheduleToList: function(scheduleId, scheduleName, actions) {
		var table = util.getE('scheduler_run_now:list:table');
		var tbody = util.createE('tbody', {id:scheduleId + ':tbody'});
		// Build header row
		var tr = util.createE('tr', {className:'header'});
		var tdLabel = util.createE('td', {colSpan:3, className:'header'});
		// var tdState = util.createE('td');
		var text = util.createT(scheduleName);
		util.chainE(tbody, [tr, [tdLabel, text]]);
		// Add the action rows
		for (var i = 0; i < actions.length; i++) {
			schedulerRun.addActionRow(tbody, scheduleId, i, actions[i]);
		}
		util.chainE(table, tbody);
	},
	addActionRow: function(tbody, scheduleId, actionIndex, action) {
		actionBaseId = scheduleId + ':' + action.id;
		var tr = util.createE('tr', {id:actionBaseId + ':tr'});
		var th = util.createE('th');
		var thText = util.createT(actionIndex + 1);
		var tdLabel = util.createE('td', {id:actionBaseId + ':label'});
		// Note, the label we get from labelBuilder is a string whcih may
		// contain HTML tags, so we use innerHTML to display the label.
		var label = schedulerActions.labelBuilder(action);
		tdLabel.innerHTML = label;
		// var tdLabelText = util.createT(labelObj.label);
		// util.chainE(tdLabel, tdLabelText);
		/*
		if (labelObj.subLabel != '') {
			var span = util.createE('span', {className:'label'});
			var spanText = util.createT(' ' + labelObj.subLabel);
			util.chainE(tdLabel, span, spanText);
		}
		*/
		var tdState = util.createE('td', {id:actionBaseId + ':state', className:'state'});
		var a = util.createE('a', {id:actionBaseId + ':remove', href:'javascript:;'});
		var aText = util.createT(langVar('lang_stats.btn.remove'));
		util.chainE(tbody, [tr, [th, thText], tdLabel, [tdState, [a, aText]]]);
	}
}
//
// scheduler.js
//
// Note, the pageInfo.section is set to "scheduler" after initialization has been done.
// This ensure that adminConfig.js doesn't check for changes upon exit unless
// everthing is initialized!
/*
DEFINED IN ADMIN MAIN PAGE!
Used by adminConfig.js
var pageInfo = {};
pageInfo.page = '';
pageInfo.saveActive = false;
pageInfo.exitActive = false;
*/
var scheduler = {
	smtpServerIsDefined: false, // Specifies if an SMTP server is defined in preferences. We use this setting to show a warning message
								// if a send_report_by_email action is defined but no SMTP server has been defined yet.
	activeUserNodeName: '',
	theList: null,
	usersDb: [], // [{name:"user_1", username:"user 1"}, ...] contains user node name with username, required for the list labels where we indicate "(by user 1)"
	actionsDb: [], 	// Contains the action names, labels and profile access rights if there are no unlimited grants, format is:
					// [{name:"send_report_by_email", label:"Send report by email", all_profiles:false, profiles:['profile_1', profile_2]}, ...]
	itemsDb: [],  // items work array
	itemsDbBackup: [], // items array in the state when first loaded or last saved, it is used upon Undo Changes
	profilesDb: [],
	reportNamesDb: [],
	reportLabelsDb: [],
	validator: null,
	// mainButtons;
	saveChangesBtn: null,
	newItemBtn: null,
	deleteBtn: null,
	duplicateBtn: null,
	undoAllChangesBtn: null,
	noItemFormIsActive: false,
	actionsMoveControlIsPositioned: false,
	activeCreatedByUser: '', // keeps the active createdByUser value because it is not a form field
	dateUnitLabelsSingular: {
		year: langVar('lang_admin.scheduler.date_filter.year'),
		quarter: langVar('lang_admin.scheduler.date_filter.quarter'),
		month: langVar('lang_admin.scheduler.date_filter.month'),
		week: langVar('lang_admin.scheduler.date_filter.week'),
		day: langVar('lang_admin.scheduler.date_filter.day')
	},
	dateUnitLabelsPlural: {
		year: langVar('lang_admin.scheduler.date_filter.years'),
		quarter: langVar('lang_admin.scheduler.date_filter.quarters'),
		month: langVar('lang_admin.scheduler.date_filter.months'),
		week: langVar('lang_admin.scheduler.date_filter.weeks'),
		day: langVar('lang_admin.scheduler.date_filter.days')
	}
}
function init() {
	var YE = YAHOO.util.Event;
	scheduler.validator = new util.Validator();
	//
	// init toolbar buttons and form controls
	//
	scheduler.saveChangesBtn = new util.ToolbarButton('save_changes', saveChanges, toolbarButtonsDb);
	scheduler.newItemBtn = new util.ToolbarButton('new_schedule', newItem, toolbarButtonsDb);
	scheduler.duplicateBtn = new util.ToolbarButton('duplicate', duplicateItem, toolbarButtonsDb);
	scheduler.deleteBtn = new util.ToolbarButton('delete', deleteItem, toolbarButtonsDb);
	scheduler.undoAllChangesBtn = new util.ToolbarButton('undo_all_changes', undoAllChanges, toolbarButtonsDb);
	YE.addListener('scheduler:label', 'keyup', updateListAndFormLabel);
	YE.addListener('scheduler:run_now_btn', 'click', schedulerRun.runNow);
	YE.addListener('scheduler:new_action_btn', 'click', schedulerActions.newItem);
	//
	// Ignore/Disable buttons according RBAC
	//
	var permissions = pageInfo.permissions;
	if (permissions.isEdit) {
		if (!permissions.isAdd) {
			scheduler.newItemBtn.disableAndIgnore();
			scheduler.duplicateBtn.disableAndIgnore();
		}
		if (!permissions.isDelete) {
			scheduler.deleteBtn.disableAndIgnore();
		}
		// Register isModifiedPageHandler in adminConfig.js
		// (We don't check for modifications if there is no edit 
		// permission because there is no Save button anyway!)
		adminConfig.getIsModifiedPageHandler = getIsModifiedPage;
	}
	else {
		scheduler.saveChangesBtn.disableAndIgnore();
		scheduler.newItemBtn.disableAndIgnore();
		scheduler.duplicateBtn.disableAndIgnore();
		scheduler.deleteBtn.disableAndIgnore();
		scheduler.undoAllChangesBtn.disableAndIgnore();
	}
	// Init help
	util.helpWindow.init('');
	//
	// Create theList object
	//
	scheduler.theList = new listcontroller.List({
		containerElementId: 'item_list_body',
		itemEvent: itemActivated,
		isSwitch1: true,
		isMoveControl: true
	});
	//
	// init Scheduler Actions List, Scheduler Actions and Date Filter Editor
	//
	schedulerActionsList.init();
	schedulerActions.init();
	schedulerDF.init();
}
function initMainDisplay() {
	util.hideE(['form_section', 'scheduler:no_smtp_server_info', 'loading_info', 'saving_info']);
	var firstItemId = scheduler.theList.getFirstItemId();
	if (firstItemId != null) {
		setItem(firstItemId);
		displayNoItemForm(false);
		handleSmtpServerWarning();
	}
	else {
		// no item exists
		displayNoItemForm(true);
	}
	util.showE('form_section');
}
function getScheduleData() {
	if (!pageInfo.exitActive) {
		// console.log('getScheduleData()');
		// invoked upon editor load
		var url = '?dp=admin_pages.scheduler.get_schedule_data';
			var dat = 'v.fp.page_token=' + pageInfo.pageToken;
		util.serverPost(url, dat);
	}
}
function getScheduleDataResponse(dat) {
	// alert('getScheduleDataResponse()');
	// Temp
	// return false;
	if (!pageInfo.exitActive) {
		scheduler.smtpServerIsDefined = dat.smtpServerIsDefined;
		scheduler.activeUserNodeName = dat.activeUserNodeName;
		scheduler.usersDb = dat.usersDb;
		// get itemsDb
		scheduler.itemsDb = dat.scheduleDb;
		scheduler.itemsDbBackup = util.cloneObject(dat.scheduleDb);
		// Get profilesDb
		scheduler.profilesDb = dat.profilesDb;
		scheduler.reportNamesDb = dat.reportNamesDb;
		scheduler.reportLabelsDb = dat.reportLabelsDb;
		scheduler.actionsDb = dat.actionsDb;
		util.createHash(scheduler.usersDb, 'name');
		util.createHash(scheduler.profilesDb, 'name');
		util.createHash(scheduler.actionsDb, 'name');
		// util.showObject(scheduler.profilesDb);
		init();
		scheduler.theList.init(scheduler.itemsDb);
		initMainDisplay();
		// Set final toolbar state
		scheduler.saveChangesBtn.enable();
		scheduler.undoAllChangesBtn.enable();
		scheduler.newItemBtn.enable();
		updateToolbarButtons(); // handles Duplicate and Delete
		adminConfig.setItemListSize();
		YAHOO.util.Event.addListener(window, 'resize', adminConfig.setItemListSize);
		pageInfo.initComplete = true;
	}
}
function displayNoItemForm(isDisplayNoItemForm) {
	if (isDisplayNoItemForm) {
		// util.updateT('item_form_label', '-');
		util.hideE('schedule_form');
		util.showE('no_item_form');
	}
	else {
		util.hideE('no_item_form');
		util.showE('schedule_form');
	}
	scheduler.noItemFormIsActive = isDisplayNoItemForm;
}
function updateToolbarButtons() {
	var isItems = scheduler.theList.isItems();
	scheduler.deleteBtn.enable(isItems);
	scheduler.duplicateBtn.enable(isItems);
}
function updateListAndFormLabel() {
	setTimeout('setListAndFormLabel()', 300);
}
function setListAndFormLabel() {
	var h = util.h;
	var label = util.getF('scheduler:label');
	if (label == '') {
		label = '-';
	}
	// We need to check if the schedule is from another user
	var item = scheduler.theList.getSelectedItem(); 
	var createdByUser = item.dat.created_by_user;
	if (createdByUser != scheduler.activeUserNodeName) {
		var usersDb = scheduler.usersDb;
		var username = (usersDb[h(createdByUser)] != null) ? usersDb[h(createdByUser)].username : langVar('lang_admin.scheduler.unknown_user');
		var byUserNameLabel = ' (' + langVar('lang_admin.general.by_user') + ')';
		byUserNameLabel = byUserNameLabel.replace(/__PARAM__1/, username);
		label += ' ' + byUserNameLabel;
	}
	// util.updateT('item_form_label', label);
	scheduler.theList.updateListLabel(label);
}
function getReportsDb(profileName) {
	// Called from schedulerActions.js, it returns a sorted array to create the report list
	function compareProfileLabels(a, b) {
		var labelA = a.label.toLowerCase();
		var labelB = b.label.toLowerCase();
		if (labelA < labelB) {return -1}
		if (labelA > labelB) {return 1}
		return 0;
	}
	var s = scheduler;
	var profile = s.profilesDb[util.h(profileName)];
	var reportsDb = [];
	if (profile != null) {
		// util.showObject(profile);
		var reportNameIndexes = profile.rN;
		var reportLabelIndexes = profile.rL;
		var reportsWithNumOfReportElements = profile.rWithNumOfReportElements;
		var reportNames = s.reportNamesDb;
		var reportLabels = s.reportLabelsDb;
		// alert('profile for which create a reportsDb: ' + profile.label);
		for (var i = 0; i < reportNameIndexes.length; i++) {
			var nameIndex = reportNameIndexes[i];
			var labelIndex = reportLabelIndexes[i];
			var numberOfReportElements = reportsWithNumOfReportElements[i];
			var name = reportNames[nameIndex];
			var label = reportLabels[labelIndex];
			reportsDb[i] = {name:name, label:label, numberOfReportElements:numberOfReportElements};
		}
		// Sort reportsDb
		reportsDb.sort(compareProfileLabels);
	}
	return reportsDb;
}
function enableRunNowButton() {
	// Function is called from schedulerRun.js as well!
	util.enableE('scheduler:run_now_btn');
}
function disableRunNowButton() {
	// Function is called from schedulerRun.js as well!
	util.disableE('scheduler:run_now_btn');
}
function setItem(itemId) {
	scheduler.theList.selectItem(itemId);
	updateForm();
}
function itemActivated(itemId) {
	if (validateActiveItem()) {
		// if selected node item is valid then move to activated node item
		// alert('itemId: ' + itemId);
		setItem(itemId);
	}
}
function newItem() {
	if (validateActiveItem()) {
		var newItemId = scheduler.theList.getNewItemId();
		var newScheduleLabel = langVar('lang_admin.scheduler.new_schedule');
		var newScheduleObj = {
			id: newItemId,
			type: '',
			switch1: true,
			label: newScheduleLabel,
			dat: {
				created_by_user: scheduler.activeUserNodeName,
				label: newScheduleLabel,
				month: '*',
				day: '*',
				hour: 0,
				minute: 0,
				actions: []
			}
		}
		// util.showObject(newScheduleObj);
		scheduler.theList.newItem(newScheduleObj);
		setItem(newItemId);
		updateToolbarButtons();
	}
}
function duplicateItem() {
	if (validateActiveItem()) {
		var theList = scheduler.theList
		var clonedItemId = theList.cloneItem();
		// Update created_by_user property
		theList.setItemDatValue(clonedItemId, 'created_by_user', scheduler.activeUserNodeName);
		setItem(clonedItemId);
	}
}	
function deleteItem() {
	var theList = scheduler.theList;
	var nextItemIdToBeSelected = theList.deleteItem();
	if (nextItemIdToBeSelected) {
		// select the next item
		// reset the validator in case that the deleted an item with error indication
		scheduler.validator.reset();
		setItem(nextItemIdToBeSelected);
	}
	else {
		// All items have been deleted
		displayNoItemForm(true);
		updateToolbarButtons();
	}
	handleSmtpServerWarning();
}
function updateForm() {
	// util.hideE(['scheduler:run_now:executing', 'scheduler:run_now:complete']);
	if (!schedulerRun.isRunning) {
		enableRunNowButton();
	}
	var item = scheduler.theList.getSelectedItem();
	var itemDat = item.dat;
	scheduler.activeCreatedByUser = itemDat.created_by_user;
	// util.showObject(item);
	var label = itemDat.label;
	util.setF('scheduler:label', label);
	util.setF('month_list', itemDat.month);
	util.setF('day_list', itemDat.day);
	util.setF('hour_list', itemDat.hour);
	util.setF('minute_list', itemDat.minute);
	// Update form label
	// util.updateT('item_form_label', label);
	//
	// Handle Actions
	// 
	schedulerActionsList.list.compose(itemDat.actions);
	if (scheduler.noItemFormIsActive) {
		displayNoItemForm(false);
	}
	// Position the Actions Move Control
	if (!scheduler.actionsMoveControlIsPositioned) {
		// We need to give it some time, else we don't get the right list position
		var s = 'scheduler:actions_list:div';
		setTimeout('schedulerActionsList.moveControl.setPosition("' + s + '")', 250);
		scheduler.actionsMoveControlIsPositioned = true;
	}
}
function handleSmtpServerWarning() {
	// Check if we need to show a no SMTP server warning
	if (!scheduler.smtpServerIsDefined) {
		var showWarning = false;
		// We show a warning message if an email report action exists
		var itemsDb = scheduler.itemsDb;
		// Note, we have to check the actions of the currently selected schedule
		// via schedulerActionsList.list.getItemsClone() because they are not up to
		// date in itemsDb
		var i;
		if (itemsDb.length > 0) {
			var activeItemId = scheduler.theList.getSelectedItemId();
			for (i = 0; i < itemsDb.length; i++) {
				// util.showObject(itemsDb[i]);
				var itemId = itemsDb[i].id;
				if (itemId != activeItemId) {
					var actions = itemsDb[i].dat.actions;
					for (var j = 0; j < actions.length; j++) {
						if (actions[j].type == 'send_report_by_email') {
							showWarning = true;
							break;
						}
					}
				}
			}
		}
		// If there is no warning yet then check the current active actions object
		if (!showWarning) {
			var activeActions = schedulerActionsList.list.getItemsReference();
			// util.showObject(activeActions);
			for (i = 0; i < activeActions.length; i++) {
				if (activeActions[i].type == 'send_report_by_email') {
					showWarning = true;
					break;
				}
			}
		}
		util.showE('scheduler:no_smtp_server_info', showWarning);
	}
}
function validateActiveItem() {
	// Only validate if isEdit permission and if items
	var theList = scheduler.theList;
	if (pageInfo.permissions.isEdit && theList.isItems()) {
		validator = scheduler.validator;
		var o = {};
		o.label = validator.isValue('scheduler:label');
		// Note, labels must be only unique among the schedules of a specific created_by_user value,
		// hence we use getLookupItemsByConstraint() and use the created_by_user value of the current item.
		// util.showObject(theList.getLookupItemsByConstraint('label', 'created_by_user', scheduler.activeCreatedByUser));
		o.label = validator.isUnique('scheduler:label', theList.getLookupItemsByConstraint('label', 'created_by_user', scheduler.activeCreatedByUser));
		o.month = util.getF('month_list');
		o.day = util.getF('day_list');
		o.hour = util.getF('hour_list');
		o.minute = util.getF('minute_list');
		o.created_by_user = scheduler.activeCreatedByUser;
		if (validator.allValid()) {
			// Get a clone of the items in actionsList
			o.actions = schedulerActionsList.list.getItemsClone();
			theList.saveItem(o);
			return true;
		}
		return false;
	}
	// No isEdit permission or no items
	return true;
}
function getActionItemDat(path, actionItem, position) {
	// Creates the output dat string for a single action item upon saveChanges()
	var handleDateFilter = false;
	var prop;
	var dat = path + '.position=' + position + '&';
	for (prop in actionItem) {
		if (prop != 'date_filter') {
			if (prop == 'options') {
				var options = actionItem.options;
				for (prop in options) {
					dat += path + '.options.' + prop + '=' + util.getEncodedURIComponent(options[prop]) + '&';
				}
			}
			else if (prop == 'filter_info') {
				var filterInfo = actionItem.filter_info;
				for (prop in filterInfo) {
					dat += path + '.filter_info.' + prop + '=' + util.getEncodedURIComponent(filterInfo[prop]) + '&';
				}
			}
			else {
				dat += path + '.' + prop + '=' + util.getEncodedURIComponent(actionItem[prop]) + '&';
			}
		}
		else {
			handleDateFilter = true;
		}
	}
	if (handleDateFilter) {
		var dateFilterObj = actionItem.date_filter;
		var dateFilterType = dateFilterObj.type;
		for (prop in dateFilterObj) {
			dat += path + '.date_filter.' + prop + '=' + util.getEncodedURIComponent(dateFilterObj[prop]) + '&';
		}
		// Get final df for options
		if (dateFilterType != 'none') {
			var df = '';
			if (dateFilterType == 'relative') {
				var relativePrefix = dateFilterObj.relative_prefix;
				var relativeDateUnit = dateFilterObj.relative_date_unit;
				var relativeDateUnitCount = dateFilterObj.relative_date_unit_count;
				if (relativePrefix != 'none') {
					df += relativePrefix;
				}
				df += relativeDateUnitCount + relativeDateUnit;
				if (relativeDateUnitCount > 1) {
					// Use plural form for date unit in df by adding 's'
					df += 's';
				}
			}
			else {
				// custom
				df = dateFilterObj.custom;
			}
			dat += path + '.options.df=' + util.getEncodedURIComponent(df) + '&';
		}
	}
	return dat;
}
function saveChanges() {
	if (validateActiveItem()) {
		var theList = scheduler.theList;
		var isModified = theList.getIsModified();
		// alert('isModified: ' + isModified);
		if (isModified) {
			util.hideE('form_section');
			util.showE('saving_info');
			var itemsDb = scheduler.itemsDb;
			var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
			if (itemsDb.length > 0) {
				for (var i = 0; i < itemsDb.length; i++) {
					// Assemble the schedule
					var path = 'v.fp.schedule.' + i;
					var scheduleItem = itemsDb[i];
					var scheduleItemDat = scheduleItem.dat;
					dat += path + '.position=' + i + '&'; // Used to sort nodes on server side
					dat += path + '.disabled=' + !scheduleItem.switch1 + '&';
					for (var prop in scheduleItemDat) {
						if (prop != 'actions') {
							dat += path + '.' + prop + '=' + util.getEncodedURIComponent(scheduleItemDat[prop]) + '&';
						}
						else {
							// Handle the actions
							var actions = scheduleItemDat.actions;
							if (actions.length > 0) {
								for (var j = 0; j < actions.length; j++) {
									var actionPath =  path + '.actions.' + j;
									dat += getActionItemDat(actionPath, actions[j], j);
								}
							}
							else {
								// No action exists. A schedule without a single action is allowed.
								dat += path + '.actions=&';
							}
						}
					}
				}
			}
			else {
				// No schedule exists
				dat += 'v.fp.schedule=&';
			}
			dat = dat.replace(/&$/, '');
			// alert(dat);
			pageInfo.saveActive = true;
			var url = '?dp=admin_pages.scheduler.save_data';
			util.serverPost(url, dat);
		}
		else {
			alert(langVar('lang_stats.general.no_changes_to_save'));
		}
	}
}
function saveChangesResponse() {
	// reset isModified
	scheduler.theList.resetIsModified();
	// the saved work itemsDb becomes now the itemsDbBackup
	scheduler.itemsDbBackup = util.cloneObject(scheduler.itemsDb);
	pageInfo.saveActive = false;
	util.hideE('saving_info');
	util.showE('form_section');
}
function undoAllChanges() {
	// alert('Undo Changes\nitemsDb length: ' + itemsDb.length + '\nitemsDbBackup length: ' + itemsDbBackup.length);
	scheduler.itemsDb = util.cloneObject(scheduler.itemsDbBackup);
	scheduler.validator.reset();
	scheduler.theList.init(scheduler.itemsDb);
	initMainDisplay();
	updateToolbarButtons();
}
function getIsModifiedPage() {
	// Note, isModified will be false if only the active item has been edited
	// but has an error, hence we also check "if (!validateActiveItem() ..."
	if (!validateActiveItem() || scheduler.theList.getIsModified()) {
		return true;
	}
	return false;
}
