//
//
// creUtil.js (Customize Report Element utilities, used in Reports and Reports Editor)
// Contains helper utilities for cre*.js files
//
// 
var creUtil = {
	getReportElementGraphsObjectFromGraphsDb: function(graphsDb) {
		// Returns a graphs object as required to save in report element. The graphs object is created
		// from an up to date graphsDb object.
		var metaGraphType = graphsDb.metaGraphType;
		var isChrono = (metaGraphType.indexOf('chrono_') != -1);
		var graphs = {};
		var graphType = '';
		if (isChrono) {
			graphs = util.cloneObject(graphsDb.chrono_bar_line_graph);
			graphType = (metaGraphType == 'chrono_bar') ? 'bar' : 'line';
		}
		else {
			if (metaGraphType != 'pie') {
				graphs = util.cloneObject(graphsDb.bar_line_graph);
			}
			else {
				graphs = util.cloneObject(graphsDb.pie_chart);
			}
			graphType = metaGraphType;
		}
		graphs.graph_type = graphType;
		return graphs;
	},
	getIsChronoGraphSupport: function(reportFieldsDb, columns) {
		var isChronoGraphSupport = false;
		var numberOfTextFields = 0;
		var textFieldName = '';
		for (var i = 0; i < columns.length; i++) {
			// We assume that show_column is true for text field
			var isTextField = (columns[i].show_percent_column == null);
			if (isTextField) {
				textFieldName = columns[i].report_field;
				numberOfTextFields++;
			}
		}
		if (numberOfTextFields == 1) {
			// alert('textFieldName: ' + textFieldName);
			var category = reportFieldsDb[util.h(textFieldName)].category;
			// alert('category: ' + category);
			if (category == 'date_time' ||
				category == 'day_of_week' ||
				category == 'hour_of_day') {
				isChronoGraphSupport = true;
			}
		}
		return isChronoGraphSupport;
	},
	buildSortDirectionList: function(selectElementId) {
		// Populate sort direction list
		var sortDirectionList = [{name:'ascending', label:langVar('lang_stats.btn.ascending')}, {name:'descending', label:langVar('lang_stats.btn.descending')}];
		util.populateSelect(selectElementId, sortDirectionList, 'name', 'label');
	},
	getSortByFirstListEntryLabel: function(reportFieldsDb, columns, label) {
		// Label is the first list entrty in sort by select element. We add some spaces
		// so that the GUI doen's flicker when columns are unchecked
		var maxLength = 0;
		for (var i = 0; i < columns.length; i++) {
			var reportField = columns[i].report_field;
			var columnLabel = reportFieldsDb[util.h(reportField)].label;
			if (columnLabel.length > maxLength) {
				maxLength = columnLabel.length;
			}
		}
		for (var i = label.length; i < maxLength + 10; i++) {
			label += '\u00a0';
		}
		return label;
	}
}
//
//
// creGeneral.js (Customize Report Element General, used in Config Reports Element Editor only)
//
//
var creGeneral = {
	descriptionBtn: null,
	heaaderBtn: null,
	footerBtn: null,
	// Keep active omitParenthesizedItems and useOverviewForTotals state
	// so that we can recover the state when switching between different
	// report element types
	omitParenthesizedItems: false,
	useOverviewForTotals: false,
	init: function() {
		var commandLinkOptions = {
			classNameEnabled: 'command-link-50',
			classNameDisabled: 'command-link-50-disabled'
		};	
		var YE = YAHOO.util.Event;
		creGeneral.descriptionBtn = new util.CommandLink('cre_obj:general:description:btn', creGeneral.setDescriptionHeaderFooterActor, true, commandLinkOptions);
		creGeneral.headerBtn = new util.CommandLink('cre_obj:general:header:btn', creGeneral.setDescriptionHeaderFooterActor, true, commandLinkOptions);
		creGeneral.footerBtn = new util.CommandLink('cre_obj:general:footer:btn', creGeneral.setDescriptionHeaderFooterActor, true, commandLinkOptions);
		YE.addListener('cre_obj:general:show_header_bar', 'click', creGeneral.toggleShowNameInHeaderBar);
		YE.addListener('cre_obj:general:omit_parenthesized_items', 'click', creGeneral.omitParenthesizedItemsActor);
		YE.addListener('cre_obj:general:use_overview_for_totals', 'click', creGeneral.useOverviewForTotalsActor);
	},
	updateForm: function(reportElementDb) {
		// Called upon initial opening of the report element editor (not upon meta type change)
		// util.showObject(reportElementDb);
		// alert('creGeneral.updateForm()');
		var reportElementType = reportElementDb.type;
		util.setF('cre_obj:general:display_side_by_side', reportElementDb.display_side_by_side);
		util.setF('cre_obj:general:show_header_bar', reportElementDb.show_header_bar);
		util.setF('cre_obj:general:report_link', reportElementDb.report_link);
		util.setF('cre_obj:general:description', reportElementDb.description);
		util.setF('cre_obj:general:header', reportElementDb.header);
		util.setF('cre_obj:general:footer', reportElementDb.footer);
		creGeneral.setReportLinkDisplay(reportElementDb.show_header_bar);
		// Disable display_side_by_side if session_paths
		util.enableE('cre_obj:general:display_side_by_side', (reportElementType != 'session_paths'));
		// omit_parenthesized_items and use_overview_for_totals checkbox
		// is further handled in setReportElementTypeDependencies.
		var omitParenthesizedItems = (reportElementType == 'table') ? reportElementDb.omit_parenthesized_items : false;
		var useOverviewForTotals = (reportElementType == 'table') ? reportElementDb.use_overview_for_totals : false;
		creGeneral.omitParenthesizedItems = omitParenthesizedItems;
		creGeneral.useOverviewForTotals = useOverviewForTotals;
		//
		// Make description the active text field panel
		//
		creGeneral.setDescriptionHeaderFooterByName('description');
	},
	setReportElementTypeDependencies: function(reportElementType) {
		// Called upon report element type (metaType) change.
		// Disable display_side_by_side if session_paths
		util.enableE('cre_obj:general:display_side_by_side', (reportElementType != 'session_paths'));
		var makeEnabled = (reportElementType == 'table');
		// Recover state if makeEnabled, else set checkboxes to false and disbale them
		var omitParenthesizedItems = makeEnabled ? creGeneral.omitParenthesizedItems : false;
		var useOverviewForTotals = makeEnabled ? creGeneral.useOverviewForTotals : false;
		util.setF('cre_obj:general:omit_parenthesized_items', omitParenthesizedItems);
		util.setF('cre_obj:general:use_overview_for_totals', useOverviewForTotals);
		util.showEV('cre_obj:general:advanced_options', makeEnabled);
		util.enableE('cre_obj:general:omit_parenthesized_items', makeEnabled);
		util.enableE('cre_obj:general:use_overview_for_totals', makeEnabled);
	},
	omitParenthesizedItemsActor: function() {
		creGeneral.omitParenthesizedItems = this.checked;
	},
	useOverviewForTotalsActor: function() {
		creGeneral.useOverviewForTotals = this.checked;
	},
	toggleShowNameInHeaderBar: function() {
		creGeneral.setReportLinkDisplay(this.checked);
	},
	setReportLinkDisplay: function(isShowReportNameInHeaderBar) {
		// Disable report_link select element and reset to "" if show_header_bar is unchecked,
		// else enable report_link.
		if (!isShowReportNameInHeaderBar) {
			// Reset report_link
			util.setF('cre_obj:general:report_link', '');
		}
		util.enableE('cre_obj:general:report_link', isShowReportNameInHeaderBar);
	},
	saveChanges: function(reportElementDb) {
		// Save changes to the given reportElementDb
		reportElementDb.display_side_by_side = util.getF('cre_obj:general:display_side_by_side');
		reportElementDb.show_header_bar = util.getF('cre_obj:general:show_header_bar');
		reportElementDb.report_link = util.getF('cre_obj:general:report_link');
		reportElementDb.description = util.getF('cre_obj:general:description');
		reportElementDb.header = util.getF('cre_obj:general:header');
		reportElementDb.footer = util.getF('cre_obj:general:footer');
		if (reportElementDb.type == 'table') {
			reportElementDb.omit_parenthesized_items = creGeneral.omitParenthesizedItems;
			reportElementDb.use_overview_for_totals = creGeneral.useOverviewForTotals;
		}
	},
	setDescriptionHeaderFooterActor: function() {
		var dat = this.id.split(':');
		var textFieldName = dat[2];
		creGeneral.setDescriptionHeaderFooterByName(textFieldName);
	},
	setDescriptionHeaderFooterByName: function(textFieldName) {
		// textFieldName is: description | header | footer
		util.hideE(['cre_obj:general:description', 'cre_obj:general:header', 'cre_obj:general:footer']);
		creGeneral.descriptionBtn.enable();
		creGeneral.headerBtn.enable();
		creGeneral.footerBtn.enable();
		util.showE('cre_obj:general:' + textFieldName);
		var fieldLabel = '';
		if (textFieldName == 'description') {
			fieldLabel = langVar('lang_stats.customize_report_element.description');
			creGeneral.descriptionBtn.disable();
		}
		else if (textFieldName == 'header') {
			fieldLabel = langVar('lang_stats.customize_report_element.header');
			creGeneral.headerBtn.disable();
		}
		else {
			fieldLabel = langVar('lang_stats.customize_report_element.footer');
			creGeneral.footerBtn.disable();
		}
		util.updateT('cre_obj:general:text_field_label', fieldLabel + ':');
	}
};
//
//
// creFilters.js (Customize Report Element Advanced, used in Config Reports Element Editor only)
//
//
var creFilters = {
	tableFilterExpression: '', // keeps the active tableFilterExpression when switching between report element types
	init: function() {
		var YE = YAHOO.util.Event;
		YE.addListener('cre_obj:filters:filter_expression:show_available_fields_btn', 'click', reportsUtil.viewAvailableReportFilterFields.open);
		YE.addListener('cre_obj:filters:filter_expression:view_help_btn', 'click', util.helpWindow.openGeneralHelp);
		YE.addListener('cre_obj:filters:table_filter_expression:show_available_fields_btn', 'click', reportsUtil.viewAvailableReportTableFilterFields.open);
		YE.addListener('cre_obj:filters::table_filter_expression:view_help_btn', 'click', util.helpWindow.openGeneralHelp);
	},
	updateForm: function(reportElementDb) {
		var reportElementType = reportElementDb.type;
		// Reset global tableFilterExpression
		creFilters.tableFilterExpression = '';
		util.setF('cre_obj:filters:date_filter', reportElementDb.date_filter);
		util.setF('cre_obj:filters:filter_expression', reportElementDb.filter_expression);
		var tableFilterExpression = '';
        if (reportElementType == 'table' ||
            reportElementType == 'log_detail') {
            tableFilterExpression = reportElementDb.table_filter_expression;
        }
		util.setF('cre_obj:filters:table_filter_expression', tableFilterExpression);
	},
	setReportElementTypeDependencies: function(reportElementType) {
		// Called from creControl upon metaType change
        // KHP 29/Sep/2010 - we allow table filter expressions in log_detail report elements
		var isTableFilterExpSupport = (reportElementType == 'table' || reportElementType == 'log_detail');
		var currentTableFilterExpression = util.getF('cre_obj:filters:table_filter_expression');
		if (!isTableFilterExpSupport) {
			if (currentTableFilterExpression != '') {
				// Save expression in global tableFilterExpression and cclear the expression field
				creFilters.tableFilterExpression = currentTableFilterExpression;
				util.setF('cre_obj:filters:table_filter_expression', '');
			}
		}
		else {
			if (currentTableFilterExpression == '' && creFilters.tableFilterExpression != '') {
				// Restore the table filter expression which has been set upon a supported report element
				util.setF('cre_obj:filters:table_filter_expression', creFilters.tableFilterExpression);
				creFilters.tableFilterExpression = '';
			}
		}
		util.enableE('cre_obj:filters:table_filter_expression', isTableFilterExpSupport);
	},
	saveChanges: function(reportElementDb) {
		// Called from creControl, save changes to the given
		reportElementDb.date_filter = util.getF('cre_obj:filters:date_filter');
		reportElementDb.filter_expression = util.getF('cre_obj:filters:filter_expression');
		var reportElementType = reportElementDb.type;
		if (reportElementType == 'table' || reportElementType == 'log_detail') {
			reportElementDb.table_filter_expression = util.getF('cre_obj:filters:table_filter_expression');
		}
	}
};
//
//
// creGraphs.js (Customize Report Element Graphs Class, used in Reports and Reports Editor)
//
// Note, the sort by list for graphs is part of the graphs panel, so it is handled in graph options
// because most of the data required by sort by already exists in graph options and we don't want
// to duplicate this data here.
// However, graphs contains a reference to the graphOptions object to handle sort_by integrity
//
//
CreGraphs = function(queryFieldsDb) {
	// Manages the graphs form controls
	var YE = YAHOO.util.Event;
	var graphFieldsContainerId = 'cre_obj:graphs:fields_container';
	this.graphFieldsContainerId = graphFieldsContainerId;
	this.queryFieldsDb = queryFieldsDb; // A reference to all report fields from where we get the Field labels
	this.reportElementDb = {}; // A reference to the active report element data.
							   // reportElementDb keeps the state of any action taken in the Graphs
							   // The variable is set upon this.build()
	// this.graphsDb = {}; // A reference to the active graphs object
	// this.graphOptionsRef = null; // A reference to the graphOptions object to call graphOptions.checkSortByIntegrity()
	this.isTable = false; // Set upon this.init()
	// this.metaGraphTypePriorTabChange = ''; // Set upon init(), it is used to check whether the meta graph type
										   // changed in graph options. If it changed we may need to update
										   // sort by and/or show legend in the Graphs tab.
	var idPrefix = util.getUniqueElementId();
	this.idPrefix = idPrefix; // Used as prefix in all element Id's (checkboxes, etc.) and as self reference id
	// util.showObject(this.reportElementDb);
	YE.addListener('cre_obj:graphs:select_deselect_all_btn', 'click', this.selectDeselectAllActor, this);
	// Assign single table event
	YE.addListener(graphFieldsContainerId, 'click', this.itemActivated, this);
}
CreGraphs.prototype = {
//	setGraphOptionsReference: function(graphOptionsRef) {
//		this.graphOptionsRef = graphOptionsRef;
//	},
	init: function(reportElementDb) {
		// Field list element ID scheme (prefix:index:type:elementName), index matches the array position
		// sort colum: ueid_20:0:sort:td --> should be an image!
		// label column: re_field_list:0:label:td --> used to set the sort direction
		//
		// Init
		//
		// var metaGraphType = graphsDb.metaGraphType;
		this.reportElementDb = reportElementDb;
		// this.graphsDb = graphsDb;
		var isTable = reportElementDb.show_table;
		this.isTable = isTable;
		// this.metaGraphTypePriorTabChange = metaGraphType;
		var queryFieldsDb = this.queryFieldsDb;
		var columns = reportElementDb.columns;
		this.buildGraphFields(queryFieldsDb, columns);
	},
	//
	//
	//
	// Graphs list handling
	//
	//
	//
	//
	itemActivated: function(evt, self) {
		var element = evt.target || evt.srcElement;
		var elementId = element.id;
		if (elementId != '') {
			// alert('creUtil.Graphs.prototype.itemActivated(): ' + elementId);
			var dat = elementId.split(':');
			var idPrefix = dat[0];
			var objectIndex = parseInt(dat[1], 10);
			var type = dat[2];
			// alert('idPrefix: ' + idPrefix);
			// var self = creUtil.TableColumnsSelfReferences[idPrefix]; // self is equal this in terms of the TableColumns object!
			var reportElementDb = self.reportElementDb;
			var column = reportElementDb.columns[objectIndex];
			// alert('list.idPrefix: ' + list.idPrefix);
			// util.showObject(item);
			var isAggregatingField = (column.show_percent_column != null);
			var isChecked;
			if (isAggregatingField) {
				if (type == 'graph_label') {
					// User clicked on the text label (note, no label element exists, so the checkbox
					// does not automatically set its state by clicking on the text)
					var graphInput = util.getE(idPrefix + ':' + objectIndex + ':graph');
					isChecked = graphInput.checked;
					graphInput.checked = !isChecked;
					column.show_graph = !isChecked;
				}
				else if (type == 'graph') {
					isChecked = util.getF(idPrefix + ':' + objectIndex + ':' + type);
					column.show_graph = isChecked;
				}
				// self.graphOptionsRef.checkSortByIntegrity();
			}
		}
	},
	selectDeselectAllActor: function(evt, self) {
		var reportElementDb = self.reportElementDb;
		var columns = reportElementDb.columns;
		var idPrefix = self.idPrefix;
		//
		// Check if we check or uncheck all fields
		//
		var makeAllChecked = false;
		var i;
		var column;
		var isAggregatingField;
		for (i = 0; i < columns.length; i++) {
			column = columns[i];
			isAggregatingField = (column.show_percent_column != null);
			if (isAggregatingField && !column.show_graph) {
				makeAllChecked = true;
				break;
			}
		}
		// alert('makeAllChecked: ' + makeAllChecked);
		//
		// Check or uncheck all fields
		//
		for (i = 0; i < columns.length; i++) {
			column = columns[i];
			isAggregatingField = (column.show_percent_column != null);
			// alert('isAggregatingField: ' + isAggregatingField);
			if (isAggregatingField) {
				column.show_graph = makeAllChecked;
				util.setF(idPrefix + ':' + i + ':graph', makeAllChecked);
			}
		}
	},
	//
	//
	// Graph fields handling
	//
	//
	buildGraphFields: function(queryFieldsDb, columns) {
		var table = util.getE(this.graphFieldsContainerId);
		// Clean up the list
		while (table.firstChild != null) {
			var n = table.firstChild;
			table.removeChild(n);
		}
		var tbody = document.createElement('tbody');
		var i;
		var column;
		var isAggregatingField;
		for (i = 0; i < columns.length; i++) {
			column = columns[i];
			var reportField = column.report_field;
			var reportFieldLabel = queryFieldsDb[util.h(reportField)].label;
			isAggregatingField = (column.show_percent_column != null);
			var tr = document.createElement('tr');
			this.addCheckboxCell(tr, i, 'graph', '');
			this.addLabelCell(tr, i, reportFieldLabel);
			tbody.appendChild(tr);
		}
		table.appendChild(tbody);
		//
		// Upadate the checkbox state
		//
		var idPrefix = this.idPrefix;
		for (i = 0; i < columns.length; i++) {
			column = columns[i];
			isAggregatingField = (column.show_percent_column != null);
			var input = util.getE(idPrefix + ':' + i + ':graph');
			if (isAggregatingField) {
				input.checked = columns[i].show_graph;
			}
			else {
				// Text field, disable checkbox and label
				input.disabled = 'disabled';
				var labelTd = util.getE(idPrefix + ':' + i + ':graph_label');
				labelTd.className = 'label disabled';
			}
		}
	},
	addLabelCell: function(tr, index, label) {
		var td = document.createElement('td');
		td.id = this.idPrefix + ':' + index + ':graph_label';
		td.className = 'label';
		var text = document.createTextNode(label);
		util.chainE(tr, td, text);
	},
	addCheckboxCell: function(tr, index, type, label) {
		var td = document.createElement('td');
		var id = this.idPrefix + ':' + index + ':' + type;
		var input = document.createElement('input');
		input.id = id;
		input.type = 'checkbox';
		if (!util.userAgent.isIE) {
			// Give checkbox default margin
			input.style.margin = '3px';
		}
		td.appendChild(input);
		if (label != '') {
			var labelElement = document.createElement('label');
			labelElement.htmlFor = id;
			var text = document.createTextNode(' ' + label);
			labelElement.appendChild(text);
			td.appendChild(labelElement);
		}
		tr.appendChild(td);
	}
}
//
//
// creTable.js (Customize Report Element Table Class, used in Reports and Reports Editor)
//
//
CreTable = function(queryFieldsDb, isCustomizeInReports, hideLogDetailSortingMessage) {
	// Manages the report fields list
	// this.containerElementId = containerElementId;
	var YE = YAHOO.util.Event;
	var tableFieldsContainerId = 'cre_obj:table:fields_container';
	this.tableFieldsContainerId = tableFieldsContainerId;
	this.queryFieldsDb = queryFieldsDb; // A reference to all report fields from where we get the Field labels
	this.isCustomizeInReports = isCustomizeInReports;
	this.hideLogDetailSortingMessage = hideLogDetailSortingMessage;
	this.reportElementDb = {}; // A reference to the active report element data.
							   // reportElementDb keeps the state of any action taken in the TableColumns
							   // The variable is set upon this.build()
	this.sortByFirstListEntryLabel = ''; // Set upon init()
	this.totalRows = 0; // Set upon init(), it is used for the log detail warning message in the Reports GUI
	this.isTable = false;
	this.isOverview = false;
	this.isLogDetail = false; 
	var idPrefix = util.getUniqueElementId();
	this.idPrefix = idPrefix; // Used as prefix in all element Id's (checkboxes, etc.) and as self reference id
	// Populate sort direction list in Table Options
	creUtil.buildSortDirectionList('cre_obj:table:sort_direction');
	// util.showObject(this.reportElementDb);
	YE.addListener('cre_obj:table:sort_by', 'click', this.sortByActor, this);
	YE.addListener('cre_obj:table:sort_direction', 'click', this.sortDirectionActor, this);
	YE.addListener('cre_obj:table:select_deselect_all_btn', 'click', this.selectDeselectAllActor, this);
	// assign table event
	YE.addListener(tableFieldsContainerId, 'click', this.itemActivated, this);
};
CreTable.prototype = {
	init: function(reportElementDb, totalRows) {
		// Field list element ID scheme (prefix:index:type:elementName), index matches the array position
		// sort colum: ueid_20:0:sort:td --> should be an image!
		// label column: re_field_list:0:label:td --> used to set the sort direction
		// util.showObject(reportElementDb);
		//
		// Init
		//
		this.reportElementDb = reportElementDb;
		this.totalRows = totalRows;
		this.isTable = (reportElementDb.type == 'table');
		this.isOverview = (reportElementDb.type == 'overview');
		this.isLogDetail = (reportElementDb.type == 'log_detail');
		var queryFieldsDb = this.queryFieldsDb;
		var columns = reportElementDb.columns;
		var basicSortByFirstListEntryLabel = '(' + langVar('lang_stats.general.none') + ')'; // --> Label gets length of max column label to avoid GUI flicker
		this.sortByFirstListEntryLabel = creUtil.getSortByFirstListEntryLabel(queryFieldsDb, columns, basicSortByFirstListEntryLabel);
		this.buildTableFields(columns, queryFieldsDb);
		util.showEV('cre_obj:table:sort_by:div', !this.isOverview);
		util.disableE(['cre_obj:table:sort_by', 'cre_obj:table:sort_direction'], this.isOverview); // Disable so that we don't get focus on it if overview
		if (!this.isOverview) {
			this.updateSortBy();
			// Set initial sort direction
			util.setF('cre_obj:table:sort_direction', reportElementDb.sort_direction);
		}
	},
	itemActivated: function(evt, self) {
		var element = evt.target || evt.srcElement;
		var elementId = element.id;
		if (elementId != '') {
			// alert('creUtil.TableColumns.prototype.itemActivated(): ' + elementId);
			var dat = elementId.split(':');
			var idPrefix = dat[0];
			var objectIndex = parseInt(dat[1], 10);
			var type = dat[2];
			// alert('idPrefix: ' + idPrefix);
			// var self = creUtil.TableColumnsSelfReferences[idPrefix]; // self is equal this in terms of the TableColumns object!
			var reportElementDb = self.reportElementDb;
			var column = reportElementDb.columns[objectIndex];
			// alert('list.idPrefix: ' + list.idPrefix);
			// util.showObject(item);
			var isAggregatingField = (column.show_percent_column != null);
			// alert('isLogDetail: ' + isLogDetail);
			var isChecked;
			// var isUpdateSortBy = false;
			// if table report element
			if (self.isTable) {
				if (isAggregatingField) {
					if (type == 'table_label') {
						// We set the checkboxes depending on current state.
						// a.) If none is checked we check all
						// b.) If one or more are checked we uncheck all
						isChecked = column.show_column;
						column.show_column = !isChecked;
						column.show_percent_column = !isChecked;
						column.show_bar_column = !isChecked;
						// Update aggregating field state
						self.updateCheckboxState(objectIndex, column);
					}
					else {
						isChecked = util.getF(idPrefix + ':' + objectIndex + ':' + type);
						if (type == 'number') {
							column.show_column = isChecked;
							if (!isChecked) {
								column.show_percent_column = false;
								column.show_bar_column = false;
							}
							util.enableE(idPrefix + ':' + objectIndex + ':percent', isChecked);
							util.enableE(idPrefix + ':' + objectIndex + ':bar', isChecked);
							// util.showObject(column);
							// Update aggregating field state
							self.updateCheckboxState(objectIndex, column);
						}
						else if (type == 'percent') {
							column.show_percent_column = isChecked;
						}
						else if (type == 'bar') {
							column.show_bar_column = isChecked;
						}
					}
				}
			}
			else {
				// overview or log_detail report element which only has text and number
				// checkboxes and where text checkboxes are not disabled.
				// Note, a log_detail report element has show_percent_column and show_bar_column
				// nodes, though they are not in use, respectively always false.
				var typeDetail = isAggregatingField ? 'number' : 'text';
				var checkbox = util.getE(idPrefix + ':' + objectIndex + ':' + typeDetail);
				isChecked = checkbox.checked;
				if (type == 'table_label') {
					column.show_column = !isChecked;
					checkbox.checked = !isChecked;
				}
				else {
					column.show_column = isChecked;
				}
			}
			//
			// Handle sort_by
			// If a column becomes unchecked we have to remove it from
			// the sort_by list
			//
			if (!self.isOverview && !column.showColumn) {
				// Check if this is the active sort field
				if (column.report_field == reportElementDb.sort_by) {
					self.autoSortBy(objectIndex);
				}
				self.updateSortBy();
			}
		}
	},
	updateCheckboxState: function(objectIndex, column) {
		// Sets the checkbox state for the specified column item
		var isAggregatingField = (column.show_percent_column != null);
		var idPart = this.idPrefix + ':' + objectIndex + ':';
		// util.showObject(column);
		// alert('isAggregatingField: ' + isAggregatingField);
		var showColumn = column.show_column;
		if (!isAggregatingField) {
			util.setF(idPart + 'text', showColumn);
			if (this.isTable) {
				util.disableE(idPart + 'text');
			}
		}
		else {
			util.setF(idPart + 'number', showColumn);
			if (this.isTable) {
				var percentInput = util.getE(idPart + 'percent');
				var percentTd = percentInput.parentNode;
				var barInput = util.getE(idPart + 'bar');
				var barTd = barInput.parentNode;
				var className = showColumn ? '' : 'disabled';
				percentInput.checked = column.show_percent_column;
				percentInput.disabled = !showColumn;
				percentTd.className = className;
				barInput.checked = column.show_bar_column;
				barInput.disabled = !showColumn;
				barTd.className = className;
			}
		}
	},
	selectDeselectAllActor: function(evt, self) {
		var reportElementDb = self.reportElementDb;
		var columns = reportElementDb.columns;
		var isTable = self.isTable;
		var isOverview = self.isOverview;
		//
		// Check if we check or uncheck all fields
		//
		var makeAllChecked = false;
		var i;
		var column;
		var isAggregatingField;
		for (i = 0; i < columns.length; i++) {
			column = columns[i];
			// isAggregatingField = (column.show_percent_column != null);
			if (!column.show_column) {
				makeAllChecked = true;
				break;
			}
		}
		// alert('makeAllChecked: ' + makeAllChecked);
		//
		// Check or uncheck all fields
		//
		for (i = 0; i < columns.length; i++) {
			column = columns[i];
			isAggregatingField = (column.show_percent_column != null);
			// alert('isAggregatingField: ' + isAggregatingField);
			if (isTable) {
				if (isAggregatingField) {
					column.show_column = makeAllChecked;
					column.show_percent_column = makeAllChecked;
					column.show_bar_column = makeAllChecked;
					self.updateCheckboxState(i, column);
				}
			}
			else {
				column.show_column = makeAllChecked;
				self.updateCheckboxState(i, column);
			}
		}
		// Handle sort by
		if (!isOverview && !makeAllChecked) {
			reportElementDb.sort_by = '';
		}
		self.updateSortBy();
	},
	sortByActor: function(evt, self) {
		var previousSortByValue = self.reportElementDb.sort_by;
		var newSortByValue = util.getF('cre_obj:table:sort_by');
		// Set newSortByValue 
		self.reportElementDb.sort_by = newSortByValue;
		// Check if we need to show the log detail sorting message
		if (self.isLogDetail &&
			!self.hideLogDetailSortingMessage &&
			previousSortByValue == '' &&
			newSortByValue != '' &&
			(self.totalRows > 1000000 || !self.isCustomizeInReports)) {
			logDetailSortingMsg.openViaCustomizeReportElement(self.isCustomizeInReports);
		}
	},
	sortDirectionActor: function(evt, self) {
		self.reportElementDb.sort_direction = util.getF('cre_obj:table:sort_direction');
	},
	autoSortBy: function(objectIndex) {
		// The column of the given objectIndex has been unchecked, so the report_field
		// can't be used for sort_by, respectively it doesn't make sense.
		// In this case we automatically set set reportElementDb.sort_by to
		// the nearest checked column
		var reportElementDb = this.reportElementDb;
		var columns = reportElementDb.columns;
		var sortBy = '';
		var i;
		// we serach first upward and then downward for a checked column
		for (i = objectIndex + 1; i < columns.length; i++) {
			if (columns[i].show_column) {
				sortBy = columns[i].report_field;
				break;
			}
		}
		// If no sortBy has been found yet
		if (sortBy == '') {
			// Serach downward
			for (i = objectIndex - 1; i >= 0; i--) {
				if (columns[i].show_column) {
					sortBy = columns[i].report_field;
					break;
				}
			}
		}
		reportElementDb.sort_by = sortBy;
	},
	updateSortBy: function() {
		// Updates sort by fields list and selected item to the state
		// as in reportElementDb
		var queryFieldsDb = this.queryFieldsDb;
		var columns = this.reportElementDb.columns;
		var sortBy = this.reportElementDb.sort_by;
		var firstListLabel = this.sortByFirstListEntryLabel;
		var list = [{name:'', label:firstListLabel}];
		for (var i = 0; i < columns.length; i++) {
			var column = columns[i];
			if (column.show_column) {
				var name = column.report_field;
				var label = queryFieldsDb[util.h(name)].label;
				list[list.length] = {name:name, label:label};
			}
		}
		util.populateSelect('cre_obj:table:sort_by', list, 'name', 'label');
		util.setF('cre_obj:table:sort_by', sortBy);
	},
	buildTableFields: function(columns, queryFieldsDb) {
		var table = util.getE(this.tableFieldsContainerId);
		// Clean up the list
		while (table.firstChild != null) {
			var n = table.firstChild;
			table.removeChild(n);
		}
		var tbody = document.createElement('tbody');
		var i;
		for (i = 0; i < columns.length; i++) {
			var column = columns[i];
			var reportField = column.report_field;
			var reportFieldLabel = queryFieldsDb[util.h(reportField)].label;
			var isAggregatingField = (column.show_percent_column != null);
			var tr = document.createElement('tr');
			if (!isAggregatingField) {
				this.addCheckboxCell(tr, i, 'text', '');
				// if report element type table
				if (this.isTable) {
					this.addEmptyCell(tr);
					this.addEmptyCell(tr);
				}
			}
			else {
				this.addCheckboxCell(tr, i, 'number', '');
				// if report element type table
				if (this.isTable) {
					this.addCheckboxCell(tr, i, 'percent', '%');
					this.addCheckboxCell(tr, i, 'bar', langVar('lang_stats.customize_report_element.bar'));
				}
			}
			this.addLabelCell(tr, i, reportFieldLabel);
			tbody.appendChild(tr);
		}
		table.appendChild(tbody);
		// Upadate the checkbox state
		for (i = 0; i < columns.length; i++) {
			this.updateCheckboxState(i, columns[i]);
		}
	},
	addLabelCell: function(tr, index, label) {
		var td = document.createElement('td');
		td.id = this.idPrefix + ':' + index + ':table_label';
		td.className = 'label';
		if (this.isTable) {
			td.style.paddingLeft = '14px';
		}
		var text = document.createTextNode(label);
		util.chainE(tr, td, text);
	},
	addCheckboxCell: function(tr, index, type, label) {
		var td = document.createElement('td');
		var id = this.idPrefix + ':' + index + ':' + type;
		var input = document.createElement('input');
		input.id = id;
		input.type = 'checkbox';
		if (!util.userAgent.isIE) {
			// Give checkbox default margin
			input.style.margin = '3px';
		}
		td.appendChild(input);
		if (label != '') {
			var labelElement = document.createElement('label');
			labelElement.htmlFor = id;
			var text = document.createTextNode(' ' + label);
			labelElement.appendChild(text);
			td.appendChild(labelElement);
		}
		tr.appendChild(td);
	},
	addEmptyCell: function(tr) {
		var td = document.createElement('td');
		tr.appendChild(td);
	}
};
//
//
// creTableOptions.js (Customize Report Element TableOptions Class, used in Reports and Reports Editor)
//
//
CreTableOptions = function(queryFieldsDb, isCustomizeInReports) {
	this.isCustomizeInReports = isCustomizeInReports; // Used to handle differences between CRE in Reports GUI and CRE in Config Reports Editor GUI
	this.queryFieldsDb = queryFieldsDb; // A reference to all report fields from where we get the Field labels
	this.reportElementDb = {}; // A reference to the active report element data.
							   // reportElementDb keeps the state of any action taken in the FieldList
							   // The variable is set upon this.init()
	this.isLogDetail = false;
	this.isSessionPaths = false;
	this.isSessionPagePaths = false;
};
CreTableOptions.prototype = {
	init: function(reportElementDb, totalRows) {
		// util.showObject(reportElementDb);
		var reportElementType = reportElementDb.type;
		var isOverview = (reportElementType == 'overview');
		var isTable = (reportElementType == 'table');
		var isLogDetail = (reportElementType == 'log_detail');
		var isSessionPaths = (reportElementType == 'session_paths');
		var isSessionPagePaths = (reportElementType == 'session_page_paths');
		this.reportElementDb = reportElementDb;
		this.isOverview = isOverview;
		this.isTable = isTable;
		this.isLogDetail = isLogDetail;
		this.isSessionPaths = isSessionPaths;
		this.isSessionPagePaths = isSessionPagePaths;
		if (!isOverview) {
			util.setF('cre_obj:table_options:number_of_rows', reportElementDb.number_of_rows);
			if (this.isCustomizeInReports && !isSessionPaths && !isSessionPagePaths) {
				util.setF('cre_obj:table_options:starting_row', reportElementDb.starting_row);
				util.setF('cre_obj:table_options:ending_row', reportElementDb.ending_row);
				// Set totalRows
				util.updateT('cre_obj:table_options:total_rows', customizeRE.totalRows);
			}
			if (isTable)  {
				util.setF('cre_obj:table_options:show_remainder_row', reportElementDb.show_remainder_row);
				util.setF('cre_obj:table_options:show_averages_row', reportElementDb.show_averages_row);
				util.setF('cre_obj:table_options:show_min_row', reportElementDb.show_min_row);
				util.setF('cre_obj:table_options:show_max_row', reportElementDb.show_max_row);
				util.setF('cre_obj:table_options:show_totals_row', reportElementDb.show_totals_row);
				util.setF('cre_obj:table_options:maximum_table_bar_graph_length', reportElementDb.maximum_table_bar_graph_length);
//				util.setF('cre_obj:table_options:display_graphs_table_side_by_side', reportElementDb.display_graphs_table_side_by_side);
			}
			else if (isSessionPaths) {
				util.setF('cre_obj:table_options:expand_paths_greater_than', reportElementDb.expand_paths_greater_than);
				util.setF('cre_obj:table_options:number_of_rows_expanded', reportElementDb.number_of_rows_expanded);
			}
		}
		else {
			// Overview report element
			var compactView = reportElementDb.compact_view;
			util.setF('cre_obj:table_options:classic_table', !compactView);
			util.setF('cre_obj:table_options:compact_view', compactView);
		}
		util.showE('cre_obj:table_options:rows', !isOverview);
		util.showE('cre_obj:table_options:misc_box', isTable);
		util.showE('cre_obj:table_options:session_paths', isSessionPaths);
		util.showE('cre_obj:table_options:overview', isOverview);
		util.showE('cre_obj:table_options:maximum_table_bar_graph_length:section', isTable);
//		util.showE('cre_obj:table_options:display_graphs_table_side_by_side:section', isTable);
	},
	saveChanges: function() {
		// Saves the active form data to reportElementDb
		// Note, in case of invalid values we keep existing values (auto fix)
		var reportElementDb = this.reportElementDb;
		if (!this.isOverview) {
			if (this.isTable) {
				reportElementDb.show_remainder_row = util.getF('cre_obj:table_options:show_remainder_row');
				reportElementDb.show_averages_row = util.getF('cre_obj:table_options:show_averages_row');
				reportElementDb.show_min_row = util.getF('cre_obj:table_options:show_min_row');
				reportElementDb.show_max_row = util.getF('cre_obj:table_options:show_max_row');
				reportElementDb.show_totals_row = util.getF('cre_obj:table_options:show_totals_row');
//				reportElementDb.display_graphs_table_side_by_side = util.getF('cre_obj:table_options:display_graphs_table_side_by_side');
				var maximumTableBarGraphLength = util.getF('cre_obj:table_options:maximum_table_bar_graph_length');
				if (maximumTableBarGraphLength != '' && util.isInteger(maximumTableBarGraphLength, 1)) {
					reportElementDb.maximum_table_bar_graph_length = maximumTableBarGraphLength;
				}
			}
			if (this.isCustomizeInReports && (this.isTable || !this.isLogDetail)) {
				var startingRow = util.getF('cre_obj:table_options:starting_row');
				var endingRow = util.getF('cre_obj:table_options:ending_row');
				if (util.isInteger(startingRow, 1) && util.isInteger(endingRow, 1)) {
					startingRow = parseInt(startingRow, 10);
					endingRow = parseInt(endingRow, 10);
					if (endingRow >= startingRow) {
						// alert('ROW NUMBERS IN creTableOptions 1' + '\nstartingRow: ' + startingRow + '\nendingRow: ' + endingRow);
						reportElementDb.starting_row = startingRow;
						reportElementDb.ending_row = endingRow;
					}
				}
			}
			var numberOfRows = util.getF('cre_obj:table_options:number_of_rows');
			if (util.isInteger(numberOfRows, 1)) {
				reportElementDb.number_of_rows = numberOfRows;
			}
			if (this.isSessionPaths) {
				var expandPathsGreaterThan = util.getF('cre_obj:table_options:expand_paths_greater_than');
				var numberOfRowsExpanded = util.getF('cre_obj:table_options:number_of_rows_expanded');
				if (util.isInteger(expandPathsGreaterThan, 0)) {
					reportElementDb.expand_paths_greater_than = expandPathsGreaterThan;
				}
				if (util.isInteger(numberOfRowsExpanded, 1)) {
					reportElementDb.number_of_rows_expanded = numberOfRowsExpanded;
				}
			}
		}
		else {
			reportElementDb.compact_view = util.getF('cre_obj:table_options:compact_view');
		}
	}
};
//
//
// crePivotTable.js (Customize Report Element PivotTable Class, used in Reports and Reports Editor)
//
//
CrePivotTable = function(queryFieldsDb) {
	var YE = YAHOO.util.Event;
	this.queryFieldsDb = queryFieldsDb; // A reference to all report fields from where we get the Field labels
	this.reportElementDb = {}; // A reference to the active report element data.
	// Init the GUI
	YE.addListener('cre_obj:pivot_table:show_pivot_table', 'click', this.toggleShowPivotTable, this);
	YE.addListener('cre_obj:pivot_table:sort_different', 'click', this.toggleSortDifferent, this);
	// populate rows list
	var rows = [5,10,20,50,100,200,500];
	var rowsList = [];
	for (var i = 0; i < rows.length; i++) {
		rowsList[i] = {name:rows[i], label:rows[i]};
	}
	util.populateSelect('cre_obj:pivot_table:number_of_rows', rowsList, 'name', 'label');
	// populate sort direction
	creUtil.buildSortDirectionList('cre_obj:pivot_table:sort_direction');
};
CrePivotTable.prototype = {
	init: function(reportElementDb) {
		this.reportElementDb = reportElementDb;
		// util.showObject(reportElementDb.columns);
		var i, len, reportField;
		//
		// Handle report fields list
		//
		// KHP 28/Feb/2013 - mainReportFieldName does not anymore exist,
		// use all fields from the columns for lookup.
		// var mainReportFieldName = reportElementDb.report_field;
		var reportElementColumns = reportElementDb.columns;
		var reportFieldsInCols = {};
		for (i = 0, len = reportElementColumns.length; i < len; i++) {
			reportFieldsInCols['_' + reportElementColumns[i].report_field] = true;
		}
		// util.showObject(reportFieldsInCols);
		var fieldList = [{name:'', label:langVar('lang_stats.customize_report_element.select_field')}];
		var queryFieldsDb = this.queryFieldsDb;
		var pivotTable = reportElementDb.pivot_table;
		var pivotReportFieldName = pivotTable.report_field;
		// util.showObject(pivotTable);
		for (i = 0, len = queryFieldsDb.length; i < len; i++) {
			var reportField = queryFieldsDb[i];
			if (!reportField.isAggregatingField) {
				var reportFieldName = reportField.name;
				// KHP 28/Feb/2013 - session categories aren't relevant anymore
				// var category = reportField.category;
				// var isNonSessionField = (category.indexOf('session_') == -1);
				if (!reportFieldsInCols.hasOwnProperty('_' + reportFieldName)) {
					fieldList[fieldList.length] = {name:reportFieldName, label:reportField.label};
				}
			}
		}
		util.populateSelect('cre_obj:pivot_table:report_field', fieldList, 'name', 'label');
		util.setF('cre_obj:pivot_table:report_field', pivotReportFieldName);
		// Handle number of rows
		util.setF('cre_obj:pivot_table:number_of_rows', pivotTable.number_of_rows);
		// Handle omit_parenthesized_items
		util.setF('cre_obj:pivot_table:omit_parenthesized_items', pivotTable.omit_parenthesized_items);
		//
		// Handle sort by
		//
		var sortBy = pivotTable.sort_by;
		var sortDirection = pivotTable.sort_direction;
		var isSortDifferent = (sortBy != '');
		var sortByList = this.getSortByList(this.queryFieldsDb, reportElementColumns);
		var sortBySelected = '';
		if (sortBy != '') {
			if (sortBy != pivotReportFieldName) {
				sortBySelected = sortBy;
			}
			else {
				sortBySelected = '_SELECTED__DRILL_DOWN_FIELD_';
			}
		}
		util.populateSelect('cre_obj:pivot_table:sort_by', sortByList, 'name', 'label');
		util.setF('cre_obj:pivot_table:sort_different', isSortDifferent);
		util.setF('cre_obj:pivot_table:sort_by', sortBySelected);
		util.setF('cre_obj:pivot_table:sort_direction', sortDirection);
		this.sortDifferent(isSortDifferent);
		//
		// Set aggregation rows
		//
		util.setF('cre_obj:pivot_table:show_averages_row', pivotTable.show_averages_row);
		util.setF('cre_obj:pivot_table:show_min_row', pivotTable.show_min_row);
		util.setF('cre_obj:pivot_table:show_max_row', pivotTable.show_max_row);
		util.setF('cre_obj:pivot_table:show_totals_row', pivotTable.show_totals_row);
		//
		// Handle show pivot table
		//
		var isShowPivotTable = pivotTable.show_pivot_table;
		util.setF('cre_obj:pivot_table:show_pivot_table', isShowPivotTable);
		this.showPivotTable(isShowPivotTable);
	},
	toggleShowPivotTable: function(evt, self) {
		var isShowPivotTable = this.checked;
		self.showPivotTable(isShowPivotTable);
	},
	toggleSortDifferent: function(evt, self) {
		var isSortDifferent = this.checked;
		self.sortDifferent(isSortDifferent);
	},
	showPivotTable: function(isShowPivotTable) {
		var a = [
			'cre_obj:pivot_table:report_field',
			'cre_obj:pivot_table:number_of_rows',
			'cre_obj:pivot_table:show_averages_row',
			'cre_obj:pivot_table:show_min_row',
			'cre_obj:pivot_table:show_max_row',
			'cre_obj:pivot_table:show_totals_row',
			'cre_obj:pivot_table:sort_different',
			'cre_obj:pivot_table:omit_parenthesized_items'
		];
		util.enableE(a, isShowPivotTable);
		// Update Sort By section
		var isSortDifferent = isShowPivotTable ? util.getF('cre_obj:pivot_table:sort_different') : false;
		// alert('isSortDifferent: ' + isSortDifferent);
		this.sortDifferent(isSortDifferent);
	},
	sortDifferent: function(isSortDifferent) {
		util.enableE(['cre_obj:pivot_table:sort_by', 'cre_obj:pivot_table:sort_direction'], isSortDifferent);
	},
	getSortByList: function(queryFieldsDb, columns) {
		// Note, the second list item in sort by sorts the 'selected drill down field'
		var list = [{name:'', label:langVar('lang_stats.customize_report_element.select_field')}, {name:'_SELECTED__DRILL_DOWN_FIELD_', label:langVar('lang_stats.customize_report_element.selected_drill_down_field')}];
		// Add any enabled aggregating column
		for (var i = 0; i < columns.length; i++) {
			var column = columns[i];
			if (column.show_percent_column != null) {
				if (column.show_column ||
					column.show_percent_column ||
					column.show_bar_column) {
					var fieldName = column.report_field;
					var fieldLabel = queryFieldsDb[util.h(fieldName)].label;
					list[list.length] = {name:fieldName, label:fieldLabel};
				}
			}
		}
		return list;
	},
	saveChanges: function() {
		// Saves active pivot table data to pivot table object in reportElementDb
		// We always save the "show_pivot_table" value.
		// All other properties are only saved if
		// show_pivot_table is true and if a report field is selected.
		var pivotTable = this.reportElementDb.pivot_table;
		var reportFieldName = util.getF('cre_obj:pivot_table:report_field');
		var isShowPivotTable = (util.getF('cre_obj:pivot_table:show_pivot_table') && (reportFieldName != ''));
		// alert('reportFieldName: ' + reportFieldName);
		pivotTable.show_pivot_table = isShowPivotTable;
		if (isShowPivotTable) {
			// Save all other data to pivot table object
			pivotTable.report_field = reportFieldName;
			pivotTable.number_of_rows = util.getF('cre_obj:pivot_table:number_of_rows');
			pivotTable.omit_parenthesized_items = util.getF('cre_obj:pivot_table:omit_parenthesized_items');
			pivotTable.show_averages_row = util.getF('cre_obj:pivot_table:show_averages_row');
			pivotTable.show_min_row = util.getF('cre_obj:pivot_table:show_min_row');
			pivotTable.show_max_row = util.getF('cre_obj:pivot_table:show_max_row');
			pivotTable.show_totals_row = util.getF('cre_obj:pivot_table:show_totals_row');
			var sortBy = util.getF('cre_obj:pivot_table:sort_by');
			var isSortDifferent = (util.getF('cre_obj:pivot_table:sort_different') && (sortBy != ''));
			var sortDirection = '';
			if (isSortDifferent) {
				if (sortBy == '_SELECTED__DRILL_DOWN_FIELD_') {
					sortBy = reportFieldName;
				}
				sortDirection = util.getF('cre_obj:pivot_table:sort_direction');
			}
			else {
				sortBy = '';
				sortDirection = 'descending';
			}
			pivotTable.sort_by = sortBy;
			pivotTable.sort_direction = sortDirection;
		}
		// util.showObject(pivotTable);
	}
};
//
//
// creGraphOptions.js (Customize Report Element GraphOptions Class, used in Reports and Reports Editor)
//
//
CreGraphOptions = function(queryFieldsDb, defaultGraphsDb) {
	// this.defaultGraphs = defaultGraphs; // A reference to defaultGraphs
	var YE = YAHOO.util.Event;
	this.queryFieldsDb = queryFieldsDb; // A reference to all report fields from where we get the labels and category
	this.defaultGraphsDb = defaultGraphsDb;
	this.reportElementDb = {};  // A reference to the active report element data.
								// Note, if no table is shown we may set sort_by and sort_direction
								// of the report element via Graph Options.
	this.activeGraphsDb = {}; // The active graphsDb object which is created upon init(). It is
							// equal the defaultGraphsDb but overriden with active reportElementDb
							// graphs data.
	this.showTable = false;  // Set upon init(), We need to know if the report element shows a table as well
							// because if no table is shown we need a full Sort By list in Graph Options
	this.isChronoGraphSupport = false; // Set upon init(), it is true if the report element supports a chrono graph
	// this.activeMetaGraphType = ''; NOT REQUIRED because the active metaGraphType is defined in activeGraphsDb!
	//
	// Init the GUI
	//
	creUtil.buildSortDirectionList('cre_obj:graph_options:sort_direction');
	YE.addListener('cre_obj:graph_options:type', 'change', this.graphTypeActor, this);
	YE.addListener('cre_obj:graph_options:sort_by', 'change', this.sortByActor, this);
	YE.addListener('cre_obj:graph_options:sort_direction', 'change', this.sortDirectionActor, this);
	// YE.addListener('cre_obj:graph_options:show_legend', 'click', this.showLegendActor, this);
};
CreGraphOptions.prototype = {
	init: function(reportElementDb) {
		// alert('creUtil.GraphOptions.prototype.init');
		this.reportElementDb = reportElementDb;
		var activeGraphsDb = this.createActiveGraphsDbObject(reportElementDb);
		this.activeGraphsDb = activeGraphsDb;
		this.showTable = reportElementDb.show_table;
		this.updateGraphTypeList(activeGraphsDb.isChronoGraphSupport);
		// util.showObject(activeGraphsDb);
		var metaGraphType = activeGraphsDb.metaGraphType;
		util.setF('cre_obj:graph_options:type', metaGraphType);
		this.updateForm(metaGraphType);
	},
	updateForm: function(metaGraphType) {
		// alert('updateForm() - metaGraphType: ' + metaGraphType);
		// var activeGraphsDb = this.activeGraphsDb;
		// util.showObject(activeGraphsDb);
		// Get a reference to the active graph object from which we read the data
		var graphObj = this.getGraphObjectReference(metaGraphType);
		// util.showObject(graphObj);
		var isChronoGraph = (metaGraphType.indexOf('chrono_') != -1);
		var isPieChart = (metaGraphType == 'pie');
		var isSortAllDescending = graphObj.sort_all_descending;
		this.updateSortByForm(isChronoGraph, isSortAllDescending);
		util.setF('cre_obj:graph_options:y_axis_height', graphObj.y_axis_height);
		util.setF('cre_obj:graph_options:x_axis_length', graphObj.x_axis_length);
		util.setF('cre_obj:graph_options:display_graphs_side_by_side', this.reportElementDb.display_graphs_side_by_side);
		util.showE('cre_obj:graphs:variables_and_legend_box', !isChronoGraph);
		var showLegend = !isChronoGraph ? graphObj.show_legend : false;
		if (!isChronoGraph) {
			util.setF('cre_obj:graph_options:max_variables', graphObj.max_variables);
			util.setF('cre_obj:graph_options:show_remainder', graphObj.show_remainder);
			// util.disableE('cre_obj:graph_options:show_remainder', isPieChart);
			util.showE('cre_obj:graph_options:show_remainder:section', !isPieChart);
			util.setF('cre_obj:graph_options:show_legend', showLegend);
			util.setF('cre_obj:graph_options:show_values_in_legend', graphObj.show_values_in_legend);
			util.setF('cre_obj:graph_options:show_percent_in_legend', graphObj.show_percent_in_legend);
			util.setF('cre_obj:graph_options:max_legend_rows', graphObj.max_legend_rows);
			if (isPieChart) {
				util.setF('cre_obj:graph_options:show_3d', graphObj.show_3d);
			}
		}
		if (!isPieChart) {
			util.setF('cre_obj:graph_options:show_percent_on_y_axis', graphObj.show_percent_on_y_axis);
		}
		util.showE('cre_obj:graph_options:show_3d_section', isPieChart);
		util.showE('cre_obj:graph_options:show_percent_on_y_axis:section', !isPieChart);
		// util.disableE('cre_obj:graph_options:show_legend', isChronoGraph);
	},
	updateSortByForm: function(isChronoGraph, isSortAllDescending) {
		if (isChronoGraph || this.showTable) {
			this.updateSortByList(isChronoGraph, isSortAllDescending);
		}
		else {
			// There is no table and no chrono graph, so we allow to select a report field
			// and the sort direction (sortAllDescending is still possible)
			this.updateSortByListWithReportFields(isSortAllDescending);
		}
	},
	graphTypeActor: function(evt, self) {
		// alert('graphTypeActor()');
		var newMetaGraphType = util.getF('cre_obj:graph_options:type');
		// Before we update the form save current data to activeGraphs
		self.saveChangesToGraphObject(self.activeGraphsDb.metaGraphType);
		// Update the form
		self.updateForm(newMetaGraphType);
		self.activeGraphsDb.metaGraphType = newMetaGraphType;
	},
	sortByActor: function(evt, self) {
		var sortBy = util.getF('cre_obj:graph_options:sort_by');
		// var graphsDb = self.graphsDb;
		var metaGraphType = self.activeGraphsDb.metaGraphType;
		var isChronoGraph = (metaGraphType.indexOf('chrono_') != -1);
		var graphObj = self.getGraphObjectReference(metaGraphType);
		var isSortAllDescending = (sortBy == '__ALL__DESC__TRUE__') ? true : false;
		if (!isChronoGraph && !self.showTable && !isSortAllDescending) {
			//
			// Handle sort by with report fields
			//
			var reportElementDb = self.reportElementDb;
			reportElementDb.sort_by = sortBy;
			// Set sort direction in case that it has not yet been set upon init
			util.setF('cre_obj:graph_options:sort_direction', reportElementDb.sort_direction);
			util.showEV('cre_obj:graph_options:sort_direction', !isSortAllDescending);
			util.enableE('cre_obj:graph_options:sort_direction', !isSortAllDescending);
		}
		graphObj.sort_all_descending = isSortAllDescending;
	},
	sortDirectionActor: function(evt, self) {
		// We update sort_direction of reportElementDb, this case is only possible
		// if no table is displayed and a report field is selected in sort_by list.
		self.reportElementDb.sort_direction = util.getF('cre_obj:graph_options:sort_direction');
	},
//	showLegendActor: function(evt, self) {
//
//		var graphObj = self.getGraphObjectReference(self.activeGraphsDb.metaGraphType);
//		// util.showObject(graphObj);
//		graphObj.show_legend = this.checked;
//	},
	//
	//
	// Check sort by integrity
	//
	//
	//checkSortByIntegrity: function() {
		// This function is called from creGraphs.js and checks the sort_by integrity when a graph field in "graphs" becomes deselected.
		// We only need to check the sort_by integrity if no table is active (this.showTable = false) and if we have a non-chrono graph,
		// in this case we need to update the sort_by list according the active graph fields.
		// alert('creGraphOptions.js - checkSortByIntegrity()');
	//},
	// 
	//
	// Utilities
	//
	//
	getGraphObjectReference: function(metaGraphType) {
		var activeGraphsDb = this.activeGraphsDb;
		var graphObj;
		if (metaGraphType == 'chrono_bar' || metaGraphType == 'chrono_line') {
			graphObj = activeGraphsDb.chrono_bar_line_graph;
		}
		else if (metaGraphType != 'pie') {
			graphObj = activeGraphsDb.bar_line_graph;
		}
		else {
			graphObj = activeGraphsDb.pie_chart;
		}
		return graphObj;
	},
	updateGraphTypeList: function(isChronoGraphSupport) {
		var list = [];
		if (isChronoGraphSupport) {
			list[0] = {name:'chrono_bar', label:langVar('lang_stats.customize_report_element.chrono_bar_graph')};
			list[1] = {name:'chrono_line', label:langVar('lang_stats.customize_report_element.chrono_line_graph')};
		}
		list[list.length] = {name:'bar', label:langVar('lang_stats.customize_report_element.bar_graph')};
		list[list.length] = {name:'line', label:langVar('lang_stats.customize_report_element.line_graph')};
		// KHP 06/Feb/2013 - Don't allow pie for chronological fields
		// KHP 28/Feb/2013 - Looks like that there are profiles out which already use a pie chart for chrono fields!
		// 					 So we need to support pie charts for chronological fields, this makes the most sense anyway.
		list[list.length] = {name:'pie', label:langVar('lang_stats.customize_report_element.pie_chart')};
		util.populateSelect('cre_obj:graph_options:type', list, 'name', 'label');
	},
	updateSortByList: function(isChronoGraph, isSortAllDescending) {
		var list = [];
		var activeSortItem = isSortAllDescending ? '__ALL__DESC__TRUE__' : '__ALL__DESC__FALSE__';
		if (isChronoGraph) {
			list[0] = {name:'__ALL__DESC__FALSE__', label:langVar('lang_stats.customize_report_element.chronological')};
			list[1] = {name:'__ALL__DESC__TRUE__', label:langVar('lang_stats.customize_report_element.reverse_chronological')};
		}
		else {
			list[0] = {name:'__ALL__DESC__TRUE__', label:langVar('lang_stats.customize_report_element.all_descending')};
			list[1] = {name:'__ALL__DESC__FALSE__', label:langVar('lang_stats.customize_report_element.as_defined_for_table')};
		}
		util.populateSelect('cre_obj:graph_options:sort_by', list, 'name', 'label');
		util.setF('cre_obj:graph_options:sort_by', activeSortItem);
		// Hide sort_direction
		util.hideEV('cre_obj:graph_options:sort_direction');
		util.disableE('cre_obj:graph_options:sort_direction');
	},
	updateSortByListWithReportFields: function(isSortAllDescending) {
		// No Table tab is shown and no chrono graph is selected,
		// so the sort_by list shows 'sort_all_descending = true'
		//  which applies to the graphs object and in addition it shows
		// all active graphs report fields of reportElementDb which
		// applies to sort_by and sort_direction.
		var list = [];
		list[0] = {name:'__ALL__DESC__TRUE__', label:langVar('lang_stats.customize_report_element.all_descending')};
		// Add all active fields
		var queryFieldsDb = this.queryFieldsDb;
		var reportElementDb = this.reportElementDb;
		var columns = reportElementDb.columns;
		// Note, for now we add all graph fields, we don't care if a field is checked or not!
		for (var i = 0; i < columns.length; i++) {
			var column = columns[i];
			// Add all text and aggreagting fields because sort_by could just be any field@
			var reportFieldName = column.report_field;
			var label = queryFieldsDb[util.h(reportFieldName)].label;
			list[list.length] = {name:reportFieldName, label:label};
		}
		//
		// Set active sort item and sort direction list
		//
		util.populateSelect('cre_obj:graph_options:sort_by', list, 'name', 'label');
		var activeSortItem;
		var showSortDirection;
		if (isSortAllDescending) {
			activeSortItem = '__ALL__DESC__TRUE__';
			showSortDirection = false;
		}
		else {
			activeSortItem = reportElementDb.sort_by;
			sort_direction = reportElementDb.sort_direction;
			util.setF('cre_obj:graph_options:sort_direction', sort_direction);
			showSortDirection = true;
		}
		util.setF('cre_obj:graph_options:sort_by', activeSortItem);
		util.showEV('cre_obj:graph_options:sort_direction', showSortDirection);
		util.enableE('cre_obj:graph_options:sort_direction', showSortDirection);
	},
	saveChangesToGraphObject: function(metaGraphType) {
		// This saves the current graphs state to activeGraphsDb but not to the reportElementDb object!
		// We save data to the graph object to keep state when switching between types
		// and to get the final graph object upon exit.
		// Note, we do form validation but auto-fix invalid form values by keeping default values
		var isChronoGraph = (metaGraphType.indexOf('chrono_') != -1);
		var graphObj = this.getGraphObjectReference(metaGraphType);
		var y = util.getF('cre_obj:graph_options:y_axis_height');
		var x = util.getF('cre_obj:graph_options:x_axis_length');
		if (util.isInteger(y, 1)) {graphObj.y_axis_height = y;}
		if (util.isInteger(x, 1)) {graphObj.x_axis_length = x;}
		if (!isChronoGraph) {
			graphObj.show_legend = util.getF('cre_obj:graph_options:show_legend');
			graphObj.show_values_in_legend = util.getF('cre_obj:graph_options:show_values_in_legend');
			graphObj.show_percent_in_legend = util.getF('cre_obj:graph_options:show_percent_in_legend');
			var maxVariables = util.getF('cre_obj:graph_options:max_variables');
			if (util.isInteger(maxVariables, 1)) {graphObj.max_variables = maxVariables;}
			graphObj.show_remainder = util.getF('cre_obj:graph_options:show_remainder');
			maxLegendRows = util.getF('cre_obj:graph_options:max_legend_rows');
			if (util.isInteger(maxLegendRows, 1)) {graphObj.max_legend_rows = maxLegendRows;}
			if (metaGraphType == 'pie') {
				graphObj.show_3d = util.getF('cre_obj:graph_options:show_3d');
			}
		}
		graphObj.show_percent_on_y_axis = util.getF('cre_obj:graph_options:show_percent_on_y_axis');
	},
	saveChanges: function() {
		// This saves the current graphs state to the reportElementDb object!
		var metaGraphType = this.activeGraphsDb.metaGraphType;
		// Save graph object
		this.saveChangesToGraphObject(metaGraphType);
		// Get graphType as saved in reportElementDb
		var isChrono = (metaGraphType.indexOf('chrono_') != -1);
		var graphType = '';
		if (isChrono) {
			graphType = (metaGraphType == 'chrono_bar') ? 'bar' : 'line';
		}
		else {
			graphType = metaGraphType;
		}
		// Get the graph object
		var graphObject = this.getGraphObjectReference(metaGraphType);
		// Clean up reportElementDb from existing graphs object.
		var reportElementDb = this.reportElementDb;
		var prop;
		// Copy graphsObj to reportElementDb.graphs
		reportElementDb.graphs = util.cloneObject(graphObject);
		// Set graphType which is not part of the graphsObj
		reportElementDb.graphs.graph_type = graphType;
		// Handle sort_by in case of show_graphs=true and show_table=false
		// In this case we need to re-check sort_by and sort_direction.
        if (!isChrono && !graphObject.sort_all_descending && !this.showTable) {
            // Get sort_by and sort_direction
            reportElementDb.sort_by = util.getF('cre_obj:graph_options:sort_by');
            reportElementDb.sort_direction = util.getF('cre_obj:graph_options:sort_direction');
        }
		reportElementDb.display_graphs_side_by_side = util.getF('cre_obj:graph_options:display_graphs_side_by_side');
        // util.showObject(graphObject, "graphObject");
        // util.showObject(reportElementDb, "reportElementDb");
	},
	createActiveGraphsDbObject: function(reportElementDb) {
		// Creates an active graphsDb object with any active report element graphs data.
		// graphsDb keeps the graph data state while customizing report element graphs data
		// util.showObject(reportElementDb);
		var activeGraphsDb = util.cloneObject(this.defaultGraphsDb);
		var isChronoGraphSupport = creUtil.getIsChronoGraphSupport(this.queryFieldsDb, reportElementDb.columns);
		var graphType = '';
		var isChronoGraph = false;
		if (reportElementDb['graphs'] && reportElementDb.graphs['graph_type']) {
			var reportElementGraphs = reportElementDb.graphs;
			// util.showObject(reportElementGraphs);
			graphType = reportElementGraphs.graph_type;
			isChronoGraph = (isChronoGraphSupport && (graphType != 'pie')  && (reportElementGraphs['show_chrono_graph'] != null) && reportElementGraphs.show_chrono_graph);
			var overrideTarget;
			if (isChronoGraph) {
				overrideTarget = activeGraphsDb.chrono_bar_line_graph;
			}
			else if (graphType != 'pie') {
				overrideTarget = activeGraphsDb.bar_line_graph;
			}
			else {
				overrideTarget = activeGraphsDb.pie_chart;
			}
			// util.showObject(overrideTarget);
			for (prop in reportElementGraphs) {
				// alert('prop: ' + prop);
				if (overrideTarget[prop] != null) {
					// Override default
					// alert('Override prop "' + prop + '" with value: ' + reportElementGraphs[prop]);
					overrideTarget[prop] = reportElementGraphs[prop];
				}
			}
		}
		else {
			// There is no report element graphs, so we use default settings as defined
			// in activeGraphsDb
			graphType = isChronoGraphSupport ? activeGraphsDb.chrono_graph_type : activeGraphsDb.graph_type;
			isChronoGraph = (isChronoGraphSupport && graphType != 'pie');
		}
		// Set meta graph type
		var metaGraphType = !isChronoGraph ? graphType : 'chrono_' + graphType;
		// alert('graphType: ' + graphType + '\nmetaGraphType: ' + metaGraphType);
		activeGraphsDb.isChronoGraphSupport = isChronoGraphSupport;
		activeGraphsDb.metaGraphType = metaGraphType;
		// util.showObject(activeGraphsDb);
		return activeGraphsDb;
	}
};
//
//
// creSessionFields.js
// This handles session fields for sessions_overview, session_paths and session_page_paths report elements,
// it is only used in Config/Reports
//
//
var creSessionFields = function() {
	//
	//
	// Return global properties and methods
	//
	//
	var allFieldsValid = false; // allFieldsValid is used in reportElement.js to verify for valid session fields
	var sessionFields = {
		// All session fields with supported report element type
		session_user_field: {sessions_overview:true, session_paths:true},
		session_date_time_field: {sessions_overview:true, session_page_paths:true},
		session_id_field: {sessions_overview:true, session_paths:true, session_page_paths:true},
		session_duration_field: {sessions_overview:true},
		session_events_field: {sessions_overview:true},
		session_entrances_field: {sessions_overview:true, session_page_paths:true},
		session_exits_field: {sessions_overview:true, session_page_paths:true},
		session_sequence_number_field: {session_paths:true},
		session_page_field: {session_paths:true, session_page_paths:true}};
	var validator = null;
	function init() {
		validator = new util.Validator();
		// Populate the session field lists with queryFieldsDb
		var a = [{name:'', label:'--- ' + langVar('lang_stats.btn.select') + ' ---'}],
			databaseFieldsDb = reports.databaseFieldsDb,
			element,
			fieldName;
		// util.showObject(databaseFieldsDb);
		// populate the lists
		for (fieldName in sessionFields) {
			element = util.getE('cre_obj:' + fieldName);
			util.populateSelect(element, a, 'name', 'label');
			util.extendSelect(element, databaseFieldsDb, 'name', 'label');
		}
	}
	function updateForm(reportElementDb) {
		// alert('creSessionFields.updateForm()');
		// util.showObject(reportElementDb);
		var boxElementId,
			fieldElementId,
			fieldName,
			fieldValue,
			reportElementType = reportElementDb.type;
		validator.reset();
		for (fieldName in sessionFields) {
			boxElementId = 'cre_obj:' + fieldName + ':box';
			fieldElementId = 'cre_obj:' + fieldName;
			if (sessionFields[fieldName][reportElementType]) {
				fieldValue = reportElementDb[fieldName];
				// Show the row and set the select element
				util.showE(boxElementId);
				util.setF(fieldElementId, fieldValue);
			}
			else {
				// Hide the session field row
				util.hideE(boxElementId);
			}
		}
	}
	function saveChanges(reportElementDb) {
		// As there are no default session fields we need to validate
		// session field existence.
		var fieldName,
			fieldElementId,
			reportElementType = reportElementDb.type;
		validator.reset();
		// creSessionFields.allFieldsValid = false;
		allFieldsValid = false;
		for (fieldName in sessionFields) {
			if (sessionFields[fieldName][reportElementType]) {
				fieldElementId = 'cre_obj:' + fieldName;
				// Save and validate the field
				reportElementDb[fieldName] = validator.isValue(fieldElementId);
			}
		}
		if (validator.allValid()) {
			allFieldsValid = true;
		}
	}
	function getAllFieldsValid() {
		// return creSessionFields.allFieldsValid;
		// alert('getAllFieldsValid() - allFieldsValid: ' + allFieldsValid);
		return allFieldsValid;
	}
	function getSessionFields() {
		return sessionFields;
	}
	return {
		getAllFieldsValid: getAllFieldsValid,
		init: init,
		updateForm: updateForm,
		saveChanges: saveChanges,
		getSessionFields: getSessionFields
	}
}();
//
//
// creControl.js 
//
// Main object to handle Customize Report Element HTML object.        
// This object requires
// creGraphs.js
// creTable.js
// creTableOptions.js
// crePivotTable.js
// creGraphOptions.js
// creUtil.js
//
var creControl = {
	tabs: null,
	tabIds: [],
	queryFieldsDb: [], // Object reference to queryFieldsDb
	isCustomizeInReports: false,
	isPivotTablePermission: true,
	isGraphOptionsPermission: true,
	reportElementDb: {}, // Object reference to the active reportElementDb, it is set by initReportElementDb
	// graphsDb contains the active graphs object (default graphs overwritten by any report element graphs data)
	// graphsDb is used by creGraphs and creGraphOptions
	// graphsDb: {}, // Object reference to the active graphs object
	// Main Customize Report Element objects
	// general: null --> This tab panel object is currently handled in reportElement.js!
	general: null,
	graphs: null,
	table: null,
	tableOptions: null,
	pivotTable: null,
	graphOptions: null,
	//
	//
	// Initializes creControl object
	//
	//
	initCreObject: function(
			queryFieldsDb,
			defaultGraphsDb,
			isCustomizeInReports,
			isPivotTablePermission,
			isGraphOptionsPermission,
			hideLogDetailSortingMessage) {
		creControl.queryFieldsDb = queryFieldsDb;
		creControl.defaultGraphsDb = defaultGraphsDb;
		creControl.isCustomizeInReports = isCustomizeInReports;
		creControl.isPivotTablePermission = isPivotTablePermission;
		creControl.isGraphOptionsPermission = isGraphOptionsPermission;
		var YE = YAHOO.util.Event;
		// util.showObject(defaultGraphsDb);
		//
		// Init tabs
		//
		if (isCustomizeInReports) {
			creControl.tabIds = ['graphs', 'table', 'table_options', 'pivot_table', 'graph_options'];
		}
		else {
			creControl.tabIds = ['general', 'filters', 'graphs', 'table', 'table_options', 'pivot_table', 'graph_options', 'session_fields'];
		}
		creControl.tabs = new util.Tabs3('cre_obj:tabs', creControl.tabIds, creControl.tabActivated);
		//
		// Init Customize Report Element objects
		//
		creControl.graphs = new CreGraphs(queryFieldsDb);
		creControl.table = new CreTable(queryFieldsDb, isCustomizeInReports, hideLogDetailSortingMessage);
		creControl.tableOptions = new CreTableOptions(queryFieldsDb, isCustomizeInReports);
		creControl.pivotTable = new CrePivotTable(queryFieldsDb);
		creControl.graphOptions = new CreGraphOptions(queryFieldsDb, defaultGraphsDb);
		// Set graphOptions reference in graphs object!
		// creControl.graphs.setGraphOptionsReference(creControl.graphOptions);
		//
		// Init General panel, Filters panel and Manage Fields Editor
		//
		if (!isCustomizeInReports) {
			creGeneral.init();
			creFilters.init();
			creSessionFields.init();
			reportElementMF.init();
			YE.addListener('cre_obj:graphs:manage_fields_btn', 'click', creControl.manageFields);
			YE.addListener('cre_obj:table:manage_fields_btn', 'click', creControl.manageFields);
		}
	},
	//
	//
	// Init a reportElementDb object (Set all tab objects according the reportElementDb object)
	// init is called upon every report element meta type change!
	//
	//
	init: function(reportElementDb, obj) {
		// obj.defaultTabId and obj.totalRows are optional, respectively they are only specified
		// if the CRE object is used in dynamic reports or in Config when init() is called
		// from Manage Fields panel.
		// alert('creControl.init() - obj.defaultTabId: ' + obj.defaultTabId);
		// util.showObject(reportElementDb);
		var isCustomizeInReports = creControl.isCustomizeInReports;
		var totalRows = obj.hasOwnProperty('totalRows') ? obj.totalRows : 0;
		var defaultTabId = obj.hasOwnProperty('defaultTabId') ? obj.defaultTabId : '';
		creControl.reportElementDb = reportElementDb;
		// util.showObject(reportElementDb);
		var reportElementType = reportElementDb.type;
		// Handle the general and filters tab dependencies
		if (!isCustomizeInReports) {
			// Note, creGeneral and creFilters form update is not required here, it is only
			// accomplished upon openeing the CRE element in callee!
			// We only handle some display issues depending on the report element type at this point
			creGeneral.setReportElementTypeDependencies(reportElementType);
			creFilters.setReportElementTypeDependencies(reportElementType);
		}
//		if (reportElementType == 'overview') {
//
//			creControl.table.init(reportElementDb);
//		}
		if (reportElementType == 'table') {
			// We update all panels regardless if all panels
			// are displayed or not because the tab panels
			// could be changed any time via show_graphs and show_table
			creControl.table.init(reportElementDb, totalRows);
			creControl.tableOptions.init(reportElementDb, totalRows);
			creControl.graphs.init(reportElementDb);
			creControl.graphOptions.init(reportElementDb);
			creControl.pivotTable.init(reportElementDb);
		}
		else if (reportElementType == 'overview' ||
				reportElementType == 'log_detail') {
			creControl.table.init(reportElementDb);
			creControl.tableOptions.init(reportElementDb, totalRows);
		}
		else if (reportElementType == 'session_paths' || reportElementType == 'session_page_paths') {
			creControl.tableOptions.init(reportElementDb, totalRows);
			if (!isCustomizeInReports) {
				creSessionFields.updateForm(reportElementDb);
			}
		}
		else if (reportElementType == 'sessions_overview' && !isCustomizeInReports) {
			creSessionFields.updateForm(reportElementDb);
		}
		//
		// Set the tab sequence
		//
		creControl.updateTabSequence(defaultTabId);
	},
	saveAndReInitGraphOptions: function() {
		// This saves and re-inits the graphs due a change in
		// reportElementDb.show_graphs and  reportElementDb.show_table property.
		// A change in this two properties (which are conrolled outside creControl)
		// most likely changes the graphs sorting parameters.
		creControl.graphOptions.saveChanges();
		creControl.graphOptions.init(creControl.reportElementDb);
	},
	//
	//
	// Save changes to reportElementDb
	//
	//
	saveChanges: function() {
		// Note, some CRE interactions are already saved in ReportElementDb,
		// so we only save the remaining data to the ReportElementDb
		var isCustomizeInReports = creControl.isCustomizeInReports;
		var reportElementDb = creControl.reportElementDb;
		var reportElementType = reportElementDb.type;
		// util.showObject(reportElementDb);
		if (!isCustomizeInReports) {
			// Save creGeneral and creFilters data
			creGeneral.saveChanges(reportElementDb);
			creFilters.saveChanges(reportElementDb);
		}
		if (reportElementType == 'table') {
			// Note, we save graphs and table, also if only
			// one of the two is actually active. This shouldn't
			// matter because default data alwyas exist for graphs
			// and table. The only exception is the log_detail report
			// element which only has table and tableOptions data.
			creControl.tableOptions.saveChanges();
			creControl.pivotTable.saveChanges();
			creControl.graphOptions.saveChanges();
		}
		else if (reportElementType == 'overview' ||
				reportElementType == 'log_detail') {
			creControl.tableOptions.saveChanges();
		}
		else if (reportElementType == 'session_paths' || reportElementType == 'session_page_paths') {
			creControl.tableOptions.saveChanges();
			if (!isCustomizeInReports) {
				creSessionFields.saveChanges(reportElementDb);
			}
		}
		else if (reportElementType == 'sessions_overview' && !isCustomizeInReports) {
			creSessionFields.saveChanges(reportElementDb);
		}
	},
	//
	//
	// Manage Fields
	//
	//
	manageFields: function() {
		// Track the tabId so that we show the right tab upon saving the fields in reportElementMF.js
		var defaultTabId = (this.id == 'cre_obj:graphs:manage_fields_btn') ? 'graphs' : 'table';
		reportElementMF.openPanel(defaultTabId);
	},
	//
	//
	// Tabs handling
	//
	//
	tabActivated: function(tabId) {
		// alert('customizeRE.tabActivated() - tabActivated: ' + tabId);
		// KHP-RC, IE6 invokes tabActivated() twice, the argument tabId is
		// first an object and the second time the actual tabId string.
		// The reason is not yet known, it could be something wrong with
		// the event function or with the self reference object in the tab3 class.
		// This has to be fixed. In the meantime we work around this 
		// problem by checking the argument to be not an object, via !isObject()
		if (!util.isObject(tabId)) { // isObject has be addded due the above described IE problem
			creControl.setTabPanel(tabId);
			// if (tabId == 'graphs') {
				// KHP-RC - resolve this function!
				// customizeRE.graphs.checkGraphsFormIntegrity();
			// }
		}
	},
	setTabPanel: function(tabId) {
		var tabIds = creControl.tabIds;
		for (var i = 0; i < tabIds.length; i++) {
			util.hideE('cre_obj:' + tabIds[i] + ':panel');
		}
		util.showE('cre_obj:' + tabId + ':panel');
		creControl.tabs.setActiveTab(tabId);
	},
	updateTabSequence: function(optionalDefaultTabId) {
		// Note, defaultTabId is optional!
		var defaultTabId = optionalDefaultTabId ? optionalDefaultTabId : '';
		// setTabSequence is called upon init() but also from outside creControl
		// upon show_table or show_graphs changes. In this case we don't need
		// to update any tab panel data but only change the tab sequence
		// and set a defaultTabId if not yet given.
		var reportElementDb = creControl.reportElementDb;
		var reportElementType = reportElementDb.type;
		var isCustomizeInReports = creControl.isCustomizeInReports;
		// var showCreObjectPanel;
		var tabSequence = [];
		if (!isCustomizeInReports) {
			// We always show the general and filters tab in Config!
			tabSequence[0] = 'general';
			tabSequence[1] = 'filters';
		}
		if (reportElementType == 'table') {
			var showGrahps = reportElementDb.show_graphs;
			var showTable = reportElementDb.show_table;
			if (showGrahps) {
				tabSequence[tabSequence.length] = 'graphs';
			}
			if (showTable) {
				tabSequence[tabSequence.length] = 'table';
				tabSequence[tabSequence.length] = 'table_options';
				if (creControl.isPivotTablePermission) {
					tabSequence[tabSequence.length] = 'pivot_table';
				}
			}
			if (showGrahps && creControl.isGraphOptionsPermission) {
				tabSequence[tabSequence.length] = 'graph_options';
			}
		}
//		else if (reportElementType == 'overview') {
//
//			tabSequence[tabSequence.length] = 'table';
//		}
		else if (reportElementType == 'overview' ||
				reportElementType == 'log_detail') {
			// overview and log_detail are limited to table and table_options
			tabSequence[tabSequence.length] = 'table';
			tabSequence[tabSequence.length] = 'table_options';
		}
		else if (reportElementType == 'session_paths'  || reportElementType == 'session_page_paths') {
			tabSequence[tabSequence.length] = 'table_options';
			if (!isCustomizeInReports) {
				tabSequence[tabSequence.length] = 'session_fields';
			}
		}
		else if (reportElementType == 'sessions_overview' && !isCustomizeInReports) {
			tabSequence[tabSequence.length] = 'session_fields';
		}
		if (defaultTabId == '') {
			// Use first tab of tabSequence
			defaultTabId = tabSequence[0];
		}
		// Set tab sequence and tab panel
		creControl.tabs.setSequence(tabSequence, defaultTabId);
		creControl.setTabPanel(defaultTabId);
	}
};
// 
// logDetailSortingMsg
//
var logDetailSortingMsg = function() {
	var GD = { // General global data
		panel: null,
		isReportsGUI: false,
		isColumnsSort: false,
		hideMessage: false,
		// Variables required for the 
		// the callee response
		reportElementName: '',
		sortBy: '',
		sortDirection: ''
	};
	function init() {
		var panelObj = {
			panelId: 'log_detail_sorting_msg:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: langVar('lang_stats.log_detail_sorting.label'),
			zIndex: 60,
			isCover: true,
			closeEvent: logDetailSortingMsg.closeSimple
		};
		GD.panel = new util.Panel3(panelObj);
		YAHOO.util.Event.addListener('log_detail_sorting_msg:continue_btn', 'click', continueSorting);
		YAHOO.util.Event.addListener('log_detail_sorting_msg:cancel_btn', 'click', closeAdvanced);
		YAHOO.util.Event.addListener('log_detail_sorting_msg:ok_btn', 'click', closeAdvanced);
	}
	function setDisplay() {
		util.showE('log_detail_sorting_msg:msg_in_reports', GD.isReportsGUI);
		util.showE('log_detail_sorting_msg:msg_in_config', !GD.isReportsGUI);
		util.showE('log_detail_sorting_msg:click_continue_to_sort', GD.isColumnsSort);
		util.showE('log_detail_sorting_msg:continue_btn_section', GD.isColumnsSort);
		util.showE('log_detail_sorting_msg:ok_btn_section', !GD.isColumnsSort);
	}
	function openViaColumnsSort(isReportsGUI, reportElementName, sortBy, sortDirection) {
		// Opened in Reports upon sort column click
		GD.isReportsGUI = isReportsGUI;
		GD.isColumnsSort = true;
		GD.reportElementName = reportElementName;
		GD.sortBy = sortBy;
		GD.sortDirection = sortDirection;
		open();
	}
	function openViaCustomizeReportElement(isReportsGUI) {
		// Opened in reports or Config via sort by change in Customize Report Element
		GD.isReportsGUI = isReportsGUI;
		GD.isColumnsSort = false;
		open();
	}
	function open() {
		// If "Don't show this message again ..." hasn't been checked yet
		if (!GD.hideMessage) {
			if (GD.panel == null) {
				init();
			}
			setDisplay();
			GD.panel.prePositionAtCenter();
			GD.panel.open();
		}
	}
	function closeSimple() {
		// Closes the window via close icon only.
		// The "Don't show message again" state is ignored.
		// Close the panel
		GD.panel.close();
	}
	function closeAdvanced() {
		// Closes the window via button but not via close icon.
		// This considers the checked state of "Don't show message again"
		// Don't show message again if checked.
		if (util.getF('log_detail_sorting_msg:hide_msg_btn')) {
			setHideLogDetailSortingMessage();
		}
		// Close the panel
		GD.panel.close();
	}
	function continueSorting() {
		// Only possible when called via openViaColumnsSort(), continue
		// the actual sort operation in reports.
		// Don't show message again if checked.
		if (util.getF('log_detail_sorting_msg:hide_msg_btn')) {
			setHideLogDetailSortingMessage();
		}
		// Continue sort operation in newReport.js
		newReport.sortColumnContinued(GD.reportElementName, GD.sortBy, GD.sortDirection);
		// Close the panel
		GD.panel.close();
	}
	function setHideLogDetailSortingMessage() {
		// This sets hide_log_detail_sorting_message_in_reports on the server side.
		// users_cache.[user node name].peferences.hide_log_detail_sorting_message_in_reports = true
		var info = GD.isReportsGUI ? reportInfo : pageInfo;
		var profileName = info.profileName;
		var pageToken = info.pageToken;
		GD.hideMessage = true;
		var url = '?dp=util.log_detail_sorting.hide_log_detail_sorting_msg';
		url += '&p=' + profileName;
		var dat = 'v.fp.page_token=' + pageToken + '&';
		dat += 'v.fp.is_reports_gui=' + GD.isReportsGUI;
		util.serverPost(url, dat);
	}
	function setHideLogDetailSortingMessageResponse() {
		// Server response, no action required
	} 
	//
	//
	// Return global properties and methods (accessible from outside the logDetailSortingMsg namespace)
	//
	//
	return {
		openViaColumnsSort: openViaColumnsSort,
		openViaCustomizeReportElement: openViaCustomizeReportElement,
		setHideLogDetailSortingMessageResponse: setHideLogDetailSortingMessageResponse
	}
}();
//
//
// reportEditor.js
//
//
var reportEditor = {
	panel: null,
	// validator: null,
	tabs: null,
	init: function() {
		// reportEditor.validator = new util.Validator();
		//
		// Init the Editor panel
		//
		var panelObj = {
			panelId: 'report_editor:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: langVar('lang_admin.report_editor.edit_report_properties'),
			left: 330,
			top: 140,
			zIndex: 20,
			isCover: true,
			isSticky: true,
			closeEvent: reportEditor.closePanel
		};
		reportEditor.panel = new util.Panel3(panelObj);
		// Init tabs
		var tabIds = ['display_options', 'filters', 'header_footer'];
		reportEditor.tabs = new util.Tabs3('report_editor:tabs', tabIds, reportEditor.display.tabActivated);
		reportEditor.tabs.setSequence(tabIds, 'display_options');
		//
		// Add events
		//
		var YE = YAHOO.util.Event;
		YE.addListener('report_editor:filter_expression:show_available_fields_btn', 'click', reportsUtil.viewAvailableReportFilterFields.open);
		YE.addListener('report_editor:filter_expression:view_help_btn', 'click', util.helpWindow.openGeneralHelp);
		YE.addListener('report_editor:remove_saved_filter_items_btn', 'click', reportEditor.removeSavedFilterItems);
		YE.addListener('report_editor:okay_btn', 'click', reportEditor.saveItem);
		YE.addListener('report_editor:cancel_btn', 'click', reportEditor.closePanel);
	},
	display: {
		tabActivated: function(tabId) {
			if (!util.isObject(tabId)) { // isObject has be addded due an IE problem, see customizeRE.js for details
				reportEditor.display.setTabPanel(tabId);
			}
		},
		setTabPanel: function(tabId) {
			// alert('setTabPanel(): ' + tabId);
			reportEditor.tabs.setActiveTab(tabId);
			// tabId is: display_options | header_footer | filters
			util.hideE(['report_editor:display_options:panel', 'report_editor:filters:panel', 'report_editor:header_footer:panel']);
			util.showE('report_editor:' + tabId + ':panel');
		}
	},
	openPanel: function() {
		// Reset form elements
		util.resetF('report_editor:form');
		// Update the form
		reportEditor.updateForm();
		// Set default tab panel
		reportEditor.display.setTabPanel('display_options');
		reportEditor.panel.open();
	},
	closePanel: function() {
		reportEditor.panel.close();
		// Close view filter fields panel in case that is open
		reportsUtil.viewAvailableReportFilterFields.close();
	},
	updateForm: function() {
		// Read data from reports.activeItemDat
		var activeItemDat = reports.activeReportItemDat;
//		util.setF('report_editor:show_header_bar', activeItemDat.show_header_bar);
		util.setF('report_editor:sync_graph_axis_with_relative_date', activeItemDat.sync_graph_axis_with_relative_date);
		util.setF('report_editor:description', activeItemDat.description);
		util.setF('report_editor:date_filter', activeItemDat.date_filter);
		util.setF('report_editor:filter_expression', activeItemDat.filter_expression);
		util.setF('report_editor:header', activeItemDat.header);
		util.setF('report_editor:footer', activeItemDat.footer);
		// Update filter items
		var isFilterItems = (activeItemDat.filter_items.length > 0);
		var filterItemsLabel = isFilterItems ? langVar('lang_admin.report_editor.exists') : langVar('lang_stats.btn.none');
		util.updateT('report_editor:saved_filter_items_info', filterItemsLabel);
		util.showE('report_editor:remove_saved_filter_items_btn', isFilterItems);
	},
	removeSavedFilterItems: function() {
		reports.activeReportItemDat.filter_items = [];
		util.updateT('report_editor:saved_filter_items_info', langVar('lang_stats.btn.none'));
		util.hideE('report_editor:remove_saved_filter_items_btn');
	},
	saveItem: function() {
		// Write data to reports.activeItemDat
		var activeItemDat = reports.activeReportItemDat;
		// activeItemDat.show_header_bar = util.getF('report_editor:show_header_bar');
		activeItemDat.sync_graph_axis_with_relative_date = util.getF('report_editor:sync_graph_axis_with_relative_date');
		activeItemDat.description = util.getF('report_editor:description');
		activeItemDat.date_filter = util.getF('report_editor:date_filter');
		activeItemDat.filter_expression = util.getF('report_editor:filter_expression');
		activeItemDat.header = util.getF('report_editor:header');
		activeItemDat.footer = util.getF('report_editor:footer');
		reportEditor.closePanel();
	}
};
//
// reportElementsList.js
//
var reportElementsList = {
	moveControl: null,
	list: null,
	init: function() {
		//
		// Create the report elements list object
		//
		var obj = {
			name: 'report_elements',
			tbodyElementId: 'reports:report_element_list:tbody',
			noItemText: langVar('lang_admin.report_editor.no_report_element_defined'),
			isDefaultItem: true,
			listChangeNotificationFunction: reportElementsList.listChangeNotification,
			editItemFunction: reportElement.editItem,
			duplicateItemFunction: reportElement.duplicateItem,
			labelBuilderFunction: reportElement.labelBuilder
		}
		reportElementsList.list = new listcontrollerB.List(obj);
		reportElementsList.moveControl = new util.MoveControl('report_element_move_control', reportElementsList.moveItem);
	},
	listChangeNotification: function(listName, selectedItemIndex, numberOfItems) {
		// Called from wide item list controller upon list change
		// Set button state of move control
		reportElementsList.moveControl.setState(selectedItemIndex, numberOfItems);
	},
	moveItem: function(direction) {
		// Invoked from MoveControl
		// var direction = reportElementsList.moveControl.getMoveDirection(this.id);
		// Move item in list
		reportElementsList.list.moveItem(direction);
	}
}
//
//
// reportElement.js (Report Element Editor)
//
//
var reportElement = {
	panel: null,
	// Customize report element objects
	// graphs: null,
	// graphOptions: null,
	// tableColumns: null,
	// tableOptions: null,
	// pivotTable: null,
	// defaultColumnsDb
	// Created upon init, it contains the default columns per report element type.
	// The default columns are cloned to activeReportElement.types by initActiveReportElementDbs()
	defaultColumnsDb: {
		overview: [],
		table: [],
		log_detail: []
	},
	//
	//
	// Active form parameters
	//
	//
	mode: '', // the active mode of an item is: new | edit | duplicate (We need to know the mode when saving the item back to the listcontrollerB
	itemIndex: 0, // refers to the active item index in listcontrollerB items array, respectively to the active row index
	labelOri: '',
	activeMetaType: '',
	activeReportElementType: '',
	activeReportElementDbs: {},
	activeDisplay: '', // '' | graphs | table | graphs_and_table
	activeGraphsDb: {}, // contains the active graphs object (default graphs overriden by any report element graphs data)
			  	// graphsDb is used by 'graphs' and 'graphOptions'!
	/*
	activeReportElementDbs: { // initialized via initActiveReportElementDbs()
		overview: {},
		table: {},
		log_detail: {},
		...
		activeReportElementDbs keeps the active reportElementDb per report element type.
		So if we switch between table and log_detail we are able to maintain state between
		different report element types. If we change i.e. a table report element from file_type to
		page_directory then we only replace the report field and the first column
		of the active "table" types and then update the Customize Report Element objects accordingly.
	},
	*/
	// newItemObj is used for common properties
	// upon New Report Element or when editing an empty Report Element object
	newItemObj: {
		// Set common properties only so that
		// we can set/reset the initial form values
		// which are not covered by Customize Report Element objects
		label: langVar('lang_admin.report_editor.new_report_element'),
		label_ori: '',
		compact_view: '',
		display_side_by_side: false,
		show_header_bar: false,
		report_link: '',
		type: '',
		description: '',
		header:'',
		footer: '',
		date_filter: '',
		filter_expression: '',
		table_filter_expression: '',
		show_graphs: true,
		show_table: true
	},
	init: function() {
		//
		// Init the Editor panel
		//
		var panelObj = {
			panelId: 'ree:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: '-',
			left: 330,
			top: 120,
			zIndex: 20,
			isCover: true,
			isSticky: true,
			closeEvent: reportElement.closePanel
		};
		reportElement.panel = new util.Panel3(panelObj);
		// Create report element meta types list
		util.populateSelect('ree:report_element_meta_type', [{name:'', label:langVar('lang_admin.report_editor.select_report_element_type')}], 'name', 'label');
		util.extendSelect('ree:report_element_meta_type', reports.reportElementMetaTypesDb, 'name', 'label');
		//
		// Init Customize Report Element object
		//
		var isCustomizeInReports = false;
		var isPivotTablePermission = true; // is always true in Config
		var isGraphOptionsPermission = true; // is always true in Config
		var hideLogDetailSortingMessage = pageInfo.hideLogDetailSortingMessage;
		creControl.initCreObject(
			reports.queryFieldsDb,
			reports.defaultGraphsDb,
			isCustomizeInReports,
			isPivotTablePermission,
			isGraphOptionsPermission,
			hideLogDetailSortingMessage
		);
		//
		// Add events
		//
		var YE = YAHOO.util.Event;
		YE.addListener('ree:report_element_meta_type', 'change', reportElement.display.metaTypeActor);
		YE.addListener('ree:report_element_display', 'change', reportElement.display.reportElementDisplayActor);
		YE.addListener('ree:okay_btn', 'click', reportElement.saveItem);
		YE.addListener('ree:cancel_btn', 'click', reportElement.closePanel);
		//
		// Create default column objects
		//
		reportElement.defaultColsUtil.create();
	},
	display: {
		//
		//
		// Report Element meta type
		//
		//
		metaTypeActor: function() {
			var metaType = util.getF('ree:report_element_meta_type');
			reportElement.display.setMetaType(metaType);
		},
		setMetaType: function(metaType) {
			// alert('setMetaType() - metaType: ' + metaType);
            // util.showObject(reports.reportElementMetaTypesDb);
			if (reportElement.activeMetaType != '') {
				// Save current data to Customize Report Element objects because
				// the customize report element objects become re-initialized.
				creControl.saveChanges();
			}
			var reportElementType = '';
			var reportElementDisplay = '';
			var enableReportElementDisplayList;
			// var showTableFilterExpression = false;
			if (metaType != '') {
				showCreObjectPanel = true;
                // util.showObject({metaType:metaType});
                // util.showObject(reports.reportElementMetaTypesDb);
				var metaTypeItem = reports.reportElementMetaTypesDb[util.h(metaType)];
				reportElementType = metaTypeItem.type;
				// alert('reportElementType: ' + reportElementType);
				var reportElementDb = reportElement.activeReportElementDbs[reportElementType];
				// util.showObject(reportElementDb);
				if (reportElementType == 'table') {
					//
					// We need to update the default reportElementDb
					//
					var reportFieldName = metaTypeItem.name;
					reportElement.display.setMainColumn(reportElementDb, reportFieldName); 
					//
					// Check default reportElementDisplay
					//
					var showGraphs = reportElementDb.show_graphs;
					var showTable = reportElementDb.show_table;
					if (showGraphs && showTable) {
						reportElementDisplay = 'graphs_and_table';
					}
					else if (showGraphs) {
						reportElementDisplay = 'graphs';
					}
					else {
						reportElementDisplay = 'table';
					}
					enableReportElementDisplayList = true;
				}
				else {
					reportElementDisplay = 'table';
					enableReportElementDisplayList = false;
				}
				//
				// Update the Customize Report Element object
				//
				var obj = {}; // no obj paramters are required upon init() in Config, so we specify an empty object
				creControl.init(reportElementDb, obj);
			}
			else {
				// No meta type is selected
				reportElementDisplay = 'table';
				enableReportElementDisplayList = false;
				showCreObjectPanel = false;
				// Update the tab sequence because we don't init a reportElementDb
				creControl.updateTabSequence();
			}
			reportElement.activeMetaType = metaType;
			reportElement.activeReportElementType = reportElementType;
			util.setF('ree:report_element_display', reportElementDisplay);
			util.enableE('ree:report_element_display', enableReportElementDisplayList);
			//util.showE('ree:table_filter_expression:section', showTableFilterExpression);
			util.hideE('ree:no_report_element_selected'); // hides the no report element selected error message
			util.showEV('ree:customize_report_element:object', showCreObjectPanel);
		},
		updateMetaTypeList: function(reportFieldName) {
			// Called from Manage Fields if report element is of type table.
			// This updates the meta type so that it displays the name of
			// the first text column.
			util.setF('ree:report_element_meta_type', reportFieldName);
		},
		setMainColumn: function(reportElementDb, reportFieldName) {
			// This sets the report field of the main column by considering corrupted 
			// report elements where the main column is not the first column.
			// reportElementDb.columns[0].report_field = reportFieldName;
			var columns = reportElementDb.columns;
			var firstColumn = columns[0];
			// util.showObject(firstColumn, 'firstColumn');
			if (firstColumn.report_field != reportFieldName) {
				var i;
				if (firstColumn.report_field == '') {
					// Report field is not yet set, set first column now.
					firstColumn.report_field = reportFieldName;
					// Make sure that the set report_field does not exist in any other column.
					// We don't check the set 1st column, so i = 1
					for (i = 1; i < columns.length; i++) {
						if (columns[i].report_field == reportFieldName) {
							// Remove the column
							columns.splice(i, 1);
                            i--; // re-adjusts array index after delete
						}
					}
				}
				else {
					// Check if a column with the report field name already exists
					var reportFieldNameExists = false;
					for (i = 1; i < columns.length; i++) {
						if (columns[i].report_field == reportFieldName) {
							reportFieldNameExists = true;
							break;
						}
					}
					// Set reportFieldName in 1st column if it doesn't exist in other column
					if (!reportFieldNameExists) {
						firstColumn.report_field = reportFieldName;
					}
				}
			}
		},
		//
		//
		// Report Element display (graphs, table or graphs_and_table)
		//
		//
		reportElementDisplayActor: function() {
			// Update the active reportElementDb.show_graphs and reportElementDb.show_table property
			// and update the tab squence
			var reportElementDisplay = util.getF('ree:report_element_display');
			var showGraphs = false;
			var showTable = false;
			switch (reportElementDisplay) {
				case 'graphs':
					showGraphs = true;
					break;
				case 'table':
					showTable = true;
					break;
				case 'graphs_and_table':
					showGraphs = true;
					showTable = true;
					break;
			}
			var reportElementType = reportElement.activeReportElementType;
			var reportElementDb = reportElement.activeReportElementDbs[reportElementType];
			reportElementDb.show_graphs = showGraphs;
			reportElementDb.show_table = showTable;
			creControl.updateTabSequence();
			if (showGraphs) {
				// We need to save and re-init the graph options
				// because graphs sorting depends if a table is displayed or not.
				creControl.saveAndReInitGraphOptions();
			}
		}
	},
	newItem: function() {
		// invoked from report.js
		reportElement.mode = 'new';
		var newItemObj = util.cloneObject(reportElement.newItemObj);
		reportElement.initForm(newItemObj);
	},
	editItem: function(itemIndex, item) {
		// invoked from listcontrollerB
		reportElement.mode = 'edit';
		reportElement.itemIndex = itemIndex;
		// If there is no item yet then create one now.
		if (!item.type) {
			item = util.cloneObject(reportElement.newItemObj);
		}
		// util.showObject(item);
		// alert('editItem with itemIndex: ' + itemIndex);
		reportElement.initForm(item);
	},
	duplicateItem: function(itemIndex, item) {
		// invoked from listcontrollerB
		// itemIndex is the index of the item we duplicate!
		// item is the original item, so don't modify object parameters such as label
		reportElement.mode = 'duplicate';
		reportElement.itemIndex = itemIndex;
		// var label = item.label;
		// item.label = label + ' ' + langVar('lang_stats.general.copy');
		var oriLabel = item.label;
		var newlabel = langVar('lang_stats.general.item_copy');
		newlabel = newlabel.replace(/__PARAM__1/, oriLabel);
		item.label = newlabel;
		// alert('editItem with itemIndex: ' + itemIndex);
		reportElement.initForm(item);
	},
	openPanel: function() {
		var mode = reportElement.mode;
		var panelLabel = '';
		if (mode == 'new') {
			panelLabel = langVar('lang_admin.report_editor.new_report_element');
		}
		else if (mode == 'edit') {
			panelLabel = langVar('lang_admin.report_editor.edit_report_element');
		}
		else {
			panelLabel = langVar('lang_admin.report_editor.edit_duplicated_report_element');
		}
		reportElement.panel.open({label:panelLabel});
	},
	closePanel: function() {
		reportElement.panel.close();
	},
	initForm: function(reportElementDb) {
		// util.showObject(reportElementDb);
		// Reset
		reportElement.labelOri = reportElementDb.label_ori;
		reportElement.activeMetaType = '';
		reportElement.activeReportElementType = '';
		// 
		// Initialize reportElementDbs
		//
		reportElement.initActiveReportElementDbs(reportElementDb);
		//
		//
		// Update form data which are not covered by Customize Report Element Objects
		//
		//
		util.setF('ree:report_element_label', reportElementDb.label);
		// Update "Link to report"
		if (reports.isUpdateReportLinkList) {
			reportElement.updateReportLinkList();
			// Don't update the "Link to report" while on the same report.
			reports.isUpdateReportLinkList = false;
		}
		//
		// 
		// Set metaType which handles the Customize Report Element Objects
		//
		//
		var metaType = '';
		var reportElementType = reportElementDb.type;		
		// Before we set the metaType init the general and filters tab (they are not handled within creControl by metaType!)
		creGeneral.updateForm(reportElementDb);
		creFilters.updateForm(reportElementDb);
		if (reportElementType != '') {
			if (reportElementType == 'table') {
				// metaType = reportElementDb.report_field;
				metaType = reportElement.getMetaTypeOfTableReportElement(reportElementDb.columns);
			}
			else {
				metaType = '__RET__' + reportElementType;
			}
		}
		util.setF('ree:report_element_meta_type', metaType);
		reportElement.display.setMetaType(metaType); // --> this handles creControl.init()
		//
		// Open the panel
		//
		reportElement.openPanel();
	},
	saveItem: function() {
		var metaType = reportElement.activeMetaType;
		if (metaType != '') {
			// Save Customize Report Element object
			creControl.saveChanges();
			var reportElementType = reportElement.activeReportElementType;
			// Validate session fields of session reports
			if (reportElementType == 'sessions_overview' ||
				reportElementType == 'session_paths' ||
				reportElementType == 'session_page_paths') {
				if (!creSessionFields.getAllFieldsValid()) {
					// Activate session fields tab
					creControl.setTabPanel('session_fields');
					return false;
				}
			}
			var reportElementDb = reportElement.activeReportElementDbs[reportElementType];
			// Note, reportElementDb only contains data of the Basics tab panel!
			// We create a new report element object which is saved to the report element db in listcontrollerB
			var obj = {};
			obj.label = util.getF('ree:report_element_label'); // Duplicate labels are allowed!
			obj.label_ori = reportElement.labelOri;
			obj.type = reportElementType;
			obj.display_side_by_side = reportElementDb.display_side_by_side;
			obj.show_header_bar = reportElementDb.show_header_bar;
			obj.report_link = reportElementDb.report_link;
			obj.description = reportElementDb.description;
			obj.header = reportElementDb.header;
			obj.footer = reportElementDb.footer;
			obj.date_filter = reportElementDb.date_filter;
			obj.filter_expression = reportElementDb.filter_expression;
			switch (reportElementType) {
                case 'table':
                    obj.number_of_rows = reportElementDb.number_of_rows;
                    obj.show_graphs = reportElementDb.show_graphs;
                    obj.show_table = reportElementDb.show_table;
					obj.display_graphs_side_by_side = reportElementDb.display_graphs_side_by_side;
//					obj.display_graphs_table_side_by_side = reportElementDb.display_graphs_table_side_by_side;
					obj.maximum_table_bar_graph_length = reportElementDb.maximum_table_bar_graph_length;
                    obj.omit_parenthesized_items = reportElementDb.omit_parenthesized_items;
                    obj.use_overview_for_totals = reportElementDb.use_overview_for_totals;
                    obj.table_filter_expression = reportElementDb.table_filter_expression;
                    obj.sort_by = reportElementDb.sort_by;
                    obj.sort_direction = reportElementDb.sort_direction;
                    obj.show_remainder_row = reportElementDb.show_remainder_row;
                    obj.show_averages_row = reportElementDb.show_averages_row;
                    obj.show_min_row = reportElementDb.show_min_row;
                    obj.show_max_row = reportElementDb.show_max_row;
                    obj.show_totals_row = reportElementDb.show_totals_row;
                    obj.graphs = util.cloneObject(reportElementDb.graphs);
                    obj.pivot_table = util.cloneObject(reportElementDb.pivot_table);
                    obj.columns =  util.cloneObject(reportElementDb.columns);
                    break;
                case 'overview':
					obj.compact_view = reportElementDb.compact_view;
                    obj.columns =  util.cloneObject(reportElementDb.columns);
                    break;
                case 'log_detail':
                    obj.number_of_rows = reportElementDb.number_of_rows;
                    obj.show_graphs = false;
                    obj.show_table = true;
                    obj.table_filter_expression = reportElementDb.table_filter_expression;
                    obj.sort_by = reportElementDb.sort_by;
                    obj.sort_direction = reportElementDb.sort_direction;
                    obj.columns =  util.cloneObject(reportElementDb.columns);
                    break;
                case 'sessions_overview':
                    obj.session_user_field = reportElementDb.session_user_field;
                    obj.session_date_time_field = reportElementDb.session_date_time_field;
                    obj.session_id_field = reportElementDb.session_id_field;
                    obj.session_duration_field =reportElementDb.session_duration_field;
                    obj.session_events_field =reportElementDb.session_events_field;
                    obj.session_entrances_field =reportElementDb.session_entrances_field;
                    obj.session_exits_field = reportElementDb.session_exits_field;
                    break;
                case 'session_paths':
                    obj.number_of_rows = reportElementDb.number_of_rows;
                    obj.session_user_field = reportElementDb.session_user_field;
                    obj.session_id_field = reportElementDb.session_id_field;
                    obj.session_sequence_number_field = reportElementDb.session_sequence_number_field;
                    obj.session_page_field = reportElementDb.session_page_field;
                    obj.number_of_rows_expanded = reportElementDb.number_of_rows_expanded;
                    obj.expand_paths_greater_than = reportElementDb.expand_paths_greater_than;
                    break;
                case 'session_page_paths':
                    obj.number_of_rows = reportElementDb.number_of_rows;
                    obj.session_date_time_field = reportElementDb.session_date_time_field;
                    obj.session_id_field = reportElementDb.session_id_field;
                    obj.session_entrances_field = reportElementDb.session_entrances_field;
                    obj.session_exits_field = reportElementDb.session_exits_field;
                    obj.session_page_field  = reportElementDb.session_page_field;
                    break;
			}
			// util.showObject(obj);
			// Save the item to the report element list (listcontrollerB handles saving the object)
			reportElementsList.list.saveItem(reportElement.mode, reportElement.itemIndex, obj);
			reportElement.closePanel();
		}
		else {
			// No report element type selected.
			// reportElement.display.setTabPanel('basics');
			util.showE('ree:no_report_element_selected');
		}
	},
	labelBuilder: function(reportElementItem) {
		var label = '';
		if (reportElementItem.label != '') {
			label = reportElementItem.label;
		}
		else {
			label = '<span>' + langVar('lang_admin.report_editor.no_name_defined') + '</span>';
		}
		// alert('label: ' + label);
		return label;
	},
	//
	//
	//
	// Utilities
	//
	//
	//
	defaultColsUtil: {
		getColumnObject: function(queryFieldItem) {
			var column = {};
			column.report_field = queryFieldItem.name;
			column.show_column = true;
			if (queryFieldItem.isAggregatingField) {
				column.show_percent_column = false;
				column.show_bar_column = false;
				column.show_graph = false;
			}
			return column;
		},
		getOverviewColumns: function(queryFields) {
			var a = [];
			var numQueryFields = queryFields.length;
			for (var i = 0; i < numQueryFields; i++) {
				var item = queryFields[i];
				if (item.isAggregatingField) {
					a[a.length] = {report_field: item.name, show_column: true};
				}
			}
			return a;
		},
		getTableColumns: function(queryFields) {
			// We add a fake column as the first column. This column
			// becomes replaced with the actual report element report field
			var a = [{report_field:'', show_column:true}]; 
			var getColObject = reportElement.defaultColsUtil.getColumnObject;
			var numQueryFields = queryFields.length;
			for (var i = 0; i < numQueryFields; i++) {
				var item = queryFields[i];
				var isAggregatingField = item.isAggregatingField;
				var sessionField = item.sessionField;
				var nonDefaultSessionFields = /session_(begin|end|entrances|exits)/;
				var isNonDefaultSesssionField = nonDefaultSessionFields.test(sessionField);
				if (isAggregatingField && !isNonDefaultSesssionField) {
					a[a.length] = getColObject(item);
				}
			}
			return a;
		},
		getLogDetailColumns: function(queryFields) {
			var a = [];
			// KHP-RC, refine which fields are allowed in the log_detail report element.
			// For now we add most of the fields, though some are definely not allowed!
			var getColObject = reportElement.defaultColsUtil.getColumnObject;
			var i;
			var item;
			//
			// Add non-aggregating fields
			//
			var numQueryFields = queryFields.length;
			for (i = 0; i < numQueryFields; i++) {
				item = queryFields[i];
				if (!item.isAggregatingField && item.reportElementTypeSupport.log_detail) {
					a[a.length] = getColObject(item);
				}
			}
			//
			// Add aggregating fields
			//
			for (i = 0; i < numQueryFields; i++) {
				item = queryFields[i];
				if (item.isAggregatingField && item.reportElementTypeSupport.log_detail) {
					a[a.length] = getColObject(item);
				}
			}
			return a;
		},
		create: function() {
			var queryFields = reports.queryFieldsDb;
			var defaultColumnsDb = reportElement.defaultColumnsDb;
			//
			// Set default columns
			//
			var cols = reportElement.defaultColsUtil;
			defaultColumnsDb.overview = cols.getOverviewColumns(queryFields);
			defaultColumnsDb.table = cols.getTableColumns(queryFields);
			defaultColumnsDb.log_detail = cols.getLogDetailColumns(queryFields);
		}
	},
	createDefaultReportElementDb: function(type) {
		// Creates a default report element.
		// Note, although the Overview and other report elements don't have i.e.
		// a sort_by property it doesn't harm if we add it to the defaultDb
		// because the property will be simply ignored.
		// However, when we save the item object back to the report element db
		// in listcontrollerB we only save the appropriate report element options!
		var db = {
			label: '',
			label_ori: '',
			compact_view: false,
			display_side_by_side: false,
			show_header_bar: false,
			report_link: '',
			type: type,
			description: '',
			header:'',
			footer: '',
			omit_parenthesized_items: false,
			use_overview_for_totals: false,
			date_filter: '',
			filter_expression: '',
			table_filter_expression: '',
			// report_field: '',
			show_graphs: true,
	        show_table: true,
			display_graphs_side_by_side: true,
//			display_graphs_table_side_by_side: false,
			maximum_table_bar_graph_length: '',
			sort_by: '',
			sort_direction: '',
	        number_of_rows: 10,
	        session_user_field: '',
	        session_date_time_field: '',
	        session_id_field: '',
	        session_duration_field: '',
	        session_events_field: '',
	        session_entrances_field: '',
	        session_exits_field: '',
	        session_sequence_number_field: '',
	        session_page_field: '',
	        number_of_rows_expanded: 10,
	        expand_paths_greater_than: 100,
	        show_remainder_row: true,
	        show_averages_row: false,
	        show_min_row: false,
	        show_max_row: false,
	        show_totals_row: true,
			pivot_table: {
				show_pivot_table: false,
				report_field: '',
				sort_by: '',
				sort_direction: '',
				number_of_rows: 10,
				show_averages_row: false,
				show_min_row: false,
				show_max_row: false,
				show_totals_row: true
			}
		};
		if (type == 'overview' || type == 'table' || type == 'log_detail') {
			// Get a clone of the default columns
			db.columns = util.cloneObject(reportElement.defaultColumnsDb[type]);
		}
		// util.showObject(db);
		return db;
	},
	initActiveReportElementDbs: function(currentReportElementDb) {
		// creates default active report element Dbs, except for the given
		// currentReportElementDb which contains the data as in the
		// opened report element item.
		// currentReportElementDb is an empty object when we create
		// a new report element.
		var reportElementDbs = {};
		var currentReportElementType = currentReportElementDb.type;
		var reportElementTypes = [
			'overview',
			'table',
			'log_detail',
			'sessions_overview',
			'entry_pages',
			'exit_pages',
			'session_pages',
			'session_users',
			'individual_sessions',
			'session_paths',
			'session_page_paths'];
		// var activeReportElementCommon = reportElement.activeReportElement.common;
		// var activeReportElementTypes = reportElement.activeReportElement.types;
		var activeReportElementDbs = reportElement.activeReportElementDbs;
		for (var i = 0; i < reportElementTypes.length; i++) {
			var type = reportElementTypes[i];
			// Clean up existing data
			activeReportElementDbs[type] = {};
			if (type != currentReportElementType) {
				//
				// Create a default report element for the given report element type
				//
				activeReportElementDbs[type] = reportElement.createDefaultReportElementDb(type);
			}
			else {
				//
				// Clone the given report element to activeReportElement
				//
				activeReportElementDbs[type] = util.cloneObject(currentReportElementDb);
			}
		}
	},
	getMetaTypeOfTableReportElement: function(columns) {
		// The metaType of a table report element is the first
		// aggreagting column
		var i,
			j,
			metaType = '',
			queryFieldsDb = reports.queryFieldsDb,
			queryFieldItem,
			reportFieldName;
			for (i = 0, j = columns.length; i < j; i++) {
				reportFieldName = columns[i].report_field;
				queryFieldItem = queryFieldsDb[util.h(reportFieldName)];
				// util.showObject(queryFieldItem);
				if (!queryFieldItem.isAggregatingField) {
					metaType = reportFieldName;
					break;
				}
			}
		return metaType;
	},
	addToReportLinkList: function(items, newReportLinkList, reportGroupLabel, reportLabelsLookup, duplicateLabels) {
		var maxItems = items.length,
			item,
			itemId,
			itemDat,
//			reportName,
			reportLabel;
		for (var i = 0; i < maxItems; i++) {
			item = items[i];
			if (item.type == 'report') {
				// Note, all report_link values have been replaced with the item id of the report
				// for editing purposes only. So we use the report id in the select element as value.
				itemDat = item.dat;
				itemId = item.id;
//				reportName = itemDat.report_name;
				reportLabel = itemDat.label;
				// Handle/check duplicate labels
				if (reportLabelsLookup.hasOwnProperty('_' + reportLabel)) {
					// This is a duplicate report label, add it to duplicateLabels
					duplicateLabels['_' + reportLabel] = true;
					isDuplicateLabels = true;
				}
				else {
					// Track label in reportLabelsLookup
					reportLabelsLookup['_' + reportLabel] = true;
				}
				// We add the groupLabel to be used for duplicate labels
				newReportLinkList[newReportLinkList.length] = {name: itemId, label: reportLabel, groupLabel: reportGroupLabel};
			}
			else {
				// This is a report group
				reportElement.addToReportLinkList(
					item.subitems,
					newReportLinkList,
					item.label,
					reportLabelsLookup,
					duplicateLabels);
			}
			itemDat = items[i].dat;
		}
	},
	updateReportLinkList: function() {
		// This updates the report_link list
		function compareLabels(a, b) {
			var labelA = a.label.toLowerCase(),
				labelB = b.label.toLowerCase();
			if (labelA < labelB) {return -1}
			if (labelA > labelB) {return 1}
			return 0;
		}
		var noneLabel = '(' + langVar('lang_stats.btn.none') + ')',
			a = [{name:'', label:noneLabel}],
			b = [],
			reportLabelsLookup = {},
			duplicateLabels = {}, // This tracks all duplicate labels
			isDuplicateLabels = false,
			reportLabel = '',
			item = null;
		reportElement.addToReportLinkList(reports.itemsDb, b, '', reportLabelsLookup, duplicateLabels);
		for (var prop in duplicateLabels) {
			isDuplicateLabels = true;
			break;
		}
		if (isDuplicateLabels) {
			// Add the report group label next to duplicate report labels.
			for (var i = 0, max = b.length; i < max; i++) {
				item = b[i];
				reportLabel = item.label;
				if (duplicateLabels.hasOwnProperty('_' + reportLabel) &&
					item.groupLabel != '') {
					// Set new label
					b[i].label = reportLabel + ' (' + item.groupLabel + ')';
				}
			}
		}
		// Sort list by label
		b.sort(compareLabels);
		// util.showObject(b);
		util.populateSelect('cre_obj:general:report_link', a, 'name', 'label');
		util.extendSelect('cre_obj:general:report_link', b, 'name', 'label');
	}
}
//
// reportElementMF.js (Report Element Manage Fields)
//
var reportElementMF = {
	panel: null,
	defaultTabId: '', // defaultTabId in CustomizeReportElement panel
	reportElementDb: null, // A reference to the active reportElementDb
	/*
	allFieldsDb
	The allFieldsDb object contains all permitted fields for the given report element type,
	it also contains the column state of the active fields (show_column, etc.) and the state
	if the field is active.
	/*
	allFieldsDb = [
		{
		id: i0,
		name: page,
		showColumn: true,
		isActiveField: false --> defines if the field is part of report element columns or not.
		),
		{
		id: i26,
		name: hits
		showColumn: true,
		showPercentColumn: false,
		showBarColumn: false
		isActiveField: true
		},
		...
	*/
	allFieldsDb: [],
	/*
	activeFieldsDb
	The activeFields object contains all active field Ids in order. It does not contain any other property.
	activeFields = ['i0', 'i15', 'i18', 'i19', 'i21'];
	*/
	activeFieldsDb: [],
	// allFieldsSelectedItemId: '',
	// activeFieldsSelectedItemId: '',
	mainReportFieldName: '',
	// activeListType defines which of the two lists are active. It is used to handle the Add/Remove button event and label
	activeListType: '', // all_fields | active_fields
	init: function() {
		//
		// Init the Editor panel
		//
		var panelObj = {
			panelId: 'remf:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: langVar('lang_admin.report_editor.manage_fields'),
			left: 360,
			top: 145,
			zIndex: 40,
			isCover: true,
			isSticky: true,
			closeEvent: reportElementMF.closePanel
		};
		reportElementMF.panel = new util.Panel3(panelObj);
		//
		// Add events
		//
		var YE = YAHOO.util.Event;
		YE.addListener('remf:all_fields_list', 'click', reportElementMF.allFields.selectActor);
		YE.addListener('remf:active_fields_list', 'click', reportElementMF.activeFields.selectActor);
		YE.addListener('remf:add_remove_btn', 'click', reportElementMF.addRemoveItem);
		YE.addListener('remf:move_up_btn', 'click', reportElementMF.activeFields.moveItem);
		YE.addListener('remf:move_down_btn', 'click', reportElementMF.activeFields.moveItem);
		YE.addListener('remf:okay_btn', 'click', reportElementMF.saveFields);
		YE.addListener('remf:cancel_btn', 'click', reportElementMF.closePanel);
	},
	openPanel: function(defaultTabId) {
		// defaultTabId is required for Customize Report Element panel upon save.
		reportElementMF.defaultTabId = defaultTabId;
		// Reset objects and buttons
		reportElementMF.allFieldsDb = [];
		reportElementMF.activeFieldsDb = [];
		reportElementMF.reportElementDb = {};
		reportElementMF.activeListType = '';
		reportElementMF.allFields.selectedItemId = '';
		reportElementMF.activeFields.selectedItemIndex = -1; // Default is -1!
		util.disableE(['remf:add_remove_btn', 'remf:move_up_btn', 'remf:move_down_btn']);
		// Get a reference to the active reportElementDb type and columns
		var metaType = reportElement.activeMetaType;
		var metaTypeLabel = reports.reportElementMetaTypesDb[util.h(metaType)].label;
		var reportElementType = reportElement.activeReportElementType;
		var reportElementDb = reportElement.activeReportElementDbs[reportElementType];
		// util.showObject(reportElementDb);
		// alert('metaTypeLabel: ' + metaTypeLabel);
		// var activeColumns = reportElementDb.columns;
		// util.showObject(activeColumns);
		reportElementMF.reportElementDb = reportElementDb;
		// Set main label
		var mainLabel = langVar('lang_admin.report_editor.report_element_type') + ': <strong>' + metaTypeLabel + '</strong>';
		var mainLabelElement = util.getE('remf:main_label');
		mainLabelElement.innerHTML = mainLabel;
		// Create allFields and activeFields object
		reportElementMF.createAllFieldsObject(reportElementType, reportElementDb.columns);
		reportElementMF.createActiveFieldsObject(reportElementDb.columns);
		// Build initial allFields and activeFields list
		reportElementMF.buildAllFieldsList();
		reportElementMF.buildActiveFieldsList();
		reportElementMF.panel.open();
	},
	closePanel: function() {
		reportElementMF.panel.close();
	},
	saveFields: function() {
		var allFieldsDb = reportElementMF.allFieldsDb;
		// util.showObject(allFieldsDb);
		var activeFieldsDb = reportElementMF.activeFieldsDb;
		var firstTextReportField = '';
		var columns = [];
		var reportElementType = reportElement.activeReportElementType;
		var isOverview = (reportElementType == 'overview');
		var isTable = (reportElementType == 'table');
		var i, len;
		for (i = 0, len = activeFieldsDb.length; i < len; i++) {
			var itemId = activeFieldsDb[i];
			var item = allFieldsDb[util.h(itemId)];
			var columnItem = {};
			columnItem.report_field = item.name;
			columnItem.show_column = item.showColumn;
			if (item.isAggregatingField) {
				if (!isOverview) {
					columnItem.show_percent_column = item.showPercentColumn;
					columnItem.show_bar_column = item.showBarColumn;
					columnItem.show_graph = item.showGraph;
				}
			}
			else if (firstTextReportField == '') {
				firstTextReportField = item.name;
			}
			columns[i] = columnItem;
		}
		if ((!isOverview && firstTextReportField != '') ||
			(isOverview && columns.length > 0)) {
			var reportElementDb = reportElementMF.reportElementDb;
			reportElementDb.columns = columns;
			// KHP 28/Feb/2013 - If a column matches the pivot report field
			// then reset the pivot table.
			if (isTable) {
				var pivotReportFieldName = reportElementDb.pivot_table.report_field;
				if (pivotReportFieldName != '') {
					for (i = 0, len = columns.length; i < len; i++) {
						if (columns[i].report_field == pivotReportFieldName) {
							// Field is already used in table, reset pivot table field
							reportElementDb.pivot_table.report_field = '';
							break;
						}
					}
				}
			}
			// Re-init the reportElementDb
			// Save changes to reportElementDb before reapplying creControl.init()
			// or all already done changes to the report element are lost!
			creControl.saveChanges();
			var obj = {};
			obj.defaultTabId = reportElementMF.defaultTabId;
			creControl.init(reportElementDb, obj);
			if (isTable) {
				// The name of the report element type may have changed when
				// a new text column has been added. So we update the report
				// element type list to be sure that it shows the report field label
				// of the first text column
				reportElement.display.updateMetaTypeList(firstTextReportField);
			}
			reportElementMF.closePanel();
		}
		else {
			var msg;
			if (!isOverview) {
				msg = langVar('lang_admin.report_editor.no_text_field_defined_msg');
			}
			else {
				msg = langVar('lang_admin.report_editor.no_field_defined_msg');
			}
			alert(msg);
		}
	},
	addRemoveItem: function() {
		if (reportElementMF.activeListType == 'all_fields') {
			// All fields
			reportElementMF.allFields.addItem();
		}
		else {
			// Active fields
			reportElementMF.activeFields.removeItem();
		}
	},
	//
	//
	//
	// allFields list handler
	//
	//
	//
	allFields: {
		selectedItemId: '',
		selectActor: function(evt) {
			// Invoked upon list click
			var element = evt.target || evt.srcElement;
			if (element.nodeName == 'LI') {
				var elementId = element.id;
				var dat = elementId.split(':');
				var itemId = dat[2];
				// alert('itemId: ' + itemId);
				reportElementMF.allFields.selectByItemId(itemId, element);
			}
		},
		selectByItemId: function(itemId, element) {
			// 'element' is optional, respectively it is only given when invoked via selectActor()
			// itemId may be empty, this is the case if no more item exists to select.
			function deselectSelectedItem(previousSelectedItemId) {
				var previousSelectedItemElement = util.getE('remf:all_fields_list:' + previousSelectedItemId);
				previousSelectedItemElement.className = '';
			}
			// Make allFields the active list if it isn't already
			if (reportElementMF.activeListType != 'all_fields') {
				// Set button label
				util.updateT('remf:add_remove_btn', langVar('lang_stats.btn.add') + ' >');
				// Deselect any activeFields list item
				reportElementMF.activeFields.deselectSilent();
				// Disable activeFields move buttons
				util.disableE(['remf:move_up_btn','remf:move_down_btn']);
				// Set list type
				reportElementMF.activeListType = 'all_fields';
			}
			var makeItemSelected;
			var previousSelectedItemId = reportElementMF.allFields.selectedItemId;
			// alert('previousSelectedItemId: ' + previousSelectedItemId);
			if (itemId != '') {
				if (!element) {
					var element = util.getE('remf:all_fields_list:' + itemId);
				}
				if (previousSelectedItemId != '') {
					if (previousSelectedItemId == itemId) {
						// Deselect the item
						makeItemSelected = false;
					}
					else {
						// A different item is already selected, deselect it
						deselectSelectedItem(previousSelectedItemId);
						makeItemSelected = true;
					}
				}
				else {
					makeItemSelected = true;
				}
				// alert('makeItemSelected: ' + makeItemSelected);
				element.className = makeItemSelected ? 'selected' : '';
			}
			else {
				// No new item is to select but we deselect any existing item
				makeItemSelected = false;
				if (previousSelectedItemId != '') {
					deselectSelectedItem(previousSelectedItemId);
				}
			}
			reportElementMF.allFields.selectedItemId = makeItemSelected ? itemId : '';
			util.enableE('remf:add_remove_btn', makeItemSelected);
		},
		deselectSilent: function() {
			// deselectSilent is called by activeFields only! It deselects any selected list item
			// without making allFields the active list.
			var previousSelectedItemId = reportElementMF.allFields.selectedItemId;
			if (previousSelectedItemId != '') {
				var previousSelectedItemElement = util.getE('remf:all_fields_list:' + previousSelectedItemId);
				previousSelectedItemElement.className = '';
				reportElementMF.allFields.selectedItemId = '';
			}
		},
		addItem: function() {
			// Adds the selected item to activeFields
			var allFieldsDb = reportElementMF.allFieldsDb;
			var selectedItemId = reportElementMF.allFields.selectedItemId;
			var selectedItem = allFieldsDb[util.h(selectedItemId)];
			// Select the next item if one exists, or the previous item if one exists,
			// or no item if this was the last in-active item
			// Create an array with all inactive items, including the selected item to
			// check which is the next selected item.
			var inactiveItemIds = [];
			var selectedItemIndex = 0;
			var nextSelectedItemId = '';
			for (var i = 0; i < allFieldsDb.length; i++) {
				var item = allFieldsDb[i];
				if (!item.isActiveField) {
					var j = inactiveItemIds.length;
					inactiveItemIds[j] = item.id;
					if (item.id == selectedItemId) {
						selectedItemIndex = j;
					}
				}
			}
			var numInactiveItems = inactiveItemIds.length;
			if (numInactiveItems > 1) {
				var nextSelectedItemIndex = (selectedItemIndex < numInactiveItems - 1) ? selectedItemIndex + 1 : selectedItemIndex - 1;
				nextSelectedItemId = inactiveItemIds[nextSelectedItemIndex];
			}
			//
			// Modify current selectedItem
			//
			selectedItem.isActiveField = true;
			// Hide the item in allFields list
			util.hideE('remf:all_fields_list:' + selectedItemId);
			// Select the next item'
			reportElementMF.allFields.selectByItemId(nextSelectedItemId);
			// Add the item to activeFields list
			reportElementMF.activeFields.addItem(selectedItem);
		}
	},
	//
	//
	//
	// activeFields list handler
	//
	//
	//
	activeFields: {
		selectedItemIndex: -1, // -1 is set if no item is selected!
		selectActor: function(evt) {
			// Invoked upon list click
			var element = evt.target || evt.srcElement;
			if (element.nodeName == 'LI') {
				var elementId = element.id;
				var dat = elementId.split(':');
				var itemIndex = parseInt(dat[2], 10);
				// alert('itemId: ' + itemId);
				reportElementMF.activeFields.selectByItemIndex(itemIndex, element);
			}
		},
		selectByItemIndex: function(itemIndex, element) {
			// 'element' is optional, respectively it is only given when invoked via selectActor()
			function deselectSelectedItem(previousSelectedItemIndex) {
				var previousSelectedItemElement = util.getE('remf:active_fields_list:' + previousSelectedItemIndex);
				previousSelectedItemElement.className = '';
			}
			var activeFieldsDb = reportElementMF.activeFieldsDb;
			var makeItemSelected;
			var previousSelectedItemIndex = reportElementMF.activeFields.selectedItemIndex;
			// Make activeFields the active list if it isn't already
			if (reportElementMF.activeListType != 'active_fields') {
				// Set button label
				util.updateT('remf:add_remove_btn', '< ' + langVar('lang_stats.btn.remove'));
				// Deselect any allFields list item
				reportElementMF.allFields.deselectSilent();
				// Set list type
				reportElementMF.activeListType = 'active_fields';
			}
			if (itemIndex != -1) {
				if (!element) {
					var element = util.getE('remf:active_fields_list:' + itemIndex);
				}
				if (previousSelectedItemIndex != -1) {
					if (previousSelectedItemIndex == itemIndex) {
						// Deselect the item
						makeItemSelected = false;
					}
					else {
						// A different item is already selected, deselect it
						deselectSelectedItem(previousSelectedItemIndex);
						makeItemSelected = true;
					}
				}
				else {
					makeItemSelected = true;
				}
				// alert('makeItemSelected: ' + makeItemSelected);
				element.className = makeItemSelected ? 'selected' : '';
			}
			else {
				// No new item is to select but we deselect any existing item
				makeItemSelected = false;
				if (previousSelectedItemIndex != -1) {
					deselectSelectedItem(previousSelectedItemIndex);
				}
			}
			reportElementMF.activeFields.selectedItemIndex = makeItemSelected ? itemIndex : -1;
			// Handle Remove and Move buttons
			reportElementMF.activeFields.setButtonState(makeItemSelected);
		},
		setButtonState: function(isSelectedItem) {
			// setButtonState() is called within selectByItemIndex,
			// it handles Remove and Move buttons state
			// We allow to remove every field (text and numeric)
			var isRemove = isSelectedItem;
			var isMoveUp = false;
			var isMoveDown = false;
			var activeFieldsDb = reportElementMF.activeFieldsDb;
			var activeFieldsDbLen = activeFieldsDb.length;
			if (isSelectedItem && activeFieldsDbLen > 1) {
				// Items can be moved in any order, i.e. agggegating fields
				// to top and non-aggregating fields to bottom.
				var selectedItemIndex = reportElementMF.activeFields.selectedItemIndex;
				isMoveUp = (selectedItemIndex != 0);
				isMoveDown =  (selectedItemIndex != (activeFieldsDbLen - 1));
			}
			util.enableE('remf:add_remove_btn', isRemove);
			util.enableE('remf:move_up_btn', isMoveUp);
			util.enableE('remf:move_down_btn', isMoveDown);
		},
		deselectSilent: function() {
			// deselectSilent is called by allFields only! It deselects any selected list item
			// without making activeFields the active list.
			var previousSelectedItemIndex = reportElementMF.activeFields.selectedItemIndex;
			if (previousSelectedItemIndex != -1) {
				var previousSelectedItemElement = util.getE('remf:active_fields_list:' + previousSelectedItemIndex);
				previousSelectedItemElement.className = '';
				reportElementMF.activeFields.selectedItemIndex = -1;
			}
		},
		addItem: function(newAllFieldsItem) {
			// Invoked from allFields.addItem(). This adds the given item
			// to the activeFields list.
			// util.showObject(newAllFieldsItem, 'activeFields.addItem() - newAllFieldsItem argument');
			var newItemId = newAllFieldsItem.id;
			var activeFieldsDb = reportElementMF.activeFieldsDb;
			// util.showObject(activeFieldsDb, 'activeFields.addItem() - activeFieldsDb');
			var insertAtItemIndex = -1;
			if (newAllFieldsItem.isAggregatingField) {
				// Insert item as last item in activeFieldsDb
				insertAtItemIndex = activeFieldsDb.length;
			}
			else {
				// This is a non-aggregating field. Insert the item 
				// after the last active non-aggregating field
				var allFieldsDb = reportElementMF.allFieldsDb;
				var activeFieldsDbHasAggregatingField = false;
				for (var i = 0; i < activeFieldsDb.length; i++) {
					var itemId = activeFieldsDb[i];
					var allFieldsItem = allFieldsDb[util.h(itemId)];
					// alert('allFieldsItem: ' + allFieldsItem);
					// util.showObject(allFieldsItem);
					if (allFieldsItem.isAggregatingField) {
						// This is the first aggreagting field,
						// we insert the new non-aggregating field at this point.
						insertAtItemIndex = i;
						activeFieldsDbHasAggregatingField = true;
						// alert('insertAtItemIndex: ' + insertAtItemIndex);
						break;
					}
				}
				if (!activeFieldsDbHasAggregatingField) {
					// The activeFieldsDb has no aggregating field yet, so we add the current field as last field
					insertAtItemIndex = activeFieldsDb.length;
				}
			}
			// Insert the item			
			activeFieldsDb.splice(insertAtItemIndex, 0, newItemId);
			// Rebuild the active fields list
			reportElementMF.buildActiveFieldsList();
		},
		removeItem: function() {
			// Invoked by remove button, removes the selected item in activeFields list.
			var activeFieldsDb = reportElementMF.activeFieldsDb;
			var allFieldsDb = reportElementMF.allFieldsDb;
			var selectedItemIndex = reportElementMF.activeFields.selectedItemIndex;
			var selectedItemId = activeFieldsDb[selectedItemIndex];
			//var s = 'selectedItemIndex: ' + selectedItemIndex;
			//s += '\nselectedItemId: ' + selectedItemId;
			//alert(s);
			var nextSelectedItemIndex = -1;
			var numberOfItems = activeFieldsDb.length;
			// Check which item becomes selected next
			if (numberOfItems > 1) {
				nextSelectedItemIndex = (selectedItemIndex < numberOfItems - 1) ? selectedItemIndex : selectedItemIndex - 1;
			}
			// Remove selected li element
			var ul = util.getE('remf:active_fields_list');
			var li = util.getE('remf:active_fields_list:' + selectedItemIndex);
			ul.removeChild(li);
			// We must update the li element Ids so that they match
			// the index number of the activeFieldsDb items!
			var liElementItems = ul.getElementsByTagName('li');
			for (var i = 0; i < liElementItems.length; i++) {
				li = liElementItems[i];
				li.id = 'remf:active_fields_list:' + i;
			}
			// Remove selected item from activeFieldsDb
			activeFieldsDb.splice(selectedItemIndex, 1);
			// As the item has been removed there is no more selected item,
			// so reset selectedItemIndex!
			reportElementMF.activeFields.selectedItemIndex = -1;
			// Re-enable the removed element in allFields list
			util.showE('remf:all_fields_list:' + selectedItemId);
			// Update the allFieldsItem state
			var allFieldsItem = allFieldsDb[util.h(selectedItemId)];
			allFieldsItem.isActiveField = false;
			// Select the next item
			reportElementMF.activeFields.selectByItemIndex(nextSelectedItemIndex);
		},
		moveItem: function() {
			var isMoveUp = (this.id == 'remf:move_up_btn');
			var activeFieldsDb = reportElementMF.activeFieldsDb;
			var selectedItemIndex = reportElementMF.activeFields.selectedItemIndex;
			var selectedItemId = activeFieldsDb[selectedItemIndex];
			var selectedLiElement = util.getE('remf:active_fields_list:' + selectedItemIndex);
			var selectedTextElement = selectedLiElement.firstChild;
			var selectedLabel = selectedTextElement.nodeValue;
			// Get the itemIndex of the element which will be replaced with the selected element
			var nextItemIndex = isMoveUp ? selectedItemIndex - 1 : selectedItemIndex + 1;
			var nextLiElement = util.getE('remf:active_fields_list:' + nextItemIndex);
			var nextTextElement = nextLiElement.firstChild;
			var nextLabel = nextTextElement.nodeValue;
			// Update item position in activeFieldsDb
			activeFieldsDb.splice(selectedItemIndex, 1); // Removes selected itemId
			activeFieldsDb.splice(nextItemIndex, 0, selectedItemId); // Re-inserts selected itemId at new position
			// Replace the list labels
			var newSelectedTextElement = util.createT(nextLabel);
			var newNextTextElement = util.createT(selectedLabel);
			selectedLiElement.replaceChild(newSelectedTextElement, selectedTextElement);
			nextLiElement.replaceChild(newNextTextElement, nextTextElement);
			// Select the next item
			reportElementMF.activeFields.selectByItemIndex(nextItemIndex);
		}
	},
	//
	//
	// Utilities
	//
	//
	createAllFieldsObject: function(reportElementType, columns) {
		function addQueryFieldToAllFieldsDb(queryField) {
			var allFieldsDbLen = allFieldsDb.length;
			var isAggregatingField = queryField.isAggregatingField;
			// We add most of the query field properties to allFieldsDb for convenience
			var obj = {};
			obj.id = 'i' + allFieldsDbLen; // id is used for HTML element IDs
			obj.name = queryField.name;
			obj.label = queryField.label;
			obj.isAggregatingField = isAggregatingField;
			obj.category = queryField.category;
			// obj.derivationMethod = queryField.derivationMethod;
			obj.isExpression = queryField.isExpression;
			obj.isActiveField = false;
			//
			// Add field column properties because they will be used
			// as default values and will keep the values of already
			// existing column fields.
			//
			obj.showColumn = true;
			if (isAggregatingField) {
				obj.showPercentColumn = false;
				obj.showBarColumn = false;
				obj.showGraph = false;
			}
			allFieldsDb[allFieldsDbLen] = obj;
		}
		var queryFields = reports.queryFieldsDb;
		var reportElementType = reportElement.activeReportElementType;
		var isOverview = (reportElementType == 'overview');
		var isTable = (reportElementType == 'table');
		var isLogDetail = (reportElementType == 'log_detail');
		var allFieldsDb = [];
		// KHP 04/Oct/2010 - Create a tempColumns object,
		// it is used to verify if all columns exist in allFieldsDb.
		// If a column is missing for whatever reason in allFieldsDb 
		// then we must add it.
		var tempColumns = util.cloneObject(columns);
		util.createHash(tempColumns, 'report_field');
		var i;
		var column;
		var queryField;
		var queryFieldsLen = queryFields.length;
		for (i = 0; i < queryFieldsLen; i++) {
			queryField = queryFields[i];
			// util.showObject(queryField);
			if (!isLogDetail || queryField.reportElementTypeSupport.log_detail) {
				addQueryFieldToAllFieldsDb(queryField);
				// Mark the added field in tempColumns
				var column = tempColumns[util.h(queryField.name)];
				if (column != null) {
					column.columnExistsInAllFieldsDb = true;
				}
			}
		}
		// util.showObject(tempColumns, 'tempColumns');
		// Check if we need to add any column which exists in columns
		// but not in allFieldsDb
		var tempColumnsLen = tempColumns.length;
		for (i = 0; i < tempColumnsLen; i++) {
			column =  tempColumns[i];
			if (column.columnExistsInAllFieldsDb == null) {
				// Add this report field to allFieldsDb
				var reportFieldName = column.report_field;
				queryField = queryFields[util.h(reportFieldName)];
				addQueryFieldToAllFieldsDb(queryField);
			}
		}
		// Set allFieldsDb
		reportElementMF.allFieldsDb = allFieldsDb;
		// Create hash
		util.createHash(reportElementMF.allFieldsDb, 'id');
	},
	createActiveFieldsObject: function(columns) {
		// We create a tempColumns object because we require createHash on which we don't want to use on reportElementDb.		
		var reportElementType = reportElement.activeReportElementType;
		var isOverview = (reportElementType == 'overview');
		var tempColumns = util.cloneObject(columns);
		util.createHash(tempColumns, 'report_field');
		// util.showObject(tempColumns);
		var i;
		//
		// Update allFields according activeFields state
		//
		var allFieldsDb = reportElementMF.allFieldsDb;
		for (i = 0; i < allFieldsDb.length; i++) {
			var allFieldsItem = allFieldsDb[i];
			var reportFieldName = allFieldsItem.name;
			// If the reportFieldName already exists in activeFields (equal tempColumns)
			// then update the allFieldsItem
			var column = tempColumns[util.h(reportFieldName)]
			if (column) {
				var itemId = allFieldsItem.id;
				// Add the itemId to the column in tempColumns because we need it for
				// the final activeFieldsDb
				column.id = itemId;
				allFieldsItem.isActiveField = true;
				allFieldsItem.showColumn = column.show_column;
				if (!isOverview && allFieldsItem.isAggregatingField) {
					allFieldsItem.showPercentColumn = column.show_percent_column;
					allFieldsItem.showBarColumn = column.show_bar_column;
					allFieldsItem.showGraph = column.show_graph;
				}
			}
		}
		//
		// Create the activeFieldsDb
		//
		var activeFieldsDb = [];
		for (i = 0; i < tempColumns.length; i++) {
			// util.showObject(tempColumns[i]);
			activeFieldsDb[i] = tempColumns[i].id;	
		}
		// Set activeFields
		reportElementMF.activeFieldsDb = activeFieldsDb;
	},
	buildAllFieldsList: function() {
		var allFieldsDb = reportElementMF.allFieldsDb;
		var ul = util.getE('remf:all_fields_list');
		util.removeChildElements(ul);
		for (var i = 0; i < allFieldsDb.length; i++) {
			var allFieldsItem = allFieldsDb[i];
			// Hide the list item if the field is already active
			var displayStyle = allFieldsItem.isActiveField ? 'none' : 'list-item';
			var elementId = 'remf:all_fields_list:' + allFieldsItem.id;
			var li = util.createE('li', {id:elementId, display:displayStyle});
			var label = util.createT(allFieldsItem.label);
			util.chainE(ul, li, label);
		}
	},
	buildActiveFieldsList: function() {
		var activeFieldsDb = reportElementMF.activeFieldsDb;		
		var allFieldsDb = reportElementMF.allFieldsDb;
		// util.showObject(allFieldsDb);
		var ul = util.getE('remf:active_fields_list');
		util.removeChildElements(ul);
		for (var i = 0; i < activeFieldsDb.length; i++) {
			var itemId = activeFieldsDb[i];
			var item = allFieldsDb[util.h(itemId)];
			// In activeFields we use the index of activeFieldsDb for the ID!
			var elementId = 'remf:active_fields_list:' + i;
			var li = util.createE('li', {id:elementId});
			var label = util.createT(item.label);
			util.chainE(ul, li, label);
		}
	}
};
//
// reportsUtil.js
//
reportsUtil = {
	initViewAvailableFields: function() {
		// This inits viewAvailableReportFilterFields and viewAvailableReportTableFilterFields
		// Create avaialble expression fields arrays, then init the view panels
		var queryFieldsDb = reports.queryFieldsDb;
		var reportFilterFields = [];
		var customFilterFields = [];
		for (var i = 0; i < queryFieldsDb.length; i++) {
			var item = queryFieldsDb[i];
			if (!item.isExpression) {
				var reportFieldLabel = item.label;
				var reportFieldName = item.name;
				if (!item.isAggregatingField) {
					var dbFieldName = item.dbFieldName;
					reportFilterFields[reportFilterFields.length] = {label:reportFieldLabel, name:dbFieldName};
				}
				customFilterFields[customFilterFields.length] = {label:reportFieldLabel, name:reportFieldName};
			}
		}
		reportsUtil.viewAvailableReportFilterFields.init(reportFilterFields);
		reportsUtil.viewAvailableReportTableFilterFields.init(customFilterFields);
	},
	viewAvailableReportFilterFields: {
		fieldsPanel: null,
		init: function(availableExpressionFieldsDb) {
			var viewExprFieldsObj = {
				panelId: 'view_report_filter_expression_fields:panel',
				tbodyId: 'view_report_filter_expression_fields:tbody',
				panelLabel: langVar('lang_admin.available_expression_fields.available_fields'),
				availableExpressionFieldsDb: availableExpressionFieldsDb
			};
			reportsUtil.viewAvailableReportFilterFields.fieldsPanel = new AvailableExprFields(viewExprFieldsObj);
			YAHOO.util.Event.addListener('view_report_filter_expression_fields:close_btn', 'click', reportsUtil.viewAvailableReportFilterFields.close);
		},
		open: function() {
			reportsUtil.viewAvailableReportFilterFields.fieldsPanel.open();
		},
		close: function() {
			reportsUtil.viewAvailableReportFilterFields.fieldsPanel.close();
		}
	},
	viewAvailableReportTableFilterFields: {
		fieldsPanel: null,
		init: function(availableExpressionFieldsDb) {
			var viewExprFieldsObj = {
				panelId: 'view_custom_expression_fields:panel',
				tbodyId: 'view_custom_expression_fields:tbody',
				panelLabel: langVar('lang_admin.available_expression_fields.available_fields'),
				availableExpressionFieldsDb: availableExpressionFieldsDb
			};
			reportsUtil.viewAvailableReportTableFilterFields.fieldsPanel = new AvailableExprFields(viewExprFieldsObj);
			YAHOO.util.Event.addListener('view_custom_expression_fields:close_btn', 'click', reportsUtil.viewAvailableReportTableFilterFields.close);
		},
		open: function() {
			reportsUtil.viewAvailableReportTableFilterFields.fieldsPanel.open();
		},
		close: function() {
			reportsUtil.viewAvailableReportTableFilterFields.fieldsPanel.close();
		}
	}
};
//
//
// reports.js
//
//
var reports = {
	theList: null,
	itemsDb: [],
	itemsDbBackup: [],
	// reportElementMetaTypesDb - contains all report element types by report field name or by type.	
	// Note, the non-table report elements contain a prefix which distinguishes them from table report elements.
	// So for table report elements the name is equal the reportFieldName, for non-table report elements the
	// name is equal "__RET__" + report_element_type
	// This approach allows us to set and get meta type list values in a simple way
	// {name:__RET__overview, type:"overview", label:"Overview", },
	// {name:file_type, , type:"table", label:"File type"},
	// {name:page, type:"table", label:"Page"},
	// {name:__RET__log_detail, type:"table", label:"Page"},
	// ...
	reportElementMetaTypesDb: [],
	databaseFieldsDb: [], // Required for creSessionFields.js
	queryFieldsDb: [], // Contains all report fields with report field properties and additional information
	defaultGraphsDb: {},
	// sessionFieldsDb: [],
	// isBasicSessionReportsSupport: false,
	// isPageSessionReportsSupport: false,
	validator: null,
	moveControl: null,
	// mainButtons;
	saveChangesBtn: null,
	newReportGroupBtn: null,
	newReportBtn: null,
	deleteBtn: null,
	duplicateBtn: null,
	undoAllChangesBtn: null,
	// noItemFormIsActive: false,
	reportElementsMoveControlIsPositioned: false,
	activeMainDisplayElementId: '',
	activeType: '', // report | report_group
	activeReportGroupItemDat: {
		label_ori: ''
	},
	// activeReportItemDat contains active form data which are used by reportEditor to read upon open and write upon save
	// and when validating the active report.
	activeReportItemDat: {
		report_name: '',
		label_ori: '',
//		show_header_bar: true,
		sync_graph_axis_with_relative_date: false,
		description: '',
		header: '',
		footer: '',
		date_filter: '',
		filter_expression: '',
		filter_items: []
	},
	deletedReportNamesDb: [], // Contains the report node names of any deleted report. The list is send to the server upon save.
	// isUpdateReportLinkList flag is set to true each time a new report is displayed
	// in the form. It is used in report elements editor to re-create an up to date
	// report list.
	isUpdateReportLinkList: true
};
function init() {
	// init new session
	// util.initSession();
	reports.validator = new util.Validator();
	//
	// init toolbar buttons and form controls
	//
	reports.saveChangesBtn = new util.ToolbarButton('save_changes', saveReportChanges, toolbarButtonsDb);		
	reports.newReportGroupBtn = new util.ToolbarButton('new_report_group', newReportGroupItem, toolbarButtonsDb);
	reports.newReportBtn = new util.ToolbarButton('new_report', newReportItem, toolbarButtonsDb);
	reports.duplicateBtn = new util.ToolbarButton('duplicate', duplicateItem, toolbarButtonsDb);
	reports.deleteBtn = new util.ToolbarButton('delete', deleteItem, toolbarButtonsDb);
	reports.undoAllChangesBtn = new util.ToolbarButton('undo_all_changes', undoAllChanges, toolbarButtonsDb);
	var YE = YAHOO.util.Event;
	YE.addListener('reports:report_group_label', 'keyup', updateItemListLabel);
	YE.addListener('reports:report_label', 'keyup', updateItemListLabel);
	YE.addListener('reports:edit_report_properties_btn', 'click', reportEditor.openPanel);
	YE.addListener('reports:new_report_element_btn', 'click', reportElement.newItem);
	//
	// Ignore/Disable buttons according RBAC
	//
	var permissions = pageInfo.permissions;
	if (permissions.isEdit) {
		if (!permissions.isAdd) {
			reports.newReportGroupBtn.disableAndIgnore();
			reports.newReportBtn.disableAndIgnore();
			reports.duplicateBtn.disableAndIgnore();
		}
		if (!permissions.isDelete) {
			reports.deleteBtn.disableAndIgnore();
		}
		// Register isModifiedPageHandler in adminConfig.js
		// (We don't check for modifications if there is no edit 
		// permission because there is no Save button anyway!)
		adminConfig.getIsModifiedPageHandler = getIsModifiedPage;
	}
	else {
		reports.saveChangesBtn.disableAndIgnore();
		reports.newReportGroupBtn.disableAndIgnore();
		reports.newReportBtn.disableAndIgnore();
		reports.duplicateBtn.disableAndIgnore();
		reports.deleteBtn.disableAndIgnore();
		reports.undoAllChangesBtn.disableAndIgnore();
	}
	//
	// Create theList object
	//
	reports.theList = new listcontroller.List({
		containerElementId: 'item_list_body',
		itemEvent: itemActivated,
		switchEvent: itemSwitchActivated,
		isSwitch1: true,
		isSwitch2: true,
		switchButtonElementId: 'reports:item_list:switch_btn',
		switchLabelElementId: 'reports:item_list:switch_label',
		switch1Label: langVar('lang_admin.report_editor.show_in_dynamic_reports'),
		switch2Label: langVar('lang_admin.report_editor.show_in_static_reports'),
		isMoveControl: true
	});
	//
	// init Report Element List, Report Editor, ...
	//
	reportElementsList.init();
	reportEditor.init();
	reportElement.init();
	// Init view available fields panels
	reportsUtil.initViewAvailableFields();
	// init OptionInfo
	optionInfo.init();
}
function initMainDisplay() {
	// alert('initMainDisplay: ' + pageInfo.reportName);
	function getItemIdForGivenReportName(items, reportName) {
		// Returns the itemId for the given reportName,
		// or an empty string if the report_name does not exist.
		var itemId = '';
		for (var i = 0, max = items.length; i < max; i++) {
			var item = items[i];
			if (item.type == 'report') {
				if (item.dat.report_name == reportName) {
					itemId = item.id;
				}
			}
			else {
				var subitems = item.subitems;
				for (var j = 0; j < subitems.length; j++) {
					var subitem = subitems[j];
					if (subitem.dat.report_name == reportName) {
						itemId = subitem.id;
						break;
					}
				}
			}
		}
		return itemId;
	}
	var itemsDb = reports.itemsDb;
	if (itemsDb.length > 0) {
		var defaultItemId = '';
		var reportName = pageInfo.reportName;
		if (reportName != '') {
			// Check if the report exists and get the itemId
			defaultItemId = getItemIdForGivenReportName(itemsDb, reportName);
			// Reset the reportName because we don't want to select
			// any defaultReport when initMainDisplay() is inoked
			// upon undoChanges, etc.
			pageInfo.reportName = '';
		}
		// alert('defaultItemId: ' + defaultItemId);
		// If we don't have a defaultItemId yet get the first itemId.
		if (defaultItemId == '') {
			defaultItemId = reports.theList.getFirstItemId();
		}
		setItem(defaultItemId);
	}
	else {
		// no item exists
		setMainDisplay('no_item_form');
	}
	// Update moveControl button state
	// reports.theList.setMoveControlState();
	util.showE('form_section');
}
function setMainDisplay(displayElementId) {
	if (reports.activeMainDisplayElementId != displayElementId) {
		util.hideE(['no_item_form', 'report_group_form', 'report_form', 'loading_info', 'saving_info']);
		util.showE(displayElementId);
		reports.activeMainDisplayElementId = displayElementId;
		// Update form label
		var formLabel = '';
		if (displayElementId == 'report_group_form') {
			formLabel = langVar('lang_admin.report_editor.report_group');
		}
		else if (displayElementId == 'report_form') {
			formLabel = langVar('lang_admin.report_editor.report');
		}
		else {
			formLabel = '-';
		}
		util.updateT('item_form_label', formLabel);
	}
}
function getReportData() {
	util.showE('loading_info');
	if (!pageInfo.exitActive) {
		// Init help
		util.helpWindow.init('');
		var url = '?dp=config_pages.reports.get_report_data';
		url += '&p=' + pageInfo.profileName;
		var dat = 'v.fp.page_token=' + pageInfo.pageToken;
		util.serverPost(url, dat);
	}
}
function createReportNamesWithItemIdLookup(items, lookup) {
	// This creates a lookup for _report_name = itemId to get
	// the itemId for each report_name.
	for (var i = 0, max = items.length; i < max; i++) {
		var item = items[i],
			reportName = '';
		if (item.type == 'report') {
			reportName = '_' + item.dat.report_name;
			lookup[reportName] = item.id;
		}
		else {
			// Report Group, call itself.
			createReportNamesWithItemIdLookup(item.subitems, lookup);
		}
	}
}
function updateItemsDb(items, reportNamesWithItemIdLookup) {
	// updateItemsDb
	// 1. Add report label property to itemsDb. The property exists but has no value yet,
	// we get the label from label_ori.
	// 2. Replace the report_name in link_to_report of report elements
	// with the unique itemId. This will be converted back to report_name
	// upon saving the report data.
	for (var i = 0, max = items.length; i < max; i++) {
		var item = items[i],
			reportElements = null,
			reportElement = null,
			reportLink = '';
		if (item.type == 'report') {
			// Update label
			item.dat.label = item.dat.label_ori;
			// Replace report_link in report_elements
			reportElements = item.dat.report_elements;
			for (var k = 0, maxK = reportElements.length; k < maxK; k++) {
				reportElement = reportElements[k];
				reportLink = reportElement.report_link;
				if (reportLink != '') {
					// Note, the report given in reportLink might not anymore exist
					if (reportNamesWithItemIdLookup.hasOwnProperty('_' + reportLink)) {
						// Replace report_name with report id
						reportElement.report_link = reportNamesWithItemIdLookup['_' + reportLink];
					}
					else {
						// Reset report_link
						reportElement.report_link = '';
					}
				}
			}
		}
		else {
			// Report Group, Recursively call updateItemsDb for subitems
			updateItemsDb(item.subitems, reportNamesWithItemIdLookup);
		}
	}
}
function getReportDataResponse(dat) {
	// alert('getReportDataResponse(): ' + dat);
	var itemsDb = dat.reportsDb,
		reportNamesWithItemIdLookup = {}; // This will be used in updateItemsDb()
	createReportNamesWithItemIdLookup(itemsDb, reportNamesWithItemIdLookup);
//	util.showObject(reportNamesWithItemIdLookup);
	updateItemsDb(itemsDb, reportNamesWithItemIdLookup);
//	util.showObject(itemsDb);
	reports.itemsDb = itemsDb;
	reports.itemsDbBackup = util.cloneObject(itemsDb);
	reports.reportElementMetaTypesDb = dat.reportElementMetaTypesDb;
    // util.showObject(reports.reportElementMetaTypesDb);
	util.createHash(reports.reportElementMetaTypesDb, 'name');
	reports.defaultGraphsDb = dat.defaultGraphsDb;
	reports.databaseFieldsDb = dat.databaseFieldsDb;
	reports.queryFieldsDb = dat.queryFieldsDb;
	util.createHash(reports.queryFieldsDb, 'name');
	// util.showObject('reports.queryFieldsDb');
	//
	// Init
	//
	init();
	reports.theList.init(reports.itemsDb);
	initMainDisplay();
	// Set final toolbar state
	reports.saveChangesBtn.enable();
	reports.undoAllChangesBtn.enable();
	reports.newReportGroupBtn.enable();
	reports.newReportBtn.enable();
	updateToolbarButtons(); // handles Duplicate and Delete
	adminConfig.setItemListSize();
	YAHOO.util.Event.addListener(window, 'resize', adminConfig.setItemListSize);
	pageInfo.initComplete = true;
}
function updateToolbarButtons() {
	var isItems = reports.theList.isItems();
	reports.deleteBtn.enable(isItems);
	reports.duplicateBtn.enable(isItems);
}
function updateItemListLabel() {
	var label = this.value;
	setTimeout('setItemListLabel("' + label + '")', 150);
}
function setItemListLabel(label) {
	if (label == '') {
		label = '-';
	}
	reports.theList.updateListLabel(label);
}
function itemActivated(itemId) {
	// util.showE('transparent_cover');
	if (validateActiveItem()) {
		setItem(itemId);
	}
}
function itemSwitchActivated(itemId) {
	// A user clicked on an item checkbox (not necessarily a selected item)
	// and we may need to update the ViewReports link
	// alert('itemSwitchActivated() - itemId: ' + itemId);
	var activeSwitchKey = reports.theList.getActiveSwitchKey();
	// If checkboxes are set for dynamic reports
	if (activeSwitchKey == 'switch1') {
		// alert('itemSwitchActivated() - itemId: ' + itemId);
		setViewReportsLink();
	}
}
function setItem(itemId) {
	// selects active item in list and displays the form
	// alert('set item: ' + itemId);
	// Set session is active
	// util.sessionActive();
	reports.theList.selectItem(itemId);
	updateForm();
}
function newReportGroupItem() {
	if (validateActiveItem()) {
		var newItemId = reports.theList.getNewItemId();
		var newReportGroupLabel = langVar('lang_admin.general.new_report_group');
		var newReportGroupObj = {
			id: newItemId,
			type: 'report_group',
			switch1: true,
			switch2: true,
			label: newReportGroupLabel,
			dat: {
				label: newReportGroupLabel,
				label_ori: ''
			},
			subitems: []
		}
		// util.showObject(newReportGroupObj);
		var insertDirective = {insertAtRootLevel: true};
		reports.theList.newItem(newReportGroupObj, insertDirective);
		setItem(newItemId);
		updateToolbarButtons();
	}
}
function newReportItem() {
	// alert('newReportItem()');
	if (validateActiveItem()) {
		var newItemId = reports.theList.getNewItemId();
		var newReportLabel = langVar('lang_admin.general.new_report');
		var newReportObj = {
			id: newItemId,
			type: 'report',
			switch1: true,
			switch2: true,
			label: newReportLabel,
			dat: {
				report_name: '', // report node name
				label: newReportLabel,
				label_ori: '',
//				show_header_bar: true,
				description: '',
				header: '',
				footer: '',
				date_filter: '',
				filter_expression: '',
				filter_items: [],
				sync_graph_axis_with_relative_date: false,
				report_elements: []
			}
		}
		// util.showObject(newReportObj);
		var insertDirective = {};
		if (reports.activeType == 'report_group') {
			insertDirective.insertAsFirstSubItem = true;
		}
		else {
			insertDirective.insertAsItemOrSubItem = true;
		}
		reports.theList.newItem(newReportObj, insertDirective);
		setItem(newItemId);
		updateToolbarButtons();
	}
}
function duplicateItem() {
	if (validateActiveItem()) {
		var theList = reports.theList
		var clonedItemId = theList.cloneItem();
		// Update cloned item properties
		// KHP-RC, we have to update all subitems if this is a report group!
		theList.setItemDatValue(clonedItemId, 'report_name', '');
		theList.setItemDatValue(clonedItemId, 'label_ori', '');
		setItem(clonedItemId);
	}
}
function deleteItem() {
	var theList = reports.theList;
	var selectedItem = theList.getSelectedItem();
	//
	// Track deleted report items
	//
	var deletedReportsDb = reports.deletedReportNamesDb;
	var reportName = '';
	if (selectedItem.type == 'report_group') {
		var subitems = selectedItem.subitems;
		for (var i = 0; i < subitems.length; i++) {
			reportName = subitems[i].dat.report_name;
			if (reportName != '') {
				deletedReportsDb[deletedReportsDb.length] = reportName;
			}
		}
	}
	else {
		// This is a report item
		reportName = selectedItem.dat.report_name;
		if (reportName != '') {
			deletedReportsDb[deletedReportsDb.length] = reportName;
		}
	}
	//
	// Delete item
	//
	var nextItemIdToBeSelected = theList.deleteItem();
	if (nextItemIdToBeSelected) {
		// reset the validator in case that the deleted item indicated an error
		reports.validator.reset();
		setItem(nextItemIdToBeSelected);
	}
	else {
		// Last item has been deleted, so no item exists.
		reports.activeType = '';
		initMainDisplay();
	}
}
function updateForm() {
	var item = reports.theList.getSelectedItem();
	var itemDat = item.dat;
//	util.showObject(item);
	// Reset isUpdateReportLinkList to true. This ensures that a new list is created
	// when opening the report element editor next time.
	reports.isUpdateReportLinkList = true;
	var type = item.type;
	reports.activeType = type;
	var displayElementId = '';
	var activeReportName = '';
	var isShowInDynamicReports = false;
	if (type == 'report_group') {
		//
		// Update report group
		// 
		displayElementId = 'report_group_form';
		util.setF('reports:report_group_label', itemDat.label);
		reports.activeReportGroupItemDat.label_ori = itemDat.label_ori;
	}
	else {
		//
		// Update report
		//
		displayElementId = 'report_form';
		util.setF('reports:report_label', itemDat.label);
		// Write active itemDat data to reports.activeReportItemDat object so that they can be read by reportsEditor.js
		var activeReportItemDat = reports.activeReportItemDat;
		for (prop in activeReportItemDat) {
			activeReportItemDat[prop] = itemDat[prop];
		}
		// util.showObject(itemDat.filter_items);
		// util.showObject(activeReportItemDat.filter_items);
		//
		// Handle report elements list
		// 
		reportElementsList.list.compose(itemDat.report_elements);
	}
	//
	// Update display
	//
	if (reports.activeMainDisplayElementId != displayElementId) {
		setMainDisplay(displayElementId);
	}
	//
	// Update View Reports link
	//
	setViewReportsLink();
	//
	// Position the Report Elements Move Control
	//
	if (!reports.reportElementsMoveControlIsPositioned && type == 'report') {
		// We need to give it some time, else we don't get the right list position
		var s = 'reports:report_element_list:div';
		setTimeout('reportElementsList.moveControl.setPosition("' + s + '")', 250);
		reports.reportElementsMoveControlIsPositioned = true;
	}
}
function setViewReportsLink() {
	// This updates the "View Reports" link in the navigation
	// so that the selected report is the default report when
	// viewing reports.
	// Make sure the link exists!
	var theLinkElement = util.getE('config_nav:view_reports_btn');
	if (theLinkElement) {
		var reportNameToView = '';
		var defaultDateFilter = pageInfo.defaultDateFilter;
		if (reports.activeType == 'report') {
			// We only set the reportName if is not a new report (because it isn't saved yet)
			// and if it is active in dynamic reports
			var theList = reports.theList;
			var isNew = theList.getSelectedItemIsNew();
			var isActive = theList.getSelectedItemIsSwitch1(); // This already considers any parent switch to be true as well
			if (!isNew && isActive) {
				reportNameToView = reports.activeReportItemDat.report_name;
			}
		}
		// else it is a group for which no viewReportsName is defined
		// Set the link
		var href = '?dp=reports&p=' + pageInfo.profileName;
		if (defaultDateFilter != '') {
			href += '&df=' + defaultDateFilter;
		}
		if (reportNameToView != '') {
			href += '&rn=' + reportNameToView;
		}
		theLinkElement.href = href;
	}
}
function validateActiveItem() {
	var theList = reports.theList;
	// Only validate if isEdit permission and if items
	if (pageInfo.permissions.isEdit && theList.isItems()) {
		validator = reports.validator;
		var isReportItem = (reports.activeType == 'report');
		var o = {};
		if (!isReportItem) {
			// Report group
			o.label = validator.isValue('reports:report_group_label');
			o.label_ori = reports.activeReportGroupItemDat.label_ori;
		}
		else {
			// Report
			// Handle the label.
			// Note, duplicate labels are allowed.
			o.label = validator.isValue('reports:report_label');
			// Should we allow duplicate labels?
			// For now we allow it and don't check for duplicates!
			// Get activeReportItemDat
			var activeReportItemDat = reports.activeReportItemDat;
			for (var prop in activeReportItemDat) {
				o[prop] = activeReportItemDat[prop];
			}
		}
		if (validator.allValid()) {
			if (isReportItem) {
				// Get a clone of the report element items
				o.report_elements = reportElementsList.list.getItemsClone();
			}
			// alert('validateActiveItem() - isModified state before saving the item: ' + theList.getIsModified());
			theList.saveItem(o);
			// alert('validateActiveItem() - isModified state after saving the item: ' + theList.getIsModified());
			return true;
		}
		return false;
	}
	else {
		// No item exists or no isEdit permission
		return true;
	}
}
//
//
// saveReportChanges utilities
//
//
function assembleReportElementItemDat(path, item, reportElementCount, itemIdWithReportNameLookup) {
	function addNumberOfRows() {
		dat += path + '.number_of_rows=' + item.number_of_rows + '&';
	}
	function addColumns() {
		var columns = item.columns;
		for (var i = 0; i < columns.length; i++) {
			var columnsObj = columns[i];
			var columnsPath = path + '.columns.' + i; 
			for (var prop in columnsObj) {
				dat += columnsPath + '.' + prop + '=' + columnsObj[prop] + '&';
			}
			// Add position node
			dat += columnsPath + '.position=' + i + '&';
		}
	}
	function addSessionFields() {
		var sessionFields = creSessionFields.getSessionFields();
		// util.showObject(sessionFields);
		var sessionFieldName;
		for (sessionFieldName in sessionFields) {
			if (sessionFields[sessionFieldName][reportElementType]) {
				// Add session field value
				dat += path + '.' + sessionFieldName + '=' + encodeURIComponent(item[sessionFieldName]) + '&';
			}
		}
	}
	var reportElementType = item.type;
	var isLogDetail = (reportElementType == 'log_detail');
	var prop;
	var graphsObj;
	var pivotTableObj;
	var dat;
	// Note, item.report_link contains the itemId of the report, convert to report_name!
	var reportLinkItemId = item.report_link;
	var reportLink = '';
	if (itemIdWithReportNameLookup.hasOwnProperty(reportLinkItemId)) {
		reportLink = itemIdWithReportNameLookup[reportLinkItemId];
//		util.showObject({reportLinkItemId:reportLinkItemId, reportLink:reportLink});
	}
	dat = path + '.position=' + reportElementCount + '&'; // Used to sort reports in order on server side
	dat += path + '.label=' + encodeURIComponent(item.label) + '&';
	dat += path + '.label_ori=' + encodeURIComponent(item.label_ori) + '&';
	dat += path + '.type=' + item.type + '&';
	dat += path + '.display_side_by_side=' + item.display_side_by_side + '&';
	dat += path + '.show_header_bar=' + item.show_header_bar + '&';
	dat += path + '.report_link=' + reportLink + '&';
	dat += path + '.description=' + encodeURIComponent(item.description) + '&';
	dat += path + '.header=' + encodeURIComponent(item.header) + '&';
	dat += path + '.footer=' + encodeURIComponent(item.footer) + '&';
	dat += path + '.date_filter.df=' + encodeURIComponent(item.date_filter) + '&';
	dat += path + '.filter.expression=' + encodeURIComponent(item.filter_expression) + '&';
	switch (reportElementType) {
		case 'overview':
			dat += path + '.compact_view=' + item.compact_view + '&';
			addColumns();
			break;
		case 'table':
			addNumberOfRows();
			dat += path + '.show_graphs=' + item.show_graphs + '&';
			dat += path + '.show_table=' + item.show_table + '&';
			dat += path + '.display_graphs_side_by_side=' + item.display_graphs_side_by_side + '&';
//			dat += path + '.display_graphs_table_side_by_side=' + item.display_graphs_table_side_by_side + '&';
			dat += path + '.maximum_table_bar_graph_length=' + item.maximum_table_bar_graph_length + '&';
			dat += path + '.omit_parenthesized_items=' + item.omit_parenthesized_items + '&';
			dat += path + '.use_overview_for_totals=' + item.use_overview_for_totals + '&';
			dat += path + '.table_filter_expression=' + encodeURIComponent(item.table_filter_expression) + '&';
			dat += path + '.sort_by=' + encodeURIComponent(item.sort_by) + '&';
			dat += path + '.sort_direction=' + item.sort_direction + '&';
			dat += path + '.show_remainder_row=' + item.show_remainder_row + '&';
			dat += path + '.show_averages_row=' + item.show_averages_row + '&';
			dat += path + '.show_min_row=' + item.show_min_row + '&';
			dat += path + '.show_max_row=' + item.show_max_row + '&';
			dat += path + '.show_totals_row=' + item.show_totals_row + '&';
			graphsObj = item.graphs;
			pivotTableObj = item.pivot_table;
			// util.showObject(graphsObj);
			for (prop in graphsObj) {
				dat += path + '.graphs.' + prop + '=' + graphsObj[prop] + '&';
			}
			for (prop in pivotTableObj) {
				dat += path + '.pivot_table.' + prop + '=' + pivotTableObj[prop] + '&';
			}
			addColumns();
			break;
		case 'log_detail':
			addNumberOfRows();
			dat += path + '.show_graphs=false&';
			dat += path + '.show_table=true&';
            dat += path + '.table_filter_expression=' + encodeURIComponent(item.table_filter_expression) + '&';
			dat += path + '.sort_by=' + encodeURIComponent(item.sort_by) + '&';
			dat += path + '.sort_direction=' + item.sort_direction + '&';
			addColumns();
			break;
		case 'sessions_overview':
			addSessionFields();
			break;
		case 'session_paths':
			addNumberOfRows();
			addSessionFields();
			dat += path + '.number_of_rows_expanded=' + item.number_of_rows_expanded + '&';
			dat += path + '.expand_paths_greater_than=' + item.expand_paths_greater_than + '&';
			break;
		case 'session_page_paths':
			addNumberOfRows();
			addSessionFields();
			break;
	}
	// util.showObject(dat);
	return dat;
}
function assembleFilterItemsDat(path, filterItems) {
	var dat = '';
	var itemValue;
	if (filterItems.length > 0) {
		for (var i = 0; i < filterItems.length; i++) {
			var itemPath = path + '.filter.filter_items.' + i;
			var item = filterItems[i];
			for (var prop in item) {
				dat += itemPath + '.' + prop + '=' + util.getEncodedURIComponent(item[prop]) + '&';
			}
		}
	}
	else {
		// No filter items
		dat = path + '.filter.filter_items=&';
	}
	return dat;
}
function assembleReportItemDat(item, reportCount, itemIdWithReportNameLookup) {
	var finalReportName = item.final_report_name;
	// var oriReportName = item.report_name;
	var path = 'v.fp.reports.' + finalReportName;
	var dat = path + '.position=' + reportCount + '&'; // Used to sort reports in order on server side
	// dat += path + '.report_name_ori=' + encodeURIComponent(item.report_name) + '&'; // Used to handle existing report dependencies and label lang variables
	dat += path + '.label=' + encodeURIComponent(item.label) + '&';
//	dat += path + '.show_header_bar=' + item.show_header_bar + '&';
	dat += path + '.description=' + encodeURIComponent(item.description) + '&';
	dat += path + '.header=' + encodeURIComponent(item.header) + '&';
	dat += path + '.footer=' + encodeURIComponent(item.footer) + '&';
	dat += path + '.date_filter.df=' + encodeURIComponent(item.date_filter) + '&';
	dat += path + '.filter.expression=' + encodeURIComponent(item.filter_expression) + '&';
	dat += assembleFilterItemsDat(path, item.filter_items);
	dat += path + '.sync_graph_axis_with_relative_date=' + item.sync_graph_axis_with_relative_date + '&';
	// util.showObject(item);
	var reportElements = item.report_elements;
	if (reportElements.length > 0) {
		for (var i = 0; i < reportElements.length; i++) {
			var reportElementPath = path + '.report_elements.' + i;
			var reportElementItemDat = assembleReportElementItemDat(reportElementPath, reportElements[i], i, itemIdWithReportNameLookup);
			dat += reportElementItemDat;
		}
	}
	else {
		// No report element exists
		dat += path + '.report_elements=&';
	}
	return dat;
}
function assembleMenuItemDat(path, position, item) {
	// util.showObject(item);
	var dat = '';
	dat += path + '.position=' + position + '&';
	dat += path + '.report=' + item.dat.final_report_name + '&';
	dat += path + '.show_in_dynamic_reports=' + item.switch1 + '&';
	dat += path + '.show_in_static_reports=' + item.switch2 + '&';
	return dat;
}
function verifyAndCreateReportNodeNames(itemsDb) {
	// Checks if we need to specify and add any new report node names
	function checkForValidExistingNodeNames(items, reportNodeNameItems) {
		// This checks for existing valid node names. A node name is valid
		// if it exists and if the label is equal label_ori. In this case we
		// keep the existing node name.
		for (var i = 0; i < items.length; i++) {
			var item = items[i];
			if (item.type == 'report') {
				var itemDat = item.dat;
				if ((itemDat.report_name != '') && (itemDat.label == itemDat.label_ori)) {
					// Use existing report name, so we add this node name to reportNodeNameItems
					reportNodeNameItems[reportNodeNameItems.length] = itemDat.report_name;;
				}
			}
			else {
				// This is a report group, check subitems recursively
				checkForValidExistingNodeNames(item.subitems, reportNodeNameItems);
			}
		}
	}
	function setFinalReportNodeNames(items, reportNodeNameItems) {
		// This sets the final report node names to be used upon saving the data
		// Note, we let the server know if we send a new node name or not.
		for (var i = 0; i < items.length; i++) {
			var item = items[i];
			if (item.type == 'report') {
				var itemDat = item.dat;
				var finalReportName = ''; // The final report node name
				if ((itemDat.report_name != '') && (itemDat.label == itemDat.label_ori)) {
					// Use existing node name, so we add this node name to reportNodeNameItems
					finalReportName = itemDat.report_name;
				}
				else {
					// Create a new report node name and add it to reportNodeNameItems
					finalReportName = util.labelToUniqueNodeName(itemDat.label, reportNodeNameItems, 'report');
					reportNodeNameItems[reportNodeNameItems.length] = finalReportName;
				}
				// Add finalReportName to items
				itemDat.final_report_name = finalReportName;
			}
			else {
				// This is a report group, check subitems recursively
				setFinalReportNodeNames(item.subitems, reportNodeNameItems);
			}
		}
	}
	var reportNodeNameItems = [];
	checkForValidExistingNodeNames(itemsDb, reportNodeNameItems);
	setFinalReportNodeNames(itemsDb, reportNodeNameItems);
	// util.showObject(reportNodeNameItems);
}
function getDeletedReportNamesDat() {
	var a = reports.deletedReportNamesDb;
	var path = 'v.fp.deleted_reports';
	var dat = '';
	if (a.length > 0) {
		for (var i = 0; i < a.length; i++) {
			dat += path + '.' + encodeURIComponent(a[i]) + '=true&';
		}
	}
	else {
		dat = path + '=&';
	}
	return dat;
}
function createItemIdWithReportNameLookup(items, lookup) {
	// This creates a lookup for itemId = report_name to get
	// the report_name for each itemId.
	for (var i = 0, max = items.length; i < max; i++) {
		var item = items[i],
			itemId = '';
		if (item.type == 'report') {
			itemId = item.id;
			lookup[itemId] = item.dat.final_report_name;
		}
		else {
			// Report Group, call itself.
			createItemIdWithReportNameLookup(item.subitems, lookup);
		}
	}
}
//
//
// saveReportChanges
//
//
function saveReportChanges() {
	if (validateActiveItem()) {
		var theList = reports.theList;
		var isModified = theList.getIsModified();
		// alert('isModified: ' + isModified);
		if (isModified) {
			util.hideE('form_section');
			util.showE('saving_info');
			// setMainDisplay('saving_info');
			var itemsDb = reports.itemsDb;
			// util.showObject(itemsDb);
			var itemIdWithReportNameLookup = {};
			var reportsDat = '';
			var reportsMenuDat = '';
			var reportItemDat;
			var menuItemDat;
			if (itemsDb.length > 0) {
				// var reportsPath = 'v.fp.reports';
				//
				// Handle report node names
				//
				verifyAndCreateReportNodeNames(itemsDb);
//				util.showObject(itemsDb);
				//
				// Create itemId with report_name lookup which will be used
				// to convert any id given in a report_element.report_link to
				// its report_name. Note, this must be done after verifyAndCreateReportNodeNames()!
				//
				createItemIdWithReportNameLookup(itemsDb, itemIdWithReportNameLookup);
//				util.showObject(itemIdWithReportNameLookup);
				//
				// Assemble the data
				//
				var reportCount = 0; // Used as subnode in reports to sort them in menu order on server side.
				for (var i = 0; i < itemsDb.length; i++) {
					var item = itemsDb[i];
					var type = item.type;
					var itemDat = item.dat;
					var reportsMenuPath = 'v.fp.reports_menu.' + i;
					if (type == 'report_group') {
						// This is a report group with zero or more report items
						reportsMenuDat += reportsMenuPath + '.position=' + i + '&';
						reportsMenuDat += reportsMenuPath + '.label=' + encodeURIComponent(itemDat.label) + '&';
						reportsMenuDat += reportsMenuPath + '.label_ori=' + encodeURIComponent(itemDat.label_ori) + '&';
						reportsMenuDat += reportsMenuPath + '.show_in_dynamic_reports=' + item.switch1 + '&';
						reportsMenuDat += reportsMenuPath + '.show_in_static_reports=' + item.switch2 + '&';
						var subitemsDb = item.subitems;
						if (subitemsDb.length > 0) {
							for (var j = 0; j < subitemsDb.length; j++) {
								// This is a single menu item with a single report
								var subitem = subitemsDb[j];
								var subItemDat = subitem.dat;
								var reportsMenuItemsPath = reportsMenuPath + '.items.' + j;
								menuItemDat = assembleMenuItemDat(reportsMenuItemsPath, j, subitem);
								reportsMenuDat += menuItemDat;
								reportItemDat = assembleReportItemDat(subItemDat, reportCount, itemIdWithReportNameLookup);
								reportsDat += reportItemDat;
								reportCount++;
							}
						}
						else {
							// This is a report group without any reports.
							reportsMenuDat += reportsMenuPath + '.items=&';
							// reportsDat += reportsMenuDat;
						}
					}
					else {
						// This is a single menu item with a single report
						menuItemDat = assembleMenuItemDat(reportsMenuPath, i, item);
						reportsMenuDat += menuItemDat;
						reportItemDat = assembleReportItemDat(itemDat, reportCount, itemIdWithReportNameLookup);
						reportsDat += reportItemDat;
						reportCount++;
					}
				}
			}
			else {
				// No item exists
				reportsDat = 'v.fp.reports=&';
				reportsMenuDat = 'v.fp.reports_menu=&';
			}
			var deletedReportNamesDat = getDeletedReportNamesDat();
			deletedReportNamesDat = deletedReportNamesDat.replace(/&$/, '');
			// reportsMenuDat = reportsMenuDat.replace(/&$/, '');
			var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
			dat += reportsDat + reportsMenuDat + deletedReportNamesDat;			
			pageInfo.saveActive = true;
			var url = '?dp=config_pages.reports.save_reports';
			url += '&p=' + pageInfo.profileName;
			// Temp
//			return false;
			util.serverPost(url, dat);
		}
		else {
			alert(langVar('lang_stats.general.no_changes_to_save'));
		}
	}
}
function saveReportChangesResponse() {
	//
	// Reset items
	//
	// Update the current itemsDb with saved report data
	// For all items:
	// itemDat.label_ori must be set to the value of itemDat.label
	// For report items only:
	// itemDat.report_name must be set to the value of itemDat.final_report_name
	function resetItems(items) {
		for (var i = 0, max = items.length; i < max; i++) {
			var item = items[i];
			item.isNew = false;
			if (item.type == 'report') {
				var itemDat = item.dat;
				itemDat.label_ori = itemDat.label;
				itemDat.report_name = itemDat.final_report_name;
				// Delete the final_report_name property because
				// it will cause isModifiedItem as it does not
				// exist when we save the item into listcontroller upon
				// validateActiveItem()
				delete itemDat.final_report_name;
			}
			else {
				// Report Group, Recursively call resetItems for subitems
				resetItems(item.subitems);
			}
		}
	}
	var itemsDb = reports.itemsDb;
	resetItems(itemsDb);
	// Reset itemsDbBackup
	reports.itemsDbBackup = util.cloneObject(itemsDb);
	//
	// Before we reset isModified we must update the form
	// of the current selected item because it may not match
	// the latest itemsDb data
	//
	if (itemsDb.length > 0) {
		updateForm();
	}
	else {
		// If updateForm() isn't executed we must handle setViewReportsLink
		setViewReportsLink();
	}
	//
	// Reset isModified
	//
	var theList = reports.theList;
	theList.resetIsModified();
	//
	// Recover the display
	//
	pageInfo.saveActive = false;
	util.hideE('saving_info');
	util.showE('form_section');
}
function undoAllChanges() {
	reports.deletedReportNamesDb = [];
	reports.itemsDb = util.cloneObject(reports.itemsDbBackup);
	reports.theList.init(reports.itemsDb);
	initMainDisplay();
}
function getIsModifiedPage() {
	// Note, isModified will be false if only the active item has been edited
	// but has an error, hence we also check "if (!validateActiveItem() ..."
	var theList = reports.theList;
	if (!validateActiveItem() || theList.getIsModified()) {
		return true;
	}
	return false;
}
