//
// xrefsList.js
//
// Handles the active fields list and available fields list in Xrefs Editor

var xrefsList = {
	
	activeFieldsDb: [], // Contains the active field items,
	activeSortBy: '', // The active sort_by field name
	// activeSortDirection: '', // The active sort_direction
	allFieldsDb: [],  // A clone of xrefs.databaseFieldsDb but ordered as it appears in the list (first all string fields followed by all numerical fields)
						// It also contains an isActiveField property so that we know which field has to be selected next when we add an item.
	isEditMode: false,
	
	activeAddRemoveButtonType: '', // neutral | add | remove (neutral just shows the arrows if no item is selected)
	
	init: function() {
		
		var YE = YAHOO.util.Event;
		
		YE.addListener('xrefs_edit_mode_btn', 'click', xrefsList.switchToEditModeActor);
		
		YE.addListener('xrefs:active_fields_list', 'click', xrefsList.activeFields.selectActor);
		YE.addListener('xrefs:all_fields_list', 'click', xrefsList.allFields.selectActor);
		
		YE.addListener('xrefs:add_remove_btn', 'click', xrefsList.addRemoveItem);
		
		YE.addListener('xrefs:move_up_btn', 'click', xrefsList.activeFields.moveItem);
		YE.addListener('xrefs:move_down_btn', 'click', xrefsList.activeFields.moveItem);

		YE.addListener('xrefs:sort_by', 'change', xrefsList.sortByActivated);
		
		// Create the allFieldsDb object
		var databaseFieldsDb = xrefs.databaseFieldsDb;
		var allFieldsDb = xrefsList.allFieldsDb;
		var i;
		var dbFieldItem;
		var allFieldsItemObj;
		
		// Add all string fields to allFieldsDb
		for (i = 0; i < databaseFieldsDb.length; i++) {
			dbFieldItem = databaseFieldsDb[i];
			if (!dbFieldItem.isAggregatingField) {
				allFieldsItemObj = util.cloneObject(dbFieldItem);
				allFieldsItemObj['isActiveField'] = false;
				allFieldsDb[allFieldsDb.length] = allFieldsItemObj;
			}
		}
		
		// Add all aggregating fields to allFieldsDb
		for (i = 0; i < databaseFieldsDb.length; i++) {
			dbFieldItem = databaseFieldsDb[i];
			if (dbFieldItem.isAggregatingField) {
				allFieldsItemObj = util.cloneObject(dbFieldItem);
				allFieldsItemObj['isActiveField'] = false;
				allFieldsDb[allFieldsDb.length] = allFieldsItemObj;
			}
		}
		
		// Create allFieldsDb hash
		util.createHash(allFieldsDb, 'id');
		
		// Build the allFields list
		xrefsList.allFields.build();
	},
	
	
	updateForm: function(fields, sortBy, sortDirection, isEditMode) {
		
		// Is called from xrefs.updateForm()
		// We ignore the all fields list unless we are in edit mode.
		// So we don't update all fields unless we are or switch to edit mode

		// util.showObject({'sortBy':sortBy, 'sortDirection':sortDirection});
		
		// Get a copy of the active fields and reset activeFields properties
		xrefsList.activeFieldsDb = util.cloneObject(fields);
		xrefsList.activeFields.selectedItemIndex = -1;
		
		// Create active field items list		
		xrefsList.activeFields.build();
		
		// Set editMode
		xrefsList.switchToEditMode(isEditMode);
		
		// Scroll the lists to the top
		var activFieldsUl = util.getE('xrefs:active_fields_list');
		var allFieldsUl = util.getE('xrefs:all_fields_list');
		activFieldsUl.scrollIntoView(true);
		allFieldsUl.scrollIntoView(true);


		// Update sortBy and sortDirection list
		xrefsList.activeSortBy = sortBy;
		xrefsList.updateSortBy();

		var activeSortDirection = sortBy != '' ? sortDirection : 'descending';
		util.setF('xrefs:sort_direction', activeSortDirection);
	},
	
	switchToEditModeActor: function() {
		xrefsList.switchToEditMode(true);
	},
	
	switchToEditMode: function(isEditMode) {
		
		if (isEditMode) {
		
			// We need to reset/update the all fields list and button state
			xrefsList.allFields.selectedFieldId = '';
			
			var i;
			var fieldId;
			
			// Create a lookup object of the active fields
			var activeFieldsDb = xrefsList.activeFieldsDb;
			var activeFieldsLookup = {};
			for (i = 0; i < activeFieldsDb.length; i++) {
				fieldId = activeFieldsDb[i];
				activeFieldsLookup[fieldId] = true;
			}
			
			//
			// Update all fields list
			//
			
			var allFieldsDb = xrefsList.allFieldsDb;
			
			for (i = 0; i < allFieldsDb.length; i++) {
				
				var allFieldsItemObj = allFieldsDb[i];
			
				fieldId = allFieldsItemObj.id;
				
				var isActiveField = activeFieldsLookup[fieldId] ? true : false;
				
				// Track isActiveField state in allFieldsDb!
				allFieldsItemObj.isActiveField = isActiveField;
				
				// We hide any active field in all fields list
				
				var displayStyle = isActiveField ? 'none' : 'list-item';
				var li = util.getE('xrefs:all_fields_list:' + fieldId);
				li.className = '';
				li.style.display = displayStyle;
			}
			
			// Update the button state
			xrefsList.updateButtonState();
		}
		
		// 
		// Set edit mode display
		//
		
		util.showE('xrefs_edit_mode_btn', !isEditMode);
	
		var editElementIds = [
			'xrefs:add_remove_btn',
			'xrefs:move_up_btn',
			'xrefs:move_down_btn',
			'xrefs:all_fields_label',
			'xrefs:all_fields_section'
		];
		
		util.showE(editElementIds, isEditMode);
		
		xrefsList.isEditMode = isEditMode;
	},
	
	updateButtonState: function() {
		
		var isAddRemove = false;
		var isMoveUp = false;
		var isMoveDown = false;
		var addRemoveButtonType = 'neutral';
		var addRemoveButtonLabel = '< >';
		var h = util.h;
		
		if (xrefsList.activeFields.selectedItemIndex != -1) {
			
			isAddRemove = true;
			addRemoveButtonType = 'remove';
			addRemoveButtonLabel = langVar('lang_stats.btn.remove') + ' >';
			
			// Check the move buttons
			// Note, we don't allow to move numerical fields above non-numerical fields and vice versa.
			var selectedItemIndex = xrefsList.activeFields.selectedItemIndex;
			
			var activeFieldsDb = xrefsList.activeFieldsDb;
			var allFieldsDb  = xrefsList.allFieldsDb;

			var selectedFieldId = activeFieldsDb[selectedItemIndex];
			var selectedItem = allFieldsDb[h(selectedFieldId)];
			
			// Handle isMoveUp
			if (selectedItemIndex > 0) {
				// The selected item and previous item must have the same isAggregatingField value
				
				var previousFieldId = activeFieldsDb[selectedItemIndex - 1];
				var previousItem = allFieldsDb[h(previousFieldId)];
				
				if (selectedItem.isAggregatingField == previousItem.isAggregatingField) {
					isMoveUp = true;
				}
			}
			
			// Handle is isMoveDown
			if (selectedItemIndex < activeFieldsDb.length - 1) {
				// The selected item and next item must have the same isAggregatingField value
				
				var nextFieldId = activeFieldsDb[selectedItemIndex + 1];
				var nextItem = allFieldsDb[h(nextFieldId)];
				
				if (selectedItem.isAggregatingField == nextItem.isAggregatingField) {
					isMoveDown = true;
				}
			}
		}
		else if (xrefsList.allFields.selectedFieldId != '') {
			isAddRemove = true;
			addRemoveButtonType = 'add';
			addRemoveButtonLabel = '< ' + langVar('lang_stats.btn.add');
		}
		
		util.enableE('xrefs:add_remove_btn', isAddRemove);
		util.enableE('xrefs:move_up_btn', isMoveUp);
		util.enableE('xrefs:move_down_btn', isMoveDown);
	
		if (xrefsList.activeAddRemoveButtonType != addRemoveButtonType) {
			util.updateT('xrefs:add_remove_btn', addRemoveButtonLabel);
			xrefsList.activeAddRemoveButtonType = addRemoveButtonType;
		}
		
		// Reset fields error in xrefs
		resetFieldsError();
	},
	
	addRemoveItem: function() {
		
		if (xrefsList.allFields.selectedFieldId != '') {
			xrefsList.addItem(xrefsList.allFields.selectedFieldId);
		}
		else if (xrefsList.activeFields.selectedItemIndex != -1) {
			xrefsList.removeItem(xrefsList.activeFields.selectedItemIndex);
		}

		// Update sort_by
		xrefsList.updateSortBy();
	},
	
	addItem: function(selectedFieldId) {
		
		// alert('selectedFieldId: ' + selectedFieldId);
		
		var allFieldsDb = xrefsList.allFieldsDb;
		
		var i;
		
		//
		// Get all the inactive fields so that we know which all fields item has to be selected next
		// Note, we include the selectedFieldId in inactiveFieldIds!
		//
		
		var inactiveFieldIds = [];
		
		for (i = 0; i < allFieldsDb.length; i++) {
			
			var allFieldsItemObj = allFieldsDb[i];
			var fieldId = allFieldsItemObj.id;
			
			if (!allFieldsItemObj.isActiveField) {
				
				inactiveFieldIds[inactiveFieldIds.length] = fieldId;
				
				if (fieldId == selectedFieldId) {
					
					// Mark this field item as active
					allFieldsItemObj.isActiveField = true;
				}
			}
		}
		
		//
		// Get the item which we select next in the all fields list
		//
		
		var nextSelectedFieldId = '';
		
		for (i = 0; i < inactiveFieldIds.length; i++) {
			
			if (inactiveFieldIds[i] == selectedFieldId) {
				
				var nextItemIndex = -1;
				
				// If there is a next item then use it
				if (i + 1 < inactiveFieldIds.length) {
					nextSelectedFieldId = inactiveFieldIds[i + 1];
				}
				else if (i != 0) {
					nextSelectedFieldId = inactiveFieldIds[i - 1];
				}
			}
		}
		
		//
		// Handle selected item
		//
		
		var selectedLi = util.getE('xrefs:all_fields_list:' + selectedFieldId);
		selectedLi.className = '';
		selectedLi.style.display = 'none';
		xrefsList.allFields.selectedFieldId = '';
		
		
		//
		// Select the next item
		// 
		
		if (nextSelectedFieldId != '') {
			xrefsList.allFields.selectByFieldId(nextSelectedFieldId);
		}
		else {
			// There is no more item in all fields, so we need to update the button state.
			xrefsList.updateButtonState();
		}
		
		//
		// Add item to active fields
		//
		
		xrefsList.activeFields.addItem(selectedFieldId);
	},
	
	removeItem: function(selectedItemIndex) {
		
		var activeFieldsDb = xrefsList.activeFieldsDb;
		var selectedFieldId = activeFieldsDb[selectedItemIndex];
		
		// Get the item which is selected next in active fields
		var nextSelectedItemIndex = -1;
		var numberOfItems = activeFieldsDb.length;
		
		if (numberOfItems > 1) {
			nextSelectedItemIndex = (selectedItemIndex < numberOfItems - 1) ? selectedItemIndex : selectedItemIndex - 1;
		}
		
		// Remove selected li element
		var ul = util.getE('xrefs:active_fields_list');
		var li = util.getE('xrefs:active_fields_list:' + selectedItemIndex);
		ul.removeChild(li);
		
		// We must update the li element Ids so that they match
		// the index number of the activeFieldsDb items!
		var liElementItems = ul.getElementsByTagName('li');
		for (var i = 0; i < liElementItems.length; i++) {
			li = liElementItems[i];
			li.id = 'xrefs:active_fields_list:' + i;
		}
		
		// Remove selected item from activeFieldsDb
		activeFieldsDb.splice(selectedItemIndex, 1);
		
		// Reset selectedItemIndex!
		xrefsList.activeFields.selectedItemIndex = -1;
		
		
		// Select next item if any
		if (nextSelectedItemIndex != -1) {
			xrefsList.activeFields.selectByItemIndex(nextSelectedItemIndex);
		}
		else {
			// There is no more item in active fields, we only update the button state.
			xrefsList.updateButtonState();
		}
	
		
		// Show the removed item in all fields list
		xrefsList.allFields.activateItem(selectedFieldId);
	},

	activeFields: {
		
		selectedItemIndex: -1, // -1 is set if no item is selected!
		
		selectActor: function(evt) {
			
			// Invoked upon list click
			var elementIdSuffix = xrefsList.getSelectedItemElementIdSuffix(evt);
			
			if (elementIdSuffix != '') {
				var itemIndex = parseInt(elementIdSuffix, 10);
				xrefsList.activeFields.selectByItemIndex(itemIndex);
			}			
		},
		
		selectByItemIndex: function(itemIndex) {
			
			// alert('selectByItemIndex: ' + itemIndex);
			
			if (xrefsList.activeFields.selectedItemIndex == itemIndex) {
				// Deselect the item
				xrefsList.activeFields.deselectField();
			}
			else {
				
				// If an all fields item is selected then deselect it now
				if (xrefsList.allFields.selectedFieldId != '') {
					xrefsList.allFields.deselectField();
				}
				
				// Deselect any selected active item
				if (xrefsList.activeFields.selectedItemIndex != -1) {
					// Deselect already selected item
					xrefsList.activeFields.deselectField();
				}
				
				// Select the item of itemIndex
				var li = util.getE('xrefs:active_fields_list:' + itemIndex);
				li.className = 'selected';
				
				xrefsList.activeFields.selectedItemIndex = itemIndex;
			}
			
			// If we are already in edit mode we only update the button state,
			// else we switch to edit mode now.
			if (xrefsList.isEditMode) {
				// Update the buttons
				xrefsList.updateButtonState();
			}
			else {
				// switchToEditMode(), includes updateButtonState()
				xrefsList.switchToEditMode(true);
			}
		},
		
		deselectField: function() {
			
			var li = util.getE('xrefs:active_fields_list:' + xrefsList.activeFields.selectedItemIndex);
			li.className = '';
			xrefsList.activeFields.selectedItemIndex = -1;
		},
		
		addItem: function(fieldId) {
			
			// Called from xrefsList.addItem(). Adds the given item to the active fields list
			var allFieldsDb = xrefsList.allFieldsDb;
			var h = util.h;
			var fieldItemToAdd = allFieldsDb[h(fieldId)];
			
			// util.showObject(fieldItemToAdd);
			
			// Add fieldId to activeFieldsDb
			var activeFieldsDb = xrefsList.activeFieldsDb;
			var insertIndex;
			if (fieldItemToAdd.isAggregatingField) {
				// Add field as last aggregating field
				insertIndex = activeFieldsDb.length;
			}
			else {
				// Add field as last non-aggregating field
				var firstAggregatingFieldIndex = -1;
				for (var i = 0; i < activeFieldsDb.length; i++) {
					var existingFieldId = activeFieldsDb[i];
					var existingFieldItem =  allFieldsDb[h(existingFieldId)];
					
					if (existingFieldItem.isAggregatingField) {
						firstAggregatingFieldIndex = i;
						break;
					}
				}
				
				if (firstAggregatingFieldIndex > -1) {
					
					if (firstAggregatingFieldIndex == 0) {
						// There is no non-aggregating field in the list,
						// so we add it as first item
						insertIndex = 0;
					}
					else {
						insertIndex = firstAggregatingFieldIndex - 1;
					}
				}
				else {
					// There is no aggregating field yet, so we add the item as last item
					insertIndex = activeFieldsDb.length;
				}
			}
			
			// Insert item into array
			activeFieldsDb.splice(insertIndex, 0, fieldId);
			
			// Rebuild the active fields list
			xrefsList.activeFields.build();
		},
		
		moveItem: function() {
		
			var isMoveUp = (this.id == 'xrefs:move_up_btn');
			
			var activeFieldsDb = xrefsList.activeFieldsDb;
			
			var selectedItemIndex = xrefsList.activeFields.selectedItemIndex;
			var selectedItemId = activeFieldsDb[selectedItemIndex];
			var selectedLiElement = util.getE('xrefs:active_fields_list:' + selectedItemIndex);
			var selectedTextElement = selectedLiElement.firstChild;
			var selectedLabel = selectedTextElement.nodeValue;
			
			// Get the itemIndex of the element which will be replaced with the selected element
			var nextItemIndex = isMoveUp ? selectedItemIndex - 1 : selectedItemIndex + 1;
			var nextLiElement = util.getE('xrefs:active_fields_list:' + nextItemIndex);
			var nextTextElement = nextLiElement.firstChild;
			var nextLabel = nextTextElement.nodeValue;
			
			// Update item position in activeFieldsDb
			activeFieldsDb.splice(selectedItemIndex, 1); // Removes selected itemId
			activeFieldsDb.splice(nextItemIndex, 0, selectedItemId); // Re-inserts selected itemId at new position
			
			// Replace the list labels
			var newSelectedTextElement = util.createT(nextLabel);
			var newNextTextElement = util.createT(selectedLabel);
			selectedLiElement.replaceChild(newSelectedTextElement, selectedTextElement);
			nextLiElement.replaceChild(newNextTextElement, nextTextElement);
			
			// Select the next item
			xrefsList.activeFields.selectByItemIndex(nextItemIndex);
		},
		
		build: function() {
			
			var allFieldsDb = xrefsList.allFieldsDb;
			var activeFieldsDb = xrefsList.activeFieldsDb;
			
			// util.showObject(allFieldsDb);
		
			var ul = util.getE('xrefs:active_fields_list');
			util.removeChildElements(ul);
			
			for (var i = 0; i < activeFieldsDb.length; i++) {
				
				var fieldId = activeFieldsDb[i];
				var fieldItem = allFieldsDb[util.h(fieldId)];
				
				// alert('fieldId: ' + fieldId);
				// util.showObject(fieldItem);
				
				// In activeFields we use the index of fields for the ID!
				var elementId = 'xrefs:active_fields_list:' + i;
				var li = util.createE('li', {id:elementId});
				var text = util.createT(fieldItem.label);
				
				if (fieldItem.isAggregatingField) {
					
					var span = util.createE('span');
					var spanLabel = ' ' + langVar('lang_admin.cross_reference_groups.numerical_abbrev');
					
					var spanText = util.createT(spanLabel);
					util.chainE(ul, [[li, text], [span, spanText]]);
				}
				else {
					
					util.chainE(ul, li, text);
				}
			}
		}
	},
	
	allFields: {
		
		// The allFields data are presented by xrefs.databaseFieldsDb!
		
		selectedFieldId: '',
		
		selectActor: function(evt) {
			
			// Invoked upon list click
			var elementIdSuffix = xrefsList.getSelectedItemElementIdSuffix(evt);
			
			if (elementIdSuffix != '') {
				var fieldId = elementIdSuffix;
				xrefsList.allFields.selectByFieldId(fieldId);
			}
		},
		
		selectByFieldId: function(fieldId) {
			
			if (xrefsList.allFields.selectedFieldId == fieldId) {
				// Deselect the item
				xrefsList.allFields.deselectField();
			}
			else {
				
				// If an active fields item is selected then deselect it now
				if (xrefsList.activeFields.selectedItemIndex != -1) {
					xrefsList.activeFields.deselectField();
				}
				
				// Deselect any selected all fields item
				if (xrefsList.allFields.selectedFieldId != '') {
					// Deselect already selected item
					xrefsList.allFields.deselectField();
				}
				
				// Select the item of fieldId
				var li = util.getE('xrefs:all_fields_list:' + fieldId);
				li.className = 'selected';
				
				xrefsList.allFields.selectedFieldId = fieldId;
			}
			
			// Update the buttons
			xrefsList.updateButtonState();
		},
		
		deselectField: function() {
			
			var li = util.getE('xrefs:all_fields_list:' + xrefsList.allFields.selectedFieldId);
			li.className = '';
			xrefsList.allFields.selectedFieldId = '';
		},
		
		activateItem: function(fieldId) {
			
			// Sets/shows the list item of the given fieldId in the all fields list.
			var li = util.getE('xrefs:all_fields_list:' + fieldId);
			li.style.display = 'list-item';
			
			var allFieldsDb = xrefsList.allFieldsDb;
			var fieldItemObj = allFieldsDb[util.h(fieldId)];
			fieldItemObj.isActiveField = false;
		},
		
		//
		// Initial list builder
		//
		
		addListItem: function(ul, dbFieldItem) {
			
			var elementId = 'xrefs:all_fields_list:' + dbFieldItem.id;
			var li = util.createE('li', {id:elementId});
			var text = util.createT(dbFieldItem.label);
			
			// Add the type within a span to the label, for numerical fields only
			if (dbFieldItem.isAggregatingField) {
			
				var span = util.createE('span');
				var spanLabel = ' ' + langVar('lang_admin.cross_reference_groups.numerical_abbrev');
				var spanText = util.createT(spanLabel);
				util.chainE(ul, [[li, text], [span, spanText]]);
			}
			else {
				util.chainE(ul, li, text);
			}
		},
		
		
		build: function() {
			
			// Builds the initial all fields list
			
			var allFieldsDb = xrefsList.allFieldsDb;
			
			var ul = util.getE('xrefs:all_fields_list');
			
			for (var i = 0; i < allFieldsDb.length; i++) {
				var allFieldsItemObj = allFieldsDb[i];
				xrefsList.allFields.addListItem(ul, allFieldsItemObj);
			}
		}
	},

	//
	//
	// Handle sort by
	//
	//

	sortByActivated: function(e) {

		// Sort by has been activated, set xrefsList.activeSortBy
		xrefsList.activeSortBy = util.getF('xrefs:sort_by');
	},

	updateSortBy: function() {

		// The sort_by field must be updated every time the
		// activeFieldsDb changes
		var activeSortBy = xrefsList.activeSortBy;
		var activeSortByFieldExists = false;
		var activeFieldsDb = xrefsList.activeFieldsDb;
		var allFieldsDb = xrefsList.allFieldsDb;
		var h = util.h;

		var a = [{name: '', label: '(' + langVar('lang_stats.general.none') + ')'}];

		for (var i = 0, len = activeFieldsDb.length; i < len; i++) {

			var item = allFieldsDb[h(activeFieldsDb[i])];
			var name = item.name;
			var label = item.label;
			a[i+1] = {name: name, label: label};

			// Check if active sort by field exists
			if (activeSortBy == name) {
				activeSortByFieldExists = true;
			}
		}

		util.populateSelect('xrefs:sort_by', a, 'name', 'label');

		// Set the activeSortBy
		if (!activeSortByFieldExists) {
			// Reset activeSortBy
			activeSortBy = '';
			xrefsList.activeSortBy = activeSortBy;
		}

		util.setF('xrefs:sort_by', activeSortBy);
	},
	
	//
	//
	// Utilities
	//
	//
	
	getSelectedItemElementIdSuffix: function(evt) {
		
		// Returns an elementIdSuffix of the selected item
		// or empty value if no item has been clicked.
		
		var elementIdSuffix = '';
			
		var element = evt.target || evt.srcElement;
		var tagName = element.nodeName;

		if (tagName == 'LI' || tagName == 'SPAN') {
			
			if (tagName == 'LI') {
				elementId = element.id;
			}
			else {
				var li = element.parentNode;
				elementId = li.id;
			}
			
			// alert('elementId: ' + elementId);
			
			var dat = elementId.split(':');
			var elementIdSuffix= dat[2];
		}
		
		return elementIdSuffix;
	}
};
