//
// reportFiltersPanel.js
// This class handles report filter items and groups
// in reports and scheduler.
//

/* global
	globalFilter: false,
	filterItemEditor: false,
	filterGroupEditor: false,
	repFiltersUtil: false */

var ReportFiltersPanel = (function() {

	'use strict';

	var YE = YAHOO.util.Event,
//		YD = YAHOO.util.Dom,
		YL = YAHOO.lang;

	var Filters = function(conf) {

//		util.showObject(conf);

		this.containerId = conf.containerId;
		this.headerClassName = conf.headerClassName || 'filter-items-panel-header';
		this.bodyClassName = conf.bodyClassName || 'filter-items-panel-body';

		// The filters list container
		this.container = null;

		// Buttons definition in panel header.
		this.addItems = conf.addItems || false;
		this.moveItems = conf.moveItems || false;
		this.importItems = conf.importItems || false;

		this.newItemBtn = null;
		this.newGroupBtn = null;
		this.saveCheckedAsGroupBtn = null;
		this.importBtn = null;
//		this.sortBtn = null;
		this.selectAllBtn = null;
		this.deselectAllBtn = null;

		// Callback if isMoveToSaved
		this.moveToSavedCallback = this.moveItems ? conf.moveToSavedCallback : null;

		// Call back to open the Import window
		this.openImportPanelCallback = this.importItems ? conf.openImportPanelCallback : null;

		// Callback when filter items become selected/deselected,
		// this is i.e. used in the import panel to enable/disable
		// the import button.
		this.filterItemsSelectCallback = conf.filterItemsSelectCallback || null;

		this.idPrefix = '';
		this.idCount = 0;

		// filterItems is a new array which includes filterGroups.
		// filterItems and filterGroups are only separate nodes
		// on the server side.
		this.filterItems = [];

		this.isModified = false;

		// The active profileName, this is required in scheduler
		// actions when importing filters.
		this.profileName = '';

		// A reference to the active queryFieldsDb
		this.queryFieldsDb = [];
	}

	Filters.prototype = {

		initPanel: function() {

			// This creates filter buttons and adds event listeners.
			var idPrefix = util.getUniqueElementId();

			this.idPrefix = idPrefix;

			// Create the buttons
			var btns = '';

			var newFilterItemBtnId = idPrefix + ':new_filter_item';
			var newFilterGroupBtnId = idPrefix + ':new_filter_group';
			var saveCheckedAsGroupBtnId = idPrefix + ':save_checked_as_group';
			var importBtnId = idPrefix + ':import_items';
//			var sortBtnId = idPrefix + ':sort_items';
			var selectAllBtnId = idPrefix + ':select_all';
			var deselectAllBtnId = idPrefix + ':deselect_all';

			if (this.addItems) {
				btns = '<a id="' + newFilterItemBtnId + '" href="javascript:;" class="command-link-50">' +
					langVar('lang_stats.global_filter.new_item') + '</a> ' +

					'<a id="' + newFilterGroupBtnId + '" href="javascript:;" class="command-link-50">' +
					langVar('lang_stats.global_filter.new_group') + '</a> ' +

					'<a id="' + saveCheckedAsGroupBtnId + '" href="javascript:;" class="command-link-50">' +
					langVar('lang_stats.global_filter.save_checked_as_group') + '</a> ';
			}

			if (this.importItems) {

				btns += '<a id="' + importBtnId + '" href="javascript:;" class="command-link-50">' +
					langVar('lang_stats.btn.import') + '</a> ';
			}

			// Disabled sort because list appears always sorted anyway.
//			if (this.addItems) {
//
//				btns += '<a id="' + sortBtnId + '" href="javascript:;" class="command-link-50">' +
//					langVar('lang_stats.btn.sort') + '</a> ';
//			}

			// Select All, Deselect All
			btns += '<a id="' + selectAllBtnId + '" href="javascript:;" class="command-link-50">' +
				langVar('lang_stats.btn.select_all') + '</a> ' +

				'<a id="' + deselectAllBtnId + '" href="javascript:;" class="command-link-50">' +
				langVar('lang_stats.btn.deselect_all') + '</a>';


			// Create HTML
			var outerContainer = util.getE(this.containerId);
			var innerContainerId = idPrefix + ':' + this.containerId;
			var html = '<div class="' + this.headerClassName + '">' + btns + '</div>' +
				'<div class="' + this.bodyClassName + '">' +
				'<div id="' + innerContainerId + '"></div>';
				'</div>';

			outerContainer.innerHTML = html;

			// Get the container for the filters list
			this.container = util.getE(innerContainerId);

			// Create button objects
			var cmlOpt = {
				classNameEnabled: 'command-link-50',
				classNameDisabled: 'command-link-50-disabled'
			};

			if (this.addItems) {

				this.newItemBtn = new util.CommandLink(newFilterItemBtnId, this._handleMainEvents, true, cmlOpt, this);
				this.newGroupBtn = new util.CommandLink(newFilterGroupBtnId, this._handleMainEvents, true, cmlOpt, this);
				this.saveCheckedAsGroupBtn = new util.CommandLink(saveCheckedAsGroupBtnId, this._handleMainEvents, true, cmlOpt, this);
//				this.sortBtn = new util.CommandLink(sortBtnId, this._handleMainEvents, true, cmlOpt, this);

				// Init editor panels
				filterItemEditor.init();
				filterGroupEditor.init();
			}

			if (this.importItems) {

				this.importBtn = new util.CommandLink(importBtnId, this._handleMainEvents, true, cmlOpt, this);
			}

			this.selectAllBtn = new util.CommandLink(selectAllBtnId, this._handleMainEvents, true, cmlOpt, this);
			this.deselectAllBtn = new util.CommandLink(deselectAllBtnId, this._handleMainEvents, true, cmlOpt, this);

			// Handle filter list events
			YE.addListener(this.container, 'click', this._handleFilterListEvents, this);
		},


		initFilters: function(profileName, queryFieldsDb, filterItems, filterGroups) {

//			console.log('reportFiltersPanel.initFilters() - profileName: ' + profileName);

			// initFilters starts over with new filters.
			// This is required when initializing filters
			// in scheduler actions where profiles can be changed without
			// a page reload.

			if (this.idPrefix === '') {
				// Initialize the panel
				this.initPanel();
			}

			this.profileName = profileName;
			this.queryFieldsDb = queryFieldsDb;

			// Combine filterItems and filterGroups and add helper properties
			this.filterItems = this._getModifiedAndCombinedFilterItems(filterItems, filterGroups);

//			util.showObject(this.filterItems);

			// Update queryFieldsDb
			filterItemEditor.setProfileSpecificData(queryFieldsDb);

			// Build the filterItem list
			this._buildFiltersList();

			this.setButtonState();
		},

		resetIsModified: function() {

			this.isModified = false;
		},

		getIsModified: function() {

			return this.isModified;
		},

		_handleMainEvents: function(evt, self) {

			var element = evt.target || evt.srcElement;
			var elementId = element.id;
			var dat = elementId.split(':');
			var key = dat[1];

//			console.log('key: ' + key);

			switch (key) {

				case 'new_filter_item':

					self._createNewFilterItem(false, '');
					break;

				case 'new_filter_group':

					filterGroupEditor.open('', '', true, false, self);
					break;

				case 'save_checked_as_group':

					filterGroupEditor.open('', '', true, true, self);
					break;

				case 'import_items':

					// Callback import window
					self.openImportPanelCallback(
						self.profileName,
						self.queryFieldsDb,
						self.filterItems
					);

					break;

//				case 'sort_items':
//
//					self._buildFiltersList();
//					break;

				case 'select_all':

					self._selectAll(true);
					break;

				case 'deselect_all':

					self._selectAll(false);
					break;
			}
		},

		_handleFilterListEvents: function(evt, self) {


			var element = evt.target || evt.srcElement;
			var elementId = element.id;

//			console.log('element.nodeName: ' + element.nodeName);
//			console.log('element.id: ' + element.id);

			if (elementId !== '') {

				var dat = elementId.split(':');
				var key = dat[1];
				var itemId = dat[2];

				switch (key) {

					case 'cb':

						// Toggle filter state
						self._toggleFilterActiveState(element, itemId);
						break;

					case 'edit':

						// Edit filter item
						self._editFilterItem(itemId);

						break;

					case 'edit_subitem':

						self._editFilterSubItem(itemId);

						break;

					case 'delete':

						// Delete saved filter item
						self._deleteFilterItem(itemId);
						break;

					case 'delete_subitem':

						// Delete saved filter item
						self._deleteFilterSubItem(itemId);
						break;

					case 'add':

						// Add filter item in filter group
						self._createNewFilterItem(true, itemId);

						break;

					case 'img':

						// Expand collapse group
						self._expandCollapseGroup(itemId);
						break;

					case 'move':

						// Move to saved
						self._moveToSaved(itemId);

						break;
				}
			}
		},


		setButtonState: function() {

			// Updates the state of number of items and enables disables control buttons
			var isItems = this.filterItems.length > 0;
			var numActiveItems = this.getNumOfActiveItems();

			if (this.addItems) {

				this.newItemBtn.enable();
				this.newGroupBtn.enable();

				// Allow to save single items as new group because the item could already be a group
				// or a single item to which other items are added manually.
				this.saveCheckedAsGroupBtn.enable((numActiveItems > 0));

//				this.sortBtn.enable(isItems);
			}

			if (this.importItems) {
				// Only exists in scheduler actions.
				// Is disabled when no profile is selected.
				this.importBtn.enable();
			}

			this.deselectAllBtn.enable(isItems);
			this.selectAllBtn.enable(isItems);

			// Fire filterItemsSelectCallback,
			// this is the best place to call the listener.
			if (this.filterItemsSelectCallback) {
				this.filterItemsSelectCallback((numActiveItems > 0));
			}
		},

		disableAll: function() {

			// Disables all buttons
			// This is called from scheduler actions when no profile is selected.
			// In this case no filter items can be added.

			// Note, the panel may not yet exist when calling disableAll, create it now.
			if (this.idPrefix === '') {
				// Initialize the panel
				this.initPanel();
			}

			if (this.addItems) {

				this.newItemBtn.disable();
				this.newGroupBtn.disable();
				this.saveCheckedAsGroupBtn.disable();
//				this.sortBtn.disable();
			}

			if (this.importItems) {

				this.importBtn.disable();
			}

			this.deselectAllBtn.disable();
			this.selectAllBtn.disable();
		},

		_toggleFilterActiveState: function(element, itemId) {

			var item = this.filterItems[util.h(itemId)];
			item.isActive = element.checked;

			this.isModified = true;
			this.setButtonState();
		},

		_selectAll: function(checkItems) {
			// Set checkbox state in array, then apply setCheckboxes

			var filterItems = this.filterItems;

			for (var i = 0, len = filterItems.length; i < len; i++) {
				var item = filterItems[i];
				item.isActive = checkItems;
				util.setF(this.idPrefix + ':cb:' + item.id , checkItems);
			}

			this.isModified = true;
			this.setButtonState();
		},

		_createNewFilterItem: function(isItemInGroup, groupItemId) {

			var isActive = isItemInGroup ? false : true;
			var isNew = true;
			var id = this._getNewItemId();

			var item = {
				id: id,
				isNegated: false,
				filterType: 'field',
				fieldName: '',
				expressionType: '',
				itemValue: ''
			};

			filterItemEditor.open(item, isActive, isNew, isItemInGroup, groupItemId, this);
		},

		addNewItemViaMoveToSaved: function(newFilterItem) {

			// This inserts a new filter item which is called from another panel
			// via Move To Saved.
			// Filter groups can be ignored.

			// The newFilterItem is only inserted if it doesn't yet exist.
			// If the item already exists set only its isActive value to the
			// value of the newFilterItem.

			var filterItems = this.filterItems;
			var isExistingItem = false;
			var existingFilterItem;
			var newFilterItemString = repFiltersUtil.getFilterItemAsString(newFilterItem);

			for (var i = 0, len = filterItems.length; i < len; i++) {

				var existingFilterItem = filterItems[i];

				if (existingFilterItem.isRootItem) {

					var existingFilterItemString = repFiltersUtil.getFilterItemAsString(existingFilterItem);

					if (existingFilterItemString === newFilterItemString) {
						isExistingItem = true;
						break;
					}
				}
			}

			if (!isExistingItem) {

				// give the new item a new id
				newFilterItem.id = this._getNewItemId();

				// Insert newFilterItem
				filterItems.push(newFilterItem);

				// Update hash
				util.createHash(filterItems, 'id');

				this._buildFiltersList();
			}
			else {

				// The item already exists. Override isActive state only
				// if inserted item isActive.

				if (newFilterItem.isActive) {

					existingFilterItem.isActive = true;
					this._updateCheckboxState();
				}
			}

			this.isModified = true;
			this.setButtonState();
		},

		addNewItemsViaImport: function(newFilterItemsAndGroups) {

			// This adds/imports new filterItems and filterGroups
			// from another reportFiltersPanel object.
			var filterItems = this.filterItems;
			var clonedItem;

			for (var i = 0, len = newFilterItemsAndGroups.length; i < len; i++) {

				clonedItem = util.cloneObject(newFilterItemsAndGroups[i]);

				// Assign new item id.
				clonedItem.id = this._getNewItemId();

				// Reset isActive or not?
//				clonedItem.isActive = true;

				// Set isExpanded of groups to false;
				if (clonedItem.isGroup) {

					clonedItem.isExpanded = false;

					// Assign new id to subitems
					var subitems = clonedItem.items;
					for (var j = 0, numSubitems = subitems.length; j < numSubitems; j++) {
						subitems[j].id = this._getNewItemId();
					}
				}

				// Add new item to filterItems
				filterItems.push(clonedItem);
			}

			// Create hash to access filters by id
			util.createHash(filterItems, 'id');

			// Build the filterItem list
			this._buildFiltersList();

			this.setButtonState();
		},

		_editFilterItem: function(itemId) {

			// Edit root item or group
			var item = this.filterItems[util.h(itemId)];

//			util.showObject(item);

			if (item.isRootItem) {

				var isActive = item.isActive;
				var isNew = false;
				var groupItemId = '';
				var isItemInGroup = false;

				filterItemEditor.open(item, isActive, isNew, isItemInGroup, groupItemId, this);
			}
			else {

				// This is a group item
				filterGroupEditor.open(itemId, item.label, false, false, this);
			}
		},

		_editFilterSubItem: function(itemId) {

			// This is an item in a group.
			// Get the itemId of the group and subitem

//			util.showObject(parentItem);
			var isActive = false;
			var isNew = false;
			var isItemInGroup = true;

			var parentItem = this._getParentFilterItem(itemId);
			var groupItemId = parentItem.id;
			var subitems = parentItem.items;

			for (var i = 0; i < subitems.length; i++) {
				if (subitems[i].id == itemId) {

					var item = subitems[i];
				}
			}

			filterItemEditor.open(item, isActive, isNew, isItemInGroup, groupItemId, this);
		},

		_deleteFilterItem: function(itemId) {

			// Deletes a filter root item or group
			util.deleteArrayObject(this.filterItems, 'id', itemId);

			this.isModified = true;
			this._buildFiltersList();
			this.setButtonState();
		},

		_deleteFilterSubItem: function(itemId) {

			// Delete a filter item of a filter group
			var parentItem = this._getParentFilterItem(itemId);
			var subitems = parentItem.items;

			util.deleteArrayObject(subitems, 'id', itemId);

			this.isModified = true;
			this._buildFiltersList();
			this.setButtonState();
		},

		_moveToSaved: function(itemId) {

			// Clone item which becomes moved.
			var filterItems = this.filterItems;
			var filterItem = filterItems[util.h(itemId)];
			var clonedFilterItem = util.cloneObject(filterItem);

			// Delete item from this panel
			util.deleteArrayObject(filterItems, 'id', itemId);

			// Move the clonedFilterItem
			this.moveToSavedCallback(clonedFilterItem);

			this.isModified = true;
			this._buildFiltersList();
			this.setButtonState();
		},

		//
		// Handle Save Checked as New Group
		//

		saveCheckedAs: function() {

			var filterLabel = util.getF('save_checked_filters_as_name');

			// alert('Filter name');

			if (filterLabel != '') {

				// Check for duplicate filter name

				var isDuplicateLabel = false;
				var filterNameItems = [];

				for (var i = 0; i < savedFilterGroups.length; i++) {

					var existingFilterLabel = savedFilterGroups[i].label;

					if (existingFilterLabel.toLowerCase() == filterLabel.toLowerCase()) {
						isDuplicateLabel = true;
					}

					filterNameItems[i] = savedFilterGroups[i].name;
				}


				if (!isDuplicateLabel) {

					var filterName = util.labelToUniqueNodeName(filterLabel, filterNameItems, 'filter');
					saveCheckedFilters(filterName, filterLabel);
				}
				else {

					if (confirm(langVar('lang_stats.global_filter.confirm_existing_filter_replacement_message'))) {

						// KHP-RC, Delete existing filter
						alert('Replace existing filter');

						return false;
					}
				}
			}
			else {

				alert(langVar('lang_stats.global_filter.missing_filter_name_message'));
			}
		},

		saveCheckedAsNewGroupProcessing: function(groupName, groupLabel) {

			// Create new group from checked saved filter items and filter groups.

			var filterItems = this.filterItems;

			var lookup = {};
			var newItems = [];

			for (var i = 0, len = filterItems.length; i < len; i++) {

				var item = filterItems[i];

				if (item.isActive) {

					if (item.isRootItem) {

						this._addCheckedFilterItem(lookup, newItems, item);
					}
					else {

						// Add subitems of filter group
						var subitems = item.items;
						for (var j = 0, numSubitems = subitems.length; j < numSubitems; j++) {

							this._addCheckedFilterItem(lookup, newItems, subitems[j]);
						}
					}
				}
			}

			// Setup new filter group

			var newGroupId = this._getNewItemId();
			var newGroup = {
				id: newGroupId,
				isRootItem: false,
				isGroup: true,
				isExpanded: false,
				name: groupName,
				label: groupLabel,
				isActive: false,
				items: newItems
			};

			// Add newGroup to savedFilterGroups

			filterItems.push(newGroup);

			// Sort the new Items
			filterItems.sort(this._compareLabels);

			// update hash
			util.createHash(filterItems, 'id');

			this.isModified = true;

			this._buildFiltersList();
			this.setButtonState();
		},

		_addCheckedFilterItem: function(lookup, newItems, filterItem) {

			// Adds filterItem to newItems if it doesn't yet exist.

			var filterId = repFiltersUtil.getFilterItemAsString(filterItem);

			if (!lookup.hasOwnProperty(filterId)) {

				var clonedFilterItem = util.cloneObject(filterItem);

				// Give the item a new id
				clonedFilterItem.id = this._getNewItemId();

				newItems.push(clonedFilterItem);
				lookup[filterId] = true;
			}
		},

		//
		// Create filter item rows
		//

		_buildFiltersList: function() {

			// Build the filter and filter group rows.
			// This is also used upon sorting the filter
			// rows.

			var item;
			var filterItems = this.filterItems;

//			util.showObject(filterItems, '_buildFiltersList()');

			var filtersHtml = '<table><tbody>';


			// Sort the items
			filterItems.sort(this._compareLabels);

			// Create the rows
			for (var i = 0, len = filterItems.length; i < len; i++) {

				item = filterItems[i];

				if (item.isRootItem) {
					filtersHtml += this._getFilterRow(item.id, item.label);
				}
				else {

					// This is a group item
					var subitems = item.items;
					var numSubitems = subitems.length;
					var groupHasItems = (numSubitems > 0);
					var isExpanded = item.isExpanded;

					filtersHtml += this._getFilterGroupRow(item.id, item.label, groupHasItems, isExpanded);

					// Sort subitems
					subitems.sort(this._compareLabels);

					for (var j = 0; j < numSubitems; j++) {

						var subItem = subitems[j];
						filtersHtml += this._getSubFilterRow(subItem.id, subItem.label, isExpanded);
					}
				}
			}

			filtersHtml += '</tbody></table>';

			this.container.innerHTML = filtersHtml;

			// Update checkbox state
			this._updateCheckboxState();
		},

		_getFilterRow: function(itemId, itemLabel) {

			var idPrefix = this.idPrefix;

			var row = '<tr id="' + idPrefix + ':row:' + itemId + '">';

			// Add empty cell for filter group icons column
			row += '<td style="width:14px"></td>';

			// Add checkbox cell
			var checkboxId = idPrefix + ':cb:' + itemId;
			var labelId = idPrefix + ':label:' + itemId;
			row += '<td style="width:14px">';
			row += '<input id="' + checkboxId + '" type="checkbox" />';
			row += '</td>';

			// Add label cell
			row += '<td>';
			row += '<label id="' + labelId + '" for="' + checkboxId + '">' +
					YL.escapeHTML(itemLabel)  + '</label>';
			row += '</td>';


			// Add buttons
			row += '<td class="filter-item-controls">';

			if (this.addItems || this.moveItems) {

				if (this.addItems) {

					// Edit
					row += this._getButton(idPrefix + ':edit:' + itemId, langVar('lang_stats.btn.edit'));
				}
				else {

					// Move to saved
					row += this._getButton(idPrefix + ':move:' + itemId, langVar('lang_stats.global_filter.move_to_saved'));
				}

				// Delete
				row += this._getButton(idPrefix + ':delete:' + itemId, langVar('lang_stats.btn.delete'));
			}

			row += '</td>';
			row += '</tr>';

			return row;
		},

		_getSubFilterRow: function(itemId, itemLabel, isExpanded) {

			// This returns a filter row of a group

			var idPrefix = this.idPrefix;
			var row = '<tr id="' + idPrefix + ':row:' + itemId + '"';
			if (!isExpanded) {row += ' style="display:none"';}
			row += '>';

			// Add empty cell for filter group icons column
			row += '<td style="width:14px"></td>';

			// Add empty cell for checkbox
			row += '<td style="width:14px"></td>';

			// Add label cell
			row += '<td>';
			row += '<span id="' + idPrefix + ':label:' + itemId + '">' + YL.escapeHTML(itemLabel)  + '</span>';
			row += '</td>';


			// Add buttons
			row += '<td class="filter-item-controls">';

			if (this.addItems) {

				// Edit
				row += this._getButton(idPrefix + ':edit_subitem:' + itemId, langVar('lang_stats.btn.edit'));

				// Delete
				row += this._getButton(idPrefix + ':delete_subitem:' + itemId, langVar('lang_stats.btn.delete'));
			}

			row += '</td>';

			row += '</tr>';

			return row;
		},

		_getFilterGroupRow: function(itemId, itemLabel, groupHasItems, isExpanded) {

			var imgSrc = '';
			if (!groupHasItems) {

				imgSrc = imgDb.gfEmptyGroup.src;
			}
			else if (!isExpanded) {
				imgSrc = imgDb.gfCollapsed.src;
			}
			else {
				// Show expanded icon
				imgSrc = imgDb.gfExpanded.src;
			}

	//		console.log('imgSrc: ' + imgSrc);

			var idPrefix =  this.idPrefix;
			var row = '<tr id="' + idPrefix + ':row:' + itemId + '">';

			// Add group icon
			row += '<td style="width:14px">';
			row += '<img id="' + idPrefix + ':img:' + itemId + '" src="' + imgSrc + '" width="31" height="15" alt="" />';
			row += '</td>';

			// Add checkbox cell
			var checkboxId = idPrefix + ':cb:' + itemId;
			var labelId = idPrefix + ':label:' + itemId;
			row += '<td style="width:14px">';
			row += '<input id="' + checkboxId + '" type="checkbox" />';
			row += '</td>';

			// Add label cell
			row += '<td>';
			row += '<label id="' + labelId + '" for="' + checkboxId + '">' +
					YL.escapeHTML(itemLabel)  + '</label>';
			row += '</td>';

			// Add buttons

			row += '<td class="filter-item-controls">';

			if (this.addItems) {

				// Add
				row += this._getButton(idPrefix + ':add:' + itemId, langVar('lang_stats.global_filter.add_new_item'));

				// Edit
				row += this._getButton(idPrefix + ':edit:' + itemId, langVar('lang_stats.btn.edit'));

				// Delete
				row += this._getButton(idPrefix + ':delete:' + itemId, langVar('lang_stats.btn.delete'));
			}

			row += '</td>';

			row += '</tr>';

			return row;
		},

		_getButton: function(id, btnLabel) {

			return ' <a id="' + id + '" href="javascript:;">' + btnLabel  + '</a>';
		},

		saveFilterItem: function(item, isNew, isItemInGroup, groupItemId) {

			// Invoked from filterItem.js!
			// If not isNew then item replaces the existing item (in this case item contains the original itemId!)

			var rebuildFiltersList = false;

			// Get itemId. All default properties already exist in item
			var itemId = item.id;

			// Get the label
			item.label = this._getFilterItemLabel(item);

			// Get the array where we insert or update the item.
			var filterItems = this.filterItems;
			var refItems;
			var parentItem;

			if (!isItemInGroup) {
				refItems = filterItems;
			}
			else {
				parentItem = filterItems[util.h(groupItemId)];
				refItems = parentItem.items;
			}


			// If not a new item then replace existing item

			if (!isNew) {

				// Get the index of the item to be replaced
				var itemIndex = util.getArrayObjectIndex(refItems, 'id', itemId);
				refItems.splice(itemIndex, 1, item);
				var labelId = this.idPrefix + ':label:' + itemId;
				util.updateT(labelId, item.label);
			}
			else {

				// This is a new item. Insert the new item as first item in array

				refItems.splice(0, 0, item);

				if (!isItemInGroup) {

					// Update hash
					util.createHash(filterItems, 'id');
				}
				else {

					// Set isExpanded of groupItem in case that this is the first item added.
					parentItem.isExpanded = true;
				}

				rebuildFiltersList = true;
			}

			this.isModified = true;

			if (rebuildFiltersList) {

				this._buildFiltersList();
				this.setButtonState();
			}
		},

		saveFilterGroup: function(itemId, groupName, groupLabel, isNew) {


			var filterItem;
			var filterItems = this.filterItems;

			if (!isNew) {

				filterItem = filterItems[util.h(itemId)];
				filterItem.name = groupName;
				filterItem.label = groupLabel;

				// Update label
				util.updateT(this.idPrefix + ':label:' + itemId, groupLabel);
			}
			else {

				var newGroupId = this._getNewItemId();
				filterItem = {
					id: newGroupId,
					name: groupName,
					label: groupLabel,
					isRootItem: false,
					isGroup: true,
					isActive: false,
					isExpanded: false,
					items: []
				};

				filterItems.push(filterItem);

				// update hash
				util.createHash(filterItems, 'id');

				this._buildFiltersList();
				this.setButtonState();
			}

			this.isModified = true;
		},

		_expandCollapseGroup: function(itemId) {

			var item = this.filterItems[util.h(itemId)];
			var isExpanded = item.isExpanded;
			var subitems = item.items;
			var numSubitems = subitems.length;

			// Expand/collapse group only if there are items
			if (numSubitems > 0) {

				var img = util.getE(this.idPrefix + ':img:' + itemId);
				img.src = isExpanded ? imgDb.gfCollapsed.src : imgDb.gfExpanded.src;

				for (var i = 0; i < numSubitems; i++) {
					// util.showObject(items[i]);
					var rowId = this.idPrefix + ':row:' + subitems[i].id;
					util.showE(rowId, !isExpanded);
				}

				item.isExpanded = !isExpanded;
			}
		},

		getFilterItems: function() {

			// Returns a new array with filterItems
			return this.getFilterOrGroupItems('isRootItem');
		},


		getFilterGroups: function() {

			// Returns a new array with filterGroups
			return this.getFilterOrGroupItems('isGroup');
		},

		getFilterOrGroupItems: function(key) {

			// Returns cloned items of the given key
			var filterItems = this.filterItems;
			var item;
			var clonedItem;
			var clonedItems = [];

			for (var i = 0, len = filterItems.length; i < len; i++) {

				var item = filterItems[i];
				if (item[key]) {
					clonedItem = util.cloneObject(item);
					clonedItems.push(clonedItem);
				}
			}

			return clonedItems;
		},

		getActiveItems: function() {

			// Returns cloned filterItems and filterGroups if isActive.
			var filterItems = this.filterItems;

			var item;
			var clonedItem;
			var clonedItems = [];

			for (var i = 0, len = filterItems.length; i < len; i++) {

				var item = filterItems[i];
				if (item.isActive) {
					clonedItem = util.cloneObject(item);
					clonedItems.push(clonedItem);
				}
			}

			return clonedItems;
		},

		//
		// Utilities
		//

		_getModifiedAndCombinedFilterItems: function(filterItems, filterGroups) {

			// This merges filterItems and filterGroups in a new array and adds
			// additional properties for list handling.

			var newItems = [];
			var newItem = {};

			// Add modified filterItems to newItems
			for (var i = 0, filterItemsLen = filterItems.length; i < filterItemsLen; i++) {

				// Create new item
				newItem = util.cloneObject(filterItems[i]);
				newItem.id = this._getNewItemId();
				newItem.label = this._getFilterItemLabel(newItem);
				newItem.isRootItem = true;
				newItem.isGroup = false;
				newItems.push(newItem);
			}

			// Add modified filterGroups to newItems
			for (var j = 0, filterGroupsLen = filterGroups.length; j < filterGroupsLen; j++) {

				// Create new item
				// Groups already have a label. Add id and isExpanded to keep expanded state.
				newItem = util.cloneObject(filterGroups[j]);
				newItem.id =  this._getNewItemId();
				newItem.isExpanded = false;
				newItem.isRootItem = false;
				newItem.isGroup = true;

				// Handle subitems
				var subitems = newItem.items;
				var subitem;
				for (var k = 0, subitemsLen = subitems.length; k < subitemsLen; k++) {
					subitem = subitems[k];
					subitem.id = this._getNewItemId();
					subitem.label = this._getFilterItemLabel(subitem);
					subitem.isRootItem = false;
					subitem.isGroup = false;
				}

				newItems.push(newItem);
			}

//			util.showObject(newItems, '_getModifiedAndCombinedFilterItems()');

			// Create hash to access filters by id
			util.createHash(newItems, 'id');

			return newItems;
		},

		_getNewItemId: function() {

			this.idCount += 1;
			return 'i' + this.idCount;
		},

		_getFieldLabel: function(fieldName) {

			var queryFieldItem = this.queryFieldsDb[util.h(fieldName)];
			return queryFieldItem.label;
		},

		_getFilterItemLabel: function(theFilterItem) {

			var filterType = theFilterItem.filterType;

			// We ignore the label if filterType is expression!

			var label = '';

			if (filterType !== 'expression') {

				var isNegated = theFilterItem.isNegated;
				var itemValue = theFilterItem.itemValue;


				switch (filterType) {

					case 'field':

						var fieldName = theFilterItem.fieldName;
						var expressionType = theFilterItem.expressionType;
						var queryFieldItem = this.queryFieldsDb[util.h(fieldName)];
						var fieldLabel = queryFieldItem.label;
						var isAggregatingField = queryFieldItem.isAggregatingField;
						var operator = '';

						// alert('expressionType: ' + expressionType);

						if (!isAggregatingField) {

							if (expressionType == 'regexp') {
								operator = !isNegated ? langVar('lang_stats.global_filter.field_matches_regular_expression') : langVar('lang_stats.global_filter.field_not_matches_regular_expression');
							}
							else if (expressionType == 'wildcard') {
								operator = !isNegated ? langVar('lang_stats.global_filter.field_matches_wildcard') : langVar('lang_stats.global_filter.field_not_matches_wildcard');
							}
							else {
								operator = !isNegated ? langVar('lang_stats.global_filter.field_is') : langVar('lang_stats.global_filter.field_is_not');

								var fieldCategory = queryFieldItem.category;

								if (fieldCategory == 'day_of_week' || fieldCategory == 'hour_of_day') {

									// item_value is integer
									var i = itemValue;
									itemValue = (fieldCategory == 'day_of_week') ? lang.weekdays[i - 1] : lang.hours[i];
								}
							}
						}
						else {

							// Aggregating field with numerical value
							operator = (expressionType == 'lt') ? langVar('lang_stats.global_filter.field_is_less_than') : langVar('lang_stats.global_filter.field_is_greater_than');
						}

						label = fieldLabel + " " + operator + " '" + itemValue + "'";
						break;


					case 'within_matches':

						label = !isNegated ? langVar('lang_stats.global_filter.field_within_field_matches_value') : langVar('lang_stats.global_filter.not_field_within_field_matches_value');

						// Compose Within/matches label
						var withinFieldLabel = _getFieldLabel(theFilterItem.withinFieldName);
						var matchesFieldLabel = _getFieldLabel(theFilterItem.matchesFieldName);

						// label += withinFieldLabel + " within " + matchesFieldLabel + " matches '" + itemValue + "'";

						label = label.replace(/__PARAM__1/, withinFieldLabel);
						label = label.replace(/__PARAM__2/, matchesFieldLabel);
						label = label.replace(/__PARAM__3/, "'" + itemValue + "'");

						break;
				}
			}
			else {
				// Nothing to modify in expressions
				label = theFilterItem.label;
			}

			return label;
		},

		_compareLabels: function(a, b) {

			// used by array sort()
			var labelA = a.label.toLowerCase();
			var labelB = b.label.toLowerCase();

			// Modify root items so that groups appear after items.
			if (a.isRootItem || a.isGroup) {
				labelA = a.isRootItem ? 'a' : 'b' + labelA;
				labelB = b.isRootItem ? 'a' : 'b' + labelB;
			}

			if (labelA < labelB) {
				return -1;
			}
			else if (labelA > labelB) {
				return 1;
			}
			else {
				return 0;
			}
		},

		_updateCheckboxState: function() {

			// Sets the isActive state of each checkbox (IE does not set the state upon checkbox creation)

			var filterItems = this.filterItems;

			// We don't need to cover filterItems in filterGroups because they don't have a checkbox!
			for (var i = 0, len = filterItems.length; i < len; i++) {

				var item = filterItems[i];
				var itemId = item.id;
				var isActive = item.isActive;
				var elementId = this.idPrefix + ':cb:' + itemId;

				util.setF(elementId, isActive);

				// Fix IE image display problem if filter group

				if (item.isGroup) {

					var imgId = this.idPrefix + ':img:' + itemId;
					var groupHasItems = (item.items.length > 0);

					setTimeout(function() {
						var img = util.getE(imgId);
						img.src = groupHasItems ? imgDb.gfCollapsed.src : imgDb.gfEmptyGroup.src;

					}, 10);
				}
			}
		},

		getNumOfActiveItems: function() {

			// Returns the number of active filter items

			var filterItems = this.filterItems;
			var numActiveItems = 0;

			for (var i = 0, len = filterItems.length; i < len; i++) {
				if (filterItems[i].isActive) {
					numActiveItems++;
				}
			}

			return numActiveItems;
		},

		_getParentFilterItem: function(subItemId) {

//			console.log('subItemId: ' + subItemId);

			// Returns the parent item of the given subItemId
			var filterItems = this.filterItems;

//			util.showObject(filterItems);

			var parentItem;

			for (var i = 0, len = filterItems.length; i < len; i++) {

				var item = filterItems[i];

				if (item.isGroup) {

					var items = item.items;

					for (var j = 0; j < items.length; j++) {

						if (items[j].id === subItemId) {

							// Return the parent item
							parentItem = item;
							break;
						}
					}
				}
			}

//			console.log('parentItem.id: ' + parentItem.id);
//			console.log('parentItem.isGroup: ' + parentItem.isGroup);

			return parentItem;
		}
	}


	// Return constructor
	return Filters;

}());