//
//
// emailUtil.js
//
// Used in Scheduler and Send Report By Email to handle recipient addresses
//
//
var emailUtil = {};
// AddressControl Class
emailUtil.AddressControl = function(containerElementId, addressElementWidth) {
	// alert('emailUtil.AddressControl');
	this.numberOfRows = 0; // The number of actuially build recipient rows
	this.addresses = []; // keeps the state upon validation and is also used to get the final email data
						// because they are written to addresses at validation time
	// Get a unique ID prefix
	var idPrefix = util.getUniqueElementId();
	this.idPrefix = idPrefix;
	this.addressDivId = idPrefix + ':addresses:div';
	this.addressTbodyId = idPrefix + ':addresses:tbody';
	this.isErrorIndication = false;
	this._build(containerElementId, idPrefix, addressElementWidth);
	YAHOO.util.Event.addListener(this.addressTbodyId, 'keyup', this._addressItemActicated, this);
}
emailUtil.AddressControl.prototype.init = function(addresses) {
	// addresses contains any default email addresses, 
	// i.e.: [{type:'to',  address:'abc@abc.com'}, {type:'cc', address:'first last <abc@abc.com>'}]
	var i;
	var idPrefix = this.idPrefix;
	if (addresses.length > 4) {
		// There are more than 4 default addresses, so we have to add additional rows
		var tbody = util.getE(this.addressTbodyId);
		var numberOfExistingRows = this.numberOfRows;
		var numberOfRequiredRows = addresses.length;
		for (i = numberOfExistingRows; i < numberOfRequiredRows; i++) {
			this._addRow(tbody, idPrefix, i);
			this._populateSelect(idPrefix, i);
		}
	}
	// Handle default addresses
	for (i = 0; i < addresses.length; i++) {
		var item =  addresses[i];
		util.setF(idPrefix + ':' + i + ':select', item.type);
		util.setF(idPrefix + ':' + i + ':input', item.address);
	}
}
emailUtil.AddressControl.prototype.validate = function() {
	// Validates the email addresses and handles error indication.
	// At least one valid email address must exist. If one of multiple
	// email addresses is not valid we show an error.
	// We validate and write email data to this.addresses.
	// this.addresses objects contains an isError property so that we know
	// if we must reset any color if an email address is indicated in Red
	// this.addresses is in the format: 
	// [{type:'to', name:'', email:'abc@abc.com', isError:false}, {type:'cc', name:'first last', email:'abc@abc.com',  isError:false}]
	var idPrefix = this.idPrefix;
	this.addresses = [];
	var isEmailAddress = false;
	var isAddressWithError = false;
	for (var i = 0; i < this.numberOfRows; i++) {
		var type = ''; // We only define a type if we find a valid email address
		var address = util.getF(idPrefix + ':' + i + ':input');
		var isError = false;
		if (address != '') {
			if (util.isEmailAddress(address)) {
				isEmailAddress = true;
				type = util.getF(idPrefix + ':' + i + ':select');
			}
			else {
				isAddressWithError = true;
				isError = true;
				var element = util.getE(idPrefix + ':' + i + ':input');
				element.style.color = 'Red';
			}
		}
		// Add object to this.addresses
		this.addresses[i] = {type:type, address:address, isError:isError};
	}
	if (isEmailAddress && !isAddressWithError) {
		return true;
	}
	else {
		// Handle error indication
		// alert('Error in recipient addresses');
		var msg = isAddressWithError ? langVar('lang_stats.general.invalid_email_address_in_recipients_msg') : langVar('lang_stats.general.no_recipient_address_message');
		util.updateT('email-address-grid:error', msg);
		util.showE('email-address-grid:error');
		this.isErrorIndication = true;
	}
	return false;
}
emailUtil.AddressControl.prototype._addressItemActicated = function(evt, self) {
	var element = evt.target || evt.srcElement;
	var elementId = element.id;
	// alert('_addressItemActicated() - id: ' + elementId);
	var dat = elementId.split(':');
	var rowIndex = parseInt(dat[1], 10);
	var elementType = dat[2];
	if (elementType == 'input') {
		// Handle error div on bottom
		if (self.isErrorIndication) {
			util.hideE('email-address-grid:error');
			self.isErrorIndication = false;
		}
		// Reset error of current row
		// Note, the row may not yet exist in this.addresses because it is only
		// added upon validation, so check for existence.
		var addresses = self.addresses;
		if (addresses[rowIndex]) {
			var address = addresses[rowIndex];
			if (address.isError) {
				element.style.color = 'Black';
				address.isError = false;
			}
		}
		// Add a new row if the current row is the last row
		var numberOfRows = self.numberOfRows;
		if (rowIndex == numberOfRows - 1) {
			var tbody = util.getE(self.addressTbodyId);
			self._addRow(tbody, self.idPrefix, numberOfRows);
			self._populateSelect( self.idPrefix, numberOfRows);
		}
	}
}
emailUtil.AddressControl.prototype.getAddresses = function() {
	// Returns all valid addresses
	// Valid addresses already exist in this.addresses since validation.
	// Valid addresses have a defined type, others not.
	var a = this.addresses;
	var b = [];
	for (var i = 0; i < a.length; i++) {
		var item = a[i];
		if (item.type != '') {
			b[b.length] = {type:item.type, address:item.address};
		}
	}
	return b;
}
emailUtil.AddressControl.prototype.reset = function() {
	// Resets the control by:
	// a.) setting the row numbers to 4
	// b.) removing any error indication
	// c.) resetting form fields to "to:" address and empty input field
	var i;
	if (this.isErrorIndication) {
		util.hideE('email-address-grid:error');
		this.isErrorIndication = false;
	}
	this.addresses = [];
	//
	// Reset rows to four rows
	//
	var numberOfRows = this.numberOfRows;
	if (this.numberOfRows > 4) {
		// Remove rows
		var tbody = util.getE(this.addressTbodyId);
		var rows = tbody.getElementsByTagName('tr');
		for (i = rows.length - 1; i > 3; i--) {
			var theRow = rows[i];
			tbody.removeChild(theRow);
		}
		this.numberOfRows = 4;
	}
	//
	// Reset select and input elements
	//
	var idPrefix = this.idPrefix;
	for (i = 0; i < 4; i++) {
		util.setF(idPrefix + ':' + i + ':select', 'to');
		var input = util.getE(idPrefix + ':' + i + ':input');
		input.value = '';
		input.style.color = 'Black';
	}
}
emailUtil.AddressControl.prototype.disable = function(isDisable) {
	// isDisable is optional
	var makeDisabled = (isDisable != null) ? isDisable : true;
	var numberOfRows = this.numberOfRows;
	var idPrefix = this.idPrefix;
	var a = [];
	for (var i = 0; i < numberOfRows; i++) {
		a[a.length] = idPrefix + ':' + i + ':select';
		a[a.length] = idPrefix + ':' + i + ':input';
	}
	util.disableE(a, makeDisabled);
}
emailUtil.AddressControl.prototype.freezeOverflow = function(isFreeze) {
	// Fixes a firefox bug, we require to set overflow to hidden when
	// a subform is displayed above the div element where overflow is set
	// to scroll
	var overflow = (isFreeze == null || isFreeze == true) ? 'hidden' : 'auto';
	var element = util.getE(this.addressDivId);
	element.style.overflow = overflow;
}
emailUtil.AddressControl.prototype._build = function(containerElementId, idPrefix, addressElementWidth) {
	// containerElementId is usually a td element
	// Builds the control
	// We create four address rows as default, starting from row 0, 1, 2, 3
	var container = util.getE(containerElementId);
	var addressDiv = util.createE('div', {id:this.addressDivId, className:'email-address-grid', width:addressElementWidth + 'px'});
	var table = util.createE('table', {className:'email-address-grid', cellSpacing:0});
	var tbody = util.createE('tbody', {id:this.addressTbodyId});
	var i;
	for (i = 0; i < 4; i++) {
		this._addRow(tbody, idPrefix, i);
	}
	// Add a div for error indication
	var errorDiv = util.createE('div', {id:'email-address-grid:error', className:'form-error'});
	util.chainE([container, [addressDiv, [table, tbody]], errorDiv]);
	// Handle select elements
	// Populate select only works after the select element is part of the dom
	for (i = 0; i < 4; i++) {
		this._populateSelect(idPrefix, i);
	}
}
emailUtil.AddressControl.prototype._addRow = function(tbody, idPrefix, rowIndex) {
	// Builds a row with select and input element
	// We create four address rows as default, starting from row 0
	var selectId = idPrefix + ':' + rowIndex + ':select';
	var inputId = idPrefix + ':' + rowIndex + ':input';
	var tr = util.createE('tr');
	var th = util.createE('th', {padding:'0px'});
	var td = util.createE('td', {padding:'0px'});
	var select = util.createE('select', {id:selectId});
	var input = util.createE('input', {id:inputId, type:'text', value:''});
	util.chainE(tbody, [tr, [th, select], [td, input]]);
	this.numberOfRows = rowIndex + 1;
}
emailUtil.AddressControl.prototype._populateSelect = function(idPrefix, rowIndex) {
	var list = [{name:'to', label:langVar('lang_stats.general.email_to') + ':'}, {name:'cc', label:langVar('lang_stats.general.email_cc') + ':'}, {name:'bcc', label:langVar('lang_stats.general.email_bcc') + ':'}];
	var selectId = idPrefix + ':' + rowIndex + ':select';
	util.populateSelect(selectId, list, 'name', 'label');
	util.setF(selectId, 'to');
}
//
// repFiltersUtil - used in reports and in scheduler
//
//
/* global
	globalFilter: false */
var repFiltersUtil = (function() {
	'use strict';
	function getIsValidFilterFields(queryFieldsDb, theFilterItem) {
		// This checks if the report field used in the filter item
		// actually exists in the queryFieldsDb.
		// Expressions are ignored, respectively assumed to be valid.
		// util.showObject(theFilterItem);
		var filterType = theFilterItem.filterType;
		var isValid = true;
//		console.log('getIsValidFilterFields() - filterType: ' + filterType);
		if (filterType === 'field') {
			var fieldName = theFilterItem.fieldName;
			isValid = typeof queryFieldsDb[util.h(fieldName)] !== 'undefined';
		}
//		console.log('getIsValidFilterFields() - isValid: ' + isValid);
		return isValid;
	}
	function getFilterItemAsString(theFilterItem) {
		// Returns a filterItem as string without isActive state.
		// This is used to check for existing filter items.
		var filterType = theFilterItem.filterType;
		var s = filterType;
		if (filterType == 'field') {
			s += theFilterItem.fieldName;
			s += theFilterItem.expressionType;
		}
		if (filterType == 'expression') {
			s += theFilterItem.label;
		}
		else {
			s += theFilterItem.isNegated;
		}
		s += theFilterItem.itemValue;
		return s;
	}
	function getFilterGroupId(theFilterGroup) {
		// Returns a filterGroup as string for comparison
//		util.showObject(theFilterGroup);
		var filterItems = theFilterGroup.items;
		var s = 'group_' + theFilterGroup.label;
		for (var i = 0, len = filterItems.length; i < len; i++) {
			s += getFilterItemAsString(filterItems[i]);
		}
		return s;
	}
	function getFilterItemDat(pathName, item, isItemInGroup) {
		var dat = '';
		var filterType = item.filterType;
		if (!isItemInGroup) {
			dat += pathName + '.is_active=' + item.isActive + '&';
		}
		dat += pathName + '.filter_type=' + filterType + '&';
		if (filterType != 'expression') {
			dat += pathName + '.is_negated=' + item.isNegated + '&';
            if (filterType == 'field') {
                dat += pathName + '.field_name=' + item.fieldName + '&';
                dat += pathName + '.expression_type=' + item.expressionType + '&';
            }
            else if (filterType == 'within_matches') {
                dat += pathName + '.within_field_name=' + encodeURIComponent(item.withinFieldName) + '&';
                dat += pathName + '.matches_field_name=' + encodeURIComponent(item.matchesFieldName) + '&';
            }
		}
		else {
			dat += pathName + '.label=' + encodeURIComponent(item.label) + '&';
		}
		dat += pathName + '.item_value=' + encodeURIComponent(item.itemValue) + '&';
		return dat;
	}
	function getGroupValuesByKey(filterItems, key, activeItemId) {
		// Creates an array of group labels or group names with all labels/names, except for activeItemId.
		// activeItemId is empty '' for new groups
		var a = [];
		for (var i = 0, len = filterItems.length; i < len; i++) {
			var item = filterItems[i];
			if (item.isGroup && item.id !== activeItemId) {
				a.push(item[key]);
			}
		}
		return a;
	}
	// Return global properties and methods
	return {
		getIsValidFilterFields: getIsValidFilterFields,
		getFilterItemAsString: getFilterItemAsString,
		getFilterGroupId: getFilterGroupId,
		getFilterItemDat: getFilterItemDat,
		getGroupValuesByKey: getGroupValuesByKey
	};
}());
//
// reportFiltersPanel.js
// This class handles report filter items and groups
// in reports and scheduler.
//
/* global
	globalFilter: false,
	filterItemEditor: false,
	filterGroupEditor: false,
	repFiltersUtil: false */
var ReportFiltersPanel = (function() {
	'use strict';
	var YE = YAHOO.util.Event,
//		YD = YAHOO.util.Dom,
		YL = YAHOO.lang;
	var Filters = function(conf) {
//		util.showObject(conf);
		this.containerId = conf.containerId;
		this.headerClassName = conf.headerClassName || 'filter-items-panel-header';
		this.bodyClassName = conf.bodyClassName || 'filter-items-panel-body';
		// The filters list container
		this.container = null;
		// Buttons definition in panel header.
		this.addItems = conf.addItems || false;
		this.moveItems = conf.moveItems || false;
		this.importItems = conf.importItems || false;
		this.newItemBtn = null;
		this.newGroupBtn = null;
		this.saveCheckedAsGroupBtn = null;
		this.importBtn = null;
//		this.sortBtn = null;
		this.selectAllBtn = null;
		this.deselectAllBtn = null;
		// Callback if isMoveToSaved
		this.moveToSavedCallback = this.moveItems ? conf.moveToSavedCallback : null;
		// Call back to open the Import window
		this.openImportPanelCallback = this.importItems ? conf.openImportPanelCallback : null;
		// Callback when filter items become selected/deselected,
		// this is i.e. used in the import panel to enable/disable
		// the import button.
		this.filterItemsSelectCallback = conf.filterItemsSelectCallback || null;
		this.idPrefix = '';
		this.idCount = 0;
		// filterItems is a new array which includes filterGroups.
		// filterItems and filterGroups are only separate nodes
		// on the server side.
		this.filterItems = [];
		this.isModified = false;
		// The active profileName, this is required in scheduler
		// actions when importing filters.
		this.profileName = '';
		// A reference to the active queryFieldsDb
		this.queryFieldsDb = [];
	}
	Filters.prototype = {
		initPanel: function() {
			// This creates filter buttons and adds event listeners.
			var idPrefix = util.getUniqueElementId();
			this.idPrefix = idPrefix;
			// Create the buttons
			var btns = '';
			var newFilterItemBtnId = idPrefix + ':new_filter_item';
			var newFilterGroupBtnId = idPrefix + ':new_filter_group';
			var saveCheckedAsGroupBtnId = idPrefix + ':save_checked_as_group';
			var importBtnId = idPrefix + ':import_items';
//			var sortBtnId = idPrefix + ':sort_items';
			var selectAllBtnId = idPrefix + ':select_all';
			var deselectAllBtnId = idPrefix + ':deselect_all';
			if (this.addItems) {
				btns = '<a id="' + newFilterItemBtnId + '" href="javascript:;" class="command-link-50">' +
					langVar('lang_stats.global_filter.new_item') + '</a> ' +
					'<a id="' + newFilterGroupBtnId + '" href="javascript:;" class="command-link-50">' +
					langVar('lang_stats.global_filter.new_group') + '</a> ' +
					'<a id="' + saveCheckedAsGroupBtnId + '" href="javascript:;" class="command-link-50">' +
					langVar('lang_stats.global_filter.save_checked_as_group') + '</a> ';
			}
			if (this.importItems) {
				btns += '<a id="' + importBtnId + '" href="javascript:;" class="command-link-50">' +
					langVar('lang_stats.btn.import') + '</a> ';
			}
			// Disabled sort because list appears always sorted anyway.
//			if (this.addItems) {
//
//				btns += '<a id="' + sortBtnId + '" href="javascript:;" class="command-link-50">' +
//					langVar('lang_stats.btn.sort') + '</a> ';
//			}
			// Select All, Deselect All
			btns += '<a id="' + selectAllBtnId + '" href="javascript:;" class="command-link-50">' +
				langVar('lang_stats.btn.select_all') + '</a> ' +
				'<a id="' + deselectAllBtnId + '" href="javascript:;" class="command-link-50">' +
				langVar('lang_stats.btn.deselect_all') + '</a>';
			// Create HTML
			var outerContainer = util.getE(this.containerId);
			var innerContainerId = idPrefix + ':' + this.containerId;
			var html = '<div class="' + this.headerClassName + '">' + btns + '</div>' +
				'<div class="' + this.bodyClassName + '">' +
				'<div id="' + innerContainerId + '"></div>';
				'</div>';
			outerContainer.innerHTML = html;
			// Get the container for the filters list
			this.container = util.getE(innerContainerId);
			// Create button objects
			var cmlOpt = {
				classNameEnabled: 'command-link-50',
				classNameDisabled: 'command-link-50-disabled'
			};
			if (this.addItems) {
				this.newItemBtn = new util.CommandLink(newFilterItemBtnId, this._handleMainEvents, true, cmlOpt, this);
				this.newGroupBtn = new util.CommandLink(newFilterGroupBtnId, this._handleMainEvents, true, cmlOpt, this);
				this.saveCheckedAsGroupBtn = new util.CommandLink(saveCheckedAsGroupBtnId, this._handleMainEvents, true, cmlOpt, this);
//				this.sortBtn = new util.CommandLink(sortBtnId, this._handleMainEvents, true, cmlOpt, this);
				// Init editor panels
				filterItemEditor.init();
				filterGroupEditor.init();
			}
			if (this.importItems) {
				this.importBtn = new util.CommandLink(importBtnId, this._handleMainEvents, true, cmlOpt, this);
			}
			this.selectAllBtn = new util.CommandLink(selectAllBtnId, this._handleMainEvents, true, cmlOpt, this);
			this.deselectAllBtn = new util.CommandLink(deselectAllBtnId, this._handleMainEvents, true, cmlOpt, this);
			// Handle filter list events
			YE.addListener(this.container, 'click', this._handleFilterListEvents, this);
		},
		initFilters: function(profileName, queryFieldsDb, filterItems, filterGroups) {
//			console.log('reportFiltersPanel.initFilters() - profileName: ' + profileName);
			// initFilters starts over with new filters.
			// This is required when initializing filters
			// in scheduler actions where profiles can be changed without
			// a page reload.
			if (this.idPrefix === '') {
				// Initialize the panel
				this.initPanel();
			}
			this.profileName = profileName;
			this.queryFieldsDb = queryFieldsDb;
			// Combine filterItems and filterGroups and add helper properties
			this.filterItems = this._getModifiedAndCombinedFilterItems(filterItems, filterGroups);
//			util.showObject(this.filterItems);
			// Update queryFieldsDb
			filterItemEditor.setProfileSpecificData(queryFieldsDb);
			// Build the filterItem list
			this._buildFiltersList();
			this.setButtonState();
		},
		resetIsModified: function() {
			this.isModified = false;
		},
		getIsModified: function() {
			return this.isModified;
		},
		_handleMainEvents: function(evt, self) {
			var element = evt.target || evt.srcElement;
			var elementId = element.id;
			var dat = elementId.split(':');
			var key = dat[1];
//			console.log('key: ' + key);
			switch (key) {
				case 'new_filter_item':
					self._createNewFilterItem(false, '');
					break;
				case 'new_filter_group':
					filterGroupEditor.open('', '', true, false, self);
					break;
				case 'save_checked_as_group':
					filterGroupEditor.open('', '', true, true, self);
					break;
				case 'import_items':
					// Callback import window
					self.openImportPanelCallback(
						self.profileName,
						self.queryFieldsDb,
						self.filterItems
					);
					break;
//				case 'sort_items':
//
//					self._buildFiltersList();
//					break;
				case 'select_all':
					self._selectAll(true);
					break;
				case 'deselect_all':
					self._selectAll(false);
					break;
			}
		},
		_handleFilterListEvents: function(evt, self) {
			var element = evt.target || evt.srcElement;
			var elementId = element.id;
//			console.log('element.nodeName: ' + element.nodeName);
//			console.log('element.id: ' + element.id);
			if (elementId !== '') {
				var dat = elementId.split(':');
				var key = dat[1];
				var itemId = dat[2];
				switch (key) {
					case 'cb':
						// Toggle filter state
						self._toggleFilterActiveState(element, itemId);
						break;
					case 'edit':
						// Edit filter item
						self._editFilterItem(itemId);
						break;
					case 'edit_subitem':
						self._editFilterSubItem(itemId);
						break;
					case 'delete':
						// Delete saved filter item
						self._deleteFilterItem(itemId);
						break;
					case 'delete_subitem':
						// Delete saved filter item
						self._deleteFilterSubItem(itemId);
						break;
					case 'add':
						// Add filter item in filter group
						self._createNewFilterItem(true, itemId);
						break;
					case 'img':
						// Expand collapse group
						self._expandCollapseGroup(itemId);
						break;
					case 'move':
						// Move to saved
						self._moveToSaved(itemId);
						break;
				}
			}
		},
		setButtonState: function() {
			// Updates the state of number of items and enables disables control buttons
			var isItems = this.filterItems.length > 0;
			var numActiveItems = this.getNumOfActiveItems();
			if (this.addItems) {
				this.newItemBtn.enable();
				this.newGroupBtn.enable();
				// Allow to save single items as new group because the item could already be a group
				// or a single item to which other items are added manually.
				this.saveCheckedAsGroupBtn.enable((numActiveItems > 0));
//				this.sortBtn.enable(isItems);
			}
			if (this.importItems) {
				// Only exists in scheduler actions.
				// Is disabled when no profile is selected.
				this.importBtn.enable();
			}
			this.deselectAllBtn.enable(isItems);
			this.selectAllBtn.enable(isItems);
			// Fire filterItemsSelectCallback,
			// this is the best place to call the listener.
			if (this.filterItemsSelectCallback) {
				this.filterItemsSelectCallback((numActiveItems > 0));
			}
		},
		disableAll: function() {
			// Disables all buttons
			// This is called from scheduler actions when no profile is selected.
			// In this case no filter items can be added.
			// Note, the panel may not yet exist when calling disableAll, create it now.
			if (this.idPrefix === '') {
				// Initialize the panel
				this.initPanel();
			}
			if (this.addItems) {
				this.newItemBtn.disable();
				this.newGroupBtn.disable();
				this.saveCheckedAsGroupBtn.disable();
//				this.sortBtn.disable();
			}
			if (this.importItems) {
				this.importBtn.disable();
			}
			this.deselectAllBtn.disable();
			this.selectAllBtn.disable();
		},
		_toggleFilterActiveState: function(element, itemId) {
			var item = this.filterItems[util.h(itemId)];
			item.isActive = element.checked;
			this.isModified = true;
			this.setButtonState();
		},
		_selectAll: function(checkItems) {
			// Set checkbox state in array, then apply setCheckboxes
			var filterItems = this.filterItems;
			for (var i = 0, len = filterItems.length; i < len; i++) {
				var item = filterItems[i];
				item.isActive = checkItems;
				util.setF(this.idPrefix + ':cb:' + item.id , checkItems);
			}
			this.isModified = true;
			this.setButtonState();
		},
		_createNewFilterItem: function(isItemInGroup, groupItemId) {
			var isActive = isItemInGroup ? false : true;
			var isNew = true;
			var id = this._getNewItemId();
			var item = {
				id: id,
				isNegated: false,
				filterType: 'field',
				fieldName: '',
				expressionType: '',
				itemValue: ''
			};
			filterItemEditor.open(item, isActive, isNew, isItemInGroup, groupItemId, this);
		},
		addNewItemViaMoveToSaved: function(newFilterItem) {
			// This inserts a new filter item which is called from another panel
			// via Move To Saved.
			// Filter groups can be ignored.
			// The newFilterItem is only inserted if it doesn't yet exist.
			// If the item already exists set only its isActive value to the
			// value of the newFilterItem.
			var filterItems = this.filterItems;
			var isExistingItem = false;
			var existingFilterItem;
			var newFilterItemString = repFiltersUtil.getFilterItemAsString(newFilterItem);
			for (var i = 0, len = filterItems.length; i < len; i++) {
				var existingFilterItem = filterItems[i];
				if (existingFilterItem.isRootItem) {
					var existingFilterItemString = repFiltersUtil.getFilterItemAsString(existingFilterItem);
					if (existingFilterItemString === newFilterItemString) {
						isExistingItem = true;
						break;
					}
				}
			}
			if (!isExistingItem) {
				// give the new item a new id
				newFilterItem.id = this._getNewItemId();
				// Insert newFilterItem
				filterItems.push(newFilterItem);
				// Update hash
				util.createHash(filterItems, 'id');
				this._buildFiltersList();
			}
			else {
				// The item already exists. Override isActive state only
				// if inserted item isActive.
				if (newFilterItem.isActive) {
					existingFilterItem.isActive = true;
					this._updateCheckboxState();
				}
			}
			this.isModified = true;
			this.setButtonState();
		},
		addNewItemsViaImport: function(newFilterItemsAndGroups) {
			// This adds/imports new filterItems and filterGroups
			// from another reportFiltersPanel object.
			var filterItems = this.filterItems;
			var clonedItem;
			for (var i = 0, len = newFilterItemsAndGroups.length; i < len; i++) {
				clonedItem = util.cloneObject(newFilterItemsAndGroups[i]);
				// Assign new item id.
				clonedItem.id = this._getNewItemId();
				// Reset isActive or not?
//				clonedItem.isActive = true;
				// Set isExpanded of groups to false;
				if (clonedItem.isGroup) {
					clonedItem.isExpanded = false;
					// Get new node name for this group
					var groupNameItems = repFiltersUtil.getGroupValuesByKey(filterItems, 'name', clonedItem.id);
					clonedItem.name = util.labelToUniqueNodeName(clonedItem.label, groupNameItems, 'filter_group');
					// Assign new id to subitems
					var subitems = clonedItem.items;
					for (var j = 0, numSubitems = subitems.length; j < numSubitems; j++) {
						subitems[j].id = this._getNewItemId();
					}
				}
				// Add new item to filterItems
				filterItems.push(clonedItem);
			}
			// Create hash to access filters by id
			util.createHash(filterItems, 'id');
			// Build the filterItem list
			this._buildFiltersList();
			this.setButtonState();
		},
		_editFilterItem: function(itemId) {
			// Edit root item or group
			var item = this.filterItems[util.h(itemId)];
//			util.showObject(item);
			if (item.isRootItem) {
				var isActive = item.isActive;
				var isNew = false;
				var groupItemId = '';
				var isItemInGroup = false;
				filterItemEditor.open(item, isActive, isNew, isItemInGroup, groupItemId, this);
			}
			else {
				// This is a group item
				filterGroupEditor.open(itemId, item.label, false, false, this);
			}
		},
		_editFilterSubItem: function(itemId) {
			// This is an item in a group.
			// Get the itemId of the group and subitem
//			util.showObject(parentItem);
			var isActive = false;
			var isNew = false;
			var isItemInGroup = true;
			var parentItem = this._getParentFilterItem(itemId);
			var groupItemId = parentItem.id;
			var subitems = parentItem.items;
			for (var i = 0; i < subitems.length; i++) {
				if (subitems[i].id == itemId) {
					var item = subitems[i];
				}
			}
			filterItemEditor.open(item, isActive, isNew, isItemInGroup, groupItemId, this);
		},
		_deleteFilterItem: function(itemId) {
			// Deletes a filter root item or group
			util.deleteArrayObject(this.filterItems, 'id', itemId);
			this.isModified = true;
			this._buildFiltersList();
			this.setButtonState();
		},
		_deleteFilterSubItem: function(itemId) {
			// Delete a filter item of a filter group
			var parentItem = this._getParentFilterItem(itemId);
			var subitems = parentItem.items;
			util.deleteArrayObject(subitems, 'id', itemId);
			this.isModified = true;
			this._buildFiltersList();
			this.setButtonState();
		},
		_moveToSaved: function(itemId) {
			// Clone item which becomes moved.
			var filterItems = this.filterItems;
			var filterItem = filterItems[util.h(itemId)];
			var clonedFilterItem = util.cloneObject(filterItem);
			// Delete item from this panel
			util.deleteArrayObject(filterItems, 'id', itemId);
			// Move the clonedFilterItem
			this.moveToSavedCallback(clonedFilterItem);
			this.isModified = true;
			this._buildFiltersList();
			this.setButtonState();
		},
		//
		// Handle Save Checked as New Group
		//
		saveCheckedAs: function() {
			var filterLabel = util.getF('save_checked_filters_as_name');
			// alert('Filter name');
			if (filterLabel != '') {
				// Check for duplicate filter name
				var isDuplicateLabel = false;
				var filterNameItems = [];
				for (var i = 0; i < savedFilterGroups.length; i++) {
					var existingFilterLabel = savedFilterGroups[i].label;
					if (existingFilterLabel.toLowerCase() == filterLabel.toLowerCase()) {
						isDuplicateLabel = true;
					}
					filterNameItems[i] = savedFilterGroups[i].name;
				}
				if (!isDuplicateLabel) {
					var filterName = util.labelToUniqueNodeName(filterLabel, filterNameItems, 'filter');
					saveCheckedFilters(filterName, filterLabel);
				}
				else {
					if (confirm(langVar('lang_stats.global_filter.confirm_existing_filter_replacement_message'))) {
						// TODO, Delete existing filter?
						alert('Replace existing filter');
						return false;
					}
				}
			}
			else {
				alert(langVar('lang_stats.global_filter.missing_filter_name_message'));
			}
		},
		saveCheckedAsNewGroupProcessing: function(groupName, groupLabel) {
			// Create new group from checked saved filter items and filter groups.
			var filterItems = this.filterItems;
			var lookup = {};
			var newItems = [];
			for (var i = 0, len = filterItems.length; i < len; i++) {
				var item = filterItems[i];
				if (item.isActive) {
					if (item.isRootItem) {
						this._addCheckedFilterItem(lookup, newItems, item);
					}
					else {
						// Add subitems of filter group
						var subitems = item.items;
						for (var j = 0, numSubitems = subitems.length; j < numSubitems; j++) {
							this._addCheckedFilterItem(lookup, newItems, subitems[j]);
						}
					}
				}
			}
			// Setup new filter group
			var newGroupId = this._getNewItemId();
			var newGroup = {
				id: newGroupId,
				isRootItem: false,
				isGroup: true,
				isExpanded: false,
				name: groupName,
				label: groupLabel,
				isActive: false,
				items: newItems
			};
			// Add newGroup to savedFilterGroups
			filterItems.push(newGroup);
			// Sort the new Items
			filterItems.sort(this._compareLabels);
			// update hash
			util.createHash(filterItems, 'id');
			this.isModified = true;
			this._buildFiltersList();
			this.setButtonState();
		},
		_addCheckedFilterItem: function(lookup, newItems, filterItem) {
			// Adds filterItem to newItems if it doesn't yet exist.
			var filterId = repFiltersUtil.getFilterItemAsString(filterItem);
			if (!lookup.hasOwnProperty(filterId)) {
				var clonedFilterItem = util.cloneObject(filterItem);
				// Give the item a new id
				clonedFilterItem.id = this._getNewItemId();
				newItems.push(clonedFilterItem);
				lookup[filterId] = true;
			}
		},
		//
		// Create filter item rows
		//
		_buildFiltersList: function() {
			// Build the filter and filter group rows.
			// This is also used upon sorting the filter
			// rows.
			var item;
			var filterItems = this.filterItems;
//			util.showObject(filterItems, '_buildFiltersList()');
			var filtersHtml = '<table><tbody>';
			// Sort the items
			filterItems.sort(this._compareLabels);
			// Create the rows
			for (var i = 0, len = filterItems.length; i < len; i++) {
				item = filterItems[i];
				if (item.isRootItem) {
					filtersHtml += this._getFilterRow(item.id, item.label);
				}
				else {
					// This is a group item
					var subitems = item.items;
					var numSubitems = subitems.length;
					var groupHasItems = (numSubitems > 0);
					var isExpanded = item.isExpanded;
					filtersHtml += this._getFilterGroupRow(item.id, item.label, groupHasItems, isExpanded);
					// Sort subitems
					subitems.sort(this._compareLabels);
					for (var j = 0; j < numSubitems; j++) {
						var subItem = subitems[j];
						filtersHtml += this._getSubFilterRow(subItem.id, subItem.label, isExpanded);
					}
				}
			}
			filtersHtml += '</tbody></table>';
			this.container.innerHTML = filtersHtml;
			// Update checkbox state
			this._updateCheckboxState();
		},
		_getFilterRow: function(itemId, itemLabel) {
			var idPrefix = this.idPrefix;
			var row = '<tr id="' + idPrefix + ':row:' + itemId + '">';
			// Add empty cell for filter group icons column
			row += '<td style="width:14px"></td>';
			// Add checkbox cell
			var checkboxId = idPrefix + ':cb:' + itemId;
			var labelId = idPrefix + ':label:' + itemId;
			row += '<td style="width:14px">';
			row += '<input id="' + checkboxId + '" type="checkbox" />';
			row += '</td>';
			// Add label cell
			row += '<td>';
			row += '<label id="' + labelId + '" for="' + checkboxId + '">' +
					YL.escapeHTML(itemLabel)  + '</label>';
			row += '</td>';
			// Add buttons
			row += '<td class="filter-item-controls">';
			if (this.addItems || this.moveItems) {
				if (this.addItems) {
					// Edit
					row += this._getButton(idPrefix + ':edit:' + itemId, langVar('lang_stats.btn.edit'));
				}
				else {
					// Move to saved
					row += this._getButton(idPrefix + ':move:' + itemId, langVar('lang_stats.global_filter.move_to_saved'));
				}
				// Delete
				row += this._getButton(idPrefix + ':delete:' + itemId, langVar('lang_stats.btn.delete'));
			}
			row += '</td>';
			row += '</tr>';
			return row;
		},
		_getSubFilterRow: function(itemId, itemLabel, isExpanded) {
			// This returns a filter row of a group
			var idPrefix = this.idPrefix;
			var row = '<tr id="' + idPrefix + ':row:' + itemId + '"';
			if (!isExpanded) {row += ' style="display:none"';}
			row += '>';
			// Add empty cell for filter group icons column
			row += '<td style="width:14px"></td>';
			// Add empty cell for checkbox
			row += '<td style="width:14px"></td>';
			// Add label cell
			row += '<td>';
			row += '<span id="' + idPrefix + ':label:' + itemId + '">' + YL.escapeHTML(itemLabel)  + '</span>';
			row += '</td>';
			// Add buttons
			row += '<td class="filter-item-controls">';
			if (this.addItems) {
				// Edit
				row += this._getButton(idPrefix + ':edit_subitem:' + itemId, langVar('lang_stats.btn.edit'));
				// Delete
				row += this._getButton(idPrefix + ':delete_subitem:' + itemId, langVar('lang_stats.btn.delete'));
			}
			row += '</td>';
			row += '</tr>';
			return row;
		},
		_getFilterGroupRow: function(itemId, itemLabel, groupHasItems, isExpanded) {
			var imgSrc = '';
			if (!groupHasItems) {
				imgSrc = imgDb.gfEmptyGroup.src;
			}
			else if (!isExpanded) {
				imgSrc = imgDb.gfCollapsed.src;
			}
			else {
				// Show expanded icon
				imgSrc = imgDb.gfExpanded.src;
			}
	//		console.log('imgSrc: ' + imgSrc);
			var idPrefix =  this.idPrefix;
			var row = '<tr id="' + idPrefix + ':row:' + itemId + '">';
			// Add group icon
			row += '<td style="width:14px">';
			row += '<img id="' + idPrefix + ':img:' + itemId + '" src="' + imgSrc + '" width="31" height="15" alt="" />';
			row += '</td>';
			// Add checkbox cell
			var checkboxId = idPrefix + ':cb:' + itemId;
			var labelId = idPrefix + ':label:' + itemId;
			row += '<td style="width:14px">';
			row += '<input id="' + checkboxId + '" type="checkbox" />';
			row += '</td>';
			// Add label cell
			row += '<td>';
			row += '<label id="' + labelId + '" for="' + checkboxId + '">' +
					YL.escapeHTML(itemLabel)  + '</label>';
			row += '</td>';
			// Add buttons
			row += '<td class="filter-item-controls">';
			if (this.addItems) {
				// Add
				row += this._getButton(idPrefix + ':add:' + itemId, langVar('lang_stats.global_filter.add_new_item'));
				// Edit
				row += this._getButton(idPrefix + ':edit:' + itemId, langVar('lang_stats.btn.edit'));
				// Delete
				row += this._getButton(idPrefix + ':delete:' + itemId, langVar('lang_stats.btn.delete'));
			}
			row += '</td>';
			row += '</tr>';
			return row;
		},
		_getButton: function(id, btnLabel) {
			return ' <a id="' + id + '" href="javascript:;">' + btnLabel  + '</a>';
		},
		saveFilterItem: function(item, isNew, isItemInGroup, groupItemId) {
			// Invoked from filterItem.js!
			// If not isNew then item replaces the existing item (in this case item contains the original itemId!)
			var rebuildFiltersList = false;
			// Get itemId. All default properties already exist in item
			var itemId = item.id;
			// Get the label
			item.label = this._getFilterItemLabel(item);
			// Get the array where we insert or update the item.
			var filterItems = this.filterItems;
			var refItems;
			var parentItem;
			if (!isItemInGroup) {
				refItems = filterItems;
			}
			else {
				parentItem = filterItems[util.h(groupItemId)];
				refItems = parentItem.items;
			}
			// If not a new item then replace existing item
			if (!isNew) {
				// Get the index of the item to be replaced
				var itemIndex = util.getArrayObjectIndex(refItems, 'id', itemId);
				refItems.splice(itemIndex, 1, item);
				var labelId = this.idPrefix + ':label:' + itemId;
				util.updateT(labelId, item.label);
			}
			else {
				// This is a new item. Insert the new item as first item in array
				refItems.splice(0, 0, item);
				if (!isItemInGroup) {
					// Update hash
					util.createHash(filterItems, 'id');
				}
				else {
					// Set isExpanded of groupItem in case that this is the first item added.
					parentItem.isExpanded = true;
				}
				rebuildFiltersList = true;
			}
			this.isModified = true;
			if (rebuildFiltersList) {
				this._buildFiltersList();
				this.setButtonState();
			}
		},
		saveFilterGroup: function(itemId, groupName, groupLabel, isNew) {
			var filterItem;
			var filterItems = this.filterItems;
			if (!isNew) {
				filterItem = filterItems[util.h(itemId)];
				filterItem.name = groupName;
				filterItem.label = groupLabel;
				// Update label
				util.updateT(this.idPrefix + ':label:' + itemId, groupLabel);
			}
			else {
				var newGroupId = this._getNewItemId();
				filterItem = {
					id: newGroupId,
					name: groupName,
					label: groupLabel,
					isRootItem: false,
					isGroup: true,
					isActive: false,
					isExpanded: false,
					items: []
				};
				filterItems.push(filterItem);
				// update hash
				util.createHash(filterItems, 'id');
				this._buildFiltersList();
				this.setButtonState();
			}
			this.isModified = true;
		},
		_expandCollapseGroup: function(itemId) {
			var item = this.filterItems[util.h(itemId)];
			var isExpanded = item.isExpanded;
			var subitems = item.items;
			var numSubitems = subitems.length;
			// Expand/collapse group only if there are items
			if (numSubitems > 0) {
				var img = util.getE(this.idPrefix + ':img:' + itemId);
				img.src = isExpanded ? imgDb.gfCollapsed.src : imgDb.gfExpanded.src;
				for (var i = 0; i < numSubitems; i++) {
					// util.showObject(items[i]);
					var rowId = this.idPrefix + ':row:' + subitems[i].id;
					util.showE(rowId, !isExpanded);
				}
				item.isExpanded = !isExpanded;
			}
		},
		getFilterItems: function() {
			// Returns a new array with filterItems
			return this.getFilterOrGroupItems('isRootItem');
		},
		getFilterGroups: function() {
			// Returns a new array with filterGroups
			return this.getFilterOrGroupItems('isGroup');
		},
		getFilterOrGroupItems: function(key) {
			// Returns cloned items of the given key
			var filterItems = this.filterItems;
			var item;
			var clonedItem;
			var clonedItems = [];
			for (var i = 0, len = filterItems.length; i < len; i++) {
				var item = filterItems[i];
				if (item[key]) {
					clonedItem = util.cloneObject(item);
					clonedItems.push(clonedItem);
				}
			}
			return clonedItems;
		},
		getActiveItems: function() {
			// Returns cloned filterItems and filterGroups if isActive.
			var filterItems = this.filterItems;
			var item;
			var clonedItem;
			var clonedItems = [];
			for (var i = 0, len = filterItems.length; i < len; i++) {
				var item = filterItems[i];
				if (item.isActive) {
					clonedItem = util.cloneObject(item);
					clonedItems.push(clonedItem);
				}
			}
			return clonedItems;
		},
		//
		// Utilities
		//
		_getModifiedAndCombinedFilterItems: function(filterItems, filterGroups) {
			// This merges filterItems and filterGroups in a new array and adds
			// additional properties for list handling.
			var newItems = [];
			var newItem = {};
			// Add modified filterItems to newItems
			for (var i = 0, filterItemsLen = filterItems.length; i < filterItemsLen; i++) {
				// Create new item
				newItem = util.cloneObject(filterItems[i]);
				newItem.id = this._getNewItemId();
				newItem.label = this._getFilterItemLabel(newItem);
				newItem.isRootItem = true;
				newItem.isGroup = false;
				newItems.push(newItem);
			}
			// Add modified filterGroups to newItems
			for (var j = 0, filterGroupsLen = filterGroups.length; j < filterGroupsLen; j++) {
				// Create new item
				// Groups already have a label. Add id and isExpanded to keep expanded state.
				newItem = util.cloneObject(filterGroups[j]);
				newItem.id =  this._getNewItemId();
				newItem.isExpanded = false;
				newItem.isRootItem = false;
				newItem.isGroup = true;
				// Handle subitems
				var subitems = newItem.items;
				var subitem;
				for (var k = 0, subitemsLen = subitems.length; k < subitemsLen; k++) {
					subitem = subitems[k];
					subitem.id = this._getNewItemId();
					subitem.label = this._getFilterItemLabel(subitem);
					subitem.isRootItem = false;
					subitem.isGroup = false;
				}
				newItems.push(newItem);
			}
//			util.showObject(newItems, '_getModifiedAndCombinedFilterItems()');
			// Create hash to access filters by id
			util.createHash(newItems, 'id');
			return newItems;
		},
		_getNewItemId: function() {
			this.idCount += 1;
			return 'i' + this.idCount;
		},
		_getFieldLabel: function(fieldName) {
			var queryFieldItem = this.queryFieldsDb[util.h(fieldName)];
			return queryFieldItem.label;
		},
		_getFilterItemLabel: function(theFilterItem) {
			var filterType = theFilterItem.filterType;
			// We ignore the label if filterType is expression!
			var label = '';
			if (filterType !== 'expression') {
				var isNegated = theFilterItem.isNegated;
				var itemValue = theFilterItem.itemValue;
				switch (filterType) {
					case 'field':
						var fieldName = theFilterItem.fieldName;
						var expressionType = theFilterItem.expressionType;
						var queryFieldItem = this.queryFieldsDb[util.h(fieldName)];
						var fieldLabel = queryFieldItem.label;
						var isAggregatingField = queryFieldItem.isAggregatingField;
						var operator = '';
						// alert('expressionType: ' + expressionType);
						if (!isAggregatingField) {
							if (expressionType == 'regexp') {
								operator = !isNegated ? langVar('lang_stats.global_filter.field_matches_regular_expression') : langVar('lang_stats.global_filter.field_not_matches_regular_expression');
							}
							else if (expressionType == 'wildcard') {
								operator = !isNegated ? langVar('lang_stats.global_filter.field_matches_wildcard') : langVar('lang_stats.global_filter.field_not_matches_wildcard');
							}
							else {
								operator = !isNegated ? langVar('lang_stats.global_filter.field_is') : langVar('lang_stats.global_filter.field_is_not');
								var fieldCategory = queryFieldItem.category;
								if (fieldCategory == 'day_of_week' || fieldCategory == 'hour_of_day') {
									// item_value is integer
									var i = itemValue;
									itemValue = (fieldCategory == 'day_of_week') ? lang.weekdays[i - 1] : lang.hours[i];
								}
							}
						}
						else {
							// Aggregating field with numerical value
							operator = (expressionType == 'lt') ? langVar('lang_stats.global_filter.field_is_less_than') : langVar('lang_stats.global_filter.field_is_greater_than');
						}
						label = fieldLabel + " " + operator + " '" + itemValue + "'";
						break;
					case 'within_matches':
						label = !isNegated ? langVar('lang_stats.global_filter.field_within_field_matches_value') : langVar('lang_stats.global_filter.not_field_within_field_matches_value');
						// Compose Within/matches label
						var withinFieldLabel = this._getFieldLabel(theFilterItem.withinFieldName);
						var matchesFieldLabel = this._getFieldLabel(theFilterItem.matchesFieldName);
						// label += withinFieldLabel + " within " + matchesFieldLabel + " matches '" + itemValue + "'";
						label = label.replace(/__PARAM__1/, withinFieldLabel);
						label = label.replace(/__PARAM__2/, matchesFieldLabel);
						label = label.replace(/__PARAM__3/, "'" + itemValue + "'");
						break;
				}
			}
			else {
				// Nothing to modify in expressions
				label = theFilterItem.label;
			}
			return label;
		},
		_compareLabels: function(a, b) {
			// used by array sort()
			var labelA = a.label.toLowerCase();
			var labelB = b.label.toLowerCase();
			// Modify root items so that groups appear after items.
			if (a.isRootItem || a.isGroup) {
				labelA = a.isRootItem ? 'a' : 'b' + labelA;
				labelB = b.isRootItem ? 'a' : 'b' + labelB;
			}
			if (labelA < labelB) {
				return -1;
			}
			else if (labelA > labelB) {
				return 1;
			}
			else {
				return 0;
			}
		},
		_updateCheckboxState: function() {
			// Sets the isActive state of each checkbox (IE does not set the state upon checkbox creation)
			var filterItems = this.filterItems;
			// We don't need to cover filterItems in filterGroups because they don't have a checkbox!
			for (var i = 0, len = filterItems.length; i < len; i++) {
				var item = filterItems[i];
				var itemId = item.id;
				var isActive = item.isActive;
				var elementId = this.idPrefix + ':cb:' + itemId;
				util.setF(elementId, isActive);
				// Fix IE image display problem if filter group
				if (item.isGroup) {
					var imgId = this.idPrefix + ':img:' + itemId;
					var groupHasItems = (item.items.length > 0);
					setTimeout(function() {
						var img = util.getE(imgId);
						img.src = groupHasItems ? imgDb.gfCollapsed.src : imgDb.gfEmptyGroup.src;
					}, 10);
				}
			}
		},
		getNumOfActiveItems: function() {
			// Returns the number of active filter items
			var filterItems = this.filterItems;
			var numActiveItems = 0;
			for (var i = 0, len = filterItems.length; i < len; i++) {
				if (filterItems[i].isActive) {
					numActiveItems++;
				}
			}
			return numActiveItems;
		},
		_getParentFilterItem: function(subItemId) {
//			console.log('subItemId: ' + subItemId);
			// Returns the parent item of the given subItemId
			var filterItems = this.filterItems;
//			util.showObject(filterItems);
			var parentItem;
			for (var i = 0, len = filterItems.length; i < len; i++) {
				var item = filterItems[i];
				if (item.isGroup) {
					var items = item.items;
					for (var j = 0; j < items.length; j++) {
						if (items[j].id === subItemId) {
							// Return the parent item
							parentItem = item;
							break;
						}
					}
				}
			}
//			console.log('parentItem.id: ' + parentItem.id);
//			console.log('parentItem.isGroup: ' + parentItem.isGroup);
			return parentItem;
		}
	}
	// Return constructor
	return Filters;
}());
//
// filterItem - global filter items editor
// used in reports and in scheduler.
//
var filterItemEditor = (function() {
	'use strict';
  	var YE = YAHOO.util.Event,
		YD = YAHOO.util.Dom,
		panel = null,
		validator = null,
		queryFieldsDb = null,
		// active filter item properties
		id = '',
		isActive = false,
		isNew = false,
		isItemInGroup = false,
		groupItemId = '',
		// The callee object
		calleeObj = null,
		// activeOperatorListType is text or numerical
		activeOperatorListType = '',
		textFieldItemOperators = [
			{
			name:'item_name',
			label:langVar('lang_stats.global_filter.is_item_name')
			},
			{
			name:'wildcard',
			label:langVar('lang_stats.global_filter.is_wildcard_expression')
			},
			{
			name:'regexp',
			label:langVar('lang_stats.global_filter.is_regular_expression')
			},
			{
			name:'not_item_name',
			label:langVar('lang_stats.global_filter.not_item_name')
			},
			{
			name:'not_wildcard',
			label:langVar('lang_stats.global_filter.not_wildcard_expression')
			},
			{
			name:'not_regexp',
			label:langVar('lang_stats.global_filter.not_regular_expression')
			}
		],
		numericalFieldItemOperators = [
			{
			name:'lt',
			label:langVar('lang_stats.global_filter.is_less_than')
			},
			{
			name:'gt',
			label:langVar('lang_stats.global_filter.is_greater_than')
			}
		];
	function init() {
		var panelObj = {
			panelId:"filter_item:panel",
			panelClassName: 'panel-50',
			panelHeaderLabel: '-',
			left: 70,
			top: 80,
			zIndex: 40,
			isCover: true,
			isSticky: true,
			closeEvent: close
		};
		panel = new util.Panel3(panelObj);
		validator = new util.Validator();
		// Populate default operator list for text field items
		updateFieldItemOperatorList('text');
		var label = '';
		// Populate day of week list
		var weekdays = [{
			name:'',
			label:langVar('lang_stats.global_filter.select_day_of_week')
			}];
		for (var i = 1; i < 8; i++) {
			label = lang.weekdays[i - 1];
			weekdays[weekdays.length] = {
				name: i,
				label: label
			};
		}
		util.populateSelect('filter_item:day_of_week_list', weekdays, 'name', 'label');
		// Populate hour of day list
		var hours = [{
			name:'',
			label:langVar('lang_stats.global_filter.select_hour')
			}];
		for (var j = 0; j < 24; j++) {
			label = lang.hours[j];
			hours[hours.length] = {
				name: j,
				label: label
			};
		}
		util.populateSelect('filter_item:hour_of_day_list', hours, 'name', 'label');
		YE.addListener('filter_item:read_about_within_matches_btn', 'click', toggleInfoPanel);
		YE.addListener('filter_item:filter_item_type_list', 'change', typeActor);
		YE.addListener('filter_item:field_list', 'change', fieldActor);
		YE.addListener('filter_item:field_item_operator_list', 'change', operatorActor);
		YE.addListener('filter_item:ok_btn', 'click', saveItem);
		YE.addListener('filter_item:cancel_btn', 'click', close);
	}
	function setProfileSpecificData(_queryFieldsDb) {
		// This sets profile related data. This is required in scheduler
		// actions where profiles can be changed at any time.
		queryFieldsDb = _queryFieldsDb;
		// Create hash
//		util.createHash(queryFieldsDb);
//		util.showObject(queryFieldsDb);
		// Populate field item list
		// Note, we don't allow date_time fields in global filter! Date time is set via df only!
		var filterFieldsDb = [{
			name:'',
			label:langVar('lang_stats.global_filter.select_field')
		}];
		var nonAggregatingFilterFieldsDb = [{
			name: '',
			label:langVar('lang_stats.global_filter.select_field')
		}];
		for (var i = 0; i < queryFieldsDb.length; i++) {
			var queryField = queryFieldsDb[i];
			var isValidField = false;
			if (queryField.category !== 'date_time' && queryField.hasDbField) {
				var isAggregatingField = queryField.isAggregatingField;
				isValidField = (!isAggregatingField || (isAggregatingField && (queryField.aggregationMethod !== 'unique')));
			}
			if (isValidField) {
				filterFieldsDb[filterFieldsDb.length] = {
					name:queryField.name,
					label:queryField.label
				};
				if (!isAggregatingField) {
					nonAggregatingFilterFieldsDb[nonAggregatingFilterFieldsDb.length] = {
						name: queryField.name,
						label:queryField.label
					};
				}
			}
		}
		util.populateSelect('filter_item:field_list', filterFieldsDb, 'name', 'label');
		util.populateSelect('filter_item:within_matches:within_field', nonAggregatingFilterFieldsDb, 'name', 'label');
		util.populateSelect('filter_item:within_matches:matches_field', nonAggregatingFilterFieldsDb, 'name', 'label');
	}
	function open(item, _isActive, _isNew, _isItemInGroup, _groupItemId, _calleeObj) {
//		util.showObject(item);
		id = item.id;
		isActive = _isActive;
		isNew = _isNew;
		isItemInGroup = _isItemInGroup;
		groupItemId = _groupItemId;
		calleeObj = _calleeObj;
		updateForm(item);
		var panelHeaderLabel = isNew ? langVar('lang_stats.global_filter.new_item') : langVar('lang_stats.global_filter.edit_item');
		var viewportWidth = YD.getViewportWidth();
		// If large display
		if (viewportWidth > 1000) {
			// Get position for filterItemEditor
			var region = YD.getRegion(calleeObj.container);
			var left = region.left < 120 ? region.left : region.left - 20;
			var top = region.top - 45;
			panel.open({
				label: panelHeaderLabel,
				left: left,
				top: top
			});
		}
		else {
			// If small display
			panel.prePositionAtCenter();
			panel.open({label: panelHeaderLabel});
		}
	}
	function close() {
		util.hideE('filter_item:info_panel');
		validator.reset();
		panel.close();
//		globalFilter.enableForm();
	}
	function updateFieldItemOperatorList(operatorListType) {
		// operatorListType is text | numerical
		if (activeOperatorListType !== operatorListType) {
			// Update operator list
			var theListItems = (operatorListType === 'text') ? textFieldItemOperators : numericalFieldItemOperators;
			util.populateSelect('filter_item:field_item_operator_list', theListItems, 'name', 'label');
			activeOperatorListType = operatorListType;
		}
	}
	function getFieldItemOperator() {
		// returns an object
		// o.isExpression = true | false
		// o.isNegated = true | false
		// o.expressionType = '' | 'wildcard' | 'regexp' | lt | gt
		// lt (less than) and gt (greater than) have been added for numerical fields
		var o = {};
		var operatorValue = util.getF('filter_item:field_item_operator_list');
		var isExpression = false;
		var isNegated = false;
		var expressionType = '';
		if (operatorValue.indexOf('wildcard') !== -1 || operatorValue.indexOf('regexp') !== -1) {
			isExpression = true;
			expressionType = (operatorValue.indexOf('wildcard') !== -1) ? 'wildcard' : 'regexp';
		}
		else if (operatorValue === 'lt' || operatorValue === 'gt') {
			expressionType = operatorValue;
		}
		if (operatorValue.indexOf('not_') !== -1) {
			isNegated = true;
		}
		o.isExpression = isExpression;
		o.isNegated = isNegated;
		o.expressionType = expressionType;
		return o;
	}
	//
	// filterItemType
	//
	function typeActor() {
		var filterType = util.getF('filter_item:filter_item_type_list');
		setType(filterType);
	}
	function setType(filterType) {
		util.hideE(['filter_item:form:field',
			// 'filter_item:form:session_start',
			'filter_item:form:within_matches',
			'filter_item:form:expression',
			'filter_item:read_about_within_matches_btn'
		]);
		util.showE('filter_item:form:' + filterType);
		if (filterType === 'within_matches') {
			util.showE('filter_item:read_about_within_matches_btn');
		}
	}
	//
	// filterItem - field, operator and item value handling
	//
	function fieldActor() {
		// Invoked upon field change
		var fieldName = util.getF('filter_item:field_list');
		if (fieldName !== '') {
			// Update the operator list (text or numerical)
			var queryField = queryFieldsDb[util.h(fieldName)];
			var isAggregatingField = queryField.isAggregatingField;
			var operatorListType = isAggregatingField ? 'numerical' : 'text';
			updateFieldItemOperatorList(operatorListType);
			var defaultOperator = isAggregatingField ? 'lt' : 'item_name';
			util.setF('filter_item:field_item_operator_list', defaultOperator);
		}
		// Check/update the item value section
		setFieldItemValueSection();
	}
	function operatorActor() {
		// Check/update the item value section
		setFieldItemValueSection();
	}
	function setFieldItemValueSection() {
		// This sets the item value section depending on the
		// selected field and the selected operator.
		util.hideE(['filter_item:field_item_value_row', 'filter_item:field_item_day_of_week_row', 'filter_item:field_item_hour_of_day_row']);
		var fieldName = util.getF('filter_item:field_list');
		if (fieldName !== '') {
			var queryField = queryFieldsDb[util.h(fieldName)];
			var fieldCategory = queryField.category;
			var isAggregatingField = queryField.isAggregatingField;
			var o = getFieldItemOperator();
			var isExpression = o.isExpression; // wildcard or regular expression
			if ( !isAggregatingField && !isExpression && (fieldCategory === 'day_of_week' || fieldCategory === 'hour_of_day')) {
				if (fieldCategory === 'day_of_week') {
					util.showE('filter_item:field_item_day_of_week_row');
				}
				else {
					util.showE('filter_item:field_item_hour_of_day_row');
				}
			}
			else {
				util.showE('filter_item:field_item_value_row');
			}
		}
		else {
			// No field selected, show field_item_value_row
			util.showE('filter_item:field_item_value_row');
		}
	}
	function toggleInfoPanel(evt) {
		var anchorElement = evt.target || evt.srcElement;
		var infoPanel = util.getE('filter_item:info_panel');
		if (infoPanel.style.display !== 'block') {
			// Handle top position
			var YD = YAHOO.util.Dom;
			var anchorRegion = YD.getRegion(anchorElement);
			// util.showObject(elementRegion);
			infoPanel.style.top = anchorRegion.bottom + 8 + 'px';
			// Handle left position and width
			var parentRegion = YD.getRegion('filter_item:panel');
			var parentWidth = parentRegion.width;
			var parentLeft = parentRegion.left;
			var offset = 10;
			// alert('parentWidth: ' + parentWidth);
			infoPanel.style.left = parentLeft + offset + 'px';
			infoPanel.style.width = parentWidth - (2 * offset) + 'px';
			infoPanel.style.zIndex = 50;
			// Display the panel
			infoPanel.style.display = 'block';
		}
		else {
		   infoPanel.style.display = 'none';
		}
		anchorElement.blur();
	}
	//
	// updateForm
	//
	function updateForm(item) {
//		util.showObject(queryFieldsDb);
		// Reset form
		util.resetF('filter_item:form');
		// The itemType is '' for unsaved filters and 'filter_item' for saved filters!
		var filterType = item.filterType;
		util.setF('filter_item:filter_item_type_list', filterType);
		switch (filterType) {
			case 'field':
				var fieldName = item.fieldName;
				util.setF('filter_item:field_list', fieldName);
				//
				// update operator list if we edit an existing field
				//
				if (fieldName !== '') {
					var queryField = queryFieldsDb[util.h(fieldName)];
					var isAggregatingField = queryField.isAggregatingField;
					var operatorListType = isAggregatingField ? 'numerical' : 'text';
					updateFieldItemOperatorList(operatorListType);
				}
				//
				// set operator value
				//
				var operatorValue;
				var expressionType = item.expressionType;
				if (expressionType === '') {
					operatorValue = 'item_name';
				}
				else {
					operatorValue = expressionType;
				}
				if (item.isNegated) {
					operatorValue = 'not_' + operatorValue;
				}
				util.setF('filter_item:field_item_operator_list', operatorValue);
				//
				// set item value
				//
				var valueElementId = 'filter_item:field_item_value';
				if (item.expressionType === '') {
					var fieldCategory = (fieldName !== '') ? queryFieldsDb[util.h(fieldName)].category : '';
					if (fieldCategory === 'day_of_week') {
						valueElementId = 'filter_item:day_of_week_list';
					}
					else if (fieldCategory === 'hour_of_day') {
						valueElementId = 'filter_item:hour_of_day_list';
					}
				}
				util.setF(valueElementId, item.itemValue);
				break;
			case 'within_matches':
				if (item.isNegated) {
					util.setF('filter_item:within_matches:use_not_operator', true);
				}
				util.setF('filter_item:within_matches:within_field', item.withinFieldName);
				util.setF('filter_item:within_matches:matches_field', item.matchesFieldName);
				util.setF('filter_item:within_matches:item_value', item.itemValue);
				break;
			case 'expression':
				util.setF('filter_item:expression_item_label', item.label);
				util.setF('filter_item:expression_item_value', item.itemValue);
				break;
		}
		setType(filterType);
		setFieldItemValueSection();
	}
	function saveItem() {
		var obj = {};
		var filterType = util.getF('filter_item:filter_item_type_list');
		obj.id = id;
		obj.isRootItem = !isItemInGroup;
		obj.isGroup = false;
		obj.filterType = filterType;
		if (!isItemInGroup) {
			obj.isActive = isActive;
		}
		switch (filterType) {
			case 'field':
				var fieldName = validator.isValue('filter_item:field_list');
				if (fieldName !== '') {
					obj.fieldName = fieldName;
					var queryField = queryFieldsDb[util.h(fieldName)];
					var isAggregatingField = queryField.isAggregatingField;
//					var operatorListType = isAggregatingField ? 'numerical' : 'text';
					var operatorObj = getFieldItemOperator();
					// util.showObject(operatorObj);
					obj.isNegated = operatorObj.isNegated;
					var expressionType = operatorObj.expressionType;
					if (!isAggregatingField) {
						var isWildcardExpression = (expressionType === 'wildcard');
						var isRegularExpression = (expressionType === 'regexp');
						if (!isWildcardExpression && !isRegularExpression) {
							// We need to get the fieldCategory to check if
							// we have a day_of_week or hour_of_day list active
							var fieldCategory = queryField.category;
							var valueElementId = 'filter_item:field_item_value';
							if (fieldCategory === 'day_of_week') {
								valueElementId = 'filter_item:day_of_week_list';
							}
							else if (fieldCategory === 'hour_of_day') {
								valueElementId = 'filter_item:hour_of_day_list';
							}
							obj.itemValue = validator.isValue(valueElementId);
						}
						else {
							obj.itemValue = validator.isValue('filter_item:field_item_value');
							if (isWildcardExpression) {
								expressionType = 'wildcard';
							}
							else {
								// isRegularExpression
								expressionType = 'regexp';
								obj.itemValue = validator.isRegularExpression('filter_item:field_item_value');
							}
						}
					}
					else {
						// Numerical field,
						// we allow int and float, so we just check for number
						obj.itemValue = validator.isNumber('filter_item:field_item_value');
					}
					obj.expressionType = expressionType;
				}
				break;
			case 'within_matches':
				obj.isNegated = util.getF('filter_item:within_matches:use_not_operator');
				obj.withinFieldName = validator.isValue('filter_item:within_matches:within_field');
				obj.matchesFieldName = validator.isValue('filter_item:within_matches:matches_field');
				obj.itemValue = validator.isValue('filter_item:within_matches:item_value');
				// Make sure the within field is different from the matches field
				if (obj.withinFieldName !== '' && (obj.withinFieldName === obj.matchesFieldName)) {
					validator.isCustom('filter_item:within_matches:matches_field', langVar('lang_stats.global_filter.equal_within_and_matches_field_message'));
				}
				/*
				  util.setF('filter_item:within_matches:use_not_operator', true);
				}
				util.setF('filter_item:within_matches:within_field', item.withinFieldName);
				util.setF('filter_item:within_matches:matches_field', item.matchesFieldName);
				util.setF('filter_item:within_matches:item_value', item.itemValue);
				*/
				break;
			case 'expression':
				obj.label = validator.isValue('filter_item:expression_item_label');
				// Check for duplicate names - We have a problem here because
				// the label only exists for expressions but not for any other FilterType,
				// so we simply allow duplicate expression labels!
				obj.itemValue = validator.isValue('filter_item:expression_item_value');
				break;
		}
		if (validator.allValid()) {
			// util.showObject(obj);
			close();
			calleeObj.saveFilterItem(obj, isNew, isItemInGroup, groupItemId);
		}
	}
	// Return global properties and methods
	return {
		init: init,
		setProfileSpecificData: setProfileSpecificData,
		open: open
	};
}());
//
// filterGropup - global filter group handling and editor
// used in reports and in scheduler.
//
/* global
	globalFilter: false */
var filterGroupEditor = (function() {
	var YE = YAHOO.util.Event,
		YD = YAHOO.util.Dom,
		panel = null,
		validator = null,
		// The callee object
		calleeObj = null,
		// active filter group properties
		id = '',
		isNew = false,
		isSaveCheckedAsNewGroup = false;
	function init() {
		var panelObj = {
			panelId:'filter_group:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: '-',
//			left: 70,
//			top: 120,
			zIndex: 40,
			isCover: true,
			isSticky: true,
			closeEvent: _close
		};
		panel = new util.Panel3(panelObj);
		validator = new util.Validator();
		YE.addListener('filter_group:label', 'keypress', _saveGroupViaEnter);
		YE.addListener('filter_group:ok_btn', 'click', _saveGroup);
		YE.addListener('filter_group:cancel_btn', 'click', _close);
	}
	function open(itemId, groupLabel, _isNew, _isSaveCheckedAsNewGroup, _calleeObj) {
		id = itemId;
		isNew = _isNew;
		isSaveCheckedAsNewGroup = _isSaveCheckedAsNewGroup;
		calleeObj = _calleeObj;
		var panelHeaderLabel;
		if (!isNew && !isSaveCheckedAsNewGroup) {
			panelHeaderLabel = langVar('lang_stats.global_filter.edit_group');
		}
		else if (isSaveCheckedAsNewGroup) {
			panelHeaderLabel = langVar('lang_stats.global_filter.save_checked_as_group');
		}
		else {
			panelHeaderLabel = langVar('lang_stats.global_filter.new_group');
		}
//		globalFilter.disableForm();
		// set the form
		util.setF('filter_group:label', groupLabel);
		var viewportWidth = YD.getViewportWidth();
		// If large display
		if (viewportWidth > 1000) {
			// Get panel position (reports : scheduler)
			var region = YD.getRegion(calleeObj.container);
			var left = region.left + 60;
			var top = region.top - 45;
			panel.open({
				label: panelHeaderLabel,
				left: left,
				top: top
			});
		}
		else {
			// If small display
			panel.prePositionAtCenter();
			panel.open({label: panelHeaderLabel});
		}
		var element = util.getE('filter_group:label');
		element.focus();
	}
	function _close() {
		validator.reset();
		panel.close();
//		globalFilter.enableForm();
	}
	function _saveGroupViaEnter(event) {
		var keyCode = (event.which) ? event.which : event.keyCode;
		if (keyCode == 13 || keyCode == 3) {
			_saveGroup();
		}
	}
	function _saveGroup() {
		// alert('save group');
		// Validate group name
		var groupLabelItems = repFiltersUtil.getGroupValuesByKey(calleeObj.filterItems, 'label', id);
		var groupLabel = validator.isValue('filter_group:label');
		groupLabel = validator.isUnique('filter_group:label', groupLabelItems);
		if (validator.allValid()) {
			// get unique node name
			var groupNameItems = repFiltersUtil.getGroupValuesByKey(calleeObj.filterItems, 'name', id);
			var groupName = util.labelToUniqueNodeName(groupLabel, groupNameItems, 'filter_group');
			if (isSaveCheckedAsNewGroup) {
				calleeObj.saveCheckedAsNewGroupProcessing(groupName, groupLabel);
			}
			else {
				calleeObj.saveFilterGroup(id, groupName, groupLabel, isNew);
			}
			_close();
		}
	}
	// Return global properties and methods
	return {
		init: init,
		open: open
	};
}());
//
// Imports filter items and filter groups from scheduler actions and reports
// of the same profile and user.
//
/* global
	scheduler: false
	ReportFiltersPanel: false */
var importReportFilters = (function() {
	'use strict';
  	var YE = YAHOO.util.Event,
		YD = YAHOO.util.Dom,
		panel = null,
		// The filtersPanel which shows the items to import.
		filtersPanel = null,
		// A reference to filter items and groups which
		// already exist as saved items or items to be saved.
		// Note, this are saved items in scheduler actions,
		// not the once loaded from users_cache.
		filtersLookup = {},
		filterGroupLabelsLookup = {},
		// Items to import which are not duplicates with
		// already saved items.
		filterItemsToImport = [],
		filterGroupsToImport = [],
		profileName = '',
		queryFieldsDb = [];
	function _init() {
		var panelObj = {
			panelId:"import_rf:panel",
			panelClassName: 'panel-50',
			panelHeaderLabel: langVar('lang_admin.scheduler.import_report_filters'),
//			left: 70,
//			top: 80,
			zIndex: 80,
			isCover: true,
			isSticky: true,
			closeEvent: _close
		};
		panel = new util.Panel3(panelObj);
		// Create a minimalistic filters panel
		filtersPanel = new ReportFiltersPanel({
			containerId: 'import_rf:container',
			filterItemsSelectCallback: importReportFilters.filterItemsSelectionListener
		});
		YE.addListener('import_rf:import_btn', 'click', _import);
		YE.addListener('import_rf:cancel_btn', 'click', _close);
	}
	function open(_profileName, _queryFieldsDb, existingFilterItems) {
		// This is called via the main reportFiltersPanel
		// NOTE, WE GET ONLY filterItems which already includes the groups!
//		util.showObject(existingFilterItems, 'existingFilterItems');
		profileName = _profileName;
		queryFieldsDb = _queryFieldsDb;
		// Reset
		filtersLookup = {};
		filterGroupLabelsLookup = {};
		filterItemsToImport = [];
		filterGroupsToImport = [];
		// Add filters which already exist to filtersLookup
		// so that we don't show duplicates to import.
		var filterId;
		for (var i = 0, numItems = existingFilterItems.length; i < numItems; i++) {
			var filterItem = existingFilterItems[i];
			if (!filterItem.hasOwnProperty('items')) {
				// This is a filterItem
				filterId = repFiltersUtil.getFilterItemAsString(filterItem);
			}
			else {
				// This is a filterGroup
				filterId = repFiltersUtil.getFilterGroupId(filterItem);
				// Add group label to filterGroupLabelsLookup
				var groupLabel = filterItem.label;
				filterGroupLabelsLookup['_' + groupLabel] = true;
			}
			filtersLookup[filterId] = true;
		}
		// Open Import panel
		if (panel === null) {
			_init();
		}
		// Set panel text
		// TODO - Getting the profileLabel looks a bit of hack because
		// we don't won't any scheduler dependencies here.
		var profileItem = scheduler.profilesDb[util.h(profileName)];
		var profileLabel = profileItem.label;
		var infoText = langVar('lang_admin.scheduler.import_report_filters_info');
		infoText = infoText.replace(/__PARAM__1/, '<strong>' + profileLabel + '</strong>');
		var infoTextContainer = util.getE('import_rf:import_report_filters_info');
		infoTextContainer.innerHTML = infoText;
		// Reset filtersPanel
		filtersPanel.initFilters(profileName, queryFieldsDb, [], []);
		var viewportWidth = YD.getViewportWidth();
		// If large display
		if (viewportWidth > 1000) {
			// Set import panel position
			var region = YD.getRegion('sa:report_filters');
			var left = region.left;
			var top = region.top - 60;
			panel.open({
				left: left,
				top: top
			});
		}
		else {
			// If small display
			panel.prePositionAtCenter();
			panel.open();
		}
		// Get filter items and filter groups already used in scheduler
		var a = _getUsedReportFiltersInScheduler();
//		util.showObject(a);
		_addToFilterItemsToImport(a.filterItems);
		_addToFilterGroupsToImport(a.filterGroups);
		// Get saved filter items and groups
		_getSavedReportFilters(profileName);
	}
	function _close() {
		panel.close();
	}
	function filterItemsSelectionListener(isSelectedItem) {
		// Enables/disables import button
		util.enableE('import_rf:import_btn', isSelectedItem);
	}
	function _import() {
		util.disableE('import_rf:import_btn');
		var activeFilterItemsAndGroups = filtersPanel.getActiveItems();
		schedulerActions.importReportFilterItemsAndGroups(activeFilterItemsAndGroups);
		// Close the Import panel.
		_close();
	}
	function _getUsedReportFiltersInScheduler() {
		// Returns all used report filters and filter groups of scheduler
		var itemsDb = scheduler.itemsDb;
		var filterItems = [];
		var filterGroups = [];
	//	util.showObject(itemsDb);
		for (var i = 0, len = itemsDb.length; i < len; i++) {
			if (itemsDb[i].dat.hasOwnProperty('actions')) {
				var actions = itemsDb[i].dat.actions;
				for (var j = 0, numActions = actions.length; j < numActions; j++) {
					var theAction = actions[j];
					if (theAction.hasOwnProperty('report_filters')) {
						var reportFilters = theAction.report_filters;
						if (reportFilters.filter_items.length > 0) {
							var existingFilterItems = util.cloneObject(reportFilters.filter_items);
							// Add existingFilterItems to filterItems
							_addFilterItemToUsedReportFilters(existingFilterItems, filterItems);
						}
						if (reportFilters.filter_groups.length > 0) {
							var existingFilterGroups = util.cloneObject(reportFilters.filter_groups);
							// Add existingFilterGroups to filterGroups
							_addFilterItemToUsedReportFilters(existingFilterGroups, filterGroups);
						}
					}
				}
			}
		}
		return {filterItems: filterItems, filterGroups: filterGroups};
	}
	function _addFilterItemToUsedReportFilters(itemsToBeAdded, theItems) {
		for (var i = 0, len = itemsToBeAdded.length; i < len; i++) {
			var filterItemOrGroup = util.cloneObject(itemsToBeAdded[i]);
			theItems.push(filterItemOrGroup);
		}
	}
	function _getSavedReportFilters(profileName) {
		var url = '?dp=admin_pages.scheduler.get_saved_report_filters';
		var dat = 'v.fp.page_token=' + pageInfo.pageToken;
		dat += '&v.fp.profile_name=' + profileName;
		util.serverPost(url, dat);
	}
	function getSavedReportFiltersResponse(dat) {
//		util.showObject(dat);
		_addToFilterItemsToImport(dat.filterItems);
		_addToFilterGroupsToImport(dat.filterGroups);
		// Show filters
		filtersPanel.initFilters(profileName, queryFieldsDb, filterItemsToImport, filterGroupsToImport);
	}
	function _addToFilterItemsToImport(filterItems) {
		// This adds non-duplicate filter items to filterItemsToImport
		for (var i = 0, len = filterItems.length; i < len; i++) {
			var filterItem = filterItems[i];
			// Check if the filter item fields actually exist for this profile.
			if (repFiltersUtil.getIsValidFilterFields(queryFieldsDb, filterItem)) {
				var filterId = repFiltersUtil.getFilterItemAsString(filterItem);
				// Add filter item if it doesn't yet exist.
				if (!filtersLookup.hasOwnProperty(filterId)) {
					var clonedFilterItem = util.cloneObject(filterItem);
					clonedFilterItem.isActive = false;
					filterItemsToImport.push(clonedFilterItem);
				}
			}
		}
	}
	function _addToFilterGroupsToImport(filterGroups) {
		// This adds non-duplicate filter groups to filterGroupsToImport
		for (var i = 0, len = filterGroups.length; i < len; i++) {
			var filterGroup = filterGroups[i];
			// If one filter item in group has an invalid field then
			// ignore the group.
			var subitems = filterGroup.items;
			var allSubitemsValid = true;
			for (var j = 0, numSubitems = subitems.length; j < numSubitems; j++) {
				var subitem = subitems[j];
				if (!repFiltersUtil.getIsValidFilterFields(queryFieldsDb, subitem)) {
					allSubitemsValid = false;
					break;
				}
			}
			if (allSubitemsValid) {
				var groupId = repFiltersUtil.getFilterGroupId(filterGroup);
				// Add filter group if it doesn't yet exist.
				if (!filtersLookup.hasOwnProperty(groupId)) {
					var clonedGroupItem = util.cloneObject(filterGroup);
					clonedGroupItem.isActive = false;
					// Reset the group node name because it becomes
					// re-generated when importing the groups in
					// reportFiltersPanel.
					clonedGroupItem.name = '';
//					util.showObject(clonedGroupItem, 'clonedGroupItem');
					// Check for unique group labels because it is
					// possible that two groups with different group ID
					// have the same label
					var groupLabel = clonedGroupItem.label;
					if (filterGroupLabelsLookup.hasOwnProperty('_' + groupLabel)) {
						groupLabel = _getUniqueGroupLabel(groupLabel);
						clonedGroupItem.label = groupLabel;
					}
					// Add final label to filterGroupLabelsLookup
					filterGroupLabelsLookup['_' + groupLabel] = true;
					filterGroupsToImport.push(clonedGroupItem);
				}
			}
		}
	}
	function _getUniqueGroupLabel(groupLabel) {
		var newGroupLabel = '';
		var isUnique = false;
		var count = 2;
		while (!isUnique) {
			newGroupLabel = groupLabel + ' (' + count + ')';
			isUnique = !filterGroupLabelsLookup.hasOwnProperty('_' + newGroupLabel);
			count++;
		}
		return newGroupLabel;
	}
	// Return global properties and methods
	return {
		open: open,
		filterItemsSelectionListener: filterItemsSelectionListener,
		getSavedReportFiltersResponse: getSavedReportFiltersResponse
	};
}());
//
// schedulerUtil.js
//
var schedulerUtil = {
	actionsEmailToEmailAddressesObject: function(options) {
		// Converts the action email parameters in options (options.rca, options.ca and options.ba)
		// to an address array with objects required by emailUtil.js
		// i.e.: [{type:'to',  address:'abc@abc.com'}, {type:'cc', address:'first last <abc@abc.com>'}]
		var emailAddresses = [];
		var shortcuts = ['rca', 'ca', 'ba'];
		var recipientTypes = ['to', 'cc', 'bcc'];
		for (var i = 0; i < shortcuts.length; i++) {
			var shortcut = shortcuts[i];
			// Note, only options with a value exists!
			// If no cc_address has been defined then no cc_address exists in options!
			if (options[shortcut]) {
				var addressString = options[shortcut];
				// Remove white space
				addressString = addressString.replace(/\s/g, '');
				var type = recipientTypes[i];
				if (addressString.indexOf(',') == -1) {
					// Single email address
					emailAddresses[emailAddresses.length] = {type:type, address:addressString};
				}
				else {
					// Multiple email addresses
					var dat = addressString.split(',');
					for (var j = 0; j < dat.length; j++) {
						emailAddresses[emailAddresses.length] = {type:type, address:dat[j]};
					}
				}
			}
		}
		return emailAddresses;
	}
}
//
// schedulerActionsList.js
//
var schedulerActionsList = {
	moveControl: null,
	list: null,
	init: function() {
		//
		// Create the actionsList object
		//
		var obj = {
			name: 'actions',
			tbodyElementId: 'scheduler:actions_list:tbody',
			noItemText: langVar('lang_admin.scheduler.no_action_defined'),
			isDefaultItem: true,
			listChangeNotificationFunction: schedulerActionsList.listChangeNotification,
			editItemFunction: schedulerActions.editItem,
			duplicateItemFunction: schedulerActions.duplicateItem,
			labelBuilderFunction: schedulerActions.labelBuilder
		}
		schedulerActionsList.list = new listcontrollerB.List(obj);
		schedulerActionsList.moveControl = new util.MoveControl('sa_move_control', schedulerActionsList.moveItem);
	},
	listChangeNotification: function(listName, selectedItemIndex, numberOfItems) {
		// Called from wide item list controller upon list change
		// Set button state of move control
		schedulerActionsList.moveControl.setState(selectedItemIndex, numberOfItems);
	},
	moveItem: function(direction) {
		// Invoked from moveControl
		// var direction = schedulerActionsList.moveControl.getMoveDirection(this.id);
		// Move item in list
		schedulerActionsList.list.moveItem(direction);
	}
}
//
// js
//
var schedulerActions = (function() {
	var YE = YAHOO.util.Event,
		YD = YAHOO.util.Dom,
		panel = null,
		validator = null,
		addressControl = null, // email address control for To, Cc and Bcc address
		//
		// Active form parameters
		//
		mode = '',// The active mode of an item is = new | edit | duplicate (We need to know the mode when saving the item back to the listcontrollerB.
		activeItemIndex = 0,// Refers to the active item index in listcontrollerB items array, respectively to the active row index.
		activeAction = '',
		isDatabaseAction = false,
		isProfileList = false,// No profile list has been populated yet.
		isExtendedProfilesList = false,// Specifies if the active profile list contains All profiles and Profile pattern list entry.
		isExtraOptions = false,// Specifies whether the extra options textarea element row is displayed.
		isUnlimitedGrants = false,
		activeReportsDb = [], // The active reportsDb of the selected profile
		activeDF = { // The active date filter object per action
			type: '', // none, relative | custom
			custom: '', // custom expression if type is custom
			relativePrefix: '', // none | recent || last
			relativeDateUnit: '', // year | quarter, month, ...
			relativeDateUnitCount: 1
		},
		reportFiltersPanel = null,
		initReportFiltersDone = false,
		isActiveReportFilters = false, // Specifies whether the report filters panel is displayed.
		reportFiltersOri = {
			// This is a backup of the original
			// report filters data. We need this to
			// save it back if report filters are
			// not modified.
			filterId: '',
			filterItems: [],
			filterGroups: []
        },
		reportFiltersId = '',
		reportFilterItems = [],
		reportFilterGroups = [],
		// profilesQueryFields contains loaded queryFieldsDb's per profile.
		// queryFieldsDb's are required for report filters.
		profilesQueryFields = {};
	function init() {
		isUnlimitedGrants = pageInfo.isRootAdmin || pageInfo.permissions.isUnlimitedGrants;
		validator = new util.Validator();
		//
		// Init the Action panel
		//
		var panelObj = {
			panelId: 'sa:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: '-',
			zIndex: 20,
			isCover: true,
			isSticky: true,
			closeEvent: closePanel
		};
		panel = new util.Panel3(panelObj);
		reportFiltersPanel = new ReportFiltersPanel({
			containerId: 'sa:report_filters',
			addItems: true,
			importItems: true,
			openImportPanelCallback: importReportFilters.open
		});
		// Create email addressControl
		addressControl = new emailUtil.AddressControl('sa:email_address_control:container', 360);
		//
		// Add events
		//
		YE.addListener('sa:action', 'change', selectActionActor);
		YE.addListener('sa:profile', 'change', selectProfileActor);
		YE.addListener('sa:report', 'change', selectReportActor);
		YE.addListener('sa:generate_report_files:output_format_type', 'change', setOutputFormatType);
		YE.addListener('sa:date_filter_text', 'click', openDateFilter);
		YE.addListener('sa:add_report_filters_btn', 'click', function() {
			setReportFilters(true);
		});
		YE.addListener('sa:remove_report_filters_btn', 'click', function() {
			setReportFilters(false);
		});
		YE.addListener(['sa:remove_db_data_by_number_of_days', 'sa:remove_db_data_by_filter_expression'], 'click', removeDatabaseDataEvent);
        YE.addListener('sa:export_csv_table:number_of_rows', 'keypress', setCvsExportNumberOfRowsToCustom);
		YE.addListener('sa:extra_options:toggle_btn', 'click', toggleExtraOptions);
		YE.addListener('sa:okay', 'click', saveItem);
		YE.addListener('sa:cancel', 'click', closePanel);
	}
	function selectActionActor() {
		var action = util.getF('sa:action');
		setActionByName(action);
	}
	function setActionByName(action) {
		// alert('setActionByName(): ' + action);
		var databaseTextPattern = /database/;
		activeAction = action;
		isDatabaseAction = databaseTextPattern.test(action);
		// Reset the validator
		validator.reset();
		// Reset the display
		var displayGroups = [
			'no_action_defined',
			'profile',
			'profile_pattern',
			'remove_database_data',
			'report',
			'export_csv_table',
			'generate_report_files',
			'generate_report_files_output_directory',
			'generate_report_files_output_file',
			'generate_report_files_pdf_not_supported',
			'email',
			'report_language',
			'extra_options',
			'execute_command_line',
			'vertical_spacer'
		];
		for (var i = 0; i < displayGroups.length; i++) {
			util.hideE('sa:' + displayGroups[i] + ':group');
		}
		var showExtendedProfileList = false;
		//
		// Handle the action
		//
		if (action != '') {
			if (action != 'execute_command_line') {
				util.showE('sa:profile:group');
				if (isDatabaseAction) {
					// database actions
					showExtendedProfileList = true;
					if (action == 'remove_database_data') {
						util.showE('sa:remove_database_data:group');
					}
					util.showE('sa:vertical_spacer:group');
				}
				else {
					// report actions
					util.showE('sa:report:group');
					if (action == 'generate_report_files') {
						util.showE('sa:generate_report_files:group');
						// initOutputFormatType();
						setOutputFormatType();
					}
					else if (action == 'export_csv_table') {
						util.showE('sa:export_csv_table:group');
					}
					else {
						util.showE('sa:email:group');
					}
					util.showE('sa:report_language:group');
				}
				// resetExtraOptions();
				util.showE('sa:extra_options:group');
			}
			else {
				util.showE('sa:execute_command_line:group');
			}
		}
		else {
			util.showE('sa:no_action_defined:group');
			util.showE('sa:vertical_spacer:group');
		}
		//
		//
		// Update the profile list
		//
		//
		var profilesDb = [];
		var baseDb = [];
		if (isUnlimitedGrants) {
			// Update profiles list if required
			if (showExtendedProfileList != isExtendedProfilesList || !isProfileList) {
				profilesDb = scheduler.profilesDb;
				baseDb[0] = {name:'', label:langVar('lang_admin.scheduler.select_profile_label')};
				if (showExtendedProfileList) {
					baseDb[1] = {name:'*', label:langVar('lang_admin.scheduler.all_profiles_label')};
					baseDb[2] = {name:'__PROFILE__PATTERN__', label:langVar('lang_admin.scheduler.use_profile_pattern')};
				}
				util.populateSelect('sa:profile', baseDb, 'name', 'label');
				util.extendSelect('sa:profile', profilesDb, 'name', 'label');
				isProfileList = true;
				isExtendedProfilesList = showExtendedProfileList;
			}
		}
		else if (action != '') {
			// The user has limited grants and profiles may vary per action, so we always
			// update the profiles list upon action change!
			// We also don't allow a profile pattern for limited grants
			// var actionsDb = scheduler.actionsDb;
			// util.showObject(scheduler.actionsDb);
			// alert('action: ' + action);
			var superAction = scheduler.actionsDb[util.h(action)];
			// util.showObject(superAction);
			var isAllProfiles = superAction.isAllProfiles;
			profilesDb = isAllProfiles ? scheduler.profilesDb : superAction.profiles;
			baseDb[0] = {name:'', label:langVar('lang_admin.scheduler.select_profile_label')};
			if (showExtendedProfileList && isAllProfiles) {
				baseDb[1] = {name:'*', label:langVar('lang_admin.scheduler.all_profiles_label')};
			}
			util.populateSelect('sa:profile', baseDb, 'name', 'label');
			util.extendSelect('sa:profile', profilesDb, 'name', 'label');
		}
		// Reset the profile list to Select Profile
		setProfileList('');
	}
	function setTabPane(tabId) {
		util.hideE(['sa:general:pane', 'sa:email:pane', 'sa:options:pane']);
		util.showE('sa:' + tabId + ':pane');
	}
	function selectProfileActor() {
		// activated upon onchange
		var profileName = util.getF('sa:profile');
		setProfileList(profileName);
	}
	function setProfileList(profileName) {
		if (profileName != '') {
			if (isDatabaseAction) {
				var profileIsPattern = (profileName == '__PROFILE__PATTERN__');
				util.showE('sa:profile_pattern:group', profileIsPattern);
			}
			else {
				// Report action, we need to update the report list
				setReportList(profileName);
				// TEMP TEST
//				getQueryFieldsData(profileName);
			}
		}
		else {
			setReportList('');
		}
		// We re-set the form value because setProfileList() could be invoked other than by event
		util.setF('sa:profile', profileName);
	}
	function selectReportActor() {
		if (activeAction == 'generate_report_files') {
			// initOutputFormatType();
			setOutputFormatType();
		}
	}
	function setReportList(profileName) {
		// util.disableE('sa:report');
		var disableSelect = false;
		var baseDb = [];
		baseDb[0] = {name:'', label:langVar('lang_admin.scheduler.select_report_label')};
		if (profileName != '') {
			activeReportsDb = getReportsDb(profileName); // in scheduler.js
			// Create hash for activeReportsDb report name
			util.createHash(activeReportsDb, 'name');
			if (activeAction == 'generate_report_files') {
				baseDb[1] = {name:'*', label:langVar('lang_admin.scheduler.all_reports_label')};
			}
			// util.showObject(reportsDb);
			util.populateSelect('sa:report', baseDb, 'name', 'label');
			util.extendSelect('sa:report', activeReportsDb, 'name', 'label');
		}
		else {
			// No profile selected
			util.populateSelect('sa:report', baseDb, 'name', 'label');
			disableSelect = true;
		}
		util.setF('sa:report', '');
		util.disableE('sa:report', disableSelect);
	}
	function setOutputFormatType() {
		// Note, PDF is only allowed if report is not "All Reports" and
		// if a report contains not more than one report element (due PDF performance)
		var outputFormatType = util.getF('sa:generate_report_files:output_format_type');
		// alert('setOutputFormatType() - outputFormatType: ' + outputFormatType);
		var isHTML = (outputFormatType == 'html');
		util.hideE(['sa:generate_report_files_output_directory:group',
			'sa:generate_report_files_output_file:group',
			'sa:generate_report_files_pdf_not_supported:group'
		]);
		if (isHTML) {
			util.showE('sa:generate_report_files_output_directory:group');
		}
		else {
			var isPDFSupport = false;
			var reportName = util.getF('sa:report');
			if (reportName == '') {
				// No report selected yet, assume we support it
				isPDFSupport = true;
			}
			else if (reportName == '*') {
				// 'All reports' selected, PDF output not supported
				isPDFSupport = false;
			}
			else {
				// Check number of report elements, only reports
				// with <= 1 report element are supported for PDF output.
				var reportItem = activeReportsDb[util.h(reportName)];
				// util.showObject(reportItem);
				if (reportItem.numberOfReportElements <= 1) {
					isPDFSupport = true;
				}
			}
			util.showE('sa:generate_report_files_output_file:group', isPDFSupport);
			util.showE('sa:generate_report_files_pdf_not_supported:group', !isPDFSupport);
		}
	}
	function openDateFilter() {
		schedulerDF.open(activeDF);
	}
	function setDateFilter(newDateFilter) {
		activeDF = util.cloneObject(newDateFilter);
	}
	function updateDateFilterText() {
		// Updates the date filter text according activeDF
		var type = activeDF.type;
		var text = '';
		if (type == 'none') {
			text = langVar('lang_admin.scheduler.date_filter.entire_date_range');
		}
		else if (type == 'relative') {
			var prefix = activeDF.relativePrefix;
			var dateUnit = activeDF.relativeDateUnit;
			var dateUnitCount = activeDF.relativeDateUnitCount;
			var param1Label = '';
			var param2Label = '';
			var dateUnitLabelsSingular = scheduler.dateUnitLabelsSingular;
			var dateUnitLabelsPlural = scheduler.dateUnitLabelsPlural;
			if (prefix != 'none') {
				if (prefix == 'recent') {
					// text = Last $param1 including $param2 of schedule execution date
					text = langVar('lang_admin.scheduler.date_filter.last_n_date_unit_including');
				}
				else {
					// text = Last $param1 excluding $param2 of schedule execution date
					text = langVar('lang_admin.scheduler.date_filter.last_n_date_unit_excluding');
				}
				param2Label = dateUnitLabelsSingular[dateUnit];
				text = text.replace(/__PARAM__2/, param2Label);
			}
			else {
				// text = Latest $param1 relative to end of log date
				text = langVar('lang_admin.scheduler.date_filter.latest_n_date_unit_relative');
			}
			// Handle param1Label
			if (dateUnitCount > 1) {
				param1Label = dateUnitCount + ' ' + dateUnitLabelsPlural[dateUnit];
			}
			else {
				param1Label = dateUnitCount + ' ' + dateUnitLabelsSingular[dateUnit];
			}
			text = text.replace(/__PARAM__1/, param1Label);
		}
		else {
			text = activeDF.custom;
		}
		util.updateT('sa:date_filter_text', text);
	}
	//
	// Handle report filters
	//
	function getQueryFieldsData(profileName) {
		var url = '?dp=admin_pages.scheduler.get_query_fields_data';
		var dat = 'v.fp.page_token=' + pageInfo.pageToken;
		dat += '&v.fp.profile_name=' + profileName;
		util.serverPost(url, dat);
	}
	function getQueryFieldsDataResponse(dat) {
//		util.showObject(dat);
		// Cache queryFieldsDb
		var profileName = dat.profileName;
		var queryFieldsDb = dat.queryFieldsDb;
		// Create hash
		util.createHash(queryFieldsDb, 'name');
		profilesQueryFields['_' + profileName] = queryFieldsDb;
		if (!initReportFiltersDone) {
			initReportFilters();
		}
		else {
			// Filters are already initialized but the profile and queryFieldsDb
			// changed. We must check filter item integrity for the new
			// queryFieldsDb.
			alert('Check filter item integrity for the new queryFieldsDb');
		}
	}
	function initReportFilters() {
		if (!initReportFiltersDone) {
			// Check if queryFieldsDb for active profile exists
			var profileName = util.getF('sa:profile');
			if (profileName !== '') {
				// If queryFieldsDb is loaded
				if (profilesQueryFields.hasOwnProperty('_' + profileName)) {
					var queryFieldsDb = profilesQueryFields['_' + profileName];
					// Init filters
					// Note, initFilters() should run only once.
					reportFiltersPanel.initFilters(
						profileName,
						queryFieldsDb,
						reportFiltersOri.filterItems,
						reportFiltersOri.filterGroups
					);
					initReportFiltersDone = true;
				}
				else {
					// Load queryFieldsDb
					getQueryFieldsData(profileName);
					// Disable filters while loading
					reportFiltersPanel.disableAll();
				}
			}
			else {
				// No profile selected, disable filters
				reportFiltersPanel.disableAll();
			}
		}
	}
	function setReportFilters(makeActive) {
		// This sets the report filters display
		isActiveReportFilters = makeActive;
		if (isActiveReportFilters && !initReportFiltersDone) {
			initReportFilters();
		}
		util.showE('sa:add_report_filters_row', !isActiveReportFilters);
		util.showE('sa:report_filters_row', isActiveReportFilters);
		util.showE('sa:report_filters', isActiveReportFilters);
		util.showE('sa:remove_report_filters_row', isActiveReportFilters);
	}
	function importReportFilterItemsAndGroups(items) {
		reportFiltersPanel.addNewItemsViaImport(items);
	}
	function handleReportFiltersState() {
		// This handles the reportFilters state.
	}
	//
	// Remove database data
	//
	function removeDatabaseDataEvent() {
		// Invoked by radio button event
		var isCustomFilterExpression = util.getF('sa:remove_db_data_by_filter_expression');
		setRemoveDatabaseData(isCustomFilterExpression);
	}
	function setRemoveDatabaseData(isCustomFilterExpression) {
		util.disableE('sa:remove_db_data_number_of_days', isCustomFilterExpression);
		util.disableE('sa:remove_db_data_filter_expression', !isCustomFilterExpression);
	}
	function setCvsExportNumberOfRowsToCustom() {
		// Makes custom radio button enabled when user types a row number
		util.setF('sa:export_csv_table:number_of_rows:custom', true);
	}
	function toggleExtraOptions() {
		setIsExtraOptions(!isExtraOptions);
	}
	function setIsExtraOptions(_isExtraOptions) {
		isExtraOptions = _isExtraOptions;
		util.showE('sa:extra_options:row', isExtraOptions);
		var linkText = isExtraOptions ? langVar('lang_admin.scheduler.remove_extra_options') : langVar('lang_admin.scheduler.add_extra_options');
		util.updateT('sa:extra_options:toggle_btn', linkText);
	}
	function newItem() {
		// invoked from scheduler.js
		mode = 'new';
		var panelLabel = langVar('lang_admin.scheduler.new_action');
		var newItemObj = {};
		openPanel(panelLabel, newItemObj);
	}
	function editItem(itemIndex, item) {
//		console.log('itemIndex I: ' + itemIndex);
		// invoked from listcontrollerB
		mode = 'edit';
		activeItemIndex = itemIndex;
//		console.log('activeItemIndex II: ' + activeItemIndex);
		var panelLabel = langVar('lang_admin.scheduler.edit_action');
		openPanel(panelLabel, item);
	}
	function duplicateItem(itemIndex, item) {
		// invoked from listcontrollerB
		// itemIndex is the index of the item we duplicate!
		// util.showObject(item);
		mode = 'duplicate';
		activeItemIndex = itemIndex;
		var panelLabel = langVar('lang_admin.scheduler.edit_duplicated_action');
		openPanel(panelLabel, item);
	}
	function openPanel(panelLabel, item) {
		// Note, the item may be an empty object, in this case we simply
		// set the initial Action form state to "Select Action"
		var sa = schedulerActions;
		// util.showObject(item);
		var isEmptyItem = (item.type == null || item.type == '');
		//
		// Reset Action form
		//
		util.resetF('sa:form');
		setActionByName('');
		// Reset df
		activeDF = {
			type: 'none',
			custom: '',
			relativePrefix: 'recent', // We set relative default date to init the date picker panel
			relativeDateUnit: 'year',
			relativeDateUnitCount: 1
		};
		// Reset report filters
		reportFiltersPanel.resetIsModified();
		initReportFiltersDone = false;
		setReportFilters(false);
		reportFiltersOri = {
			filterId: '',
			filterItems: [],
			filterGroups: []
		};
		setRemoveDatabaseData(false);
		setIsExtraOptions(false);
		//
		// Reset email addresses
		//
		addressControl.reset();
		//
		// Set data
		//
		if (!isEmptyItem) {
			updateForm(item);
		}
		//
		// Update date filter text
		//
		updateDateFilterText();
		//
		// Open the panel 
		//
		fixPanel(true);
		var viewportWidth = YD.getViewportWidth();
		// If small display
		if (viewportWidth > 1000) {
			panel.open({
				label: panelLabel,
				left: 316,
				top: 160
			});
		}
		else {
			panel.prePositionAtCenter();
			panel.open({label: panelLabel});
		}
	}
	function updateForm(item) {
		//util.showObject(item);
		var sa = schedulerActions;
		var action = item.type;
		setActionByName(action);
		util.setF('sa:action', action);
        if (action != 'execute_command_line') {
            var profile = item.profile;
            var profileIsPattern = item.profile_is_pattern;
            // alert('updateForm() - action: ' + action);
            if (isDatabaseAction) {
                if (profileIsPattern) {
                    profile = '__PROFILE__PATTERN__';
                    util.showE('sa:profile_pattern:group', profileIsPattern);
                    util.setF('sa:profile_pattern', item.profile);
                }
                util.setF('sa:profile', profile);
                if (action == 'remove_database_data') {
                    var filterInfo = item.filter_info;
                    var isCustomFilterExpression = !filterInfo.number_of_days_active;
                    if (!isCustomFilterExpression) {
                        util.setF('sa:remove_db_data_by_number_of_days', true);
                        util.setF('sa:remove_db_data_number_of_days', filterInfo.number_of_days);
                    }
                    else {
                        util.setF('sa:remove_db_data_by_filter_expression', true);
                        util.setF('sa:remove_db_data_filter_expression', item.options.f);
                    }
                    setRemoveDatabaseData(isCustomFilterExpression);
                }
            }
            else {
                //
                // Report action
                //
                util.setF('sa:profile', profile);
                var options = item.options;
                //
                // Set the report list
                //
                var reportName = options.rn;
                // Update the report list
                setReportList(profile);
                util.setF('sa:report', reportName);
                //
                // Set Date filter
                //
                var dfObj = item.date_filter;
                // util.showObject(dfObj);
                var dateFilterType = dfObj.type;
                if (dateFilterType != 'none') {
                    // Set activeDF which is then manipulated by schedulerDF.js
                    activeDF.type = dateFilterType;
                    if (dateFilterType == 'relative') {
                        activeDF.relativePrefix = dfObj.relative_prefix;
                        activeDF.relativeDateUnit = dfObj.relative_date_unit;
                        activeDF.relativeDateUnitCount = dfObj.relative_date_unit_count;
                    }
                    else {
                        activeDF.custom = dfObj.custom;
                    }
                }
				//
				// Set report filter
				//
				reportFiltersOri = {
					filterId: item.report_filters.filter_id,
					filterItems: item.report_filters.filter_items,
					filterGroups: item.report_filters.filter_groups
				};
				// Only init reportFilter items and groups if filter exists.
				if (reportFiltersOri.filterItems.length > 0 || reportFiltersOri.filterGroups.length > 0) {
					// Show/init filters
					setReportFilters(true);
				}
                //
                // Set misc. report action options
                //
                if (action == 'export_csv_table') {
                    util.setF('sa:export_csv_table:output_file', options.of);
                    var endingRow = options.er;
                    var endingRowText = '';
                    if (endingRow == 0) {
                        util.setF('sa:export_csv_table:number_of_rows:default', true);
                    }
                    else if (endingRow == -1) {
                        // All rows
                        util.setF('sa:export_csv_table:number_of_rows:all', true);
                    }
                    else {
                        // Custom row number
                        endingRowText = endingRow;
                        util.setF('sa:export_csv_table:number_of_rows:custom', true);
                    }
                    util.setF('sa:export_csv_table:number_of_rows', endingRowText);
                }
                else if (action == 'generate_report_files') {
                    var outputFormatType = options.oft;
                    if (outputFormatType == 'html') {
                        // HTML
                        util.setF('sa:generate_report_files:output_directory', options.od);
                    }
                    else {
                        // PDF
                        util.setF('sa:generate_report_files:output_file', options.of);
                    }
                    util.setF('sa:generate_report_files:output_format_type', outputFormatType);
                    // initOutputFormatType();
                    setOutputFormatType();
                }
                else {
                    // send report_by_email
                    util.setF('sa:return_address', options.rna);
                    var emailAddresses = schedulerUtil.actionsEmailToEmailAddressesObject(options);
                    addressControl.init(emailAddresses);
                    // util.setF('sa:recipient_address', options.rca);
                    // util.setF('sa:cc_address', options.ca);
                    // util.setF('sa:bcc_address', options.ba);
                    util.setF('sa:report_email_subject', options.res);
                }
                // Set language
                util.setF('sa:report_language', item.language);
            }
            if (item.extra_options != '') {
                setIsExtraOptions(true);
                util.setF('sa:extra_options', item.extra_options);
            }
        }
        else {
            // execute_command_line
            util.setF('sa:execute_command_line:command_line', item.command_line);
        }
	}
	function saveItem() {
		// We save the item object back to the items array in listcontrollerB
		// alert('saveItem()');
		var action = activeAction;
		var o = {};
        var isValidEmailAddresses = true;
		if (action != '') {
			o.type = action;
            if (action != 'execute_command_line') {
                var profileIsPattern = false;
                o.profile = validator.isValue('sa:profile');
                if (o.profile == '__PROFILE__PATTERN__') {
                    o.profile = validator.isValue('sa:profile_pattern');
                    profileIsPattern = true;
                }
                o.profile_is_pattern = profileIsPattern;
                if (isDatabaseAction) {
                    if (action == 'remove_database_data') {
                        var number_of_days_active = util.getF('sa:remove_db_data_by_number_of_days');
                        var number_of_days = 0;
                        var f_option;
                        if (number_of_days_active) {
                            // number of days
                            number_of_days = validator.isInteger('sa:remove_db_data_number_of_days', 1);
                            f_option = '(date_time < now() - 60*60*24*' + number_of_days + ')';
                        }
                        else {
                            // custom filter expression
                            f_option = validator.isValue('sa:remove_db_data_filter_expression');
                        }
                        o.options = {f: f_option};
                        o.filter_info = {
                            number_of_days_active: number_of_days_active,
                            number_of_days: number_of_days
                        }
                    }
                }
                else {
                    //
                    // This is a report action
                    //
                    o.options = {};
                    var reportName = validator.isValue('sa:report');
                    o.options.rn = reportName;
                    //
                    // Get the date filter
                    //
                    o.date_filter = {
						type: activeDF.type,
						custom: activeDF.custom,
						relative_prefix: activeDF.relativePrefix,
						relative_date_unit: activeDF.relativeDateUnit,
						relative_date_unit_count: activeDF.relativeDateUnitCount
					};
					//
					// Save report filters if modified or added.
					//
					// Set defaults for no report filters
					var filterId = '';
					var filterItems = [];
					var filterGroups = [];
					if (isActiveReportFilters) {
//						console.log('isModified reportFilters: ' + reportFiltersPanel.getIsModified());
						if (reportFiltersPanel.getIsModified())  {
							// Filters have been added or modified.
							// filterId remains empty string which
							// forces expression creation on server.
							filterItems = reportFiltersPanel.getFilterItems();
							filterGroups = reportFiltersPanel.getFilterGroups();
						}
						else {
							// Nothing modified, use original filter data
							filterId = reportFiltersOri.filterId;
							filterItems = reportFiltersOri.filterItems;
							filterGroups = reportFiltersOri.filterGroups;
						}
					}
					o.report_filters = {
						// Reset filter_id so that the filter becomes
						// re-created in filters_cache.
						filter_id: filterId,
						filter_items: filterItems,
						filter_groups: filterGroups
					};
//					util.showObject(o.report_filters);
                    //
                    // Handle misc. options
                    //
                    switch (action) {
                        case 'export_csv_table':
                            o.options.of = validator.isValue('sa:export_csv_table:output_file');
                            // Handle ending row
                            var endingRow;
                            if (util.getF('sa:export_csv_table:number_of_rows:default')) {
                                // default, use number of rows of report element
                                endingRow = 0;
                            }
                            else if (util.getF('sa:export_csv_table:number_of_rows:all')) {
                                // all rows
                                endingRow = -1;
                            }
                            else {
                                endingRow = validator.isInteger('sa:export_csv_table:number_of_rows', 1);
                            }
                            o.options.er = endingRow;
                            break;
                        case 'generate_report_files':
                            var ouputFormatType = util.getF('sa:generate_report_files:output_format_type');
                            o.options.oft = ouputFormatType;
                            if (ouputFormatType == 'html') {
                                o.options.od = validator.isValue('sa:generate_report_files:output_directory');
                            }
                            else {
                                // Output is PDF
                                // Make sure we throw a hidden error if output format is PDF but the selected report
                                // doesn't support it. Note, the error message is already displayed but we need to
                                // prevent the user from saving the form!
                                // We simply reset the 'sa:generate_report_files:output_file' field, which will fire an error anyway!
                                if (reportName != '') {
                                    isPDFOutputSupport = true;
                                    if (reportName == '*') {
                                        isPDFOutputSupport = false;
                                    }
                                    else {
                                        var reportItem = activeReportsDb[util.h(reportName)];
                                        isPDFOutputSupport = (reportItem.numberOfReportElements <= 1);
                                    }
                                    if (!isPDFOutputSupport) {
                                        // Reset the 'sa:generate_report_files:output_file' so that it throws an error and avoids saving the action
                                        util.setF('sa:generate_report_files:output_file', '');
                                    }
                                }
                                o.options.of = validator.isValue('sa:generate_report_files:output_file');
                            }
                            break;
                        case 'send_report_by_email':
                            o.options.rna = validator.isValue('sa:return_address');
                            isValidEmailAddresses = addressControl.validate();
                            if (isValidEmailAddresses) {
                                var addressControlAddresses = addressControl.getAddresses();
                                var rca = ''; // To
                                var ca = ''; // Cc
                                var ba = ''; // Bcc
                                for (var i = 0; i < addressControlAddresses.length; i++) {
                                    var addressItem = addressControlAddresses[i];
                                    var address = addressItem.address;
									// Separate multiple email addresses by
									// by comma only. A space after the comma
									// causes problems in some mail servers.
                                    if (addressItem.type == 'to') {
                                        rca += address + ',';
                                    }
                                    else if (addressItem.type == 'cc') {
                                        ca += address + ',';
                                    }
                                    else {
                                        ba += address + ',';
                                    }
                                }
                                // Add only defined addresses to options!
                                if (rca != '') {
                                    o.options.rca = rca.replace(/(,)$/, '');
                                }
                                if (ca != '') {
                                    o.options.ca = ca.replace(/(,)$/, '');
                                }
                                if (ba != '') {
                                    o.options.ba = ba.replace(/(,)$/, '');
                                }
                                // util.showObject(o.options);
                            }
                            o.options.res = validator.isValue('sa:report_email_subject');
                            break;
                    }
                    o.language = util.getF('sa:report_language');
                }
                // get extra options
                o.extra_options = isExtraOptions ? util.getF('sa:extra_options') : '';
            }
            else {
                // validate execute_command_line action
                o.command_line = validator.isValue('sa:execute_command_line:command_line');
            }
		}
		else {
			o.action = validator.isValue('sa:action');
		}
		if (validator.allValid() && isValidEmailAddresses) {
			// alert('all valid, save the item');
			// Save the item to the actions list (listcontrollerB handles save of the object)
			var sa = schedulerActions;
//			console.log('itemIndex III: ' + activeItemIndex);
			schedulerActionsList.list.saveItem(mode, activeItemIndex, o);
			closePanel();
			// Check SMTP server warning
			handleSmtpServerWarning();
		}
	}
	function closePanel() {
		var sa = schedulerActions;
		validator.reset();
		panel.close();
		fixPanel(false);
	}
	function fixPanel(isFix) {
		// IE 6 fix, show/hide select boxes in Scheduler form!
		util.showEV('schedule_date_time:tr', !isFix);
		// Firefox fix, change overlow property of underlying actions list to hidden (otherwise form field focus becomes lost!)
		var element = util.getE('scheduler:actions_list:div');
		element.style.overflow = isFix ? 'hidden' : 'scroll';
	}
	function getReportLabel(profile, reportName) {
		var profilesDb = scheduler.profilesDb;
		var profileItem = profilesDb[util.h(profile)];
		var reportLabel = '';
		if (profileItem != null) {
			var reportNames = profileItem.rN;
			var reportLabels = profileItem.rL;
			var reportNamesDb = scheduler.reportNamesDb;
			var reportLabelsDb =  scheduler.reportLabelsDb;
			// We need the index of the reportName in rN of the profile
			for (var i = 0; i < reportNames.length; i++) {
				var reportNameIndex = reportNames[i];
				if (reportNamesDb[reportNameIndex] == reportName) {
					var reportLabelIndex = reportLabels[i];
					reportLabel = reportLabelsDb[reportLabelIndex];
				}
			}
		}
		if (reportLabel == '') {
			reportLabel = langVar('lang_admin.scheduler.unknown_report');
		}
		return reportLabel;
	}
	function labelBuilder(actionItem) {
		// Invoked from listcontrollerB
		// Returns a text label with HTML markup
		// util.showObject(actionItem);
		var action = actionItem.type;
		var label = scheduler.actionsDb[util.h(action)].label;
        var finalLabel = '';
        var subLabel = '(';
		//
		// Create the label which is displayed within the span element
		//
        if (action != 'execute_command_line') {
            var profilesDb = scheduler.profilesDb;
            var profileItem = null;
            var profile = actionItem.profile;
            var profileLabel = '';
            if (profile != '*' && !actionItem.profile_is_pattern) {
                profileItem = profilesDb[util.h(profile)];
                profileLabel = (profileItem != null) ? profileItem.label : langVar('lang_admin.scheduler.unknown_profile');
            }
            else {
                if (!actionItem.profile_is_pattern) {
                    profileLabel = langVar('lang_admin.scheduler.all_profiles_label');
                }
                else {
                    profileLabel = profile;
                }
            }
            subLabel += profileLabel + ', ';
            if (action == 'export_csv_table' ||
                action == 'generate_report_files' ||
                action == 'send_report_by_email') {
                var reportName = actionItem.options.rn;
                var reportLabel = getReportLabel(profile, reportName);
                subLabel += reportLabel + ', ';
                // KHP-RC, show date filter label if any
                if (action == 'send_report_by_email') {
                    var recipientAddress = actionItem.options.rca;
                    subLabel += recipientAddress + ', ';
                }
            }
            subLabel = subLabel.replace(/(,\s)$/, ')');
            finalLabel = label + ' <span>' + subLabel + '</span>';
        }
        else {
            // execute_command_line
            // TODO - check if we should add a part of the command_line as sublabel
            finalLabel = label;
        }
		return finalLabel;
	}
	return {
		init: init,
		setDateFilter: setDateFilter,
		updateDateFilterText: updateDateFilterText,
		getQueryFieldsDataResponse: getQueryFieldsDataResponse,
		newItem: newItem,
		editItem: editItem,
		duplicateItem: duplicateItem,
		labelBuilder: labelBuilder,
		importReportFilterItemsAndGroups: importReportFilterItemsAndGroups
	}
}());
//
// schedulerDF.js (Scheduler date filter)
//
var schedulerDF = {
	panel: null,
	validator: null,
	init: function() {
		var YE = YAHOO.util.Event;
		var panelObj = {
			panelId: 'sdf:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: langVar('lang_admin.scheduler.date_filter.edit_report_date'),
			left: 400,
			top: 190,
			zIndex: 40,
			isCover: true,
			isSticky: true,
			closeEvent: schedulerDF.closePanel
		};
		schedulerDF.panel = new util.Panel3(panelObj);
		schedulerDF.validator = new util.Validator();
		YE.addListener('sdf:meta_type', 'change', schedulerDF.setDateFilterTypeByEvent);
		YE.addListener('sdf:relative_date_unit', 'change', schedulerDF.setDateUnitByEvent);
		YE.addListener('sdf:okay_btn', 'click', schedulerDF.save);
		YE.addListener('sdf:cancel_btn', 'click', schedulerDF.closePanel);
	},
	open: function(activeDF) {
		// Reset form fields
		util.setF('sdf:relative_date_unit_count', 1);
		util.setF('sdf:relative_date_unit', 'year');
		util.setF('sdf:relative_prefix', 'recent');
		util.setF('sdf:custom_date_filter', '');
		schedulerDF.updateForm(activeDF);
		schedulerDF.panel.open();
	},
	setDateFilterTypeByEvent: function() {
		schedulerDF.setDateFilterTypeByName(this.value);
	},
	setDateFilterTypeByName: function(metaType) {
		// dateType is:  none | relative_with_prefix | relative_without_prefix | custom
		schedulerDF.validator.reset();
		util.setF('sdf:meta_type', metaType);
		util.hideE(['sdf:group:none', 'sdf:group:relative', 'sdf:group:custom']);
		if (metaType == 'none') {
			util.showE('sdf:group:none');
		}
		else if (metaType == 'custom') {
			util.showE('sdf:group:custom');
		}
		else {
			var isWithPrefix = (metaType == 'relative_with_prefix');
			util.showE('sdf:relative_with_prefix:label', isWithPrefix);
			util.showE('sdf:relative_without_prefix:label', !isWithPrefix);
			util.showEV('sdf:relative_prefix', isWithPrefix);
			util.enableE('sdf:relative_prefix', isWithPrefix);
			if (isWithPrefix) {
				// update date unit in sdf:relative_prefix element
				schedulerDF.updateRelativePrefixDateUnit();
			}
			util.showE('sdf:group:relative');
		}
	},
	setDateUnitByEvent: function() {
		if (util.getF('sdf:meta_type') == 'relative_with_prefix') {
			schedulerDF.updateRelativePrefixDateUnit();
		}
	},
	updateRelativePrefixDateUnit: function() {
		// Updates the date unit in
		// 'including __DATE__UNIT__ of schedule date' and
		// 'excluding __DATE__UNIT__ of schedule date'
		var dateUnit = util.getF('sdf:relative_date_unit');
		var dateUnitLabelsSingular = scheduler.dateUnitLabelsSingular;
		var dateUnitLabel = dateUnitLabelsSingular[dateUnit];
		// 'including __DATE__UNIT__ of schedule execution date';
		var recentPrefixText = langVar('lang_admin.scheduler.date_filter.including_date_unit_of_exec');
		// 'excluding __DATE__UNIT__ of schedule execution date'
		var lastPrefixText = langVar('lang_admin.scheduler.date_filter.excluding_date_unit_of_exec');
		recentPrefixText = recentPrefixText.replace(/__PARAM__1/, dateUnitLabel);
		lastPrefixText = lastPrefixText.replace(/__PARAM__1/, dateUnitLabel);
		var list = [{name:'recent', label:recentPrefixText}, {name:'last', label:lastPrefixText}];
		// Get activePrefix which is set again after we re-populate the list 
		var activePrefix = util.getF('sdf:relative_prefix');
		util.populateSelect('sdf:relative_prefix', list, 'name', 'label');
		util.setF('sdf:relative_prefix', activePrefix);
	},
	updateForm: function(activeDF) {
		// util.showObject(activeDF);
		var type = activeDF.type;
		var metaType = '';
		if (type == 'none') {
			metaType = type;
		}
		else if (type == 'relative') {
			var isPrefix = (activeDF.relativePrefix != 'none');
			var metaType = isPrefix ? 'relative_with_prefix' : 'relative_without_prefix';
			util.setF('sdf:relative_date_unit_count', activeDF.relativeDateUnitCount);
			util.setF('sdf:relative_date_unit', activeDF.relativeDateUnit);
			if (isPrefix) {
				util.setF('sdf:relative_prefix', activeDF.relativePrefix);
			}
		}
		else {
			// custom
			metaType = type;
			util.setF('sdf:custom_date_filter', activeDF.custom);
		}
		schedulerDF.setDateFilterTypeByName(metaType);
	},
	closePanel: function() {
		schedulerDF.panel.close();
	},
	save: function() {
		var validator = schedulerDF.validator;
		validator.reset();
		var type = '';
		var custom = '';
		var relativePrefix = '';
		var relativeDateUnit = '';
		var relativeDateUnitCount = '1';
		var metaType = util.getF('sdf:meta_type');
		if (metaType == 'none') {
			type = metaType;
		}
		else if (metaType == 'custom') {
			type = metaType;
			custom = validator.isValue('sdf:custom_date_filter');
		}
		else {
			// Relative
			var isPrefix = (metaType == 'relative_with_prefix');
			// alert('isPrefix: ' + isPrefix);
			type = 'relative';
			relativePrefix = isPrefix ? util.getF('sdf:relative_prefix') : 'none';
			relativeDateUnit = util.getF('sdf:relative_date_unit');
			relativeDateUnitCount = validator.isInteger('sdf:relative_date_unit_count', 1);
		}
		if (validator.allValid()) {
			var newDateFilter = {
				type: type,
				custom: custom,
				relativePrefix: relativePrefix,
				relativeDateUnit: relativeDateUnit,
				relativeDateUnitCount: relativeDateUnitCount
			};
			// util.showObject(newDateFilter);
			schedulerDF.closePanel();
			// Update date filter and date text in Actions form
			schedulerActions.setDateFilter(newDateFilter);
			schedulerActions.updateDateFilterText();
		}
	}
}
//
// schedulerRun.js (Scheduler - Run Now)
//
//
// Note, any action added in Run Now keeps its state as of the time it has been added.
// If the same action is edited in the Actions from right afterwards and the action
// has not yet been run it would not affect the action in Run Now!
var schedulerRun = {
	panel: null,
	isOpenPanel: false,
	idCount: 0, // Used for element baseIds
	schedules: [], // Keeps schedule objects to be run, i.e.
	/*
		{
		id: 's0', 			--> ID issued in schedulerRun
		name: 'schedule A', --> The name we got from the schedule form
		actions: [] 		--> A clone of the actions at the time the user clicked Run Now
		},
		...
		Actions rows and elements contain a combined id as follows:
		scheduleId:actionId:..., i.e.:
		s0:i0:...
	*/
	basketId: '', // Used to check sync for the current basket array between client and server
	basket: [], // A flat array with schedule and action references to schedules array,
						// it is populated upon start()
	isRunning: false,
	isCancel: false,
	isComplete: false,
	runId: '', // Used for responses to check if the response is in sync with server per action
	runningBasketIndex: 0, // The index of the running action in the basket index.
							// Used upon cancel when schedule is running to indicate Canceled state
	// runningScheduleId: '',
	// runningActionId: '',
	// TempTimeMeasurement
	// startTime: 0,
	init: function() {
		var YE = YAHOO.util.Event;
		//
		// Init Run Now panel
		//
		var panelObj = {
			panelId: 'scheduler_run_now:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: 'Run Now',
			right: 30,
			top: 5,
			zIndex: 5,
			isCover: false,
			isSticky: true,
			closeEvent: schedulerRun.cancel
		};
		schedulerRun.panel = new util.Panel3(panelObj);
		// Add single event to list element for remove items
		YE.addListener('scheduler_run_now:list:table', 'click', schedulerRun.removeActionItem);
		YE.addListener('scheduler_run_now:start_btn', 'click', schedulerRun.start);
		YE.addListener('scheduler_run_now:cancel_btn', 'click', schedulerRun.cancel);
	},
	setPanelState: function(panelState) {
		// panelState is: empty | start | running | complete
		var listLabel = '';
		var cancelBtnLabel = '';
		var isEnabledStartBtn = false;
		switch (panelState) {
			case 'start':
				listLabel = langVar('lang_admin.scheduler.run.click_start_to_run_schedules');
				cancelBtnLabel = langVar('lang_stats.btn.cancel');
				isEnabledStartBtn = true;
				break;
			case 'running':
				listLabel = langVar('lang_admin.scheduler.run.running_schedules');
				cancelBtnLabel = langVar('lang_stats.btn.cancel');
				isEnabledStartBtn = false;
				break;
			case 'complete':
				listLabel = langVar('lang_admin.scheduler.run.run_schedule_completed');
				cancelBtnLabel = langVar('lang_stats.btn.close');
				isEnabledStartBtn = false;
				break;
			case 'cancel': 
				// Canceled while an action is running
				listLabel = langVar('lang_admin.scheduler.run.canceled_schedule_info');
				cancelBtnLabel = langVar('lang_stats.btn.close');
				isEnabledStartBtn = false;
				break;
		}
		util.updateT('scheduler_run_now:list:label', listLabel);
		util.enableE('scheduler_run_now:start_btn', isEnabledStartBtn);
		// var cancelBtn = util.getE('scheduler_run_now:cancel_btn');
		// cancelBtn.value = cancelBtnLabel;
		util.updateT('scheduler_run_now:cancel_btn', cancelBtnLabel);
	},
	reset: function() {
		// Reset items
		var sr = schedulerRun;
		sr.idCount = 0;
		sr.schedules = [];
		sr.basket = [];
		sr.isRunning = false;
		sr.isCancel = false;
		sr.isComplete = false;
		// Clear list items
		util.removeChildElements('scheduler_run_now:list:table');
	},
	cancel: function() {
		// Invoked upon cancel or close
		if (schedulerRun.isRunning) {
			// Cancel without to close the panel
			// alert('Schedule is still running. The current running action will be completed in the background.');
			schedulerRun.isCancel = true;
			schedulerRun.setPanelState('cancel');
			// Indicated remaining actions as canceled
			var runningBasketIndex = schedulerRun.runningBasketIndex;
			var basket = schedulerRun.basket;
			for (var i = (runningBasketIndex + 1); i < basket.length; i++) {
				var baseId = basket[i].baseId;
				var stateElement = util.getE(baseId + ':state');
				util.removeChildElements(stateElement);
				var stateText = util.createT(langVar('lang_admin.scheduler.run.canceled'));
				stateElement.appendChild(stateText);
			}
		}
		else {
			// Close the panel
			schedulerRun.reset();
			schedulerRun.setPanelState('start');
			schedulerRun.panel.close();
			enableRunNowButton();
			schedulerRun.isOpenPanel = false;
		}
	},
	runNow: function() {
		// Note, runNow adds items to be run but does not run them yet!
		// Check if we have any actions defined
		var numOfActions = schedulerActionsList.list.getNumberOfItems();
		if (numOfActions > 0) {
			if (!schedulerRun.panel) {
				schedulerRun.init();
			}
			// Check if we need to reset an open panel
			if (schedulerRun.isOpenPanel && (schedulerRun.isCancel || schedulerRun.isComplete)) {
				// Panel is open with a completed or cancled list and 
				// the user adds a new schedule. So we start over.
				schedulerRun.reset();
				schedulerRun.setPanelState('start');
			}
			// alert('runNow() - numOfActions: ' + numOfActions);
			// util.disableE('scheduler:run_now_btn');
			disableRunNowButton();
			//
			// Create schedule object
			//
			var scheduleId = schedulerRun.getNewBaseId();
			var scheduleName = util.getF('scheduler:label');
			if (scheduleName == '') {
				scheduleName = langVar('lang_admin.scheduler.run.schedule');
			}
			var actions = schedulerActionsList.list.getItemsClone();
			// Add an id to each scheduler action (We use the id for element creation and deletion)
			for (var i = 0; i < actions.length; i++) {
				actions[i].id = 'i' + i;
			}
			var obj = {};
			obj.id = scheduleId;
			obj.name = scheduleName;
			obj.actions = actions;
			// Add schedule object to schedules
			var schedules = schedulerRun.schedules;
			schedules[schedules.length] = obj;
			// Build schedule object in list
			schedulerRun.addScheduleToList(scheduleId, scheduleName, actions);
			// util.showObject(actions);
			if (!schedulerRun.isOpenPanel) {
				schedulerRun.setPanelState('start');
				schedulerRun.panel.open();
				schedulerRun.isOpenPanel = true;
			}
		}
		else {
			alert(langVar('lang_admin.scheduler.run.no_action_defined_msg'));
		}
	},
	start: function() {
		// Start to run the schedule(s)
		// Check if there are any actions in the list as all could have been removed!
		// Create basket array with schedule and action references for simple sequential processing
		// Hide the remove buttons in schedule now list
		var schedules = schedulerRun.schedules;
		schedulerRun.basket = [];
		var basket = schedulerRun.basket;
		for (var i = 0; i < schedules.length; i++) {
			var scheduleId = schedules[i].id;
			var actions = schedules[i].actions;
			for (var j = 0; j < actions.length; j++) {
				var actionId = actions[j].id;
				var baseId = scheduleId + ':' + actionId;
				basket[basket.length] = {baseId:baseId, action:actions[j]};
				util.hideEV(baseId + ':remove');
			}
		}
		if (basket.length > 0) {
			// Issue new basketId
			var theDate = new Date();
			var basketId = 'basket_' + theDate.getTime();
			schedulerRun.basketId = basketId;
			disableRunNowButton();
			schedulerRun.isRunning = true;
			// schedulerRun.runId = schedulerRun.runId + 1;
			// update panel
			schedulerRun.setPanelState('running');
			schedulerRun.runScheduleAction(0);
		}
		else {
			alert('lang_admin.scheduler.run.no_schedule_defined_msg');
		}
	},
	runScheduleAction: function(basketIndex) {
		schedulerRun.runningBasketIndex = basketIndex;
		// TempTimeMeasurement
		/*
		var startDate = new Date();
		schedulerRun.startTime = startDate.getTime();
		*/
		var basketItem = schedulerRun.basket[basketIndex];
		var baseId = basketItem.baseId;
		var action = basketItem.action;
		//
		// Handle state cell
		//
		var stateElement = util.getE(baseId + ':state');
		util.removeChildElements(stateElement);
		var stateText = util.createT(langVar('lang_admin.scheduler.run.running'));
		stateElement.appendChild(stateText);
		// Issue a new runId
		var theDate = new Date();
		var runId = 'run_' + theDate.getTime();
		schedulerRun.runId = runId;
		// util.showObject(action);
		var url = '?dp=admin_pages.scheduler.run_schedule_start';
		// Assemble URL dat
var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
		dat += 'v.fp.run_id=' + runId + '&';
		dat += 'v.fp.basket_id=' + schedulerRun.basketId + '&';
		dat += 'v.fp.basket_index=' + basketIndex + '&';
		var path = 'v.fp.schedule';
		dat += path + '.disabled=false&';
		dat += path + '.label=&';
		dat += path + '.month=*&';
		dat += path + '.day=*&';
		dat += path + '.hour=0&';
		dat += path + '.minute=0&';
		// Add the action
		var actionPath = path + '.actions.0';
		dat += getActionItemDat(actionPath, action); // called in scheduler.js
		dat = dat.replace(/&$/, '');
		util.serverPost(url, dat);
	},
	runScheduleActionResponse: function(dat) {
		if (dat.runId == schedulerRun.runId && dat.basketId == schedulerRun.basketId) {
			schedulerRun.runScheduleProgress();
		}
	},
	runScheduleProgress: function() {
		var url = '?dp=admin_pages.scheduler.run_schedule_progress';
		// Checks if the current action completed
		var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
		dat += 'v.fp.run_id=' + schedulerRun.runId;
		util.serverPost(url, dat);
	},
	runScheduleProgressResponse: function(dat) {
		// alert('runScheduleProgressResponse()');
		// util.showObject(dat);
		// return false;
		if (dat.runId == schedulerRun.runId && dat.basketId == schedulerRun.basketId) {
			// var isComplete = (dat.isComplete == true);
			if (dat.isComplete) {
				// TempTimeMeasurement
				/*
				var endDate = new Date();
				var endTime = endDate.getTime();
				var startTime = schedulerRun.startTime;
				var deltaTime = endTime - startTime;
				var seconds = deltaTime / 1000;
				var minutes = seconds / 60;
				var msg = 'The action took\nseconds: ' + seconds + '\nminutes: ' +  minutes;
				alert(msg);
				*/
				// processed item data
				var basketIndex = dat.basketIndex;
				var isError = dat.isError;
				var stateText = !isError ? langVar('lang_admin.scheduler.run.complete') : langVar('lang_admin.scheduler.run.error');
				var basket = schedulerRun.basket;
				var baseId = basket[basketIndex].baseId;
				util.updateT(baseId + ':state', stateText);
				if (isError) {
					schedulerRun.handleActionErrorDisplay(baseId, dat.errorId);
				}
				//
				// Handle next basket item
				//
				if (!schedulerRun.isCancel) {
					nextBasketIndex = basketIndex + 1;
					if (nextBasketIndex < basket.length) {
						schedulerRun.runScheduleAction(nextBasketIndex);
					}
					else {
						// All actions have been run. Set completed state
						schedulerRun.setPanelState('complete');
						schedulerRun.isRunning = false;
						schedulerRun.isComplete = true;
						enableRunNowButton();
					}
				}
				else {
					// This must be the running action upon cancel
					// We just set isRunning
					schedulerRun.isRunning = false;
				}
			}
			else {
				// Continue progress calls
				setTimeout('schedulerRun.runScheduleProgress()', 1500);
			}
		}
	},
	handleActionErrorDisplay: function(baseId, errorId) {
		// Change error state column to Red
		var tdState = util.getE(baseId + ':state');
		tdState.style.color = 'Red';
		// Add error message to label cell
		var tdLabel = util.getE(baseId + ':label');
		var br = util.createE('br');
		var span = util.createE('span', {color:'Red'});
		var spanText = util.createT(langVar('lang_admin.scheduler.run.error_occured_msg') + ' ');
		var href = '?dp=alert&ei=' + errorId;
		var a = util.createE('a', {href:href, target:'_blank'});
		var aText = util.createT(langVar('lang_admin.scheduler.run.click_here_to_view_alert'));
		util.chainE([tdLabel, br, [span, spanText], [a, aText]]);
	},
	getNewBaseId: function() {
		var i = schedulerRun.idCount;
		var id = 'srn_' + i;
		schedulerRun.idCount = i + 1;
		return id;
	},
	//
	//
	// List handling functions
	//
	//
	removeActionItem: function(evt) {
		// Invoked upon list click, we have to check if the user actually clicked a link
		var element = evt.target || evt.srcElement;
		var elementId = element.id;
		if (elementId != '' && (elementId.indexOf(':remove') != -1)) {
			var dat = elementId.split(':');
			var scheduleId = dat[0];
			var actionId = dat[1];
			var schedules = schedulerRun.schedules;
			for (var i = 0; i < schedules.length; i++) {
				var theSchedule = schedules[i];
				if (theSchedule.id == scheduleId) {
					var actions = theSchedule.actions;
					var tbody = util.getE(scheduleId + ':tbody');
					if (actions.length > 1) {
						// Remove action from actions array
						for (var j = 0; j < actions.length; j++) {
							if (actions[j].id == actionId) {
								actions.splice(j, 1);
								break;
							}
						}
						// Remove action row
						var tr = util.getE(scheduleId + ':' + actionId + ':tr');
						tbody.removeChild(tr);					
					}
					else {
						// There is only one action, so we remove the entire schedule
						// Remove schedule from schedules array
						schedules.splice(i, 1);
						// Remove schedule from list
						var table = util.getE('scheduler_run_now:list:table');
						table.removeChild(tbody);
					}
				}
			}
		}
	},
	addScheduleToList: function(scheduleId, scheduleName, actions) {
		var table = util.getE('scheduler_run_now:list:table');
		var tbody = util.createE('tbody', {id:scheduleId + ':tbody'});
		// Build header row
		var tr = util.createE('tr', {className:'header'});
		var tdLabel = util.createE('td', {colSpan:3, className:'header'});
		// var tdState = util.createE('td');
		var text = util.createT(scheduleName);
		util.chainE(tbody, [tr, [tdLabel, text]]);
		// Add the action rows
		for (var i = 0; i < actions.length; i++) {
			schedulerRun.addActionRow(tbody, scheduleId, i, actions[i]);
		}
		util.chainE(table, tbody);
	},
	addActionRow: function(tbody, scheduleId, actionIndex, action) {
		actionBaseId = scheduleId + ':' + action.id;
		var tr = util.createE('tr', {id:actionBaseId + ':tr'});
		var th = util.createE('th');
		var thText = util.createT(actionIndex + 1);
		var tdLabel = util.createE('td', {id:actionBaseId + ':label'});
		// Note, the label we get from labelBuilder is a string whcih may
		// contain HTML tags, so we use innerHTML to display the label.
		var label = schedulerActions.labelBuilder(action);
		tdLabel.innerHTML = label;
		// var tdLabelText = util.createT(labelObj.label);
		// util.chainE(tdLabel, tdLabelText);
		/*
		if (labelObj.subLabel != '') {
			var span = util.createE('span', {className:'label'});
			var spanText = util.createT(' ' + labelObj.subLabel);
			util.chainE(tdLabel, span, spanText);
		}
		*/
		var tdState = util.createE('td', {id:actionBaseId + ':state', className:'state'});
		var a = util.createE('a', {id:actionBaseId + ':remove', href:'javascript:;'});
		var aText = util.createT(langVar('lang_stats.btn.remove'));
		util.chainE(tbody, [tr, [th, thText], tdLabel, [tdState, [a, aText]]]);
	}
}
//
// scheduler.js
//
// Note, the pageInfo.section is set to "scheduler" after initialization has been done.
// This ensure that adminConfig.js doesn't check for changes upon exit unless
// everthing is initialized!
/*
DEFINED IN ADMIN MAIN PAGE!
Used by adminConfig.js
var pageInfo = {};
pageInfo.page = '';
pageInfo.saveActive = false;
pageInfo.exitActive = false;
*/
// Define global lang variable
var lang = lang || {};
var scheduler = {
	smtpServerIsDefined: false, // Specifies if an SMTP server is defined in preferences. We use this setting to show a warning message
								// if a send_report_by_email action is defined but no SMTP server has been defined yet.
	activeUserNodeName: '',
	theList: null,
	usersDb: [], // [{name:"user_1", username:"user 1"}, ...] contains user node name with username, required for the list labels where we indicate "(by user 1)"
	actionsDb: [], 	// Contains the action names, labels and profile access rights if there are no unlimited grants, format is:
					// [{name:"send_report_by_email", label:"Send report by email", all_profiles:false, profiles:['profile_1', profile_2]}, ...]
	itemsDb: [],  // items work array
	itemsDbBackup: [], // items array in the state when first loaded or last saved, it is used upon Undo Changes
	profilesDb: [],
	reportNamesDb: [],
	reportLabelsDb: [],
	validator: null,
	// mainButtons;
	saveChangesBtn: null,
	newItemBtn: null,
	deleteBtn: null,
	duplicateBtn: null,
	undoAllChangesBtn: null,
	noItemFormIsActive: false,
	actionsMoveControlIsPositioned: false,
	activeCreatedByUser: '', // keeps the active createdByUser value because it is not a form field
	dateUnitLabelsSingular: {
		year: langVar('lang_admin.scheduler.date_filter.year'),
		quarter: langVar('lang_admin.scheduler.date_filter.quarter'),
		month: langVar('lang_admin.scheduler.date_filter.month'),
		week: langVar('lang_admin.scheduler.date_filter.week'),
		day: langVar('lang_admin.scheduler.date_filter.day')
	},
	dateUnitLabelsPlural: {
		year: langVar('lang_admin.scheduler.date_filter.years'),
		quarter: langVar('lang_admin.scheduler.date_filter.quarters'),
		month: langVar('lang_admin.scheduler.date_filter.months'),
		week: langVar('lang_admin.scheduler.date_filter.weeks'),
		day: langVar('lang_admin.scheduler.date_filter.days')
	},
	showTip: false
}
function init() {
	var YE = YAHOO.util.Event;
	scheduler.validator = new util.Validator();
	//
	// init toolbar buttons and form controls
	//
	scheduler.saveChangesBtn = new util.ToolbarButton('save_changes', saveChanges, toolbarButtonsDb);
	scheduler.newItemBtn = new util.ToolbarButton('new_schedule', newItem, toolbarButtonsDb);
	scheduler.duplicateBtn = new util.ToolbarButton('duplicate', duplicateItem, toolbarButtonsDb);
	scheduler.deleteBtn = new util.ToolbarButton('delete', deleteItem, toolbarButtonsDb);
	scheduler.undoAllChangesBtn = new util.ToolbarButton('undo_all_changes', undoAllChanges, toolbarButtonsDb);
	YE.addListener('scheduler:label', 'keyup', updateListAndFormLabel);
	YE.addListener('scheduler:run_now_btn', 'click', schedulerRun.runNow);
	YE.addListener('scheduler:new_action_btn', 'click', schedulerActions.newItem);
	//
	// Ignore/Disable buttons according RBAC
	//
	var permissions = pageInfo.permissions;
	if (permissions.isEdit) {
		if (!permissions.isAdd) {
			scheduler.newItemBtn.disableAndIgnore();
			scheduler.duplicateBtn.disableAndIgnore();
		}
		if (!permissions.isDelete) {
			scheduler.deleteBtn.disableAndIgnore();
		}
		// Register isModifiedPageHandler in adminConfig.js
		// (We don't check for modifications if there is no edit 
		// permission because there is no Save button anyway!)
		adminConfig.getIsModifiedPageHandler = getIsModifiedPage;
	}
	else {
		scheduler.saveChangesBtn.disableAndIgnore();
		scheduler.newItemBtn.disableAndIgnore();
		scheduler.duplicateBtn.disableAndIgnore();
		scheduler.deleteBtn.disableAndIgnore();
		scheduler.undoAllChangesBtn.disableAndIgnore();
	}
	// Init best practice tip
	if (scheduler.showTip) {
		YE.addListener('best_practice_tip:scheduled_tasks', 'click', util.helpWindow.openGeneralHelp);
		YE.addListener('best_practice_tip:do_not_show_again', 'click', toggleDoNotShowAgain);
		YE.addListener('best_practice_tip:hide_btn', 'click', hideBestPracticeSection);
		util.showE('scheduler:best_practice_tip:section');
	}
	// Init help
	util.helpWindow.init('');
	//
	// Create theList object
	//
	scheduler.theList = new listcontroller.List({
		containerElementId: 'item_list_body',
		itemEvent: itemActivated,
		isSwitch1: true,
		isMoveControl: true
	});
	//
	// init Scheduler Actions List, Scheduler Actions,
	// Date Filter Editor and report filters
	//
	schedulerActionsList.init();
	schedulerActions.init();
	schedulerDF.init();
}
function initMainDisplay() {
	util.hideE(['form_section', 'scheduler:no_smtp_server_info', 'loading_info', 'saving_info']);
	var firstItemId = scheduler.theList.getFirstItemId();
	if (firstItemId != null) {
		setItem(firstItemId);
		displayNoItemForm(false);
		handleSmtpServerWarning();
	}
	else {
		// no item exists
		displayNoItemForm(true);
	}
	util.showE('form_section');
}
function getScheduleData() {
	if (!pageInfo.exitActive) {
//		console.log('getScheduleData()');
		// invoked upon editor load
		var url = '?dp=admin_pages.scheduler.get_schedule_data';
		var dat = 'v.fp.page_token=' + pageInfo.pageToken;
		util.serverPost(url, dat);
	}
}
function getScheduleDataResponse(dat) {
	// alert('getScheduleDataResponse()');
	//util.showObject(dat);
	if (!pageInfo.exitActive) {
		// Following lang variables are required by
		// filterItemEditor.js
		lang.weekdays = dat.weekdays;
		lang.hours = dat.hours;
		scheduler.smtpServerIsDefined = dat.smtpServerIsDefined;
		scheduler.activeUserNodeName = dat.activeUserNodeName;
		scheduler.usersDb = dat.usersDb;
		// get itemsDb
		scheduler.itemsDb = dat.scheduleDb;
		scheduler.itemsDbBackup = util.cloneObject(dat.scheduleDb);
		// Get profilesDb
		scheduler.profilesDb = dat.profilesDb;
		scheduler.reportNamesDb = dat.reportNamesDb;
		scheduler.reportLabelsDb = dat.reportLabelsDb;
		scheduler.actionsDb = dat.actionsDb;
		scheduler.showTip = dat.showTip;
		util.createHash(scheduler.usersDb, 'name');
		util.createHash(scheduler.profilesDb, 'name');
		util.createHash(scheduler.actionsDb, 'name');
		// util.showObject(scheduler.profilesDb);
		init();
		scheduler.theList.init(scheduler.itemsDb);
		initMainDisplay();
		// Set final toolbar state
		scheduler.saveChangesBtn.enable();
		scheduler.undoAllChangesBtn.enable();
		scheduler.newItemBtn.enable();
		updateToolbarButtons(); // handles Duplicate and Delete
		adminConfig.setItemListSize();
		YAHOO.util.Event.addListener(window, 'resize', adminConfig.setItemListSize);
		pageInfo.initComplete = true;
	}
}
function displayNoItemForm(isDisplayNoItemForm) {
	if (isDisplayNoItemForm) {
		// util.updateT('item_form_label', '-');
		util.hideE('schedule_form');
		util.showE('no_item_form');
	}
	else {
		util.hideE('no_item_form');
		util.showE('schedule_form');
	}
	scheduler.noItemFormIsActive = isDisplayNoItemForm;
}
function updateToolbarButtons() {
	var isItems = scheduler.theList.isItems();
	scheduler.deleteBtn.enable(isItems);
	scheduler.duplicateBtn.enable(isItems);
}
function updateListAndFormLabel() {
	setTimeout('setListAndFormLabel()', 300);
}
function setListAndFormLabel() {
	var h = util.h;
	var label = util.getF('scheduler:label');
	if (label == '') {
		label = '-';
	}
	// We need to check if the schedule is from another user
	var item = scheduler.theList.getSelectedItem(); 
	var createdByUser = item.dat.created_by_user;
	if (createdByUser != scheduler.activeUserNodeName) {
		var usersDb = scheduler.usersDb;
		var username = (usersDb[h(createdByUser)] != null) ? usersDb[h(createdByUser)].username : langVar('lang_admin.scheduler.unknown_user');
		var byUserNameLabel = ' (' + langVar('lang_admin.general.by_user') + ')';
		byUserNameLabel = byUserNameLabel.replace(/__PARAM__1/, username);
		label += ' ' + byUserNameLabel;
	}
	// util.updateT('item_form_label', label);
	scheduler.theList.updateListLabel(label);
}
function getReportsDb(profileName) {
	// Called from schedulerActions.js, it returns a sorted array to create the report list
	function compareProfileLabels(a, b) {
		var labelA = a.label.toLowerCase();
		var labelB = b.label.toLowerCase();
		if (labelA < labelB) {return -1}
		if (labelA > labelB) {return 1}
		return 0;
	}
	var s = scheduler;
	var profile = s.profilesDb[util.h(profileName)];
	var reportsDb = [];
	if (profile != null) {
		// util.showObject(profile);
		var reportNameIndexes = profile.rN;
		var reportLabelIndexes = profile.rL;
		var reportsWithNumOfReportElements = profile.rWithNumOfReportElements;
		var reportNames = s.reportNamesDb;
		var reportLabels = s.reportLabelsDb;
		// alert('profile for which create a reportsDb: ' + profile.label);
		for (var i = 0; i < reportNameIndexes.length; i++) {
			var nameIndex = reportNameIndexes[i];
			var labelIndex = reportLabelIndexes[i];
			var numberOfReportElements = reportsWithNumOfReportElements[i];
			var name = reportNames[nameIndex];
			var label = reportLabels[labelIndex];
			reportsDb[i] = {name:name, label:label, numberOfReportElements:numberOfReportElements};
		}
		// Sort reportsDb
		reportsDb.sort(compareProfileLabels);
	}
	return reportsDb;
}
function enableRunNowButton() {
	// Function is called from schedulerRun.js as well!
	util.enableE('scheduler:run_now_btn');
}
function disableRunNowButton() {
	// Function is called from schedulerRun.js as well!
	util.disableE('scheduler:run_now_btn');
}
function setItem(itemId) {
	scheduler.theList.selectItem(itemId);
	updateForm();
}
function itemActivated(itemId) {
	if (validateActiveItem()) {
		// if selected node item is valid then move to activated node item
		// alert('itemId: ' + itemId);
		setItem(itemId);
	}
}
function newItem() {
	if (validateActiveItem()) {
		var newItemId = scheduler.theList.getNewItemId();
		var newScheduleLabel = langVar('lang_admin.scheduler.new_schedule');
		var newScheduleObj = {
			id: newItemId,
			type: '',
			switch1: true,
			label: newScheduleLabel,
			dat: {
				created_by_user: scheduler.activeUserNodeName,
				label: newScheduleLabel,
				month: '*',
				day: '*',
				hour: 0,
				minute: 0,
				actions: []
			}
		};
		// util.showObject(newScheduleObj);
		scheduler.theList.newItem(newScheduleObj);
		setItem(newItemId);
		updateToolbarButtons();
	}
}
function duplicateItem() {
	if (validateActiveItem()) {
		var theList = scheduler.theList;
		var clonedItemId = theList.cloneItem();
		// Update created_by_user property
		theList.setItemDatValue(clonedItemId, 'created_by_user', scheduler.activeUserNodeName);
		setItem(clonedItemId);
	}
}	
function deleteItem() {
	var theList = scheduler.theList;
	var nextItemIdToBeSelected = theList.deleteItem();
	if (nextItemIdToBeSelected) {
		// select the next item
		// reset the validator in case that the deleted an item with error indication
		scheduler.validator.reset();
		setItem(nextItemIdToBeSelected);
	}
	else {
		// All items have been deleted
		displayNoItemForm(true);
		updateToolbarButtons();
	}
	handleSmtpServerWarning();
}
function updateForm() {
	// util.hideE(['scheduler:run_now:executing', 'scheduler:run_now:complete']);
	if (!schedulerRun.isRunning) {
		enableRunNowButton();
	}
	var item = scheduler.theList.getSelectedItem();
	var itemDat = item.dat;
	scheduler.activeCreatedByUser = itemDat.created_by_user;
	// util.showObject(item);
	var label = itemDat.label;
	util.setF('scheduler:label', label);
	util.setF('month_list', itemDat.month);
	util.setF('day_list', itemDat.day);
	util.setF('hour_list', itemDat.hour);
	util.setF('minute_list', itemDat.minute);
	// Update form label
	// util.updateT('item_form_label', label);
	//
	// Handle Actions
	// 
	schedulerActionsList.list.compose(itemDat.actions);
	if (scheduler.noItemFormIsActive) {
		displayNoItemForm(false);
	}
	// Position the Actions Move Control
	if (!scheduler.actionsMoveControlIsPositioned) {
		// We need to give it some time, else we don't get the right list position
		var s = 'scheduler:actions_list:div';
		setTimeout('schedulerActionsList.moveControl.setPosition("' + s + '")', 250);
		scheduler.actionsMoveControlIsPositioned = true;
	}
}
function handleSmtpServerWarning() {
	// Check if we need to show a no SMTP server warning
	if (!scheduler.smtpServerIsDefined) {
		var showWarning = false;
		// We show a warning message if an email report action exists
		var itemsDb = scheduler.itemsDb;
		// Note, we have to check the actions of the currently selected schedule
		// via schedulerActionsList.list.getItemsClone() because they are not up to
		// date in itemsDb
		var i;
		if (itemsDb.length > 0) {
			var activeItemId = scheduler.theList.getSelectedItemId();
			for (i = 0; i < itemsDb.length; i++) {
				// util.showObject(itemsDb[i]);
				var itemId = itemsDb[i].id;
				if (itemId != activeItemId) {
					var actions = itemsDb[i].dat.actions;
					for (var j = 0; j < actions.length; j++) {
						if (actions[j].type == 'send_report_by_email') {
							showWarning = true;
							break;
						}
					}
				}
			}
		}
		// If there is no warning yet then check the current active actions object
		if (!showWarning) {
			var activeActions = schedulerActionsList.list.getItemsReference();
			// util.showObject(activeActions);
			for (i = 0; i < activeActions.length; i++) {
				if (activeActions[i].type == 'send_report_by_email') {
					showWarning = true;
					break;
				}
			}
		}
		util.showE('scheduler:no_smtp_server_info', showWarning);
	}
}
function validateActiveItem() {
	// Only validate if isEdit permission and if items
	var theList = scheduler.theList;
	if (pageInfo.permissions.isEdit && theList.isItems()) {
		validator = scheduler.validator;
		var o = {};
		o.label = validator.isValue('scheduler:label');
		// Note, labels must be only unique among the schedules of a specific created_by_user value,
		// hence we use getLookupItemsByConstraint() and use the created_by_user value of the current item.
		// util.showObject(theList.getLookupItemsByConstraint('label', 'created_by_user', scheduler.activeCreatedByUser));
		o.label = validator.isUnique('scheduler:label', theList.getLookupItemsByConstraint('label', 'created_by_user', scheduler.activeCreatedByUser));
		o.month = util.getF('month_list');
		o.day = util.getF('day_list');
		o.hour = util.getF('hour_list');
		o.minute = util.getF('minute_list');
		o.created_by_user = scheduler.activeCreatedByUser;
		if (validator.allValid()) {
			// Get a clone of the items in actionsList
			o.actions = schedulerActionsList.list.getItemsClone();
			theList.saveItem(o);
			return true;
		}
		return false;
	}
	// No isEdit permission or no items
	return true;
}
function getDateFilterDat(path, dateFilterObj) {
	var dat = '';
	var dateFilterType = dateFilterObj.type;
	for (var prop in dateFilterObj) {
		dat += path + '.date_filter.' + prop + '=' + util.getEncodedURIComponent(dateFilterObj[prop]) + '&';
	}
	// Get final df for options
	if (dateFilterType != 'none') {
		var df = '';
		if (dateFilterType == 'relative') {
			var relativePrefix = dateFilterObj.relative_prefix;
			var relativeDateUnit = dateFilterObj.relative_date_unit;
			var relativeDateUnitCount = dateFilterObj.relative_date_unit_count;
			if (relativePrefix != 'none') {
				df += relativePrefix;
			}
			df += relativeDateUnitCount + relativeDateUnit;
			if (relativeDateUnitCount > 1) {
				// Use plural form for date unit in df by adding 's'
				df += 's';
			}
		}
		else {
			// custom
			df = dateFilterObj.custom;
		}
		dat += path + '.options.df=' + util.getEncodedURIComponent(df) + '&';
	}
	return dat;
}
function getReportFilterItemsDat(path, filterItems) {
//	util.showObject(filterItems);
	var dat = '';
	var numItems = filterItems.length;
	if (numItems > 0) {
		for (var i = 0; i < numItems; i++) {
			var item = filterItems[i];
			dat += repFiltersUtil.getFilterItemDat(path + '.report_filters.filter_items.' + i, item, false);
		}
	}
	else {
		dat = path + '.report_filters.filter_items=&';
	}
//	console.log('filterItems: ' + filterItems);
	return dat;
}
function getReportFilterGroupsDat(path, filterGroups) {
	var dat = '';
	var numGroupItems = filterGroups.length;
	if (numGroupItems > 0) {
		for (var i = 0; i < numGroupItems; i++) {
			var groupItem = filterGroups[i];
			var filterItems = groupItem.items;
			var numFilterItems = filterItems.length;
			var groupPath = path + '.report_filters.filter_groups.' + i;
			dat += groupPath + '.name=' + encodeURIComponent(groupItem.name) + '&';
			dat += groupPath + '.label=' + encodeURIComponent(groupItem.label) + '&';
			dat += groupPath + '.is_active=' + groupItem.isActive + '&';
			if (numFilterItems > 0) {
				for (var j = 0; j < numFilterItems; j++) {
					var itemPath = groupPath + '.filter_items.' + j;
					dat += repFiltersUtil.getFilterItemDat(itemPath, filterItems[j], true);
				}
			}
			else {
				dat += groupPath + '.filter_items=&';
			}
		}
	}
	else {
		dat = path + '.report_filters.filter_groups=&';
	}
	return dat;
}
function getActionItemDat(path, actionItem, position) {
	// Creates the output dat string for a single action item upon saveChanges()
	var handleDateFilter = false;
	var dat = path + '.position=' + position + '&';
	for (var prop in actionItem) {
		switch (prop) {
			case 'date_filter':
				dat += getDateFilterDat(path, actionItem.date_filter);
				break;
			case 'report_filters':
				dat +=  path + '.report_filters.filter_id=' + actionItem.report_filters.filter_id + '&';
				dat += getReportFilterItemsDat(path, actionItem.report_filters.filter_items);
				dat += getReportFilterGroupsDat(path, actionItem.report_filters.filter_groups);
				break;
			case 'options':
				var options = actionItem.options;
				for (var optProp in options) {
					dat += path + '.options.' + optProp + '=' + util.getEncodedURIComponent(options[optProp]) + '&';
				}
				break;
			case 'filter_info':
				var filterInfo = actionItem.filter_info;
				for (var fiInProp in filterInfo) {
					dat += path + '.filter_info.' + fiInProp + '=' + util.getEncodedURIComponent(filterInfo[fiInProp]) + '&';
				}
				break;
			default:
				dat += path + '.' + prop + '=' + util.getEncodedURIComponent(actionItem[prop]) + '&';
		}
	}
	return dat;
}
function saveChanges() {
	if (validateActiveItem()) {
		var theList = scheduler.theList;
		var isModified = theList.getIsModified();
		// alert('isModified: ' + isModified);
		if (isModified) {
			util.hideE('form_section');
			util.showE('saving_info');
			var itemsDb = scheduler.itemsDb;
			var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
			if (itemsDb.length > 0) {
				for (var i = 0; i < itemsDb.length; i++) {
					// Assemble the schedule
					var path = 'v.fp.schedule.' + i;
					var scheduleItem = itemsDb[i];
					var scheduleItemDat = scheduleItem.dat;
					dat += path + '.position=' + i + '&'; // Used to sort nodes on server side
					dat += path + '.disabled=' + !scheduleItem.switch1 + '&';
					for (var prop in scheduleItemDat) {
						if (prop != 'actions') {
							dat += path + '.' + prop + '=' + util.getEncodedURIComponent(scheduleItemDat[prop]) + '&';
						}
						else {
							// Handle the actions
							var actions = scheduleItemDat.actions;
							if (actions.length > 0) {
								for (var j = 0; j < actions.length; j++) {
									var actionPath =  path + '.actions.' + j;
									dat += getActionItemDat(actionPath, actions[j], j);
								}
							}
							else {
								// No action exists. A schedule without a single action is allowed.
								dat += path + '.actions=&';
							}
						}
					}
				}
			}
			else {
				// No schedule exists
				dat += 'v.fp.schedule=&';
			}
			dat = dat.replace(/&$/, '');
			// alert(dat);
			pageInfo.saveActive = true;
			var url = '?dp=admin_pages.scheduler.save_data';
			util.serverPost(url, dat);
		}
		else {
			alert(langVar('lang_stats.general.no_changes_to_save'));
		}
	}
}
function saveChangesResponse() {
	// reset isModified
	scheduler.theList.resetIsModified();
	// the saved work itemsDb becomes now the itemsDbBackup
	scheduler.itemsDbBackup = util.cloneObject(scheduler.itemsDb);
	pageInfo.saveActive = false;
	util.hideE('saving_info');
	util.showE('form_section');
}
function undoAllChanges() {
	// alert('Undo Changes\nitemsDb length: ' + itemsDb.length + '\nitemsDbBackup length: ' + itemsDbBackup.length);
	scheduler.itemsDb = util.cloneObject(scheduler.itemsDbBackup);
	scheduler.validator.reset();
	scheduler.theList.init(scheduler.itemsDb);
	initMainDisplay();
	updateToolbarButtons();
}
function getIsModifiedPage() {
	// Note, isModified will be false if only the active item has been edited
	// but has an error, hence we also check "if (!validateActiveItem() ..."
	return (!validateActiveItem() || scheduler.theList.getIsModified());
}
// Best practice tip utilities
function toggleDoNotShowAgain(evt) {
	// If checked this changes the text of the Hide Messages link
	// to "Hide And Don't Show Again"
	var text = '';
	if (this.checked) {
		text = langVar('lang_stats.btn.hide_and_do_not_show_again');
	}
	else {
		text = langVar('lang_stats.btn.hide_message');
	}
	util.updateT('best_practice_tip:hide_btn', text);
}
function hideBestPracticeSection() {
	// If don't show again is checked set appropriate system node
	if (util.getF('best_practice_tip:do_not_show_again')) {
		var url = '?dp=util.set_and_save_node';
		var dat = 'v.fp.active_page=' + pageInfo.page;
		dat += '&v.fp.page_token=' + pageInfo.pageToken;
		dat += '&v.fp.main_node_name=users_cache_preferences';
		dat += '&v.fp.new_nodes.show_tips.scheduler=false';
		util.serverPost(url, dat);
	}
	util.hideE('scheduler:best_practice_tip:section');
}
