//
// scheduler.js
//


// Note, the pageInfo.section is set to "scheduler" after initialization has been done.
// This ensure that adminConfig.js doesn't check for changes upon exit unless
// everthing is initialized!

/*
DEFINED IN ADMIN MAIN PAGE!
Used by adminConfig.js
var pageInfo = {};
pageInfo.page = '';
pageInfo.saveActive = false;
pageInfo.exitActive = false;
*/


// Define global lang variable
var lang = lang || {};

var scheduler = {
	
	smtpServerIsDefined: false, // Specifies if an SMTP server is defined in preferences. We use this setting to show a warning message
								// if a send_report_by_email action is defined but no SMTP server has been defined yet.
	activeUserNodeName: '',
	theList: null,
	
	usersDb: [], // [{name:"user_1", username:"user 1"}, ...] contains user node name with username, required for the list labels where we indicate "(by user 1)"
	actionsDb: [], 	// Contains the action names, labels and profile access rights if there are no unlimited grants, format is:
					// [{name:"send_report_by_email", label:"Send report by email", all_profiles:false, profiles:['profile_1', profile_2]}, ...]
	itemsDb: [],  // items work array
	itemsDbBackup: [], // items array in the state when first loaded or last saved, it is used upon Undo Changes
	profilesDb: [],
	reportNamesDb: [],
	reportLabelsDb: [],
	validator: null,
	
	// mainButtons;
	saveChangesBtn: null,
	newItemBtn: null,
	deleteBtn: null,
	duplicateBtn: null,
	undoAllChangesBtn: null,
	
	noItemFormIsActive: false,
	actionsMoveControlIsPositioned: false,
	
	activeCreatedByUser: '', // keeps the active createdByUser value because it is not a form field
	
	dateUnitLabelsSingular: {
		year: langVar('lang_admin.scheduler.date_filter.year'),
		quarter: langVar('lang_admin.scheduler.date_filter.quarter'),
		month: langVar('lang_admin.scheduler.date_filter.month'),
		week: langVar('lang_admin.scheduler.date_filter.week'),
		day: langVar('lang_admin.scheduler.date_filter.day')
	},
		
	dateUnitLabelsPlural: {
		year: langVar('lang_admin.scheduler.date_filter.years'),
		quarter: langVar('lang_admin.scheduler.date_filter.quarters'),
		month: langVar('lang_admin.scheduler.date_filter.months'),
		week: langVar('lang_admin.scheduler.date_filter.weeks'),
		day: langVar('lang_admin.scheduler.date_filter.days')
	},

	showTip: false
}

function init() {
	
	var YE = YAHOO.util.Event;
	
	scheduler.validator = new util.Validator();
	
	//
	// init toolbar buttons and form controls
	//
	
	scheduler.saveChangesBtn = new util.ToolbarButton('save_changes', saveChanges, toolbarButtonsDb);
	scheduler.newItemBtn = new util.ToolbarButton('new_schedule', newItem, toolbarButtonsDb);
	scheduler.duplicateBtn = new util.ToolbarButton('duplicate', duplicateItem, toolbarButtonsDb);
	scheduler.deleteBtn = new util.ToolbarButton('delete', deleteItem, toolbarButtonsDb);
	scheduler.undoAllChangesBtn = new util.ToolbarButton('undo_all_changes', undoAllChanges, toolbarButtonsDb);
	
	YE.addListener('scheduler:label', 'keyup', updateListAndFormLabel);
	YE.addListener('scheduler:run_now_btn', 'click', schedulerRun.runNow);
	YE.addListener('scheduler:new_action_btn', 'click', schedulerActions.newItem);
	
	
	//
	// Ignore/Disable buttons according RBAC
	//
	
	var permissions = pageInfo.permissions;
	
	if (permissions.isEdit) {
		
		if (!permissions.isAdd) {
			scheduler.newItemBtn.disableAndIgnore();
			scheduler.duplicateBtn.disableAndIgnore();
		}
		
		if (!permissions.isDelete) {
			scheduler.deleteBtn.disableAndIgnore();
		}
		
		// Register isModifiedPageHandler in adminConfig.js
		// (We don't check for modifications if there is no edit 
		// permission because there is no Save button anyway!)
		adminConfig.getIsModifiedPageHandler = getIsModifiedPage;
	}
	else {
		scheduler.saveChangesBtn.disableAndIgnore();
		scheduler.newItemBtn.disableAndIgnore();
		scheduler.duplicateBtn.disableAndIgnore();
		scheduler.deleteBtn.disableAndIgnore();
		scheduler.undoAllChangesBtn.disableAndIgnore();
	}

	// Init best practice tip
	if (scheduler.showTip) {

		YE.addListener('best_practice_tip:scheduled_tasks', 'click', util.helpWindow.openGeneralHelp);
		YE.addListener('best_practice_tip:do_not_show_again', 'click', toggleDoNotShowAgain);
		YE.addListener('best_practice_tip:hide_btn', 'click', hideBestPracticeSection);

		util.showE('scheduler:best_practice_tip:section');
	}

	
	// Init help
	util.helpWindow.init('');
	
	//
	// Create theList object
	//
	
	scheduler.theList = new listcontroller.List({
		containerElementId: 'item_list_body',
		itemEvent: itemActivated,
		isSwitch1: true,
		isMoveControl: true
	});
	
	//
	// init Scheduler Actions List, Scheduler Actions,
	// Date Filter Editor and report filters
	//
	
	schedulerActionsList.init();
	schedulerActions.init();
	schedulerDF.init();
}

function initMainDisplay() {
	
	util.hideE(['form_section', 'scheduler:no_smtp_server_info', 'loading_info', 'saving_info']);
	
	var firstItemId = scheduler.theList.getFirstItemId();
	
	if (firstItemId != null) {
		
		setItem(firstItemId);
		displayNoItemForm(false);
		
		handleSmtpServerWarning();
	}
	else {
		// no item exists
		displayNoItemForm(true);
	}
	
	util.showE('form_section');
}

function getScheduleData() {
	
	if (!pageInfo.exitActive) {
		
//		console.log('getScheduleData()');
		
		// invoked upon editor load
		var url = '?dp=admin_pages.scheduler.get_schedule_data';
		var dat = 'v.fp.page_token=' + pageInfo.pageToken;
		util.serverPost(url, dat);
	}
}

function getScheduleDataResponse(dat) {
	
	// alert('getScheduleDataResponse()');
	
	//util.showObject(dat);

	if (!pageInfo.exitActive) {

		// Following lang variables are required by
		// filterItemEditor.js
		lang.weekdays = dat.weekdays;
		lang.hours = dat.hours;
		
		scheduler.smtpServerIsDefined = dat.smtpServerIsDefined;
		scheduler.activeUserNodeName = dat.activeUserNodeName;
		scheduler.usersDb = dat.usersDb;
		
		// get itemsDb
		scheduler.itemsDb = dat.scheduleDb;
		scheduler.itemsDbBackup = util.cloneObject(dat.scheduleDb);
		
		// Get profilesDb
		scheduler.profilesDb = dat.profilesDb;
		scheduler.reportNamesDb = dat.reportNamesDb;
		scheduler.reportLabelsDb = dat.reportLabelsDb;
		
		scheduler.actionsDb = dat.actionsDb;

		scheduler.showTip = dat.showTip;
		
		util.createHash(scheduler.usersDb, 'name');
		util.createHash(scheduler.profilesDb, 'name');
		util.createHash(scheduler.actionsDb, 'name');
		
		// util.showObject(scheduler.profilesDb);
		
		init();
		scheduler.theList.init(scheduler.itemsDb);
		initMainDisplay();
		
		// Set final toolbar state
		scheduler.saveChangesBtn.enable();
		scheduler.undoAllChangesBtn.enable();
		scheduler.newItemBtn.enable();
		updateToolbarButtons(); // handles Duplicate and Delete
		
		adminConfig.setItemListSize();
		YAHOO.util.Event.addListener(window, 'resize', adminConfig.setItemListSize);
		
		pageInfo.initComplete = true;
	}
}

function displayNoItemForm(isDisplayNoItemForm) {
	
	if (isDisplayNoItemForm) {
		// util.updateT('item_form_label', '-');
		util.hideE('schedule_form');
		util.showE('no_item_form');
	}
	else {
		util.hideE('no_item_form');
		util.showE('schedule_form');
	}

	scheduler.noItemFormIsActive = isDisplayNoItemForm;
}

function updateToolbarButtons() {
	
	var isItems = scheduler.theList.isItems();
	scheduler.deleteBtn.enable(isItems);
	scheduler.duplicateBtn.enable(isItems);
}

function updateListAndFormLabel() {
	
	setTimeout('setListAndFormLabel()', 300);
}

function setListAndFormLabel() {
	
	var h = util.h;
	var label = util.getF('scheduler:label');
	if (label == '') {
		label = '-';
	}
	
	// We need to check if the schedule is from another user
	var item = scheduler.theList.getSelectedItem(); 
	var createdByUser = item.dat.created_by_user;
	
	if (createdByUser != scheduler.activeUserNodeName) {
		
		var usersDb = scheduler.usersDb;
		var username = (usersDb[h(createdByUser)] != null) ? usersDb[h(createdByUser)].username : langVar('lang_admin.scheduler.unknown_user');
		var byUserNameLabel = ' (' + langVar('lang_admin.general.by_user') + ')';
		byUserNameLabel = byUserNameLabel.replace(/__PARAM__1/, username);
		label += ' ' + byUserNameLabel;
	}
	
	// util.updateT('item_form_label', label);
	scheduler.theList.updateListLabel(label);
}

function getReportsDb(profileName) {
	
	// Called from schedulerActions.js, it returns a sorted array to create the report list
	
	function compareProfileLabels(a, b) {
		var labelA = a.label.toLowerCase();
		var labelB = b.label.toLowerCase();
		if (labelA < labelB) {return -1}
		if (labelA > labelB) {return 1}
		return 0;
	}
	
	var s = scheduler;
	
	var profile = s.profilesDb[util.h(profileName)];
	var reportsDb = [];
	
	if (profile != null) {
		
		// util.showObject(profile);
	
		var reportNameIndexes = profile.rN;
		var reportLabelIndexes = profile.rL;
		var reportsWithNumOfReportElements = profile.rWithNumOfReportElements;
		var reportNames = s.reportNamesDb;
		var reportLabels = s.reportLabelsDb;
		
		// alert('profile for which create a reportsDb: ' + profile.label);
		
		for (var i = 0; i < reportNameIndexes.length; i++) {
			
			var nameIndex = reportNameIndexes[i];
			var labelIndex = reportLabelIndexes[i];
			var numberOfReportElements = reportsWithNumOfReportElements[i];
			
			var name = reportNames[nameIndex];
			var label = reportLabels[labelIndex];
			
			reportsDb[i] = {name:name, label:label, numberOfReportElements:numberOfReportElements};
		}
		
		// Sort reportsDb
		reportsDb.sort(compareProfileLabels);
	}
	
	return reportsDb;
}

function enableRunNowButton() {
	// Function is called from schedulerRun.js as well!
	util.enableE('scheduler:run_now_btn');
}

function disableRunNowButton() {
	// Function is called from schedulerRun.js as well!
	util.disableE('scheduler:run_now_btn');
}

function setItem(itemId) {

	scheduler.theList.selectItem(itemId);
	updateForm();
}

function itemActivated(itemId) {
	
	if (validateActiveItem()) {
		
		// if selected node item is valid then move to activated node item
		// alert('itemId: ' + itemId);
		setItem(itemId);
	}
}

function newItem() {
	
	if (validateActiveItem()) {
	
		var newItemId = scheduler.theList.getNewItemId();
		var newScheduleLabel = langVar('lang_admin.scheduler.new_schedule');
		
		var newScheduleObj = {
			id: newItemId,
			type: '',
			switch1: true,
			label: newScheduleLabel,
			dat: {
				created_by_user: scheduler.activeUserNodeName,
				label: newScheduleLabel,
				month: '*',
				day: '*',
				hour: 0,
				minute: 0,
				actions: []
			}
		};
		
		// util.showObject(newScheduleObj);
		scheduler.theList.newItem(newScheduleObj);
		setItem(newItemId);
		updateToolbarButtons();
	}
}

function duplicateItem() {
	
	if (validateActiveItem()) {
		
		var theList = scheduler.theList;
		var clonedItemId = theList.cloneItem();
		
		// Update created_by_user property
		theList.setItemDatValue(clonedItemId, 'created_by_user', scheduler.activeUserNodeName);
		
		setItem(clonedItemId);
	}
}	

function deleteItem() {
	
	var theList = scheduler.theList;
	
	var nextItemIdToBeSelected = theList.deleteItem();
	
	if (nextItemIdToBeSelected) {
		
		// select the next item
		// reset the validator in case that the deleted an item with error indication
		scheduler.validator.reset();
		setItem(nextItemIdToBeSelected);
	}
	else {
		
		// All items have been deleted
		displayNoItemForm(true);
		updateToolbarButtons();
	}
	
	handleSmtpServerWarning();
}

function updateForm() {

	// util.hideE(['scheduler:run_now:executing', 'scheduler:run_now:complete']);
	
	if (!schedulerRun.isRunning) {
		enableRunNowButton();
	}
	
	var item = scheduler.theList.getSelectedItem();
	var itemDat = item.dat;
	
	scheduler.activeCreatedByUser = itemDat.created_by_user;
	
	// util.showObject(item);
	var label = itemDat.label;
		
	util.setF('scheduler:label', label);
	util.setF('month_list', itemDat.month);
	util.setF('day_list', itemDat.day);
	util.setF('hour_list', itemDat.hour);
	util.setF('minute_list', itemDat.minute);

	
	// Update form label
	// util.updateT('item_form_label', label);
	
	
	//
	// Handle Actions
	// 
	
	schedulerActionsList.list.compose(itemDat.actions);
	
	if (scheduler.noItemFormIsActive) {
		displayNoItemForm(false);
	}
	
	// Position the Actions Move Control
	if (!scheduler.actionsMoveControlIsPositioned) {
		
		// We need to give it some time, else we don't get the right list position
		var s = 'scheduler:actions_list:div';
		setTimeout('schedulerActionsList.moveControl.setPosition("' + s + '")', 250);
		scheduler.actionsMoveControlIsPositioned = true;
	}
}

function handleSmtpServerWarning() {
	
	// Check if we need to show a no SMTP server warning
	if (!scheduler.smtpServerIsDefined) {
		
		var showWarning = false;
		
		// We show a warning message if an email report action exists
		var itemsDb = scheduler.itemsDb;
		
		// Note, we have to check the actions of the currently selected schedule
		// via schedulerActionsList.list.getItemsClone() because they are not up to
		// date in itemsDb
		
		var i;
		
		if (itemsDb.length > 0) {
			
			var activeItemId = scheduler.theList.getSelectedItemId();
			
			for (i = 0; i < itemsDb.length; i++) {
				
				// util.showObject(itemsDb[i]);
				var itemId = itemsDb[i].id;
				
				if (itemId != activeItemId) {
				
					var actions = itemsDb[i].dat.actions;
					
					for (var j = 0; j < actions.length; j++) {
						
						if (actions[j].type == 'send_report_by_email') {
							showWarning = true;
							break;
						}
					}
				}
			}
		}
		
		// If there is no warning yet then check the current active actions object
		
		if (!showWarning) {
			
			var activeActions = schedulerActionsList.list.getItemsReference();
			
			// util.showObject(activeActions);
			
			for (i = 0; i < activeActions.length; i++) {
				
				if (activeActions[i].type == 'send_report_by_email') {
					showWarning = true;
					break;
				}
			}
		}
		
		util.showE('scheduler:no_smtp_server_info', showWarning);
	}
}

function validateActiveItem() {
	
	// Only validate if isEdit permission and if items
	
	var theList = scheduler.theList;
	
	if (pageInfo.permissions.isEdit && theList.isItems()) {
		
		validator = scheduler.validator;
		
		var o = {};
		o.label = validator.isValue('scheduler:label');
		
		// Note, labels must be only unique among the schedules of a specific created_by_user value,
		// hence we use getLookupItemsByConstraint() and use the created_by_user value of the current item.
		
		// util.showObject(theList.getLookupItemsByConstraint('label', 'created_by_user', scheduler.activeCreatedByUser));
		
		o.label = validator.isUnique('scheduler:label', theList.getLookupItemsByConstraint('label', 'created_by_user', scheduler.activeCreatedByUser));
		o.month = util.getF('month_list');
		o.day = util.getF('day_list');
		o.hour = util.getF('hour_list');
		o.minute = util.getF('minute_list');
		
		o.created_by_user = scheduler.activeCreatedByUser;
		
		if (validator.allValid()) {
			
			// Get a clone of the items in actionsList
			o.actions = schedulerActionsList.list.getItemsClone();
	
			theList.saveItem(o);
			return true;
		}
	
		return false;
	}
	
	// No isEdit permission or no items
	return true;
}

function getDateFilterDat(path, dateFilterObj) {

	var dat = '';
	var dateFilterType = dateFilterObj.type;

	for (var prop in dateFilterObj) {

		dat += path + '.date_filter.' + prop + '=' + util.getEncodedURIComponent(dateFilterObj[prop]) + '&';
	}

	// Get final df for options

	if (dateFilterType != 'none') {

		var df = '';

		if (dateFilterType == 'relative') {

			var relativePrefix = dateFilterObj.relative_prefix;
			var relativeDateUnit = dateFilterObj.relative_date_unit;
			var relativeDateUnitCount = dateFilterObj.relative_date_unit_count;

			if (relativePrefix != 'none') {
				df += relativePrefix;
			}

			df += relativeDateUnitCount + relativeDateUnit;

			if (relativeDateUnitCount > 1) {
				// Use plural form for date unit in df by adding 's'
				df += 's';
			}
		}
		else {
			// custom
			df = dateFilterObj.custom;
		}

		dat += path + '.options.df=' + util.getEncodedURIComponent(df) + '&';
	}

	return dat;
}

function getReportFilterItemsDat(path, filterItems) {

//	util.showObject(filterItems);

	var dat = '';
	var numItems = filterItems.length;

	if (numItems > 0) {

		for (var i = 0; i < numItems; i++) {

			var item = filterItems[i];
			dat += repFiltersUtil.getFilterItemDat(path + '.report_filters.filter_items.' + i, item, false);
		}
	}
	else {

		dat = path + '.report_filters.filter_items=&';
	}

//	console.log('filterItems: ' + filterItems);

	return dat;
}

function getReportFilterGroupsDat(path, filterGroups) {

	var dat = '';
	var numGroupItems = filterGroups.length;

	if (numGroupItems > 0) {

		for (var i = 0; i < numGroupItems; i++) {

			var groupItem = filterGroups[i];
			var filterItems = groupItem.items;
			var numFilterItems = filterItems.length;
			var groupPath = path + '.report_filters.filter_groups.' + i;

			dat += groupPath + '.name=' + encodeURIComponent(groupItem.name) + '&';
			dat += groupPath + '.label=' + encodeURIComponent(groupItem.label) + '&';
			dat += groupPath + '.is_active=' + groupItem.isActive + '&';

			if (numFilterItems > 0) {

				for (var j = 0; j < numFilterItems; j++) {

					var itemPath = groupPath + '.filter_items.' + j;
					dat += repFiltersUtil.getFilterItemDat(itemPath, filterItems[j], true);
				}
			}
			else {

				dat += groupPath + '.filter_items=&';
			}
		}
	}
	else {

		dat = path + '.report_filters.filter_groups=&';
	}

	return dat;
}

function getActionItemDat(path, actionItem, position) {
	
	// Creates the output dat string for a single action item upon saveChanges()
	
	var handleDateFilter = false;
	
	var dat = path + '.position=' + position + '&';

	for (var prop in actionItem) {

		switch (prop) {

			case 'date_filter':

				dat += getDateFilterDat(path, actionItem.date_filter);
				break;

			case 'report_filters':

				dat +=  path + '.report_filters.filter_id=' + actionItem.report_filters.filter_id + '&';
				dat += getReportFilterItemsDat(path, actionItem.report_filters.filter_items);
				dat += getReportFilterGroupsDat(path, actionItem.report_filters.filter_groups);
				break;

			case 'options':

				var options = actionItem.options;

				for (var optProp in options) {

					dat += path + '.options.' + optProp + '=' + util.getEncodedURIComponent(options[optProp]) + '&';
				}

				break;

			case 'filter_info':

				var filterInfo = actionItem.filter_info;
				for (var fiInProp in filterInfo) {
					dat += path + '.filter_info.' + fiInProp + '=' + util.getEncodedURIComponent(filterInfo[fiInProp]) + '&';
				}

				break;

			default:

				dat += path + '.' + prop + '=' + util.getEncodedURIComponent(actionItem[prop]) + '&';
		}
	}

	return dat;
}


function saveChanges() {
	
	if (validateActiveItem()) {
		
		var theList = scheduler.theList;
		var isModified = theList.getIsModified();
		
		// alert('isModified: ' + isModified);

		if (isModified) {
			
			util.hideE('form_section');
			util.showE('saving_info');
			
			var itemsDb = scheduler.itemsDb;
			
			var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
			
			if (itemsDb.length > 0) {
			
				for (var i = 0; i < itemsDb.length; i++) {
					
					// Assemble the schedule
					
					var path = 'v.fp.schedule.' + i;
					var scheduleItem = itemsDb[i];
					var scheduleItemDat = scheduleItem.dat;
					
					dat += path + '.position=' + i + '&'; // Used to sort nodes on server side
					dat += path + '.disabled=' + !scheduleItem.switch1 + '&';
					
					for (var prop in scheduleItemDat) {
						
						if (prop != 'actions') {
						
							dat += path + '.' + prop + '=' + util.getEncodedURIComponent(scheduleItemDat[prop]) + '&';
						}
						else {
							
							// Handle the actions
							
							var actions = scheduleItemDat.actions;
							
							if (actions.length > 0) {
								
								for (var j = 0; j < actions.length; j++) {
									
									var actionPath =  path + '.actions.' + j;
									dat += getActionItemDat(actionPath, actions[j], j);
								}
							}
							else {
								// No action exists. A schedule without a single action is allowed.
								dat += path + '.actions=&';
							}
						}
					}
				}
			}
			else {
				// No schedule exists
				dat += 'v.fp.schedule=&';
			}
			
			dat = dat.replace(/&$/, '');
			
			// alert(dat);
			
			pageInfo.saveActive = true;
			
			var url = '?dp=admin_pages.scheduler.save_data';
			util.serverPost(url, dat);
		}
		else {
			
			alert(langVar('lang_stats.general.no_changes_to_save'));
		}
	}
}

function saveChangesResponse() {
	
	// reset isModified
	scheduler.theList.resetIsModified();
	
	// the saved work itemsDb becomes now the itemsDbBackup
	scheduler.itemsDbBackup = util.cloneObject(scheduler.itemsDb);
	
	pageInfo.saveActive = false;
	
	util.hideE('saving_info');
	util.showE('form_section');
}

function undoAllChanges() {
	
	// alert('Undo Changes\nitemsDb length: ' + itemsDb.length + '\nitemsDbBackup length: ' + itemsDbBackup.length);
	
	scheduler.itemsDb = util.cloneObject(scheduler.itemsDbBackup);
	
	scheduler.validator.reset();
	scheduler.theList.init(scheduler.itemsDb);
	initMainDisplay();
	updateToolbarButtons();
}

function getIsModifiedPage() {
	
	// Note, isModified will be false if only the active item has been edited
	// but has an error, hence we also check "if (!validateActiveItem() ..."
	return (!validateActiveItem() || scheduler.theList.getIsModified());
}

// Best practice tip utilities

function toggleDoNotShowAgain(evt) {

	// If checked this changes the text of the Hide Messages link
	// to "Hide And Don't Show Again"
	var text = '';

	if (this.checked) {
		text = langVar('lang_stats.btn.hide_and_do_not_show_again');
	}
	else {
		text = langVar('lang_stats.btn.hide_message');
	}

	util.updateT('best_practice_tip:hide_btn', text);
}

function hideBestPracticeSection() {

	// If don't show again is checked set appropriate system node
	if (util.getF('best_practice_tip:do_not_show_again')) {

		var url = '?dp=util.set_and_save_node';
		var dat = 'v.fp.active_page=' + pageInfo.page;
		dat += '&v.fp.page_token=' + pageInfo.pageToken;
		dat += '&v.fp.main_node_name=users_cache_preferences';
		dat += '&v.fp.new_nodes.show_tips.scheduler=false';

		util.serverPost(url, dat);
	}

	util.hideE('scheduler:best_practice_tip:section');
}




