//
//
// listcontrollerB.js
//
//

//
// Note, row numbers are counted as array object indexes, starting from 0, 1, 2, ...
//
// Element Id handling per row
// i.e. base ID prefix is "wil" and baseId is "wil8"
// <tr id="wil8:tr"> --> We need the ID in tr to get the actual row number because the number, i.e. "8", does not specify the row number, it is only a unique number among all rows!
// <th><a id="wil8:index" --> the index column
// <td><a id="wil8:label" ...> --> the main label column
// <td id="wil8:td:edit"><a id="wil8:a:edit" --> the edit button
// <td id="wil8:td:duplicate"><a id="wil8:a:duplicate" --> the duplicate button
// <td id="wil8:td:delete"> <a id="wil8:a:delete" --> the delete button

// We added the id to the td cell as well so that we catch any cell event,
// especially for the index column where we Select/Deselect the row

// Log Filters List Notes
// In case of multiple wide item list objects (such as in log filters) 
// we create different HTML lists. IMPORTANT, there is a single list in
// log filters (If, Then, Else), though they are actually three different list objects,
// respectively there exist three different tbody elements as container!

var listcontrollerB = {};

listcontrollerB.List = function(obj) {

    // this.tableElementId = obj.tableElementId;
    var listName = obj.name;
    this.name = listName; // We require the name as identifier for multiple tbody lists such as in log filters
                          // We use the name also for the baseIdPrefix!
    this.tbodyElementId = obj.tbodyElementId;
    this.noItemText = obj.noItemText;
    this.isDefaultItem = obj.isDefaultItem; // Creates a default item row if items length is 0, with the text of noItemText (as i.e. in Scheduler)
                                            // If isDefaultItem is false then no default item row will be created (as i.e. in Log filters).

    this.customSequence = obj.customSequence ? obj.customSequence : []; // Array with a custom sequence for row indication, i.e. ['A', 'B', 'C', ...]
                                                                        // So far it is currently used in the log filter condition list

    this.isCustomSequence = (this.customSequence.length > 0);

    this.listChangeNotificationFunction = obj.listChangeNotificationFunction; // We call this function in the callee to handle move control state and multiple lists in the callee.
                                                      // The listChangeNotificationFunction will be invoked
                                                      // a.) upon selection/de-selection
                                                      // b.) when adding, duplicating or deleting an item
    this.editItemFunction = obj.editItemFunction;
    this.duplicateItemFunction = obj.duplicateItemFunction;
    this.labelBuilderFunction = obj.labelBuilderFunction;

    // baseIdPrefix
    // We require the baseIdPrefix in case of multiple wideItemLists such as in log filters,
    // where we require a separate list for If/Then/Else expressions.
    this.baseIdPrefix = 'wil_' + listName.toLowerCase() + '_';

    this.selectedRowIndex = -1; // The index of the active selected row

    this.tbody = util.getE(obj.tbodyElementId);
    this.items = [];


    // Assign the list event
    YAHOO.util.Event.addListener(obj.tbodyElementId, 'click', this.listItemActivated, this);

    // alert('listcontrollerB - object has been created');
};

listcontrollerB.List.prototype = {

    listItemActivated: function(evt, self) {

        var element = evt.target || evt.srcElement;
        var elementId = element.id;

        // alert('itemListActivated - elementId: ' + elementId);
        // alert('element.nodeName: ' + element.nodeName);

        var tagName = element.nodeName;

        // alert('tagName: ' + tagName);


        if (tagName != 'TBODY' && tagName != 'TR' && tagName != 'TD') {

            var parentElement;

            //
            // Get the row number of the selected item
            //

            var rowId = '';
            var rowElement;

            // Get the row
            parentElement = element.parentNode;

            while (rowId == '') {

                if (parentElement.nodeName == 'TR') {
                    rowId = parentElement.id;
                    rowElement = parentElement;
                }
                else {
                    parentElement = parentElement.parentNode;
                }
            }

            // alert('rowId: ' + rowId);

            //
            // Get the activeItemIndex (is equal row number)
            //

            var activeItemIndex = 0;
            var rows = self.tbody.getElementsByTagName('tr');
            for (var i = 0; i < rows.length; i++) {
                // alert('loop through rows - row id: ' + rows[i].id);
                if (rows[i].id == rowId) {
                    activeItemIndex = i;
                    break;
                }
            }


            if (tagName == 'TH') {

                //
                // Select/Deselect row
                //

                // alert('Select/Deselect row');

                if (activeItemIndex == self.selectedRowIndex) {
                    // Row is already selected, deselect it
                    rowElement.className = 'list-120';
                    self.selectedRowIndex = -1;
                    newSelectedRowIndex = -1;
                }
                else {

                    // Deselect any other selected row
                    self.deselectRow();

                    // Select the clicked row
                    rowElement.className = 'list-120-select';
                    self.selectedRowIndex = activeItemIndex;
                }

                self.sendListChangeNotification();
            }
            else {

                //
                // Get the anchor ID of the clicked item in case
                // that the clicked element isn't the anchor element
                //

                var anchorElementId = '';

                if (tagName == 'A') {
                    anchorElementId = element.id;
                }
                else {

                    // We check for a parent anchor. In case the user clicked into a spacer
                    // column we limit the check up to the TR parent element

                    parentElement = element.parentNode;

                    if (parentElement.nodeName == 'A') {

                        anchorElementId = parentElement.id;
                    }
                    else {

                        // alert('parentElement.nodeName 1: ' + parentElement.nodeName);

                        while (anchorElementId == '' && parentElement.nodeName != 'TR') {

                            if (parentElement.nodeName == 'A') {
                                anchorElementId = parentElement.id;
                            }
                            else {
                                parentElement = parentElement.parentNode;
                            }

                            // alert('parentElement.nodeName 2: ' + parentElement.nodeName);
                        }
                    }
                }

                // alert('anchorElementId: ' + anchorElementId);

                //
                // Check if we select a row, edit, duplicate or delete
                //

                if (anchorElementId != '') {

                    var dat = anchorElementId.split(':');
                    var baseId = dat[0];
                    var operationType = dat[1];

                    var items = self.items;
                    var theItem;

                    switch (operationType) {

                        case 'label':

                            theItem = (items.length > 0) ? items[activeItemIndex] : {};
                            self.editItemFunction(activeItemIndex, theItem);

                            break;

                        case 'duplicate':

                            // verify that there is 1 item in the list, respectievly that we don't duplicate the Not Defined item!
                            if (items.length >= 1) {
                                // alert('duplicate item');
                                // Note, we send the activeItemIndex to the duplicateItemFunction panel,
                                // a new itemIndex is only given when the item is saved
                                theItem = items[activeItemIndex];

                                // Clone the item because we may modify data such as the label
                                // in the function we call from here, so we can't send "theItem"
                                var duplicateItemObj = util.cloneObject(theItem);

                                self.duplicateItemFunction(activeItemIndex, duplicateItemObj);
                            }

                            break;


                        case 'delete':

                            // verify that there is more than 1 item in the list
                            // Update, this depends if we have a default row or not.
                            // alert('delete - !self.isDefaultItem: ' + !(self.isDefaultItem));

                            if (items.length > 1 || !self.isDefaultItem) {
                                self.deleteItem(activeItemIndex, baseId);
                            }

                            break;

                        // default:

                        // operationType is label or edit, both invoke edit
                        // If this is an undefined item then use an empty object for the item
                        // theItem = (items.length > 0) ? items[activeItemIndex] : {};
                        // self.editItemFunction(activeItemIndex, theItem);

                        // break;
                    }
                }
            }
        }
    },

    deselectRow: function() {

        // Deselects any selected row
        if (this.selectedRowIndex != -1) {

            var baseIdOfAlreadySelectedRow = this.getBaseIdByRowNumber(this.selectedRowIndex);
            var alreadySelectedRowElement = util.getE(baseIdOfAlreadySelectedRow + ':tr');
            alreadySelectedRowElement.className = 'list-120';
            this.selectedRowIndex = -1;
        }
    },

    sendListChangeNotification: function(trigger /* optional */) {

        // trigger: delete | move
        trigger = trigger || '';

        // TEMP
        /*
        var s = 'sendListChangeNotification()';
        s += '\nthis.name: ' + this.name;
        s += '\nthis.selectedRowIndex: ' + this.selectedRowIndex;
        s += '\nthis.items.length: ' + this.items.length;
        alert(s);
        */

        this.listChangeNotificationFunction(
            this.name,
            this.selectedRowIndex,
            this.items.length,
            trigger);
    },

    getBaseIdByRowNumber: function(itemIndex) {

        // Returns the baseId (i.e.: "wil8") for the given itemIndex

        var rows = this.tbody.getElementsByTagName('tr');
        var tr = rows[itemIndex];
        var trId = tr.id;
        var dat = trId.split(':');
        var baseId = dat[0];

        return baseId;
    },

    updateLabel: function(baseId, label) {

        // Updates the label of the given baseId with label
        // Note, label is a simple text string or string with HTML markup, i.e.:
        // "No item "
        // "Build database, <span>(profile name)</span>"

        // Add edit image to label string
        var editLabel = langVar('lang_stats.btn.edit');
        var imgString = '<img src="' + imgDb.listEditItem.src + '" width="16" height="16" alt="' + editLabel + '" title="' + editLabel + '" /> ';

        var a = util.getE(baseId + ':label');

        // Remove existing label
        while (a.firstChild != null)  {
            var y = a.firstChild;
            a.removeChild(y);
        }

        a.innerHTML = imgString + label;
    },


    compose: function(items) {

        // Compose creates/updates the list for a new list session
        // Items is an array with objects, i.e. scheduler actions.

        // alert('compose');

        // Reset row selection
        this.selectedRowIndex = -1;

        var numberOfItems = items.length;
        this.items = (numberOfItems > 0) ? util.cloneObject(items) : [];

        // alert('numberOfItems: ' + numberOfItems);

        // if items has 0 length we know that there is no item defined,
        // yet we create a row for it, with the label as given in this.noItemText
        // UPDATE - above is only true if this.isDefaultItem is true

        var numberOfRequiredRows = numberOfItems > 0 ? numberOfItems : 1;

        if (numberOfItems > 0) {
            numberOfRequiredRows = numberOfItems;
        }
        else {
            numberOfRequiredRows = this.isDefaultItem ? 1 : 0;
        }


        //
        // Check if we display the right number of rows
        //

        this.checkRows(numberOfRequiredRows);

        //
        // Update rows text and row selection
        //

        // util.showObject(items);

        var baseId;
        var itemLabel = '';
        var tr;

        if (numberOfItems > 0) {

            for (var i = 0; i < numberOfItems; i++) {

                baseId = this.getBaseIdByRowNumber(i);
                itemLabel = this.labelBuilderFunction(items[i]);
                this.updateLabel(baseId, itemLabel);

                tr = util.getE(baseId + ':tr');
                tr.className = 'list-120';
            }

        }
        else if (this.isDefaultItem) {

            // There is no item yet but we created a row, so we give it a "No item defined" label
            baseId = this.getBaseIdByRowNumber(0); // We get the baseId for row 0
            var noItemLabel = '[ ' + this.noItemText + ' ]';
            this.updateLabel(baseId, noItemLabel);

            tr = util.getE(baseId + ':tr');
            tr.className = 'list-120';
        }

        this.updateListButtonsState();
        this.sendListChangeNotification();
    },

    updateListButtonsState: function() {

        // Updates the state of the duplicate and delete button in each row.
        var rows = this.tbody.getElementsByTagName('tr');

        for (var i = 0; i < rows.length; i++) {

            var rowId = rows[i].id;
            var dat = rowId.split(':');
            var baseId = dat[0];

            var duplicateAnchor = util.getE(baseId + ':duplicate');
            var duplicateImg = duplicateAnchor.firstChild;
            var deleteAnchor = util.getE(baseId + ':delete');
            var deleteImg = deleteAnchor.firstChild;

            if (i > 0 || !this.isDefaultItem) {
                duplicateImg.src = imgDb.listDuplicateItem.src;
                deleteImg.src = imgDb.listDeleteItem.src;
            }
            else {
                // Handle first row for lists which always have a default item row
                duplicateImg.src = this.items.length > 0 ? imgDb.listDuplicateItem.src : imgDb.listDuplicateItemDis.src;
                deleteImg.src = rows.length > 1 ? imgDb.listDeleteItem.src : imgDb.listDeleteItemDis.src;
            }

            /*
            var duplicateAnchor = util.getE(baseId + ':a:duplicate');
            var deleteAnchor = util.getE(baseId + ':a:delete');

            if (i > 0 || !this.isDefaultItem) {
                duplicateAnchor.className = '';
                deleteAnchor.className = '';
            }
            else {
                // Handle first row for lists which always have a default item row
                duplicateAnchor.className = this.items.length > 0 ? '' : 'disabled';
                deleteAnchor.className = rows.length > 1 ? '' : 'disabled';
            }
            */
        }
    },

    saveItem: function(mode, itemIndex, freshItem) {

        // freshItem is a new object without any reference to this.items
        // mode is: edit | new | duplicate
        // itemIndex refers to the item we have edited or duplicated, it
        // is only relevant for edit and duplicate mode.

        // if mode is "edit" we replace the existing item with the freshItem
        // if mode is "new" we add the freshItem below any selected item or at the end
        // if mode is "duplicate" we add the freshItem next to the duplicated item, in this case the itemIndex specifies the item which we duplicated.

        // alert('saveItem - mode: ' + mode);
        // alert('saveItem - itemIndex: ' + itemIndex);

    //	console.log('mode: ' + mode);
    //	console.log('itemIndex: ' + itemIndex);
    //	util.showObject(freshItem, 'freshItem');

        var items = this.items;

    //	util.showObject(items, 'items before saving freshItem');

        // Get a new label for the fresh item
        var itemLabel = this.labelBuilderFunction(freshItem);

        var baseId = '';
        var tbody = this.tbody;
        var tr = null;

        if (mode == 'edit') {

            // replace existing item
            items.splice(itemIndex, 1, freshItem);

            // Get the baseId of the row
            baseId = this.getBaseIdByRowNumber(itemIndex);
        }
        else if (mode == 'new') {

            // insert the item as last item by now

            // Get new itemIndex where we insert the item
            itemIndex = items.length;

            // If no item exists yet we simply modify the existing 'No item defined' row (if isDefaultItem only),
            // else we add a new row.

            if (itemIndex == 0 && this.isDefaultItem) {

                // The row already exists
                baseId = this.getBaseIdByRowNumber(0);
            }
            else {

                // We need to add a new row
                tr = this.createRow(itemIndex + 1);
                tbody.appendChild(tr);
                baseId = this.getBaseIdByRowNumber(itemIndex);
            }

            // Add the item to items
            items[itemIndex] = freshItem;
        }
        else {

            // mode = 'duplicate'

            // Insert the newItem as next item of the given itemIndex
            // If the duplicated item becomes the last item in the list
            // then we can use createRow() to add an row, else we insert
            // a new row by insertRow()

            // Create the row
            tr = this.createRow(itemIndex + 2); // The rowNumber (itemIndex + 2) is only valid when adding the row as last row

            // If the source item is the last item
            if (itemIndex == (items.length - 1)) {

                // Add row as last row
                tbody.appendChild(tr);

                // Add freshItem to array
                items[itemIndex + 1] = freshItem;
            }
            else {

                // Insert the row before the next row
                var baseIdOfNextRow = this.getBaseIdByRowNumber(itemIndex + 1);
                var nextTr = util.getE(baseIdOfNextRow + ':tr');
                tbody.insertBefore(tr, nextTr);

                // Insert fresh item after the item given by itemIndex
                items.splice(itemIndex + 1, 0, freshItem);

                // Update the row numbers display
                this.updateRowNumbersDisplay();
            }

            baseId = this.getBaseIdByRowNumber(itemIndex + 1);
        }

        //
        // Update the label
        //
        this.updateLabel(baseId, itemLabel);
        this.updateListButtonsState();
        this.sendListChangeNotification();
    },

    deleteItem: function(itemIndex, baseId) {

        // Delete item in this.items and the row
        var items = this.items;
        items.splice(itemIndex, 1);

        var tr = util.getE(baseId + ':tr');
        this.tbody.removeChild(tr);

        this.updateRowNumbersDisplay();
        this.updateListButtonsState();

        // Handle row selection
        var selectedRowIndex = this.selectedRowIndex;
        if (selectedRowIndex != -1) {

            if (selectedRowIndex == itemIndex) {
                // We deleted the selected row
                // alert('deleted selected row, this.selectedRowIndex becomes -1');
                this.selectedRowIndex = -1;
            }
            else if (itemIndex < selectedRowIndex) {
                // A row before the selected row has been deleted,
                // so we need to subtract 1 from this.selectedRowIndex
                // alert('deleted a row before the selected row, this.selectedRowIndex becomes this.selectedRowIndex MINUS 1');
                this.selectedRowIndex = selectedRowIndex - 1;
            }

            // alert('this.selectedRowIndex: ' + this.selectedRowIndex);
        }

        this.sendListChangeNotification('delete');
    },

    moveItem: function(direction) {

        // Temp
        /*
        var s = 'moveItem()';
        s += '\nlistName: ' + this.name;
        s += '\ndirection: ' + direction;
        alert(s);
        */

        var selectedItemIndex = this.selectedRowIndex;
        var items = this.items;

        // alert('move item in listcontrollerB: ' + direction);

        var newItemIndex = 0; // new item index position
        var isUpDirection = false;

        switch (direction) {
            case 'top':
                newItemIndex = 0;
                isUpDirection = true;
                break;
            case 'up':
                newItemIndex = selectedItemIndex - 1;
                isUpDirection = true;
                break;
            case 'down':
                newItemIndex = selectedItemIndex + 1;
                break;
            case 'bottom':
                newItemIndex = items.length - 1;
                break;
        }

        var updateRowIndexStart = isUpDirection ? newItemIndex : selectedItemIndex;
        var updateRowIndexEnd = isUpDirection ? selectedItemIndex : newItemIndex;


        //
        // Remove selected item from items
        //
        var movedObjArray = items.splice(selectedItemIndex, 1);
        var movedObj = movedObjArray[0];
        // util.showObject(movedObj);
        // alert('splicedObject isArray: ' + util.isArray(movedObj));

        //
        // Insert removed item at new position
        //
        items.splice(newItemIndex, 0, movedObj);

        //
        // Update the rows display and row selection
        //

        // We need to update all rows between updateRowIndexStart and updateRowIndexEnd

        var rows = this.tbody.getElementsByTagName('tr');


        for (var i = updateRowIndexStart; i <= updateRowIndexEnd; i++) {

            var tr = rows[i];
            var trId = tr.id;
            var dat = trId.split(':');
            var baseId = dat[0];

            this.updateSingleRow(baseId, i, items[i]);

            // Handle row selection
            if (i == selectedItemIndex) {
                tr.className = 'list-120';
            }

            if (i == newItemIndex) {
                tr.className = 'list-120-select';
            }
        }

        this.selectedRowIndex = newItemIndex;
        this.sendListChangeNotification('move');
    },

    checkRows: function(numberOfRequiredRows) {

        var existingRows = this.tbody.getElementsByTagName('tr');
        var numberOfExistingRows = existingRows.length;

        if (numberOfRequiredRows != numberOfExistingRows) {

            // alert('delete or build rows');

            var i;
            var tbody = this.tbody;
            var tr;

            if (numberOfRequiredRows > numberOfExistingRows) {

                // Add new rows
                var numberOfRowsToAdd = numberOfRequiredRows - numberOfExistingRows;
                var rowNumberInDisplay = numberOfExistingRows + 1;

                for (i = 0; i < numberOfRowsToAdd; i++) {
                    tr = this.createRow(rowNumberInDisplay);
                    tbody.appendChild(tr);
                    rowNumberInDisplay++;
                }
            }
            else {

                // Remove rows
                var numberOfRowsToRemove = numberOfExistingRows - numberOfRequiredRows;

                for (i = 0; i < numberOfRowsToRemove; i++) {
                    tr = tbody.lastChild;
                    tbody.removeChild(tr);
                }
            }
        }
    },

    getNumberOfItems: function() {
        // returns the number of items
        return this.items.length;
    },

    getItemsClone: function() {

        // returns a clone of the up to date items in this.items
        var itemsClone;
        var items = this.items;

        if (items.length > 0) {
            itemsClone = util.cloneObject(items);
        }
        else {
            itemsClone = [];
        }

        return itemsClone;
    },

    getItemsReference: function() {
        // returns a reference to this.items
        return this.items;
    },

    getRowNumberText: function(index) {

        // Row numbers may be indicated by numbers or letters
        var rowNumberText = '';

        if (!this.isCustomSequence) {
            rowNumberText = index + 1;
        }
        else {
            // Get the sequence letter
            // rowNumberText = this.customSequence[index];
            rowNumberText = util.getRepetitionSequenceLetter(this.customSequence, index);
        }

        return rowNumberText;
    },

    updateSingleRow: function(baseId, index, item) {

        // Updates all elements of a single row of the given item
        var rowNumberText = this.getRowNumberText(index);
        util.updateT(baseId + ':index', rowNumberText);

        var itemLabel = this.labelBuilderFunction(item);
        this.updateLabel(baseId, itemLabel);
    },

    updateRowNumbersDisplay: function() {

        // Updates the row numbers in the first column

        var ths = this.tbody.getElementsByTagName('th');
        for (var i = 0; i < ths.length; i++) {
            var th = ths[i];
            var oldText = th.firstChild;
            var rowNumberText = this.getRowNumberText(i);
            var newText = util.createT(rowNumberText);
            th.replaceChild(newText, oldText);
        }
    },

    createIndexCell: function(tr, baseId, rowNumberInDisplay) {

        // var th = util.createE('th', {id:baseId + ':th'});

        var id = baseId + ':index';

        var th = util.createE('th', {id:id});
        // var a = util.createE('a', {id:id, className:'list-120', href:'javascript:;'});
        var rowNumberText = this.getRowNumberText(rowNumberInDisplay - 1); // We require the item index, hence -1
        var text = util.createT(rowNumberText);
        util.chainE(tr, th, text);
    },

    createLabelCell: function(tr, baseId) {

        var id = baseId + ':label';
        var td = util.createE('td', {whiteSpace:'normal', width:'90%'});
        var a = util.createE('a', {id:id, href:'javascript:;'});

        util.chainE(tr, td, a);

        /*
        var td = document.createElement('td');
        td.id = baseId + ':td:label';
        td.style.whiteSpace = 'normal';
        td.style.width = '90%';

        tr.appendChild(td);
        */
    },

    createButtonCell: function(tr, baseId, buttonName, buttonLabel) {
        // buttonName duplicate | delete


        var td = document.createElement('td');
        // td.id = baseId + ':td:' + buttonName;

        if (buttonName == 'delete') {
            // Add class name so that we show the right border in the last cell
            td.className = 'delete';
        }

        var id = baseId + ':' + buttonName;
        var a = util.createE('a', {id:id, href:'javascript:;'});

        var imgSrc = (buttonName == 'duplicate') ? imgDb.listDuplicateItem.src : imgDb.listDeleteItem.src;
        var img = util.createE('img', {src:imgSrc, width:16, height:16, alt:buttonLabel, title:buttonLabel});

        util.chainE(tr, td, a, img);

        // var text = document.createTextNode(buttonLabel);

        // a.appendChild(text);
        // td.appendChild(a);
        // tr.appendChild(td);
    },

    getUniqueBaseId: function() {

        // Returns a new unqiue base id by checking for existing baseIds in tr elements
        var baseIdPrefix = this.baseIdPrefix;
        var idCount = 0;
        var isUniqueId = false;
        var newBaseId = '';

        while (!isUniqueId) {

            if (util.getE(baseIdPrefix + idCount + ':tr') == null) {
                newBaseId = baseIdPrefix + idCount;
                isUniqueId = true;
            }

            idCount++;
        }

        return newBaseId;
    },

    createRow: function(rowNumberInDisplay) {

        // Returns a new tr element

        var newBaseId = this.getUniqueBaseId();

        var tr = document.createElement('tr');
        tr.id = newBaseId +  ':tr';
        tr.className = 'list-120';

        this.createIndexCell(tr, newBaseId, rowNumberInDisplay);
        this.createLabelCell(tr, newBaseId);

        // Insert a spacer column
        var td = document.createElement('td');
        td.className = 'spacer';
        var text = document.createTextNode('\u00a0'); // creates &nbsp;
        td.appendChild(text);
        tr.appendChild(td);

        // this.createButtonCell(tr, newBaseId, 'edit', 'Edit');
        this.createButtonCell(tr, newBaseId, 'duplicate', langVar('lang_stats.btn.duplicate'));
        this.createButtonCell(tr, newBaseId, 'delete', langVar('lang_stats.btn.delete'));

        return tr;
    }
};

