/* global store: false */

var profilesStorage = (function() {

	'use strict';

	function getIsUpToDate() {

		// Returns true if the profilesDb is up to date.
		var profilesListChecksum = pageInfo.profilesListChecksum;
		var storedProfilesListChecksum = store.get('profilesListChecksum');

		// ToDo, add time check. If profilesListChecksum in Storage is newer
		// than the one in pageInfo than we can also assume that we are
		// up to date! Add epoc timestamp to pageInfo and profileChanges!

		return (profilesListChecksum === storedProfilesListChecksum);
	}

	function get(property) {

		return store.get(property);
	}

	function addNewProfilesDb(profilesListChecksum, numAllProfiles, numPermittedProfiles, profilesDb) {

		// A new profilesDb has been loaded, update all storage data.

		store.set('profilesListChecksum', profilesListChecksum);
		store.set('numAllProfiles', numAllProfiles);
		store.set('numPermittedProfiles', numPermittedProfiles);
		store.set('profilesDb', profilesDb);

		// Set pageInfo.profilesListChecksum for the case that it is not up to date
		pageInfo.profilesListChecksum = profilesListChecksum;
	}

	function update(profileChanges) {

		// This updates local storage after a profile has been saved.
		// This is necessary because the file modification date causes
		// a new checksum. However, in most cases we only need to update
		// the checksum and the single profile item, so we don't need to
		// reload all profiles.

		var profilesListChecksum = profileChanges.profilesListChecksum;

//		console.log('profilesStorage.update() - new profilesListChecksum: ' + profilesListChecksum);
//		console.log('profilesStorage.update() - profileChanges.isNewProfilesList: ' + profileChanges.isNewProfilesList);

		if (!profileChanges.isNewProfilesList) {

			// This causes a minimal update in local storage.
			var profileItem = profileChanges.profileItem;
			var profileName = profileItem[0];

			var profilesDb = store.get('profilesDb');
			for (var i = 0, len = profilesDb.length; i < len; i++) {
				if (profilesDb[i][0] === profileName) {
					profilesDb.splice(i, 1, profileItem);
				}
			}

			// Save up to date profilesListChecksum
			// and modified profilesDb in storage.

			store.set('profilesListChecksum', profilesListChecksum);
			store.set('profilesDb', profilesDb);

			// Set pageInfo.profilesListChecksum so that it is in sync
			// with storage profilesListChecksum.

			pageInfo.profilesListChecksum = profilesListChecksum;
		}
		else {

			// We loaded a new profilesDb. Most likely profile
			// files have been modified manually or by other users.
			addNewProfilesDb(
				profilesListChecksum,
				profileChanges.numAllProfiles,
				profileChanges.numPermittedProfiles,
				profileChanges.profilesDb
			);
		}
	}

	function reset() {

		// This clears local storage profile data
		var profilesListChecksum = '';
		var numAllProfiles = 0;
		var numPermittedProfiles = 0;
		var profilesDb = [];

		addNewProfilesDb(
			profilesListChecksum,
			numAllProfiles,
			numPermittedProfiles,
			profilesDb
		);
	}

	//
	// General profileArrayItem utilities.
	// This utilities are also used when profilesDb is
	// is not yet saved to local storage.
	//

	function profileArrayItemToObject(a) {

		// Returns a profile item object derived from
		// profile item array a.

		// "a" is an profile item in the format ["profile_1", "profile_2", 0, "", ...]
		// which becomes converted to an object so that all properties can be accessed
		// by name.


		// Database type numbers to database type strings
		// 0 = internal_sql,
		// 1 = mysql,
		// 2 = odbc_mssql
		// 3 = odbc_oracle
		var dbTypes = ['internal_sql', 'mysql', 'odbc_mssql', 'odbc_oracle'];

		return {
			name: a[0],
			label: a[1],
			databaseType: dbTypes[(a[2])],
			databaseName: a[3],
			username: a[4],
			df: a[5],
//			isViewReports: a[6],
//			isViewConfig: a[7],
//			isRename: a[8],
//			isDelete: a[9],
			version: a[6],
			isValidProfile: (a[6] === '')
		};
	}

	function setProfileArrayItemValues(profileArrayItem, obj) {

		// This sets the profileArrayItem with object values at the correct
		// array position.
		// Currently only the 'version' property is supported.

		for (var prop in obj) {

			if (prop === 'version') {
				profileArrayItem[6] = obj[prop];
			}
		}
	}

	// Return global properties and methods
	return {
		getIsUpToDate: getIsUpToDate,
		get: get,
		addNewProfilesDb: addNewProfilesDb,
		update: update,
		reset: reset,
		profileArrayItemToObject: profileArrayItemToObject,
		setProfileArrayItemValues: setProfileArrayItemValues
	};

}());
