/* global
	util: false,
	YAHOO: false */


/**
*
* ToolbarButton class (could eventually replace the CommandLink class, so that we only use one class!)
*
* Creates an object for a toolbar button composed of an anchor with an image.
* The ToolbarButton object considers rbac state by defining a "ignore" method.
* I.e.:, saveBtn.ignore() will hide the button and ignore it any other method
* which is applied so that hoverOn(), disable() or enable() does not have any effect.
* This allows us to code buttons as in non-rbac mode and simply ignore them in case
* that there is no permission to use the saveBtn or any other RBAC feature.

*
* Arguments:
* 	buttonItemName: This is not the element ID but a simple button item name which acts as a referece in buttonsDb and from which we compose the elementId
* 	buttonEvent: The event which is fired upon onclick
* 	buttonsDb: Includes general button properties and all toolbar buttons, it has following format:
*	 buttonsDb = {
*		classNameEnabled: 'btn-10',
*		classNameDisabled: 'btn-10-disabled',
*		classNameHover: 'btn-10-hover',
*   	enabled: {"_save_changes:"new Image(), "_new_role":new Image()},
*   	disabled: {"_save_changes:"new Image(), "_new_role":new Image()},
*	}
*	buttonsDb.enabled._save_changes.src = "/picts/toolbars/save_changes.gif",
*	buttonsDb.enabled._new_role.src = "/picts/toolbars/new_role.gif",
*	buttonsDb.disabled._save_changes.src = "/picts/toolbars/save_changes_dis.gif",
*	buttonsDb.disabled._new_role.src = "/picts/toolbars/new_role_dis.gif"
*
*
*	// Note, all button item Id's start internally with an underbar to avoid any name conflicts
*	// The elementId is a composistion of "toolbar:buttonItemID"
*
*/

(function() {

	'use strict';

//	util.namespace('util');

	util.ToolbarButton = function(buttonName, buttonEvent, buttonsDb) {

		// By default we set all buttons to the disabled state, that's how
		// the html code is specified.

		// ignore=true will ignore this button upon all applied methods.
		// We also set buttonName to true if the button does not exist due
		// RBAC or licensing features. So we must always check for button
		// existence!

		var buttonId = '_' + buttonName;
		var elementId = 'toolbar:' + buttonName;

		var btn = util.getE(elementId);

//		console.log('(btn !== null: ' + (btn === null));


		this.elementId = elementId;
		this.buttonEvent = buttonEvent;
		this.isDisabled = true;
		this.ignore = (btn === null);
		this.classNameEnabled = buttonsDb.classNameEnabled;
		this.classNameDisabled = buttonsDb.classNameDisabled;
		this.classNameHover = buttonsDb.classNameHover;

		// Images are optional
		this.srcEnabled = buttonsDb.enabled[buttonId]['src'];
		this.srcDisabled = buttonsDb.disabled[buttonId]['src'];
	};


	util.ToolbarButton.prototype = {

		enable: function(/* optional */makeEnabled) {

			if (!this.ignore) {

				if (typeof makeEnabled === 'undefined') {
					makeEnabled = true;
				}

				// Only enable the button if it is not yet enabled
				if (makeEnabled) {
					if (this.isDisabled) {

						var btn = util.getE(this.elementId);
						YAHOO.util.Event.addListener(btn, 'click', this.buttonEvent);
						btn.className = this.classNameEnabled;

						var img = btn.firstChild;
						img.src = this.srcEnabled;

						this.isDisabled = false;
					}
				}
				else {
					this.disable();
				}
			}
		},

		disable: function() {

			// Don't disable the button if it is already disabled

			if (!this.ignore && !this.isDisabled) {

				// alert('disable button');
				var btn = util.getE(this.elementId);
				YAHOO.util.Event.removeListener(btn, 'click', this.buttonEvent);
				btn.className = this.classNameDisabled;

				var img = btn.firstChild;
				img.src = this.srcDisabled;

				this.isDisabled = true;
			}
		},

		disableAndIgnore: function() {

			// This sets a button to disabled and ignored state

			if (!this.ignore) {

				// if not yet disavled
				if (!this.isDisabled) {
					this.disable();
				}

				this.ignore = true;
			}
		}

//		hoverOn: function() {
//
//			if (!this.ignore) {
//				// TODO
//			}
//		},
//
//		hoverOff: function() {
//
//			if (!this.ignore) {
//				// TODO
//			}
//		}

	};

}());