/* global
	langVar: false,
	pageInfo: false,
	users: users,
	usersAM: false,
	usersUtil: false,
	util: false */
var usersUtil = (function() {
    'use strict';
	//
	// Access List Utilities
	//
	// Element Id's used in access table rows
	// The baseId, i.e. "acc0" does not specify the row number, it is random!
	// We get the active row number by checking the rows array of tbody!
	// ":op:" within the id stands for operate, so upon click on a cell which contains :op: we do something, else not
	// <tr id="acc0:tr" --> Id of access table row
	// <td id ="acc0:op:edit_profiles" --> Edit Access with profiles tab as default
	// <td id ="acc0:op:edit_roles" --> Edit Access with roles tab as default
	// <td id ="acc0:info" --> Info cell which contains text like "(by user 2)" or ("Owned by other user")
	// <td id ="acc0:op:edit" --> Edit Access with profiles tab as default
	// <td id ="acc0:op:delete" --> Delete the access item
    var isRootAdmin = false;
    var isEnterprise = false;
    var isUnlimitedGrants = false;
    function init() {
        isRootAdmin = pageInfo.isRootAdmin;
        isEnterprise = pageInfo.licensingTier === 'enterprise';
        isUnlimitedGrants = pageInfo.permissions.isUnlimitedGrants;
    }
	function accessListLinkActivated(evt) {
		// Note, we use the elementId to check if this is an operational button or link
		// and we use the className to check if the item is in enabled or disabled state!
		var element = evt.target || evt.srcElement;
		var elementId = element.id;
		var className = element.className;
		// alert('accessListLinkActivated - elementId: ' + elementId);
		if ((className.indexOf('disabled') === -1) && (elementId.indexOf(':op:') !== -1)) {
			// alert('linkActivated - elementId: ' + elementId);
			var dat = elementId.split(':');
			var baseId = dat[0];
			var accessOperation = dat[2];
			var accessItemIndex = _getRowNumberFromBaseId(baseId);
			// alert('accessListLinkActivated - accessItemIndex: ' + accessItemIndex);
			var activeAccess = users.get('activeAccess');
			//
			// Check if the user has permission to edit the item
			//
            var activeUserNodeName = users.get('activeUserNodeName');
			var isPermission = (isRootAdmin || isUnlimitedGrants);
			if (!isPermission) {
				isPermission = (activeAccess[accessItemIndex].created_by_user === activeUserNodeName);
			}
			if (isPermission) {
				if (accessOperation !== 'delete') {
					// alert('open access manager');
					var activeTabIsRoles = (accessOperation === 'edit_roles');
					var isNewAccessPair = false;
					var accessItem = util.cloneObject(activeAccess[accessItemIndex]);
					usersAM.open(accessItemIndex, accessItem, activeTabIsRoles, isNewAccessPair);
				}
				else {
					//
					// Delete item
					//
					// Delete item from activeAccess array
					activeAccess.splice(accessItemIndex, 1);
					// Delete last row (we have to update the rows anyway due delete button display, so we simply delete the last row)
					var tbody = util.getE('users:access:tbody');
					var lastTr = tbody.lastChild;
					tbody.removeChild(lastTr);
					// Update all rows
					var numberOfAccessItems = activeAccess.length;
					for (var i = 0; i < numberOfAccessItems; i++) {
						updateAccessRow(i, activeAccess[i], numberOfAccessItems);
					}
				}
			}
			else {
				alert(langVar('lang_admin.users.no_permission_msg'));
			}
		}
    }
	function hoverAccessListItem(evt) {
		var element = evt.target || evt.srcElement;
		var elementId = element.id;
		var className = element.className;
		// Note, we use the elementId to check if this is an operational button or link
		// and we use the className to check if the item is in enabled or disabled state!
		// Hover off upon mouseout must be always possible, even if disabled because
		// it possible that we accidently hover a disabled item after delete.
		// alert('hoverAccessListItem - elementId: ' + elementId + '\nevt.type: ' + evt.type);
		if (elementId.indexOf(':op:') !== -1) {
			var eventType = evt.type;
			var isMouseOver = (evt.type === 'mouseover');
			if (!isMouseOver || (className.indexOf('disabled') === -1)) {
				element.style.backgroundColor = isMouseOver ? '#F2F2F2' : 'White';
				element.style.cursor = isMouseOver ? 'pointer' : 'default';
			}
		}
	}
	function _getSortedAccessItemProfileLabels(numberOfProfiles, profiles) {
		var profilesDb = users.get('profilesDb');
        var profileName;
        var profileItem;
		var a = [];
		for (var i = 0; i < numberOfProfiles; i++) {
			profileName = profiles[i];
			profileItem = profilesDb[util.h(profileName)];
			var profileLabel = (profileItem !== null) ? profileItem.label : langVar('lang_admin.users.unknown_profile');
			a[i] = (profileItem !== null) ? profileItem.label :
                langVar('lang_admin.users.unknown_profile');
		}
		a.sort();
		return a;
	}
	function _cleanUpAccessCell(element) {
		// alert('cleanUpAccessCell - element: ' + element);
		while (element.lastChild !== null) {
			var theChild = element.lastChild;
			element.removeChild(theChild);
		}
	}
	function updateAccessRow(index, accessItem, numberOfAccessItems) {
		// util.showObject(accessItem);
		var br;
		var rolesDb = users.get('rolesDb');
		var activeUserNodeName = users.get('activeUserNodeName');
		var createdByUserNodeName = accessItem.created_by_user;
		var isOwnedByOtherUser = (!isRootAdmin && !isUnlimitedGrants && (activeUserNodeName !== createdByUserNodeName));
		//
		// Get the baseId of the row
		// 
		var tbody = util.getE('users:access:tbody');
		var allRows = tbody.getElementsByTagName('tr');
		var trId = allRows[index].id;
		var dat = trId.split(':');
		var baseId = dat[0];
		//
		//
		// Handle Profiles cell
		//
		//
		var isAllProfiles = accessItem.all_profiles;
		var profiles = accessItem.profiles;
		var numberOfProfiles = profiles.length;
		var profileTd = util.getE(baseId + ':op:edit_profiles');
        // Don't remove, the className is used to check hover and link
        // activation, both are ignored if the className contains disabled!
		profileTd.className = !isOwnedByOtherUser ? '' : 'disabled';
		_cleanUpAccessCell(profileTd);
		var profileLabel;
		var profileText;
		if (isAllProfiles || (numberOfProfiles === 0)) {
			profileLabel = isAllProfiles ? langVar('lang_admin.users.all_profiles') : '[ ' + langVar('lang_admin.users.no_profile_defined') + ' ]';
			profileText = document.createTextNode(profileLabel);
			profileTd.appendChild(profileText);
		}
		else {
			// One or more profiles in list
			// KHP 14/Sep/2010 - Make sure we get sorted labels
			var sortedProfileLabels = _getSortedAccessItemProfileLabels(numberOfProfiles, profiles);
			for (var i = 0; i < numberOfProfiles; i++) {
				profileText = document.createTextNode(sortedProfileLabels[i]);
				profileTd.appendChild(profileText);
				// Add br tag if this is not the last profile
				if (numberOfProfiles > 1 && i + 1 !== numberOfProfiles) {
					br = document.createElement('br');
					profileTd.appendChild(br);
				}
			}
		}
		//
		//
		// Handle Roles cell
		//
		//
		var roleTd = util.getE(baseId + ':op:edit_roles');
		roleTd.className = !isOwnedByOtherUser ? '' : 'disabled';
		_cleanUpAccessCell(roleTd);
		var roles = accessItem.roles;
		var numberOfRoles = roles.length;
		var roleName;
		var roleLabel;
		var roleText;
		if (numberOfRoles > 0) {
			for (var j = 0; j < numberOfRoles; j++) {
				roleName = roles[j];
				roleLabel = rolesDb[util.h(roleName)].label;
				roleText = document.createTextNode(roleLabel);
				roleTd.appendChild(roleText);
				// Add br tag if this is not the last role
				if (numberOfRoles > 1 && j + 1 !== numberOfRoles) {
					br = document.createElement('br');
					roleTd.appendChild(br);
				}
			}
		}
		else {
			// No role defined
			roleText = document.createTextNode('[ ' + langVar('lang_admin.users.no_role_defined') + ' ]');
			roleTd.appendChild(roleText);
		}
		//
		// Handle info cell
		//
		var infoTd = util.getE(baseId + ':info');
		_cleanUpAccessCell(infoTd);
		var infoLabel = '';
		if (createdByUserNodeName === activeUserNodeName) {
			// No info, add &nbsp;
			infoLabel = '\u00a0';
		}
		else {
			if (!isOwnedByOtherUser) {
				// Add info by whom the access pair has been created
				var username = _getUsernameByUserNodeName(createdByUserNodeName);
				infoLabel = '(' + langVar('lang_admin.general.by_user')  + ')';
				infoLabel = infoLabel.replace(/__PARAM__1/, username);
			}
			else {
				// This are limited grants, there is no permission to edit this item
				infoLabel = '(' + langVar('lang_admin.users.owned_by_other_user') + ')';
			}
		}
		var infoText = document.createTextNode(infoLabel);
		infoTd.appendChild(infoText);
		//
		// Handle edit cell
		//
		var editTd = util.getE(baseId + ':op:edit');
		editTd.className = !isOwnedByOtherUser ? 'link' : 'link-disabled';
		//
		// Handle delete cell
		//
		if (isEnterprise) {
			var deleteTd = util.getE(baseId + ':op:delete');
			deleteTd.className = (!isOwnedByOtherUser && numberOfAccessItems > 1) ? 'link' : 'link-disabled';
		}
	}
	function _addAccessCell(tr, elementId, className, label) {
		// type is: edit_profiles | edit_roles
		var td = document.createElement('td');
		if (className !== '') {
            td.className = className;
        }
		td.id = elementId;
		var text = document.createTextNode(label);
		td.appendChild(text);
		tr.appendChild(td);
	}
	function createAccessRow(tbody) {
		// alert('createAccessRow);
		// Get new unique baseId (i.e. acc8)
		var idCount = 0;
		var isUniqueId = false;
		var baseId = '';
		while (!isUniqueId) {
			if (util.getE('acc' + idCount + ':tr') === null) {
				baseId = 'acc' + idCount;
				isUniqueId = true;
			}
			idCount++;
		}
		var tr = document.createElement('tr');
		tr.id = baseId + ':tr';
		_addAccessCell(tr, baseId + ':op:edit_profiles', '', '-');
		_addAccessCell(tr, baseId + ':op:edit_roles', '', '-');
		_addAccessCell(tr, baseId + ':info', 'info', '-');
		_addAccessCell(tr, baseId + ':op:edit', 'link', langVar('lang_stats.btn.edit'));
        if (isEnterprise) {
			_addAccessCell(tr, baseId + ':op:delete', 'link', langVar('lang_stats.btn.delete'));
		}
		tbody.appendChild(tr);
	}
	function updateAccessList() {
		// Updates the access list
		// var table = util.getE('users:access:container');
		var tbody = util.getE('users:access:tbody');
		// Get the number of existing rows
		var allRows = tbody.getElementsByTagName('tr');
		var numberOfExistingAccessRows = allRows.length;
		var access = users.get('activeAccess');
		var numberOfAccessItems = access.length;
		// alert('numberOfExistingAccessRows: ' + numberOfExistingAccessRows + '\nnumberOfAccessItems: ' + numberOfAccessItems);
		//
		// Create/remove access rows
		//
        var i;
		if (numberOfExistingAccessRows !== numberOfAccessItems) {
			if (numberOfExistingAccessRows < numberOfAccessItems) {
				// Add new rows
				// alert('Add new rows!');
				for (i = numberOfExistingAccessRows; i < numberOfAccessItems; i++) {
					createAccessRow(tbody);
				}
			}
			else {
				// Remove rows
				// alert('Remove rows!');
				for (i = numberOfExistingAccessRows; i > numberOfAccessItems; i--) {
					// alert('Remove row: ' + i);
					var tr = tbody.lastChild;
					tbody.removeChild(tr);
				}
			}
		}
		//
		// Update the access cells
		//
		for (i = 0; i < numberOfAccessItems; i++) {
			var accessItem = access[i];
			updateAccessRow(i, accessItem, numberOfAccessItems);
		}
	}
	//
	//
	//
	// Miscellaneous utilities
	//
	//
	//
	function _getRowNumberFromBaseId(baseId) {
		var tbody = util.getE('users:access:tbody');
		var allRows = tbody.getElementsByTagName('tr');
		var trBaseId = baseId + ':tr';
		var rowNumberIndex = 0;
		for (var i = 0; i < allRows.length; i++) {
			if (trBaseId === allRows[i].id) {
				rowNumberIndex = i;
				break;
			}
		}
		return rowNumberIndex;
	}
	function _getUsernameByUserNodeName(userNodeName) {
		var usersDb = users.get('usersDb');
		var username = '';
		for (var i = 0; i < usersDb.length; i++) {
			if (userNodeName === usersDb[i].dat.node_name) {
				username = usersDb[i].dat.username;
				break;
			}
		}
		if (username === '') {
			username = langVar('lang_admin.users.unknown_user');
		}
		return username;
	}
	function getNewUserNodeName(usersDb, deletedUsersDb) {
		// Returns a new unique node name (with timestamp) so that it
		// is unique forever.
		//
		// Create a node name lookup
		//
		var usedNodeNames = {};
		var nodeName;
		for (var i = 0; i < usersDb.length; i++) {
			nodeName = usersDb[i].dat.node_name;
			nodeName = '_' + nodeName.toLowerCase();
			usedNodeNames[nodeName] = true;
		}
		for (var j = 0; j < deletedUsersDb.length; j++) {
			nodeName = deletedUsersDb[j];
			nodeName = '_' + nodeName.toLowerCase();
			usedNodeNames[nodeName] = true;
		}
		//
		// Create new node name
		//
		// KHP 05/Jan/2011 - use timestamp to ensure unique user node name forever,
		// it should not be possible to create the same user node name even after
		// deletion of the user because there might be references of the node name
		// in "created_by_user" nodes.
		var d = new Date();
		var timestamp = d.getTime();
		var newNodeName = 'user_' + timestamp;
		// util.showObject({newNodeName__1: newNodeName});
		if (usedNodeNames['_' + newNodeName]) {
			// A user node name with this timestamp already exists.
			// This case shouldn't be possible but we cover it.
			var isUniqueNodeName = false;
			var count = 1;
			while (usedNodeNames['_' + newNodeName]) {
				newNodeName = 'user_' + timestamp + count;
				count++;
			}
		}
		return newNodeName;
	}
    // Return global methods
	return {
        init: init,
		accessListLinkActivated: accessListLinkActivated,
        hoverAccessListItem: hoverAccessListItem,
        updateAccessRow: updateAccessRow,
        createAccessRow: createAccessRow,
        updateAccessList: updateAccessList,
        getNewUserNodeName: getNewUserNodeName
	};
}());
/* global
	langVar: false,
	pageInfo: false,
	users: users,
	usersUtil: false,
	util: false,
	YAHOO: false */
// usersAM.js (handles the users Access Manager Panel
var usersAM = (function() {
    'use strict';
	var YE = YAHOO.util.Event,
        isRootAdmin = false,
        isUnlimitedGrants = false,
        panel = null,
        tabs = null,
        isNewAccessPair = false,
        // The index of the active access pair in users.activeAccess.
        // The index is only relevant when isNewAccessPair is false.
        activeAccessItemIndex = 0,
        activeCreatedByUser = '';
	function init() {
		//
		// Init the panel
		//
		var panelObj = {
			panelId: 'users:am:panel',
			panelClassName: 'panel-50',
			panelHeaderLabel: '-',
			left: 300,
			top: 160,
			zIndex: 20,
			isCover: true,
			isSticky: true,
			closeEvent: close
		};
		panel = new util.Panel3(panelObj);
		//
		// Init the tabs
		//
		var tabList = ['users:am:profiles:tab', 'users:am:roles:tab'];
		tabs = new util.Tabs3('users:am:tabs', tabList, tabActivated);
		tabs.setSequence(tabList, 'users:am:profiles:tab');
		//
		//
		// Init form controls
		//
		//
		isRootAdmin = pageInfo.isRootAdmin;
		isUnlimitedGrants = pageInfo.permissions.isUnlimitedGrants;
		var isLimitedGrants = (!isRootAdmin && !isUnlimitedGrants);
		//
		// Handle roles list first so that we can populate a profilesLookup if
		// isLimitedGrants
		//
		var rolesDb = users.get('rolesDb');
		var profilesDb = users.get('profilesDb');
//        util.showObject(rolesDb);
//        util.showObject(profilesDb);
		var rolesList = [];
		var profilesList = [];
		var profilesLookup = {};
		var isAllProfilesPermission = false;
        var numProfiles;
		if (!isLimitedGrants) {
			rolesList = rolesDb;
			isAllProfilesPermission = true;
		}
		else {
			for (var i = 0, numRoles = rolesDb.length; i < numRoles; i++) {
				var roleItem = rolesDb[i];
				if (roleItem.isPermission) {
					rolesList[rolesList.length] = {name:roleItem.name, label:roleItem.label};
					if (roleItem.isAllProfiles) {
						isAllProfilesPermission = true;
					}
					else {
						// Remember any profile listed in profiles in profilesLookup
                        numProfiles = roleItem.profiles.length;
						for (var j = 0; j < numProfiles; j++) {
							profilesLookup['_' + roleItem.profiles[j]] = true;
						}
					}
				}
			}
		}
		//
		//
		// Create profiles list
		//
		//
		if (isAllProfilesPermission) {
			profilesList[0] = {name: '__ALL__PROFILES__', label: langVar('lang_admin.users.all_profiles_accesses')};
		}
        numProfiles = profilesDb.length;
		for (var k = 0; k < numProfiles; k++) {
			var profileItem = profilesDb[k];
			var profileName = profileItem.name;
			if (isAllProfilesPermission || (profilesLookup['_' + profileName] !== null)) {
				// We add a space to profile labels so that there is some offset from the All Profiles label
				var profileLabel = isAllProfilesPermission ? ' ' + profileItem.label : profileItem.label;
				profilesList[profilesList.length] = {name: profileName, label: profileLabel}; 
			}
		}
		util.populateSelect('users:am:profiles:select', profilesList, 'name', 'label');
		util.populateSelect('users:am:roles:select', rolesList, 'name', 'label');
		YE.addListener('users:am:okay', 'click', saveAccessPair);
		YE.addListener('users:am:cancel', 'click', close);
	}
	function open(itemIndex, accessItem, activeTabIsRoles, _isNewAccessPair) {
		// isNewAccessPair is true when Access Manager is opened via the
		// New Access Pair button.
		// reset the form
		util.resetF('users_am_form');
		//
		//
		// Set form values according accessItem
		//
		//
		activeAccessItemIndex = itemIndex;
		isNewAccessPair = _isNewAccessPair;
		activeCreatedByUser = accessItem.created_by_user;
		var isAllProfiles = accessItem.all_profiles;
		var profiles = accessItem.profiles;
		var roles = accessItem.roles;
		// var selectOptions;
		if (!isAllProfiles) { 
			for (var i = 0, numProfiles = profiles.length; i < numProfiles; i++) {
				util.setF('users:am:profiles:select', profiles[i]);
			}
		}
		else {
			// Check the All profiles radio button
			// util.setF('users:am:all_profiles_btn', true);
			// Select All Profiles entry
			var profilesSelect = util.getE('users:am:profiles:select');
			// alert('profilesSelect: ' + profilesSelect);
			profilesSelect.selectedIndex = 0;
			// profilesSelect.options[0].selected = true;
		}
		for (var j = 0, numRoles = roles.length; j < numRoles; j++) {
			util.setF('users:am:roles:select', roles[j]);
		}
		// Preset tabs
		var tabId = activeTabIsRoles ? 'users:am:roles:tab' : 'users:am:profiles:tab';
		tabs.setActiveTab(tabId);
		setDisplay(activeTabIsRoles);
		// Open panel
		var panelLabel = !isNewAccessPair ? langVar('lang_admin.users.edit_access_pair') : langVar('lang_admin.users.new_access_pair');
		panel.open({label: panelLabel});
	}
	function addNewAccessPair() {
		var itemIndex = 0; // is not relevant when adding a new item
		var isNewAccessPair = true;
		var activeTabIsRoles = false;
		var newAccessItem = {
			all_profiles: false,
			profiles: [],
			roles: []
			// created_by_user: users.activeUserNodeName
		};
		open(itemIndex, newAccessItem, activeTabIsRoles, isNewAccessPair);
	}
	function setDisplay(activeTabIsRoles) {
		if (activeTabIsRoles) {
			util.hideE('users:am:profiles:section');
			util.showE('users:am:roles_section');
		}
		else {
			util.hideE('users:am:roles_section');
			util.showE('users:am:profiles:section');
		}
	}
	function tabActivated(tabId) {
//		var tabId = this.id;
		var activeTabIsRoles = (tabId === 'users:am:roles:tab');
		tabs.setActiveTab(tabId);
		setDisplay(activeTabIsRoles);
	}
	function getIsValidAccessPair(profiles, roles) {
		// Only used if there are no unlimited grants. It is
		// only relevant if there are different profiles with different role permissions,
		// i.e. different profile manager roles are assigned to different profiles.
		var allValid = true;
		var isValidRoleProfileRelation;
		var rolesDb = users.get('rolesDb');
		var profilesDb = users.get('profilesDb');
		var info = '';
        var profileName = '';
		for (var i = 0, numRoles = roles.length; i < numRoles; i++) {
			var roleName = roles[i];
			var theRoleInDb = rolesDb[util.h(roleName)];
			var isAllProfiles = theRoleInDb.isAllProfiles;
			for (var j = 0, numProfiles = profiles.length; j < numProfiles; j++) {
				profileName = profiles[j];
				isValidRoleProfileRelation = false;
				if (isAllProfiles) {
					isValidRoleProfileRelation = true;
				}
				else {
					// Check if the profile exists in theRoleInDb.profiles
					for (var k = 0, numItems = theRoleInDb.profiles.length; k < numItems; k++) {
						if (theRoleInDb.profiles[k] === profileName) {
							isValidRoleProfileRelation = true;
							break;
						}
					}
				}
				// We add any invalid profile/role relations to info
				// and show them in an alert.
				if (!isValidRoleProfileRelation) {
					allValid = false;
					var roleLabel = theRoleInDb.label;
					var profileLabel = profilesDb[util.h(profileName)].label;
					info += profileLabel + ' - ' + roleLabel + '\n';
				}
			}
		}
		if (!allValid) {
			alert(langVar('lang_admin.users.profiles_roles_assignment_conflict_msg') + '\n' + info);
		}
		return allValid;
	}
	function saveAccessPair() {
		var profilesSelect = util.getE('users:am:profiles:select');
		var profilesOptions = profilesSelect.options;
		var isAllProfiles = ((profilesOptions[0].value === '__ALL__PROFILES__') && profilesOptions[0].selected);
		var profiles = [];
		var roles = [];
		// alert('isAllProfiles: ' + isAllProfiles);
		if (!isAllProfiles) {
			for (var i = 0, numProOpt = profilesOptions.length; i < numProOpt; i++) {
				if (profilesOptions[i].selected) {
					profiles[profiles.length] = profilesOptions[i].value;
				}
			}
		}
		var rolesSelect = util.getE('users:am:roles:select');
		var options = rolesSelect.options;
		for (var j = 0, numOptions = options.length; j < numOptions; j++) {
			if (options[j].selected) {
				roles[roles.length] = options[j].value;
			}
		}
		var isValidAccessPair = (isRootAdmin || isUnlimitedGrants) ? true : getIsValidAccessPair(profiles, roles);
		if (isValidAccessPair) {
			var accessItem;
			var accessItemIndex = 0;
			var activeAccess = users.get('activeAccess');
			var numberOfAccessItems = activeAccess.length;
			if (!isNewAccessPair) {
				// Update the existing access object
				accessItemIndex = activeAccessItemIndex;
				accessItem = activeAccess[accessItemIndex];
				accessItem.all_profiles = isAllProfiles;
				accessItem.profiles = profiles;
				accessItem.roles = roles;
				// Update the single row in access list
				usersUtil.updateAccessRow(accessItemIndex, accessItem, numberOfAccessItems);
			}
			else {
				// Add a new access item object
				numberOfAccessItems += 1;
				accessItem = {
					all_profiles: isAllProfiles,
					profiles: profiles,
					roles: roles,
					created_by_user: users.get('activeUserNodeName')
				};
				// Add new item to users.activeAccess
				accessItemIndex = activeAccess.length;
				activeAccess[accessItemIndex] = accessItem;
				// util.showObject(activeAccess);
				// Create a new row
				var tbody = util.getE('users:access:tbody');
				usersUtil.createAccessRow(tbody, accessItemIndex);
				// If the list already shows a delete button (final numberOfAccessItems > 2)
				// then we only need to update the last row, else we update all rows.
				if (numberOfAccessItems > 2) {
					// Update single row
					usersUtil.updateAccessRow(accessItemIndex, accessItem, numberOfAccessItems);
				}
				else {
					// Update all rows so that we get the delete button in every row
					for (var k = 0; k < numberOfAccessItems; k++) {
						usersUtil.updateAccessRow(k, activeAccess[k], numberOfAccessItems);
					}
				}
			}
			// Close the panel
			close();
		}
	}
	function close() {
		panel.close();
	}
     // Return global methods
	return {
		init: init,
        open: open,
        addNewAccessPair: addNewAccessPair
	};
}());
/* global
    adminConfig: false,
	langVar: false,
	listcontroller,
	optionInfo: false,
	pageInfo: false,
	toolbarButtonsDb: false,
	usersAM: false,
	usersUtil: false,
	util: false,
	YAHOO: false */
var users = (function() {
	'use strict';
	var YE = YAHOO.util.Event,
        isRootAdmin = false,
        isEnterprise = false,
        isUnlimitedGrants = false,
        defaultDividerLabels = [],
        languageInPreferences = '',
        languageOfDividerLists = '',
        activeUserNodeName = '',
        rootAdminUsername = '', // Used to check for duplicate username because the rootAdmin is not part of usersDb
        usersDb = [],
        usersDbBackup = [],
        profilesDb = [],
        rolesDb = [], // Contains the role names, labels and profile access rights if there are no unlimited grants, format is:
                     // [{name:"role_1", label:"Administrator", isPermission:true, all_profiles:false, profiles:['profile_1', profile_2]}, ...]
                     // Note, rolesDb contains all roles even if unlimited_grants is equal false because we need all roles for display!
                     // We use the "isPermission" property to check whether or not a role is granted.
        deletedUsersDb = [], // Keeps the node names of any deleted user, i.e. ['user_1', 'user_12']
        theList = null,
        validator = null,
        // isModifiedUsers = false,
        saveChangesBtn = null,
        newUserBtn = null,
        deleteBtn = null,
        duplicateBtn = null,
        undoAllChangesBtn = null,
        noItemFormIsActive = false,
        // Active form data:
        changePasswordIsActive = false,
        activeItemIsNew = false,
        activeItemNodeName = '',
        activeCreatedByUser = '',
        activeAccess = [],
        isSelectUserName = false, // used for username default selected upon new or duplicate
        // is true if the report filter expression field is shown on the item form
        isVisibleReportFilterExpressionField = false;
    function init() {
        //
        // Main init routine
        //
        isRootAdmin = pageInfo.isRootAdmin;
        isEnterprise = (pageInfo.licensingTier === 'enterprise');
        isUnlimitedGrants = pageInfo.permissions.isUnlimitedGrants;
        validator = new util.Validator();
        //
        // init toolbar buttons
        //
        saveChangesBtn = new util.ToolbarButton('save_changes', saveChanges, toolbarButtonsDb);
        newUserBtn = new util.ToolbarButton('new_user', newUser, toolbarButtonsDb);
        duplicateBtn = new util.ToolbarButton('duplicate', duplicateUser, toolbarButtonsDb);
        deleteBtn = new util.ToolbarButton('delete', deleteUser, toolbarButtonsDb);
        undoAllChangesBtn = new util.ToolbarButton('undo_all_changes', undoAllChanges, toolbarButtonsDb);
        //
        // Ignore/Disable buttons according RBAC
        //
        var permissions = pageInfo.permissions;
        if (permissions.isEdit) {
            if (!permissions.isAdd) {
                newUserBtn.disableAndIgnore();
                duplicateBtn.disableAndIgnore();
            }
            if (!permissions.isDelete) {
                deleteBtn.disableAndIgnore();
            }
            // Register isModifiedPageHandler in adminConfig.js
            // (We don't check for modifications if there is no edit
            // permission because there is no Save button anyway!)
            adminConfig.getIsModifiedPageHandler = getIsModifiedPage;
        }
        else {
            saveChangesBtn.disableAndIgnore();
            newUserBtn.disableAndIgnore();
            duplicateBtn.disableAndIgnore();
            deleteBtn.disableAndIgnore();
            undoAllChangesBtn.disableAndIgnore();
        }
        // init OptionInfo
        optionInfo.init();
        // Init help
        util.helpWindow.init('');
        //
        // Create the theList object
        //
        theList = new listcontroller.List({
            containerElementId: 'item_list_body',
            itemEvent: itemActivated
        });
        // Add events
        YE.addListener('users:username', 'keyup', updateListAndFormLabel);
        YE.addListener('users:change_password_btn', 'click', changeCancelPassword);
        YE.addListener('users:cancel_change_password_btn', 'click', changeCancelPassword);
        YE.addListener('users:access:tbody', 'click', usersUtil.accessListLinkActivated);
        YE.addListener('users:access:tbody', 'mouseover', usersUtil.hoverAccessListItem);
        YE.addListener('users:access:tbody', 'mouseout', usersUtil.hoverAccessListItem);
        YE.addListener('users:language', 'change', languageActor);
        //
        // Handle enterprise specific features
        //
        if (isEnterprise) {
            YE.addListener('users:access:new_access_pair_btn', 'click', usersAM.addNewAccessPair);
            YE.addListener('users:add_report_filter:btn', 'click', addReportFilter);
            YE.addListener('users:report_filter_expression:view_help_btn', 'click', util.helpWindow.openGeneralHelp);
            util.showE('users:add_report_filter:section');
        }
        //
        // Init Access Manager Panel
        //
        usersAM.init();
        usersUtil.init();
    }
    function initMainDisplay() {
        // alert('initMainDisplay()');
        util.hideE(['form_section', 'users_form', 'no_item_form', 'loading_info', 'saving_info']);
        var firstItemId = theList.getFirstItemId();
        // alert('initMainDisplay() - firstItemId: ' + firstItemId);
        if (firstItemId !== null) {
            // select the first item
            setItem(firstItemId);
            displayNoItemForm(false);
        }
        else {
            // no item exists
            displayNoItemForm(true);
        }
        util.showE('form_section');
    }
    function getUsersData() {
        if (!pageInfo.exitActive) {
            var url = '?dp=admin_pages.users.get_data';
            var dat = 'v.fp.page_token=' + pageInfo.pageToken;
            util.serverPost(url, dat);
        }
    }
    function getUsersDataResponse(dat) {
//        util.showObject(dat);
        if (!pageInfo.exitActive) {
            defaultDividerLabels = dat.defaultDividerLabels;
            languageInPreferences = dat.languageInPreferences;
            activeUserNodeName = dat.activeUserNodeName;
            rootAdminUsername = dat.rootAdminUsername;
            usersDb = dat.usersDb;
            usersDbBackup = util.cloneObject(dat.usersDb);
            profilesDb = dat.profilesDb;
            rolesDb = dat.rolesDb;
            // util.showObject(defaultDividerLabels);
            util.createHash(defaultDividerLabels, 'language');
            util.createHash(profilesDb, 'name');
            util.createHash(rolesDb, 'name');
            // Init
            if (!pageInfo.initComplete) {
                // This is the initial page load
                init();
                theList.init(usersDb);
                initMainDisplay();
                pageInfo.initComplete = true;
            }
            // Set final toolbar state
            saveChangesBtn.enable();
            undoAllChangesBtn.enable();
            newUserBtn.enable();
            updateToolbarButtons(); // handles Duplicate and Delete
            adminConfig.setItemListSize();
            YAHOO.util.Event.addListener(window, 'resize', adminConfig.setItemListSize);
        }
    }
    function displayNoItemForm(isDisplayNoItemForm) {
        if (isDisplayNoItemForm) {
            // util.updateT('item_form_label', '-');
            util.hideE('users_form');
            util.showE('no_item_form');
        }
        else {
            util.hideE('no_item_form');
            util.showE('users_form');
        }
        noItemFormIsActive = isDisplayNoItemForm;
    }
    function updateToolbarButtons() {
        var isItems = theList.isItems();
        deleteBtn.enable(isItems);
        duplicateBtn.enable(isItems);
    }
    function updateListAndFormLabel() {
        setTimeout(function() {
            var label = util.getF('users:username');
            if (label === '') {
                label = '-';
            }
            if (!isRootAdmin) {
                var item = theList.getSelectedItem();
                var userNodeName = item.dat.node_name;
                if (userNodeName === activeUserNodeName) {
                    label += ' (' + langVar('lang_admin.users.logged_in') + ')';
                }
            }
            // util.updateT('item_form_label', label);
            theList.updateListLabel(label);
        }, 300);
    }
//    function setListAndFormLabel() {
//
//
//    }
    function changeCancelPassword() {
        var isChangePassword = !changePasswordIsActive;
        // util.showE('users:password_container', isChangePassword);
        // util.showE('users:change_password_btn', !isChangePassword);
        // util.showE('users:cancel_change_password_btn', isChangePassword);
        util.showE('users:password_section', isChangePassword);
        util.showE('users:change_password_section', !isChangePassword);
        changePasswordIsActive = isChangePassword;
        // Reset any errors
        validator.reset();
    }
    function languageActor() {
        var language = util.getF('users:language');
        updateLanguageOptions(language);
    }
    function updateLanguageOptions(language) {
        // language is the language of current user
        // This updates the select element label
        // of number_thousands_divider and number_decimal_divider
        // so that the first list item shows the language specific
        // default dividers.
        function updateOptionText(elementId, label) {
            var element = util.getE(elementId);
            element.options[0].text = label;
        }
        if (language === '') {
            // Use default language as set it preferences
            language = languageInPreferences;
        }
        // Check if we need to update the default divider text
        if (language !== languageOfDividerLists) {
            var dividerItem = defaultDividerLabels[util.h(language)];
            updateOptionText('users:number_thousands_divider', dividerItem.defaultThousandDividerLabel);
            updateOptionText('users:number_decimal_divider', dividerItem.defaultDecimalDividerLabel);
            // Remember the language of divider lists
            languageOfDividerLists = language;
        }
    }
    function addReportFilter() {
        util.hideE('users:add_report_filter:section');
        util.showE('users:report_filter_expression:section');
        isVisibleReportFilterExpressionField = true;
    }
    function itemActivated(itemId) {
        if (validateActiveItem()) {
            setItem(itemId);
        }
    }
    function setItem(itemId) {
        // selects active item in list and displays the form
        theList.selectItem(itemId);
        updateForm(itemId);
        //
        // set delete button
        //
        /*
        if (itemId != activeUserItemId) {
            deleteButton.enable();
        }
        else {
            deleteButton.disable();
        }
        */
    }
    function newUser() {
        if (validateActiveItem()) {
            var newItemId = theList.getNewItemId();
            var newNodeName = usersUtil.getNewUserNodeName(usersDb, deletedUsersDb);
            // Give a default role in the pro version, the role_2 (Statistics visitor)
            var defaultRoles = isEnterprise ? [] : ['role_2'];
            var accessObj = [{
                all_profiles: false,
                profiles: [],
                roles: defaultRoles,
                created_by_user: activeUserNodeName
            }];
            var newUserLabel = langVar('lang_admin.users.new_user');
            var userObj = {
                id: newItemId,
                type: '',
                label: newUserLabel,
                dat: {
                    is_new: true,
                    node_name: newNodeName,
                    username: newUserLabel,
                    password: '',
                    email_address: '',
                    language: '',
                    number_thousands_divider: '',
                    number_decimal_divider: '',
                    created_by_user: activeUserNodeName,
                    access: accessObj,
                    auto_direct_to_reports_after_login: false,
                    report_filter_expression: ''
                }
            };
            theList.newItem(userObj);
            isSelectUserName = true;
            setItem(newItemId);
            updateToolbarButtons();
        }
    }
    function duplicateUser() {
        if (validateActiveItem()) {
            var item = theList.getSelectedItem();
            var username = item.dat.username;
            // Get new node name
            var newNodeName = usersUtil.getNewUserNodeName(usersDb, deletedUsersDb);
            var newUsername = langVar('lang_stats.general.item_copy');
            newUsername = newUsername.replace(/__PARAM__1/, username);
            // Clone item
            var clonedItemId = theList.cloneItem();
            //
            // Reset item data
            //
            theList.setItemDatValue(clonedItemId, 'is_new', true);
            theList.setItemDatValue(clonedItemId, 'node_name', newNodeName);
            theList.setItemDatValue(clonedItemId, 'username', newUsername);
            theList.setItemDatValue(clonedItemId, 'password', '');
            theList.setItemDatValue(clonedItemId, 'created_by_user', activeUserNodeName);
            isSelectUserName = true;
            setItem(clonedItemId);
        }
    }
    function deleteUser() {
        var item = theList.getSelectedItem();
        var itemId = item.id;
        deletedUsersDb[deletedUsersDb.length] = item.dat.node_name;
        // alert('deleteRole(): ' + itemId);
        //
        // select the next item
        //
        var nextItemIdToBeSelected = theList.deleteItem();
        if (nextItemIdToBeSelected !== null) {
            // reset the validator in case that the deleted an item with error indication
            validator.reset();
            setItem(nextItemIdToBeSelected);
        }
        else {
            // All items have been deleted
            // alert('Last item has been deleted, no more items to select.');
            displayNoItemForm(true);
            updateToolbarButtons();
        }
    }
    function updateForm(itemId) {
        var item = theList.getSelectedItem();
        var itemDat = item.dat;
//        util.showObject(item);
        // util.showObject(itemDat);
        // util.updateT('item_form_label', itemDat.username);
        activeItemIsNew = itemDat.is_new;
        activeItemNodeName = itemDat.node_name;
        //
        // Handle username
        //
        util.setF('users:username', itemDat.username);
        // var isLoggedInUser = (itemDat.node_name == activeUserNodeName);
        // util.showE('users:active_user_info', isLoggedInUser);
        //
        // Handle password
        //
        // util.hideE(['users:password_container', 'users:change_password_btn', 'users:cancel_change_password_btn']);
        util.hideE(['users:change_password_section', 'users:password_section']);
        var password = itemDat.password;
        changePasswordIsActive = false;
        util.setF('users:password', password);
        if (!itemDat.is_new) {
            // Existing user item
            if (password === '') {
                // util.hideE('users:password_container');
                // util.showE('users:change_password_btn');
                util.hideE('users:password_section');
                util.showE('users:change_password_section');
            }
            else {
                // password changed already, allow to cancel the change
                // util.showE('users:password_container');
                // util.showE('users:cancel_change_password_btn');
                util.showE('users:password_section');
                util.showE('users:cancel_change_password_btn');
                changePasswordIsActive = true;
            }
        }
        else {
            // New user item
            // util.showE('users:password_container');
            util.hideE('users:cancel_change_password_btn');
            util.showE('users:password_section');
        }
        util.setF('users:email_address', itemDat.email_address);
        //
        // Handle language options & created_by_user
        //
        util.setF('users:language', itemDat.language);
        updateLanguageOptions(itemDat.language);
        util.setF('users:number_thousands_divider', itemDat.number_thousands_divider);
        util.setF('users:number_decimal_divider', itemDat.number_decimal_divider);
        activeCreatedByUser = itemDat.created_by_user;
        //
        //
        // Handle access
        //
        //
        // Clone the active access array to activeAccess, we use
        // the activeAccess to create and manipulate the access list
        // per user item.
        activeAccess = util.cloneObject(itemDat.access);
        // util.showObject(activeAccess);
        usersUtil.updateAccessList();
        if (noItemFormIsActive) {
            displayNoItemForm(false);
        }
        // Set auto_direct_to_reports_after_login
        util.setF('users:auto_direct_to_reports_after_login', itemDat.auto_direct_to_reports_after_login);
        //
        //
        // Handle report filter
        //
        //
        if (isEnterprise) {
            var reportFilterExpression = itemDat.report_filter_expression;
            var reportFilterExpressionExists = (reportFilterExpression !== '');
            util.setF('users:report_filter_expression', reportFilterExpression);
            if (reportFilterExpressionExists !== isVisibleReportFilterExpressionField) {
                util.showE('users:add_report_filter:section', !reportFilterExpressionExists);
                util.showE('users:report_filter_expression:section', reportFilterExpressionExists);
                isVisibleReportFilterExpressionField = reportFilterExpressionExists;
            }
        }
        // Assign tabIndex (hard coded html tabindex has a problem with password field, hence we do via js!
        /*
        KHP 25/Nov/2010 - disabled because tabIndex looks to work  without js
        and it misses number_thousands_divider, number_decimal_divider and report filter
        anyway.
        var f1 = util.getE('users:username');
        var f2 = util.getE('users:password');
        var f3 = util.getE('users:access:container');
        var f4 = util.getE('users:language');
        f1.tabIndex = 1;
        f2.tabIndex = 2;
        f3.tabIndex = 3;
        f4.tabIndex = 4;
        if (isSelectUserName) {
            f1.select();
            // Reset isSelectUserName
            isSelectUserName = false;
        }
        */
    }
    function validateActiveItem() {
        // Only validate if isEdit permission and if items
        if (pageInfo.permissions.isEdit && theList.isItems()) {
            validator.reset();
            var o = {};
            // Transfer is_new and node_name from existing object
            o.is_new = activeItemIsNew;
            o.node_name = activeItemNodeName;
            o.username = validator.isValue('users:username');
            // Make sure the username is not equal the rootAdmin username
            if (o.username !== rootAdminUsername) {
                // The username is case sensitive, so "user a" is not equal "USER A", both are unique!
                o.username = validator.isUnique('users:username', theList.getLookupItems('username'));
            }
            else {
                // Same username as rootAdmin, we show a custom error
                var msg = '';
                if (isRootAdmin) {
                    msg = langVar('lang_admin.users.is_root_admin_username_msg');
                }
                else {
                    msg = langVar('lang_admin.users.is_reserved_username_msg');
                }
                validator.isCustom('users:username', msg);
            }
            // alert('o.username: ' + o.username);
            if (activeItemIsNew || changePasswordIsActive) {
                o.password = validator.isValue('users:password');
            }
            else {
                o.password = '';
            }
            o.email_address = util.getF('users:email_address');
            // Validate for valid email address if any is defined
            if (o.email_address !== '') {
                validator.isEmailAddress('users:email_address');
            }
            o.language = util.getF('users:language');
            o.number_thousands_divider = util.getF('users:number_thousands_divider');
            o.number_decimal_divider = util.getF('users:number_decimal_divider');
            o.created_by_user = activeCreatedByUser;
            //
            // Handle access (get active access array from activeAccess, it is ready to go)
            //
            // Note, we allow zero profiles and zero roles, there is no validation for profiles and roles!
            o.access = util.cloneObject(activeAccess);
            o.auto_direct_to_reports_after_login = util.getF('users:auto_direct_to_reports_after_login');
            o.report_filter_expression = (isVisibleReportFilterExpressionField) ? util.getF('users:report_filter_expression') : '';
            if (validator.allValid()) {
                theList.saveItem(o);
                return true;
            }
            return false;
        }
        // No isEdit permission or no items
        return true;
    }
    //
    // Save Changes
    //
    function getDeletedUsersDat() {
        var dat = '';
        if (deletedUsersDb.length > 0) {
            for (var i = 0, numItems = deletedUsersDb.length; i < numItems; i++) {
                dat += 'v.fp.deleted_users.' + deletedUsersDb[i] + '=true&';
            }
        }
        else {
            dat += 'v.fp.deleted_users=&';
        }
        return dat;
    }
    function getUsersAccessItemsDat(userPath, access) {
        var dat = '';
        var accessItem;
        var accessItemPath;
        var accessProfiles;
        var accessRoles;
        for (var i = 0, numItems = access.length; i < numItems; i++) {
            accessItem = access[i];
            accessItemPath = userPath + '.access.' + i;
            dat += accessItemPath + '.all_profiles=' + accessItem.all_profiles + '&';
            dat += accessItemPath + '.created_by_user=' + accessItem.created_by_user + '&';
            accessProfiles = accessItem.profiles;
            accessRoles = accessItem.roles;
            if (accessProfiles.length > 0) {
                for (var j = 0, numProfiles = accessProfiles.length; j < numProfiles; j++) {
                    dat += accessItemPath + '.profiles.' + j + '=' + accessProfiles[j] + '&';
                }
            }
            else {
                dat += accessItemPath + '.profiles=&';
            }
            if (accessRoles.length > 0) {
                for (var k = 0, numRoles = accessRoles.length; k < numRoles; k++) {
                    dat += accessItemPath + '.roles.' + k + '=' + accessRoles[k] + '&';
                }
            }
            else {
                dat += accessItemPath + '.roles=&';
            }
        }
        return dat;
    }
    function getUsersDat() {
        var dat = '';
        var numOfUsers = usersDb.length;
        var itemDat = '';
        if (numOfUsers > 0) {
            for (var i = 0; i < numOfUsers; i++) {
                itemDat = usersDb[i].dat;
//              util.showObject(itemDat);
                var userPath = 'v.fp.users.' + itemDat.node_name;
                dat += userPath + '.is_new=' + itemDat.is_new + '&';
                dat += userPath + '.username=' + encodeURIComponent(itemDat.username) + '&';
                dat += userPath + '.password=' + encodeURIComponent(itemDat.password) + '&';
                dat += userPath + '.email_address=' + encodeURIComponent(itemDat.email_address) + '&';
                dat += userPath + '.language=' + itemDat.language + '&';
                dat += userPath + '.number_thousands_divider=' + encodeURIComponent(itemDat.number_thousands_divider) + '&';
                dat += userPath + '.number_decimal_divider=' + encodeURIComponent(itemDat.number_decimal_divider) + '&';
                dat += userPath + '.created_by_user=' + itemDat.created_by_user + '&';
                dat += getUsersAccessItemsDat(userPath, itemDat.access);
                dat += userPath + '.auto_direct_to_reports_after_login=' + itemDat.auto_direct_to_reports_after_login + '&';
                // Add report_filter_expression
                dat += userPath + '.report_filters.all_profiles.filter_expression=' + encodeURIComponent(itemDat.report_filter_expression) + '&';
            }
        }
        else {
            dat += 'v.fp.users=&';
        }
        return dat;
    }
    function getUsersDbBackupDat() {
        var numOfUsers2 = usersDbBackup.length;
        var dat = '';
        var itemDat;
        if (numOfUsers2 > 0) {
            for (var i = 0; i < numOfUsers2; i++) {
                itemDat = usersDbBackup[i].dat;
                dat += 'v.fp.users_backup.' + i + '=' + encodeURIComponent(itemDat.username) + '&';
            }
        }
        else {
            dat += 'v.fp.users_backup=&';
        }
        return dat;
    }
    function saveChanges() {
        if (validateActiveItem()) {
            var isModified = theList.getIsModified();
            // alert('saveChanges() - isModified: ' + isModified);
            if (isModified) {
                util.hideE('form_section');
                util.showE('saving_info');
                var url = '?dp=admin_pages.users.save_data';
                var dat = 'v.fp.page_token=' + pageInfo.pageToken + '&';
                // Handle deleted users
                dat += getDeletedUsersDat();
                // Handle users
                dat += getUsersDat();
                // Handle usersDbBackup
                dat += getUsersDbBackupDat();
                dat = dat.replace(/&$/, '');
                util.serverPost(url, dat);
            }
            else {
                alert(langVar('lang_stats.general.no_changes_to_save'));
            }
        }
    }
    function saveChangesResponse() {
        // alert('saveChangesResponse()');
        // Note, we keep the system at current state, we do not re-initialize theList
        usersDbBackup = util.cloneObject(usersDb);
        deletedUsersDb = [];
        theList.resetIsModified();
        util.hideE('saving_info');
        util.showE('form_section');
    }
    function undoAllChanges() {
        usersDb = util.cloneObject(usersDbBackup);
        deletedUsersDb = [];
        theList.init(usersDb);
        initMainDisplay();
        updateToolbarButtons();
    }
    function getIsModifiedPage() {
        // Note, isModified will be false if only the active item has been edited
        // but has an error, hence we also check "if (!validateActiveItem() ..."
        return !validateActiveItem() || theList.getIsModified();
    }
    function get(name) {
        // Returns a reference or value
        if (name === 'profilesDb') {
            return profilesDb;
        }
        else if (name === 'rolesDb') {
            return rolesDb;
        }
        else if (name === 'usersDb') {
            return usersDb;
        }
        else if (name === 'activeAccess') {
            return activeAccess;
        }
        else if (name === 'activeUserNodeName') {
            return activeUserNodeName;
        }
    }
    // Return global methods
	return {
		init: init,
        getUsersData: getUsersData,
        getUsersDataResponse: getUsersDataResponse,
        saveChangesResponse: saveChangesResponse,
        get: get
	};
}());
