/* global
	convertProfile: false,
	database: false,
	deleteProfile: false,
	langVar: false,
	pageInfo: false,
	profilesFilter: false,
	profilesList: false,
	profilesStorage: false,
	profilesDashboard: false,
	profilesDashboardEditor: false,
	profilesUtil: false,
	scrollUtil: false,
	toolbarButtonsDb: false,
	util: false,
	YAHOO: false */

var profiles = (function() {

	'use strict';

	var YE = YAHOO.util.Event,
		YD = YAHOO.util.Dom,

		// profilesDb is an array with profile item arrays:
		// [
		// profile name
		// profile label
		// database type 1=internal_sql 2=mysql 3=odbc_mssql 4=odbc_oracle
		// database name
		// username
		// date filter
		// not supported version number (empty is valid profile)
		// ]
		profilesDb = [],

		// profilesView = {}, profilesView is defined in pageInfo
		numLicensedProfiles = 0,
		numAllProfiles = 0,
		numPermittedProfiles = 0,
		newProfileBtn = null,
		viewBtn = null,
//		editViewBtn = null,
		convertProfilesBtn = null,
		newProfileWizardWindow = null,

//		scrollControl = null,

		// Valid/invalid profile parameters
		// are set when loading new profile lists.
		isInvalidVersion7Profiles = false,
		isInvalidVersion8Profiles = false,

		// isInvalidProfiles is true if one invalid v7 or v8 profile exists
		isInvalidProfiles = false,

		// isValidProfiles is true if one or more valid profile exists,
		// regardless of invalid profiles.
		isValidProfiles = true,

		editViewActive = false,

		busyPanel = null;

	function init() {

		var isRootAdmin = pageInfo.isRootAdmin;
		var permissions = pageInfo.permissions;

		// Create simple progress panel
		busyPanel = new util.BusyPanel();
		newProfileBtn = new util.ToolbarButton('new_profile', createNewProfile, toolbarButtonsDb);
		viewBtn = new util.ToolbarButton('view', toggleToolbarMenu, toolbarButtonsDb);

		util.dropDownMenu.add('toolbar:view', 'profiles_toolbar:drop_down:view');

		YE.addListener('profiles_toolbar:drop_down:toggle_profiles_filter_btn', 'click', toggleProfilesFilter);

		if (permissions.isViewDatabaseInfoColumn) {
			YE.addListener('profiles_toolbar:drop_down:toggle_db_info_column_btn', 'click', toggleDatabaseInfoColumn);
		}

		if (permissions.isViewCreatedByColumn) {
			YE.addListener('profiles_toolbar:drop_down:toggle_created_by_column_btn', 'click', toggleCreatedByColumn);
		}

		YE.addListener('profiles_toolbar:drop_down:refresh_view', 'click', refreshView);

		if (isRootAdmin || permissions.isAdd) {

			YE.addListener('profiles:no_profile_info_btn', 'click', createNewProfile);
			YE.addListener('profiles_toolbar:drop_down:edit_dashboard_btn', 'click', editView);
		}

		if (isRootAdmin) {

			var commandLinkOptions = {
				classNameEnabled: '',
				classNameDisabled: 'disabled'
			};

			convertProfilesBtn = new util.CommandLink('profiles:convert_profiles_btn', convertProfile.confirmProfileConversion, true, commandLinkOptions);
		}

		YE.addListener('profiles:v7_profiles_list', 'click', profileLinkActivated);
		YE.addListener('profiles_profiles_list', 'click', profileLinkActivated);

		// Init help
		util.helpWindow.init('');

		// Init profilesUtil and profilesList
		profilesUtil.init();
		profilesList.init();

		// Init the scrollControl
//		scrollControl = new scrollUtil.Scroller('', '', profiles.scrollingList);

		numLicensedProfiles = pageInfo.numLicensedProfiles;

		if (profilesStorage.getIsUpToDate()) {

			// Note, we assume that there are no invalid v7 and v8 profiles
			// in profilesStorage.

//			console.log('Profile Data: UP TO DATE ON CLIENT - NO FRESH DATA');

//			alert('getting profiles data from storage!');
			// profilesDb already exists in storage
			numAllProfiles = profilesStorage.get('numAllProfiles');

			// Note, we need to create the hash as upon
			// initial loading.
			var theProfiles = profilesStorage.get('profilesDb');
			addProfilesToProfilesDb(theProfiles);

//			util.showObject(profilesDb, 'profilesDb from store');
//			console.log('Profile Data: FROM STORAGE');

			// Note, createDashboardData must be generated before the
			// the profilesList becomes generated.
			profilesDashboard.createDashboardData();

			profilesList.create(profilesDb);
			updateProfilesDisplay();

			profilesList.checkDashboardDataForCurrentView();
		}
		else {

//			console.log('Profile Data: FRESH FROM SERVER');

			// Get fresh profile data
//			getProfileData(0, 40);
			// Initial loading of profile data requires to load all
			// profiles at once because it may contain invalid
			// version 7, version 8.0 and version 8.1 profiles.
			getProfileData();
		}

		// Init profilesFilter
		profilesFilter.init({
			profilesDb: profilesDb,
			searchElementId: 'profiles:filter',
			clearSearchBtnId: 'profiles:clear_filter_btn',
			onSearchCallback: profiles.createFilteredProfilesList
		});

		// Init show_before_you_start_info
		if (pageInfo.showBeforeYouStartInfo) {

			YE.addListener(['before_you_start:antivirus', 'before_you_start:large_datasets'], 'click', toggleShowBeforeYouStartSection);

			var infoHelpLinks = ['before_you_start:server_sizing', 'before_you_start:database_setup', 'before_you_start:profile_tuning'];
			YE.addListener(infoHelpLinks, 'click', util.helpWindow.openGeneralHelp);

			YE.addListener('before_you_start:hide_btn', 'click', hideBeforeYouStartSection);
			YE.addListener('before_you_start:do_not_show_again', 'click', toggleDoNotShowAgain);
		}
	}

	function getProfileData(isGetNewChecksum /* option */) {

		isGetNewChecksum = isGetNewChecksum || false;

//		console.log('isGetNewChecksum: ' + isGetNewChecksum);

		if (!pageInfo.exitActive) {

			busyPanel.showLoading();

			var url = '?dp=util.profiles.get_profile_data';
			var dat = 'v.fp.active_page=' + pageInfo.page;
			dat += '&v.fp.page_token=' + pageInfo.pageToken;
			dat += '&v.fp.response_function=profiles.getProfileDataResponse';
			dat += '&v.fp.get_new_checksum=' + isGetNewChecksum;

			util.serverPost(url, dat);
		}
	}

	function getProfileDataResponse(dat) {

		if (!pageInfo.exitActive) {

//			util.showObject(dat);

			// Reset profilesDb, it may contain invalid profiles
			// which have been converted.
			profilesDb = [];

			// Set numAllProfiles
			numAllProfiles = dat.numAllProfiles;
			numPermittedProfiles = dat.numPermittedProfiles;

			// Sort the profiles by label
			dat.profilesDb.sort(compareProfileLabels);

			addProfilesToProfilesDb(dat.profilesDb);

			// Check if all all profiles are valid before we
			// save the profiles on client side.
			setIsInvalidProfiles();

			// Handle any active profile conversion.
			if (pageInfo.profileConversionState.isActive) {
				// Set conversion state parameters in case that
				// they are not yet set by setIsInvalidProfiles()
				isInvalidVersion8Profiles = true;
				isInvalidProfiles = false;
			}

			if (!isInvalidProfiles) {

				// Save profile data in local storage, only if no single invalid profile exists

				// If we requested a new profilesListChecksum then use it!
				var profilesListChecksum = '';

				// If a new profilesListChecksum
				if (dat.profilesListChecksum !== '') {

					profilesListChecksum = dat.profilesListChecksum;

					// Update pageInfo.profilesListChecksum
					pageInfo.profilesListChecksum = profilesListChecksum;

//					console.log('Using new profilesListChecksum: ' + profilesListChecksum);
				}
				else {

					// No new checksum requested, use pageInfo.profilesListChecksum.
					profilesListChecksum = pageInfo.profilesListChecksum;

//					console.log('Using existing profilesListChecksum: ' + profilesListChecksum);
				}

				profilesStorage.addNewProfilesDb(
					profilesListChecksum,
					numAllProfiles,
					numPermittedProfiles,
					profilesDb);

				profilesDashboard.createDashboardData();
				profilesList.create(profilesDb);
				updateProfilesDisplay();
				profilesList.checkDashboardDataForCurrentView();
			}
			else {

				// There is one or more invalid profiles
				profilesList.createWithInvalidProfiles(profilesDb);
				updateProfilesDisplay();

				// Handle active profile conversion, if any.
				if (pageInfo.profileConversionState.isActive) {
					convertProfile.handleActiveProfileConversion();
				}
			}

			busyPanel.stop();
		}
	}

	function updateProfilesDisplay() {

		// New profile data have been loaded or a profile has been deleted,
		// renamed or copied.

		util.hideE([
			'profiles:no_profile_info',
			'max_profiles_exceeded_info',
			'profiles:v7_profiles_info',
			'profiles:convert_profiles_info',
			'profiles_profiles_list',
			'saving_info',
			'deleting_info'
		]);

		if (numAllProfiles > 0) {

			if (isInvalidVersion7Profiles) {

				util.showE('profiles:v7_profiles_info');
			}

			if (isInvalidVersion8Profiles) {

				// Set text in convert_profiles_btn
				util.showE('profiles:convert_all_profiles_info', !isValidProfiles);
				util.showE('profiles:convert_some_profiles_info', isValidProfiles);

				util.showE('profiles:convert_profiles_info');


				if (pageInfo.isRootAdmin) {
					// util.enableE('profiles:convert_profiles_info', isInvalidVersion8Profiles && !convertProfile.isActive);
					convertProfilesBtn.enable();
				}
			}

			if (isValidProfiles || isInvalidVersion8Profiles) {

				util.showE('profiles_profiles_list');
			}

			// Enable/disable profiles filters input element
			util.enableE('profiles:filter', !isInvalidProfiles);
		}
		else {

			util.showE('profiles:no_profile_info');
		}

		// Check if there is a limit on the number of profiles
		// Note, check for number_of_licensed_profiles element existence because the element is not always available
		if (numLicensedProfiles !== 0 && (util.getE('licensed_profiles_info') !== null)) {

			// Update number of licensed profiles
			util.updateT('licensed_profiles_info:number_of_licensed_profiles', numLicensedProfiles);
			util.updateT('licensed_profiles_info:number_of_used_profiles', numAllProfiles);
			util.showE('licensed_profiles_info');

			// If all profiles exceeds number of licensed profiles
			if (numAllProfiles > numLicensedProfiles) {

				util.updateT('max_profiles_exceeded_info:number_of_profiles', numAllProfiles);
				util.updateT('max_profiles_exceeded_info:number_of_licensed_profiles', numLicensedProfiles);
				util.showE('max_profiles_exceeded_info');
			}
		}

		// TODO, we may need to wrap !convertProfile.isActive into a function!
		// ToDO, revise newProfileBtn permission
		newProfileBtn.enable(!isInvalidVersion8Profiles && !convertProfile.isActive);
		viewBtn.enable(!isInvalidVersion8Profiles && !convertProfile.isActive);

		util.showE('form_section');
	}

	function scrollingList(verticalScrollLevel) {

//		console.log('scrollingList() fired - verticalScrollLevel: ' + verticalScrollLevel);
		// ToDO
//		getProfileData();
	}

	function profileLinkActivated(evt) {

		// Event on profiles list table, used for menus, database info and
		// delete when showing simple rows which include invalid profiles.
		var element = evt.target || evt.srcElement;

//		console.log('element clicked: ' + element.id);
//		console.log('element node: ' + element.nodeName);
//		console.log('element parent node: ' + element.parentNode.nodeName);
//		console.log('element parent node ID: ' + element.parentNode.id);

		// Check if we need to handle the parent element,
		// for example multiple span elements within an anchor element
		// where id is defined on anchor element.
		if (element.id === '' && element.parentNode.id !== '') {
			element = element.parentNode;
		}

		var elementId = element.id;
//		var profileItem = {};

		var pattern = /^.+:(menu|db|delete)$/;

		if (pattern.test(elementId)) {

			var dat = elementId.split(':');
			var profileName = dat[1];
			var operationKeyWord = dat[2];

			switch (operationKeyWord) {

			case 'menu':

				profilesUtil.openProfileOptionsMenu(element, profileName);
				break;

			case 'db':

				// Handle update database, build database and database info
				database.toggleDatabaseMenu(element, profileName);

				break;

			case 'delete':

				// This delete button is only activated via simple rows
				// in case of invalid version 7 or version 8 profiles.
				deleteProfile.confirmDelete(profileName);
				break;

//			case 'toggle_invalid_profile_text':
//
//				toggleInvalidProfileText(profileName);
//				break;
			}
		}
	}

	function createNewProfile() {

		// Verify that width and height are within available viewport size
		var viewPortWidth = YD.getViewportWidth();
		var viewPortHeight = YD.getViewportHeight();

		var width = (770 < viewPortWidth) ? 770 : viewPortWidth;
		var height = (510 < viewPortHeight) ? 510 : viewPortHeight;

		var windowName = 'new_profile_wizard';
		var url = '?dp=new_profile_wizard.index';

		var left = parseInt((screen.availWidth / 2) - (width / 2), 10);
		var top = parseInt((screen.availHeight / 2) - (height / 2), 10);
		newProfileWizardWindow = window.open(url, windowName, 'width=' + width + ',height=' + height + ',left=' + left + ',top=' + top + ', status=yes, scrollbars=yes, resizable=yes');
		newProfileWizardWindow.focus();
	}

	function closeNewProfileWizardWindow() {

		// Close new profile wizard window
		// Hack for Firefox problem which doesn't close the window immediately,
		// though blur moves it to the background!

		newProfileWizardWindow.blur();
		newProfileWizardWindow.close();
	}

	function handleDeleteProfileCompleted(profileChanges) {

//		util.showObject(profileChanges);

		if (!profileChanges.isNewProfilesList) {

			// We got a new profilesListChecksum but don't
			// need a new list. Delete the profile from
			// profilesDb and update local storage
			// with the new checksum and profilesDb so that
			// it is in sync with the server side.

			var profilesListChecksum = profileChanges.profilesListChecksum;

			// Update pageInfo.profilesListChecksum
			pageInfo.profilesListChecksum = profilesListChecksum;

			var deletedProfileName = profileChanges.profileName;
			for (var i = 0, len = profilesDb.length; i < len; i++) {
				if (profilesDb[i][0] === deletedProfileName) {
					profilesDb.splice(i, 1);
					break;
				}
			}

			numAllProfiles = numAllProfiles - 1;
			numPermittedProfiles = numPermittedProfiles - 1;

			if (!isInvalidProfiles) {

				// Update local storage
				profilesStorage.addNewProfilesDb(
					profilesListChecksum,
					numAllProfiles,
					numPermittedProfiles,
					profilesDb);

				profilesList.create(profilesDb);
				updateProfilesDisplay();
				profilesList.checkDashboardDataForCurrentView();

			}
			else {

				// The list contained one or more invalid profiles.
				// Recheck profilesDb for invalid profiles
				setIsInvalidProfiles();


				// If there are still invalid profiles
				if (isInvalidProfiles) {

					profilesList.createWithInvalidProfiles(profilesDb);
					updateProfilesDisplay();
				}
				else {

					// No more invalid profile, reload the profiles.
					getProfileData();
				}
			}
		}
		else {

			// We got a new profilesDb

			if (!isInvalidProfiles) {

				handleNewProfilesDb(profileChanges);
			}
			else {

				// The list contains or contained invalid profiles
				// and we got a new profilesDB, this use case should be very rare.
				// For convenience reload the profiles.
				location.reload(true);
			}
		}
	}

	function handleNewProfile(profileChanges) {

		// This is called after a new profile has been added
		// via new profile wizard or duplicate profile.

//		console.log('handleNewProfile - profileChanges.isNewProfilesList: ' + profileChanges.isNewProfilesList);

		if (!profileChanges.isNewProfilesList) {

			// Add a single new profile item
			// to existing profilesDb.

//			util.showObject(profileChanges, 'profileChanges');

			var profilesListChecksum = profileChanges.profilesListChecksum;

			// Add item to profilesDb
			profilesDb[profilesDb.length] = profileChanges.profileItem;

			// Sort profilesDb
			profilesDb.sort(compareProfileLabels);

			// Note, we must re-add all items to profilesDb after sort
			// so that we get the right lookup names!
			var theProfiles = util.cloneObject(profilesDb);
			profilesDb = [];
			addProfilesToProfilesDb(theProfiles);
			theProfiles = [];

			numAllProfiles = numAllProfiles + 1;
			numPermittedProfiles = numPermittedProfiles + 1;

			// Update local storage
			profilesStorage.addNewProfilesDb(
				profilesListChecksum,
				numAllProfiles,
				numPermittedProfiles,
				profilesDb);


			// TODO - Revise how to handle dashboard after addition of a new profile
			// since we don't want to reload all data.
			profilesDashboard.createDashboardData();

			profilesList.create(profilesDb);
			updateProfilesDisplay();

			profilesList.checkDashboardDataForCurrentView();
		}
		else {

			// We got a new profilesDb
			handleNewProfilesDb(profileChanges);
		}
	}

	function handleNewProfilesDb(profileChanges) {

		// We received a new profilesDb after delete, rename, copy or new
		profilesDb = [];

		var profilesListChecksum = profileChanges.profilesListChecksum;

		addProfilesToProfilesDb(profileChanges.profilesDb);

		// Sort profilesDb
		profilesDb.sort(compareProfileLabels);

		numAllProfiles = profileChanges.numAllProfiles;
		numPermittedProfiles = profileChanges.numPermittedProfiles;

		// Update local storage
		profilesStorage.addNewProfilesDb(
			profilesListChecksum,
			numAllProfiles,
			numPermittedProfiles,
			profilesDb);

		// TODO - Revise how to handle dashboard after addition of a new profile
		// since we don't want to reload all data.
		profilesDashboard.createDashboardData();

		profilesList.create(profilesDb);
		updateProfilesDisplay();

		profilesList.checkDashboardDataForCurrentView();
	}

	function editView() {

		// Don't allow editView while we have invalid profiles.
		// This is a workaround for the dropdown menu which is functional
		// although the  button is disabled.
		// TODO - Remove isInvalidProfiles check when adding other drop down menus.
		if (!isInvalidProfiles) {

			editViewActive = true;

			newProfileBtn.disable();
			viewBtn.disable();

			profilesList.clear();

			// Remove scroll listener in profilesList
			profilesList.editViewActivated();

			// Create edit profiles view
			var theProfiles = getActiveProfilesDb();
			profilesDashboardEditor.create(theProfiles);

			// Close database info, if any
			database.closePanel();
		}
	}

	function editViewClosed(isCancel) {

		editViewActive = false;

		// Update dashboard data
		profilesDashboard.createDashboardData();

		var theProfiles = getActiveProfilesDb();
		profilesList.create(theProfiles);
		updateProfilesDisplay();

		newProfileBtn.enable();
		viewBtn.enable();

		profilesList.checkDashboardDataForCurrentView();

//		if (!isCancel) {

			// TODO
			// the profilesView has been saved and most likely changed.
			// Handle profilesView data update?
//		}
	}

	function profilesDashboardDataUpdated() {

		// profilesDashboard data have been updated.
		// Rebuild the current view
		var theProfiles = getActiveProfilesDb();
		profilesList.create(theProfiles);
	}

	function createFilteredProfilesList(theProfiles) {

		// Called from profilesFilter

		if (!editViewActive) {

//			console.log('createFilteredProfilesList() - theProfiles: ' + theProfiles);

			profilesList.create(theProfiles);
			profilesList.checkDashboardDataForCurrentView();
		}
		else {
			profilesDashboardEditor.update(theProfiles);
		}
	}

	// Toggle profiles filter
	function toggleProfilesFilter(evt) {

		// Don't allow toggle while we have invalid profiles.
		// This is a workaround for the drop-down menu which is functional
		// although the  button is disabled.
		// TODO - Remove isInvalidProfiles check when adding better drop down menus.
		if (!isInvalidProfiles) {

			var profilesFilter = util.getE('profiles:profiles_filter_box');
			var showProfilesFilter = profilesFilter.className === 'profiles-filter-box-hidden';

			var btnText = '';

			if (showProfilesFilter) {

				btnText = langVar('lang_stats.btn.hide_profiles_filter');
				profilesFilter.className = 'profiles-filter-box';
			}
			else {

				btnText = langVar('lang_stats.btn.show_profiles_filter');
				profilesFilter.className = 'profiles-filter-box-hidden';
			}

			util.updateT('profiles_toolbar:drop_down:toggle_profiles_filter_btn', btnText);

			// Set users_cache preferences value
			var url = '?dp=admin_pages.profiles.toggle_profiles_filter';
			var dat = 'v.fp.page_token=' + pageInfo.pageToken;
			dat += '&v.fp.show_profiles_filter=' + showProfilesFilter;

			util.serverPost(url, dat);
		}
	}

	function toggleProfilesFilterResponse() {
		return false;
	}

	// Toggle Database Info or Created By column
	function toggleDatabaseInfoColumn(evt) {

		var showColumn = !pageInfo.showDatabaseInfoColumn;
		var btnText = '';

		if (showColumn) {

			btnText = langVar('lang_admin.profiles.hide_database_info_column');
		}
		else {

			btnText = langVar('lang_admin.profiles.show_database_info_column');
		}

		util.updateT('profiles_toolbar:drop_down:toggle_db_info_column_btn', btnText);

		pageInfo.showDatabaseInfoColumn = showColumn;

		// Save state in user preferences
		saveColumnStateInUserPreferences('profiles.show_database_info_column=' + showColumn);

		// Re-create profiles view
		profilesDashboard.createDashboardData();
		profilesList.create(profilesDb);
		updateProfilesDisplay();
		profilesList.checkDashboardDataForCurrentView();
	}

	function toggleCreatedByColumn(evt) {

		var showColumn = !pageInfo.showCreatedByColumn;
		var btnText = '';

		if (showColumn) {

			btnText = langVar('lang_admin.profiles.hide_created_by_column');
		}
		else {

			btnText = langVar('lang_admin.profiles.show_created_by_column');
		}

		util.updateT('profiles_toolbar:drop_down:toggle_created_by_column_btn', btnText);

		pageInfo.showCreatedByColumn = showColumn;

		// Save state in user preferences
		saveColumnStateInUserPreferences('profiles.show_created_by_column=' + showColumn);

		profilesList.create(profilesDb);
		updateProfilesDisplay();
		profilesList.checkDashboardDataForCurrentView();
	}

	function saveColumnStateInUserPreferences(userPreferencesProperty) {

		var url = '?dp=util.set_and_save_node';
		var dat = 'v.fp.active_page=' + pageInfo.page;
		dat += '&v.fp.page_token=' + pageInfo.pageToken;
		dat += '&v.fp.main_node_name=users_cache_preferences';
		dat += '&v.fp.new_nodes.' + userPreferencesProperty;

		util.serverPost(url, dat);
	}

	function refreshView() {

		// This clears local storage and reloads all profile data
		var isGetNewChecksum = true;
		profilesList.clear();
		profilesStorage.reset();
		getProfileData(isGetNewChecksum);
	}


	//
	// Utilities
	//

	function setIsInvalidProfiles() {

		// Reset isValid/isInvalid properties
		isInvalidVersion7Profiles = false;
		isInvalidVersion8Profiles = false;
		isInvalidProfiles = false;
		isValidProfiles = false;

		for (var i = 0, len = profilesDb.length; i < len; i++) {

			var item = profilesStorage.profileArrayItemToObject(profilesDb[i]);

//			util.showObject(item);

			if (item.isValidProfile) {

				isValidProfiles = true;
			}
			else if (item.version === '8.0') {

				// This must be an invalid version 8.0 or 8.1 profile
				isInvalidVersion8Profiles = true;
			}
			else {

				// This must be an invalid version 7.0 profile
				isInvalidVersion7Profiles = true;
			}
		}

		isInvalidProfiles = isInvalidVersion8Profiles || isInvalidVersion7Profiles;

//		console.log('isInvalidProfiles: ' + isInvalidProfiles);
	}

	function addProfilesToProfilesDb(theProfiles) {

		// This adds theProfiles to profilesDb. It also creates
		// a hash to access a profile by profile name.

//		util.showObject(theProfiles, 'addProfilesToProfilesDb()');
		var len = theProfiles.length;

		if (len > 0) {

			// Add profile items to profilesDb
			for (var i = 0; i < len; i++) {

				var item = theProfiles[i];

				var lookupName = '_' + item[0];

				var targetIndex = profilesDb.length;
				profilesDb[targetIndex] = item;

				// Create lookup by profile name
				profilesDb[lookupName] = profilesDb[targetIndex];
			}
		}
	}

	function getActiveProfilesDb() {

		// Returns a filtered profilesDb if filter is active,
		// or the regular profilesDb if no filter is active.
		var obj = profilesFilter.getFilteredProfilesDb();

		if (obj.isFiltered) {
			return obj.filteredProfilesDb;
		}

		return profilesDb;
	}

	function getProfilesDb() {
		return profilesDb;
	}

	function getProfileItem(profileName) {

		var lookupName = '_' + profileName;
		return profilesStorage.profileArrayItemToObject(profilesDb[lookupName]);
	}

	function compareProfileLabels(a, b) {

//		console.log('sorting profiles ...');

		var labelA = a[1].toLowerCase();
		var labelB = b[1].toLowerCase();
		if (labelA < labelB) {
			return -1;
		}
		else if (labelA > labelB) {
			return 1;
		}
		return 0;
	}

	function getIsEditViewActive() {
		return editViewActive;
	}

	function toggleToolbarMenu() {
		// This is a fake function for the View button drop down.
		// TODO - remove this when adding better drop down menu support.
	}

	/*
		Show before you start info utilities
	*/

	function toggleShowBeforeYouStartSection(evt) {

		var element = evt.target || evt.srcElement;
		var elementId = element.id;
		var sectionElement = util.getE(elementId + ':section');
//		console.log('elementId: ' + elementId);

		sectionElement.style.display = (sectionElement.style.display === 'block') ? 'none' : 'block';
		element.blur();
	}

	function toggleDoNotShowAgain(evt) {

		// If checked this changes the text of the Hide Messages link
		// to "Hide And Don't Show Again"
        var element = evt.target || evt.srcElement;
		var text = '';

		if (element.checked) {
			text = langVar('lang_stats.btn.hide_and_do_not_show_again');
		}
		else {
			text = langVar('lang_stats.btn.hide_messages');
		}

		util.updateT('before_you_start:hide_btn', text);
	}

	function hideBeforeYouStartSection(evt) {

		// If don't show again is checked set appropriate system node
		if (util.getF('before_you_start:do_not_show_again')) {

			var url = '?dp=util.set_and_save_node';
			var dat = 'v.fp.active_page=' + pageInfo.page;
			dat += '&v.fp.page_token=' + pageInfo.pageToken;
			dat += '&v.fp.main_node_name=system';
			dat += '&v.fp.new_nodes.show_before_you_start_info.profiles=false';

			util.serverPost(url, dat);
		}

		util.hideE('before_you_start:panel');
	}

	// Return global properties and methods
	return {
		init: init,
		createNewProfile: createNewProfile,
		handleDeleteProfileCompleted: handleDeleteProfileCompleted,
		handleNewProfile: handleNewProfile,
		handleNewProfilesDb: handleNewProfilesDb,
		getProfileData: getProfileData,
		getProfileDataResponse: getProfileDataResponse,
//		scrollingList: scrollingList,
		getProfilesDb: getProfilesDb,
		getProfileItem: getProfileItem,
		createFilteredProfilesList: createFilteredProfilesList,
		closeNewProfileWizardWindow: closeNewProfileWizardWindow,
		editViewClosed: editViewClosed,
		profilesDashboardDataUpdated: profilesDashboardDataUpdated,
		getIsEditViewActive: getIsEditViewActive,
		toggleProfilesFilterResponse: toggleProfilesFilterResponse
	};

}());

YAHOO.util.Event.onDOMReady(profiles.init);
