/* global
	profiles: false,
	profilesUtil: false,
	profilesStorage: false */

var profilesDashboard = (function() {

	'use strict';

	// This handles profiles dashboard data

	// dashboardData
	// contains databaseInfo and report data.
	// dashboardData is created on client only.
	//
	// [
	// 		{
	// 		name: profile_name
	// 		viewDashboard: view report data permission
	// 		viewDatabaseInfo: view database info permission
	// 		dbInfo: [
	// 			bool isLoaded,
	// 			int db_state 0=not build 1=building 2=built,
	// 			int last_modification_time in epoc
	// 			]
	//		reportCellsCode:
	// 			0 = ignore report cells,
	// 			1 = has or possibly has report cells (will be verified on server)
	// 			2 = requested/generating (labels are already loaded, report is generated on server)
	// 			3 = loaded
	//			// TODO - how to handle report errors?
	//		reportCells: [[bool showValue, string label, string value], [showValue], [showValue, label, value]]
	//		}
	//		showValue in report cell means that a value can be shown or will be shown after loading.
	// ]
	var dashboardData = [];

	var profilesInViewRef = [];
//	var lastRequestDateInMilSec = 0;

	function createDashboardData() {

		// ToDO, if dashboardData already exists
		// then create a backup and use its values for
		// the new one. This would avoid reloading of
		// all database and report data.
		// Though fresh data would'nt be bad either!
		// Report data are cached anyway.

		var isRootAdmin = pageInfo.isRootAdmin;
		var isViewDashboardPermission = isRootAdmin;
		var isViewDatabaseInfoPermission = isRootAdmin;

		dashboardData = [];
		var profilesDb = profiles.getProfilesDb();
		var dashboardView = pageInfo.profilesDashboardView;

//		util.showObject(dashboardView, 'createDashboardData() - dashboardView');

//		var showDatabaseInfo = dashboardView.show_database_info;
		var showDatabaseInfo = pageInfo.permissions.isViewDatabaseInfoColumn && pageInfo.showDatabaseInfoColumn;

//		console.log('showDatabaseInfo: ' + showDatabaseInfo);
		var columns = dashboardView.columns;
		var numColumns = columns.length;

		if (showDatabaseInfo || numColumns > 0) {

			// Handle basic data
			for (var i = 0, len = profilesDb.length; i < len; i++) {

				var profileItem = profilesStorage.profileArrayItemToObject(profilesDb[i]);

				var profileName = profileItem.name;
				var obj = {name:profileName};

				// Check if the profile has permission to view dashboard data.
				if (!isRootAdmin) {

					var profileGrants = profilesUtil.getProfileGrants(profileName);

//					util.showObject(profileGrants);

					isViewDashboardPermission = profileGrants.viewProfilesDashboard;
					isViewDatabaseInfoPermission = profileGrants.viewDatabaseInfoColumn;
				}

				obj.viewDashboard = isViewDashboardPermission;
				obj.viewDatabaseInfo = isViewDatabaseInfoPermission;

				// Always set/get database info because it is required for
				// report generation anyway. So the database info becomes
				// automatically loaded when one or more report columns
				// are shown.
				obj.dbInfo = [false, '', ''];

				// Handle report fields defined in columns
				// which are valid for all rows, unless overridden per profile.

				// Handle columns where report_field is defined
				// in column header, respectively set default
				// columns object.
				if (numColumns > 0) {

					_setDefaultReportCells(columns, numColumns, obj);
				}

				dashboardData[i] = obj;


				// Create lookup by profile name
				var lookupName = '_' + profileName;
				dashboardData[lookupName] = dashboardData[i];
			}

			// Handle individual profile cells
			_setDefaultProfileReportCells(columns, numColumns);
		}

//		util.showObject(dashboardData, 'dashboardData');
	}

	function _setDefaultReportCells(columns, numColumns, obj) {

		// Sets default columns in obj.

		var reportCellsCode = 0; // Default 0=ignore columns
		var reportCells = [];

		for (var i = 0; i < numColumns; i++) {

			var colItem = columns[i];

			// console.log('colItem.expression: ' + colItem.expression);

			if (obj.viewDashboard && colItem.report_field !== '') {

				reportCells[i] = [true, '', ''];

				// Set reportCellsCode to "1 = has or possibly has report cells"
				// (this applies to all cells in row)
				reportCellsCode = 1;
			}
			else {
				reportCells[i] = [false];
			}
		}

		obj.reportCellsCode = reportCellsCode;
		obj.reportCells = reportCells;
	}

	function _setDefaultProfileReportCells(columns, numColumns) {

		// This sets individual profile rows which are defined
		// in dashboardView.columns[i].profiles.
		// Individual profile columns are only loaded if the user has
		// permission. So we don't need to check for permissions here.

//		util.showObject(columns, '_setDefaultProfileReportCells / num of columns: ' + numColumns);

		for (var i = 0; i < numColumns; i++) {

			var colItem = columns[i];
			var colItemProfiles = colItem.profiles;

//			util.showObject(colItemProfiles, 'colItemProfiles');

			for (var k = 0, len = colItemProfiles.length; k < len; k++) {

				var cellItem = colItemProfiles[k];

//				util.showObject(cellItem, 'cellItem');
//				console.log('dashboardItem.reportCellsCode: ' + dashboardItem.reportCellsCode);

				if (cellItem.report_field !== '') {

					var profileName = cellItem.profile_name;
					var dashboardItem = dashboardData[('_' + profileName)];

//					util.showObject(dashboardItem, 'dashboardItem');

					// Set reportCellsCode "1 = has or possibly has report cells"
					// (applies global to all columns in row)
					dashboardItem.reportCellsCode = 1;

					// Set showCell=true
					dashboardItem.reportCells[i] = [true, '', ''];
				}
			}
		}
	}

	function getDashboardDataRef() {

		return dashboardData;
	}

	function getDashboardItemDbState(profileName) {

		var dbInfo = dashboardData['_' + profileName].dbInfo;
		return dbInfo[1];
	}

	function updateDashboardItemDbState(profileName, dbState, dbLastMod, dbOperationCompleted) {

		// This changes the database state. It is called when receiving database
		// info or when updating/building the database.
		var dashboardItem = dashboardData['_' + profileName];

//		util.showObject({dbOperationCompleted:dbOperationCompleted});
//		util.showObject(dashboardItem);

		var dbInfo = dashboardItem.dbInfo;
		dbInfo[1] = dbState;
		dbInfo[2] = dbLastMod;

		// Update profiles row
		profilesList.updateDbInfoCell(profileName, dbState, dbLastMod);

		// Update or Build database completed, update report data, if any.
		if (dbOperationCompleted && dashboardItem.hasOwnProperty('reportCellsCode')) {

			var reportCellsCode = dashboardItem.reportCellsCode;

			// reportCellsCode could be 0 because no database has been built,
			// so we need to ignore reportCellsCode!
//			if (reportCellsCode !== 0) {

				// Update profiles row
//				profilesList.setReportCellsToChangedDataState(profileName);

			// Reset reportCells to 1
			dashboardItem.reportCellsCode = 1;
			_verifyDashboardDataForProfilesInViewFinal();
		}
	}

	function verifyDashboardDataForProfilesInView(profilesInView) {

		// This checks if all dashboard data exist for the current
		// visible profiles. If all data exist then no further action
		// is required, else the missing data are loaded and the
		// profilesList becomes updated.

		// Keep a reference to profilesInView
		profilesInViewRef = profilesInView;

		_verifyDashboardDataForProfilesInViewFinal();
	}

	function _verifyDashboardDataForProfilesInViewFinal() {

		// This checks if all dashboard data exist and are up to date
		// for the current visible profiles.
		// If all data exist and if they are up to date then no further
		// action is required, else the missing data are loaded and the
		// profilesList becomes updated.

//		var dashboardView = pageInfo.profilesDashboardView;

//		console.log('_verifyDashboardDataForProfilesInViewFinal()')
//		util.showObject(dashboardView);
//		util.showObject(profilesInViewRef, 'profilesInViewRef');
//		util.showObject(dashboardData, 'dashboardData');

		// var showDatabaseInfo = dashboardView.show_database_info;
		var showDatabaseInfo = pageInfo.permissions.isViewDatabaseInfoColumn && pageInfo.showDatabaseInfoColumn;
//		console.log('showDatabaseInfo 2: ' + showDatabaseInfo);

		var profilesDat = '';

		// It is possible that there are no dashboardData
		if (dashboardData.length > 0) {

			for (var i = 0, len = profilesInViewRef.length; i < len; i++) {

				var profileName = profilesInViewRef[i];

				var dashboardItem = dashboardData[('_' + profileName)];

//				util.showObject(dashboardItem);

				// If view permission
				if (dashboardItem.viewDashboard || dashboardItem.viewDatabaseInfo) {

					var dbInfo = dashboardItem.dbInfo;
					var dbInfoLoaded = dbInfo[0];
					var dbIsBuilding = (dbInfoLoaded && dbInfo[1] === 1);

					var reportCellsCode = dashboardItem.reportCellsCode;
					var isPossibleReportData = (reportCellsCode === 1);
					var waitingForReportData = (reportCellsCode === 2);

					var requiresDbInfo = (
						(dashboardItem.viewDatabaseInfo && showDatabaseInfo && !dbInfoLoaded) ||
						dbIsBuilding ||
						isPossibleReportData);

					if (requiresDbInfo || isPossibleReportData || waitingForReportData) {

						var path = '&v.fp.profiles.' + profileName;

						profilesDat += path + '.requires_db_info=' + requiresDbInfo;
						profilesDat += path + '.is_possible_report_data=' + isPossibleReportData;
						profilesDat += path + '.waiting_for_report_data=' + waitingForReportData;
					}
				}
			}
		}

		if (profilesDat !== '') {

			_getDashboardData(profilesDat);
		}
	}


	function _getIsLoadDashboardItemData(showDatabaseInfo, dashboardItem) {

		// Returns true if data for dashboardItem needs to be loaded.

		if ((showDatabaseInfo && !dashboardItem.dbInfo[0]) ||
			(dashboardItem.reportCellsCode === 1) ||
			dashboardItem.reportCellsCode === 2) {
			return true;
		}

		return false;
	}

	function _getDashboardData(profilesDat) {

		// Load dashboard data
		var url = '?dp=admin_pages.profiles_dashboard.get_dashboard_data';

		var dat = 'v.fp.active_page=' + pageInfo.page;
		dat += '&v.fp.page_token=' + pageInfo.pageToken;
		dat += profilesDat;

		util.serverPost(url, dat);
	}

	function getDashboardDataResponse(dat) {

		// Update dashboardData
//		util.showObject(dat, 'getDashboardDataResponse()');

		var dashboardView = pageInfo.profilesDashboardView;

//		util.showObject(dashboardView, 'dashboardView');

		var numColumns = dashboardView.columns.length;
		var isProcessing = false;

		// Create defaultReportCells
		var defaultReportCells = [];
		for (var k = 0; k < numColumns; k++) {
			defaultReportCells[k] = [false];
		}

		for (var i = 0, len = dat.length; i < len; i++) {

			var datItem = dat[i];
			var profileName = datItem.name;
			var dbBuilt = datItem.dbBuilt;
			var dbBuilding = datItem.dbBuilding;
			var dbLastMod = datItem.dbLastMod;

			var dbState = 0;

			if (dbBuilding) {
				dbState = 1;
				isProcessing = true;
			}
			else if (dbBuilt) {
				dbState = 2;
			}

			// var newReportCells = [];

			var dashboardItem = dashboardData['_' + profileName];
			var dbInfo = dashboardItem.dbInfo;

			dbInfo[0] = true; // isLoaded
			dbInfo[1] = dbState;
			dbInfo[2] = dbLastMod;

			// Set reportCells
//			console.log('profileName: ' + profileName);
//			console.log('reportCellsCode: ' + reportCellsCode);

			if (numColumns > 0) {

				var reportState = datItem.reportState;
				var reportCells = datItem.reportCells;

				if (reportState === 'ignore') {

					dashboardItem.reportCellsCode = 0;
					dashboardItem.reportCells = util.cloneObject(defaultReportCells);

				}
				else if (reportState === 'update') {

					// Report has been requested, labels are available, if any.
					dashboardItem.reportCellsCode = 2;
					dashboardItem.reportCells = util.cloneObject(reportCells);

					isProcessing = true;
				}
				else if (reportState === 'ready') {

					dashboardItem.reportCellsCode = 3;
					dashboardItem.reportCells = util.cloneObject(reportCells);
				}
				else {

					// reportState === 'unchanged'
					// Report is not yet ready
					isProcessing = true;
				}
			}
		}

		// Update the profilesList if not in edit mode
		if (!profiles.getIsEditViewActive()) {

			profiles.profilesDashboardDataUpdated();

			if (isProcessing) {

				// Re-check dashboard data
				setTimeout(function() {_verifyDashboardDataForProfilesInViewFinal()}, 1000);
			}
		}
	}

	// Public methods
	return {
		createDashboardData: createDashboardData,
		getDashboardDataRef: getDashboardDataRef,
		getDashboardItemDbState: getDashboardItemDbState,
		verifyDashboardDataForProfilesInView: verifyDashboardDataForProfilesInView,
		getDashboardDataResponse: getDashboardDataResponse,
		updateDashboardItemDbState: updateDashboardItemDbState
	};
}());

