/* global
	langVar: false,
	pageInfo: false,
	users: users,
	usersAM: false,
	usersUtil: false,
	util: false */

var usersUtil = (function() {

    'use strict';

	//
	// Access List Utilities
	//
	
	// Element Id's used in access table rows
	// The baseId, i.e. "acc0" does not specify the row number, it is random!
	// We get the active row number by checking the rows array of tbody!
	// ":op:" within the id stands for operate, so upon click on a cell which contains :op: we do something, else not
	// <tr id="acc0:tr" --> Id of access table row
	// <td id ="acc0:op:edit_profiles" --> Edit Access with profiles tab as default
	// <td id ="acc0:op:edit_roles" --> Edit Access with roles tab as default
	// <td id ="acc0:info" --> Info cell which contains text like "(by user 2)" or ("Owned by other user")
	// <td id ="acc0:op:edit" --> Edit Access with profiles tab as default
	// <td id ="acc0:op:delete" --> Delete the access item

    var isRootAdmin = false;
    var isEnterprise = false;
    var isUnlimitedGrants = false;

    function init() {

        isRootAdmin = pageInfo.isRootAdmin;
        isEnterprise = pageInfo.licensingTier === 'enterprise';
        isUnlimitedGrants = pageInfo.permissions.isUnlimitedGrants;
    }
	
	function accessListLinkActivated(evt) {
		
		// Note, we use the elementId to check if this is an operational button or link
		// and we use the className to check if the item is in enabled or disabled state!
		
		var element = evt.target || evt.srcElement;
		var elementId = element.id;
		var className = element.className;
		
		// alert('accessListLinkActivated - elementId: ' + elementId);
		
		if ((className.indexOf('disabled') === -1) && (elementId.indexOf(':op:') !== -1)) {
			
			// alert('linkActivated - elementId: ' + elementId);
			
			var dat = elementId.split(':');
			var baseId = dat[0];
			var accessOperation = dat[2];
			var accessItemIndex = _getRowNumberFromBaseId(baseId);
			
			// alert('accessListLinkActivated - accessItemIndex: ' + accessItemIndex);
			
			var activeAccess = users.get('activeAccess');
			
			//
			// Check if the user has permission to edit the item
			//

            var activeUserNodeName = users.get('activeUserNodeName');
			
			var isPermission = (isRootAdmin || isUnlimitedGrants);
			if (!isPermission) {
				isPermission = (activeAccess[accessItemIndex].created_by_user === activeUserNodeName);
			}
			
			
			if (isPermission) {
			
				if (accessOperation !== 'delete') {
					
					// alert('open access manager');
					
					var activeTabIsRoles = (accessOperation === 'edit_roles');
					var isNewAccessPair = false;
					var accessItem = util.cloneObject(activeAccess[accessItemIndex]);
					usersAM.open(accessItemIndex, accessItem, activeTabIsRoles, isNewAccessPair);
				}
				else {
					
					//
					// Delete item
					//
			
					// Delete item from activeAccess array
					activeAccess.splice(accessItemIndex, 1);
					
					// Delete last row (we have to update the rows anyway due delete button display, so we simply delete the last row)
					var tbody = util.getE('users:access:tbody');
					var lastTr = tbody.lastChild;
					tbody.removeChild(lastTr);
					
					// Update all rows
					var numberOfAccessItems = activeAccess.length;

					for (var i = 0; i < numberOfAccessItems; i++) {

						updateAccessRow(i, activeAccess[i], numberOfAccessItems);
					}
				}
			}
			else {
				
				alert(langVar('lang_admin.users.no_permission_msg'));
			}
		}
    }
	
	function hoverAccessListItem(evt) {
		
		var element = evt.target || evt.srcElement;
		var elementId = element.id;
		var className = element.className;
		
		// Note, we use the elementId to check if this is an operational button or link
		// and we use the className to check if the item is in enabled or disabled state!
		
		// Hover off upon mouseout must be always possible, even if disabled because
		// it possible that we accidently hover a disabled item after delete.
		
		// alert('hoverAccessListItem - elementId: ' + elementId + '\nevt.type: ' + evt.type);
		
		if (elementId.indexOf(':op:') !== -1) {
			
			var eventType = evt.type;
			var isMouseOver = (evt.type === 'mouseover');
			
			if (!isMouseOver || (className.indexOf('disabled') === -1)) {
							
				element.style.backgroundColor = isMouseOver ? '#F2F2F2' : 'White';
				element.style.cursor = isMouseOver ? 'pointer' : 'default';
			}
		}
	}
	
	function _getSortedAccessItemProfileLabels(numberOfProfiles, profiles) {
		
		var profilesDb = users.get('profilesDb');
        var profileName;
        var profileItem;
		
		var a = [];
		
		for (var i = 0; i < numberOfProfiles; i++) {
			
			profileName = profiles[i];
			profileItem = profilesDb[util.h(profileName)];
			var profileLabel = (profileItem !== null) ? profileItem.label : langVar('lang_admin.users.unknown_profile');
			
			a[i] = (profileItem !== null) ? profileItem.label :
                langVar('lang_admin.users.unknown_profile');
		}

		a.sort();
		
		return a;
	}
	
	function _cleanUpAccessCell(element) {
		
		// alert('cleanUpAccessCell - element: ' + element);
		while (element.lastChild !== null) {
			var theChild = element.lastChild;
			element.removeChild(theChild);
		}
	}
	
	function updateAccessRow(index, accessItem, numberOfAccessItems) {

		// util.showObject(accessItem);

		var br;
		
		var rolesDb = users.get('rolesDb');

		var activeUserNodeName = users.get('activeUserNodeName');
		var createdByUserNodeName = accessItem.created_by_user;
		
		var isOwnedByOtherUser = (!isRootAdmin && !isUnlimitedGrants && (activeUserNodeName !== createdByUserNodeName));
		
		
		//
		// Get the baseId of the row
		// 
		
		var tbody = util.getE('users:access:tbody');
		var allRows = tbody.getElementsByTagName('tr');
		var trId = allRows[index].id;

		var dat = trId.split(':');
		var baseId = dat[0];

		
		//
		//
		// Handle Profiles cell
		//
		//
		
		var isAllProfiles = accessItem.all_profiles;
		var profiles = accessItem.profiles;
		var numberOfProfiles = profiles.length;
		
		var profileTd = util.getE(baseId + ':op:edit_profiles');

        // Don't remove, the className is used to check hover and link
        // activation, both are ignored if the className contains disabled!
		profileTd.className = !isOwnedByOtherUser ? '' : 'disabled';
		
		_cleanUpAccessCell(profileTd);
		
		var profileLabel;
		var profileText;
		
		if (isAllProfiles || (numberOfProfiles === 0)) {
			
			profileLabel = isAllProfiles ? langVar('lang_admin.users.all_profiles') : '[ ' + langVar('lang_admin.users.no_profile_defined') + ' ]';
			profileText = document.createTextNode(profileLabel);
			profileTd.appendChild(profileText);
		}
		else {
			
			// One or more profiles in list
			// KHP 14/Sep/2010 - Make sure we get sorted labels
			var sortedProfileLabels = _getSortedAccessItemProfileLabels(numberOfProfiles, profiles);
			
			for (var i = 0; i < numberOfProfiles; i++) {
				
				profileText = document.createTextNode(sortedProfileLabels[i]);
				
				profileTd.appendChild(profileText);
				
				// Add br tag if this is not the last profile
				if (numberOfProfiles > 1 && i + 1 !== numberOfProfiles) {

					br = document.createElement('br');
					profileTd.appendChild(br);
				}
			}
		}
		
		//
		//
		// Handle Roles cell
		//
		//
		
		var roleTd = util.getE(baseId + ':op:edit_roles');
		roleTd.className = !isOwnedByOtherUser ? '' : 'disabled';
		_cleanUpAccessCell(roleTd);
		
		var roles = accessItem.roles;
		var numberOfRoles = roles.length;
		
		var roleName;
		var roleLabel;
		var roleText;
		
		if (numberOfRoles > 0) {
			
			for (var j = 0; j < numberOfRoles; j++) {
				
				roleName = roles[j];
				roleLabel = rolesDb[util.h(roleName)].label;
				roleText = document.createTextNode(roleLabel);
				roleTd.appendChild(roleText);
				
				// Add br tag if this is not the last role
				if (numberOfRoles > 1 && j + 1 !== numberOfRoles) {

					br = document.createElement('br');
					roleTd.appendChild(br);
				}
			}
		}
		else {
			
			// No role defined
			roleText = document.createTextNode('[ ' + langVar('lang_admin.users.no_role_defined') + ' ]');
			roleTd.appendChild(roleText);
		}
		

		//
		// Handle info cell
		//
		
		
		var infoTd = util.getE(baseId + ':info');
		_cleanUpAccessCell(infoTd);
		var infoLabel = '';
		
		if (createdByUserNodeName === activeUserNodeName) {
			// No info, add &nbsp;
			infoLabel = '\u00a0';
		}
		else {
			if (!isOwnedByOtherUser) {
				// Add info by whom the access pair has been created
				var username = _getUsernameByUserNodeName(createdByUserNodeName);
				infoLabel = '(' + langVar('lang_admin.general.by_user')  + ')';
				infoLabel = infoLabel.replace(/__PARAM__1/, username);
			}
			else {
				// This are limited grants, there is no permission to edit this item
				infoLabel = '(' + langVar('lang_admin.users.owned_by_other_user') + ')';
			}
		}
		
		var infoText = document.createTextNode(infoLabel);
		infoTd.appendChild(infoText);
		
		
		//
		// Handle edit cell
		//
		
		var editTd = util.getE(baseId + ':op:edit');
		editTd.className = !isOwnedByOtherUser ? 'link' : 'link-disabled';
		
		
		//
		// Handle delete cell
		//
		
		if (isEnterprise) {
		
			var deleteTd = util.getE(baseId + ':op:delete');
			deleteTd.className = (!isOwnedByOtherUser && numberOfAccessItems > 1) ? 'link' : 'link-disabled';
		}
	}
	
	function _addAccessCell(tr, elementId, className, label) {
		
		// type is: edit_profiles | edit_roles
		var td = document.createElement('td');

		if (className !== '') {
            td.className = className;
        }

		td.id = elementId;
		var text = document.createTextNode(label);
		td.appendChild(text);
		tr.appendChild(td);
	}
	
	function createAccessRow(tbody) {
		
		// alert('createAccessRow);
		
		// Get new unique baseId (i.e. acc8)
		var idCount = 0;
		var isUniqueId = false;
		var baseId = '';
		while (!isUniqueId) {
			if (util.getE('acc' + idCount + ':tr') === null) {
				baseId = 'acc' + idCount;
				isUniqueId = true;
			}
			idCount++;
		}
	
		
		var tr = document.createElement('tr');
		tr.id = baseId + ':tr';
		
		_addAccessCell(tr, baseId + ':op:edit_profiles', '', '-');
		_addAccessCell(tr, baseId + ':op:edit_roles', '', '-');
		_addAccessCell(tr, baseId + ':info', 'info', '-');
		_addAccessCell(tr, baseId + ':op:edit', 'link', langVar('lang_stats.btn.edit'));

        if (isEnterprise) {
			_addAccessCell(tr, baseId + ':op:delete', 'link', langVar('lang_stats.btn.delete'));
		}
		
		tbody.appendChild(tr);
	}
	
	function updateAccessList() {
		
		// Updates the access list
		
		// var table = util.getE('users:access:container');
		var tbody = util.getE('users:access:tbody');
		
		// Get the number of existing rows
		var allRows = tbody.getElementsByTagName('tr');
		var numberOfExistingAccessRows = allRows.length;
		
		var access = users.get('activeAccess');
		var numberOfAccessItems = access.length;
		
		// alert('numberOfExistingAccessRows: ' + numberOfExistingAccessRows + '\nnumberOfAccessItems: ' + numberOfAccessItems);
		
		//
		// Create/remove access rows
		//

        var i;
		
		if (numberOfExistingAccessRows !== numberOfAccessItems) {
			
			if (numberOfExistingAccessRows < numberOfAccessItems) {
				// Add new rows
				// alert('Add new rows!');
				for (i = numberOfExistingAccessRows; i < numberOfAccessItems; i++) {
					createAccessRow(tbody);
				}
			}
			else {
				// Remove rows
				// alert('Remove rows!');
				for (i = numberOfExistingAccessRows; i > numberOfAccessItems; i--) {
					// alert('Remove row: ' + i);
					var tr = tbody.lastChild;
					tbody.removeChild(tr);
				}
			}
		}
		
		//
		// Update the access cells
		//
		
		for (i = 0; i < numberOfAccessItems; i++) {
			
			var accessItem = access[i];
			updateAccessRow(i, accessItem, numberOfAccessItems);
		}
	}
	
	
	//
	//
	//
	// Miscellaneous utilities
	//
	//
	//
	
	function _getRowNumberFromBaseId(baseId) {
		
		var tbody = util.getE('users:access:tbody');
		var allRows = tbody.getElementsByTagName('tr');
		var trBaseId = baseId + ':tr';
		var rowNumberIndex = 0;
		
		for (var i = 0; i < allRows.length; i++) {
			if (trBaseId === allRows[i].id) {
				rowNumberIndex = i;
				break;
			}
		}
		
		return rowNumberIndex;
	}
	
	function _getUsernameByUserNodeName(userNodeName) {
		
		var usersDb = users.get('usersDb');
		var username = '';
		
		for (var i = 0; i < usersDb.length; i++) {
			
			if (userNodeName === usersDb[i].dat.node_name) {
				username = usersDb[i].dat.username;
				break;
			}
		}
		
		if (username === '') {
			username = langVar('lang_admin.users.unknown_user');
		}
		
		return username;
	}
	
	function getNewUserNodeName(usersDb, deletedUsersDb) {
		
		// Returns a new unique node name (with timestamp) so that it
		// is unique forever.
		
		//
		// Create a node name lookup
		//
		
		var usedNodeNames = {};
		var nodeName;
		
		for (var i = 0; i < usersDb.length; i++) {
			
			nodeName = usersDb[i].dat.node_name;
			nodeName = '_' + nodeName.toLowerCase();
			usedNodeNames[nodeName] = true;
		}
		
		for (var j = 0; j < deletedUsersDb.length; j++) {
			
			nodeName = deletedUsersDb[j];
			nodeName = '_' + nodeName.toLowerCase();
			usedNodeNames[nodeName] = true;
		}
		
		//
		// Create new node name
		//
		
		// KHP 05/Jan/2011 - use timestamp to ensure unique user node name forever,
		// it should not be possible to create the same user node name even after
		// deletion of the user because there might be references of the node name
		// in "created_by_user" nodes.
		
		var d = new Date();
		var timestamp = d.getTime();
		var newNodeName = 'user_' + timestamp;
		
		// util.showObject({newNodeName__1: newNodeName});
		
		if (usedNodeNames['_' + newNodeName]) {
			
			// A user node name with this timestamp already exists.
			// This case shouldn't be possible but we cover it.
			
			var isUniqueNodeName = false;
			var count = 1;
			while (usedNodeNames['_' + newNodeName]) {
				
				newNodeName = 'user_' + timestamp + count;
				count++;
			}
		}
		
		return newNodeName;
	}

    // Return global methods
	return {
        init: init,
		accessListLinkActivated: accessListLinkActivated,
        hoverAccessListItem: hoverAccessListItem,
        updateAccessRow: updateAccessRow,
        createAccessRow: createAccessRow,
        updateAccessList: updateAccessList,
        getNewUserNodeName: getNewUserNodeName
	};
    
}());
