//
//
// datePickerCal.js (date picker calendar)
//
//
//

// This object builds and manages the calendar within datePicker

var datePickerUtil = {

	getIsSingleDatePattern: function(df) {

		// This checks if the date filter pattern is a single date
		// which can be set with the date picker GUI (non-custom date filter)

        if (typeof df == 'number') {
           // Regex, etc. requires string value!
           df = df.toString();
        }

		if (/^\d{4}$/.test(df) || // matches year
			/^q(1|2|3|4)\/\d{4}$/.test(df) || // matches quarter/year
			/^\d{0,2}(jan|feb|mar|apr|may|jun|jul|aug|sep|oct|nov|dec)\d{4}$/.test(df)) { // matches day/month/year or month/year

			return true;
		}

		return false;
	},

	getIsDateRangePattern: function(df) {

		// This checks if the date filter pattern is a date range
		// which can be set with the date picker GUI (non-custom date filter)

		var isDateRangePattern = false;
        if (typeof df == 'number') {
            // Regex, etc. requires string value!
            df = df.toString();
        }

		if (df.indexOf('-') != -1) {

			var dat = df.split('-');

			if (dat.length == 2 &&
				datePickerUtil.getIsSingleDatePattern(dat[0]) &&
				datePickerUtil.getIsSingleDatePattern(dat[1])) {

                isDateRangePattern = true;
			}
		}

        // util.showObject({"isDateRangePattern": isDateRangePattern});

		return isDateRangePattern;
	},

	getIsRelativeDatePattern: function(df) {

		// This checks if the date filter pattern is a relative date filter
		// which can be set with the relative date picker GUI (non-custom date filter)

        var isDateRangePattern = false;
        if (typeof df == 'number') {
            // Regex, etc. requires string value!
            df = df.toString();
        }

		var relativeDfPattern = /^(recent|last)[0-9]+(year|quarter|month|week|day)(s)?$/;
        var isRelativeDfPattern = relativeDfPattern.test(df);

        // util.showObject({isRelativeDfPattern: isRelativeDfPattern});

		return isRelativeDfPattern;
	},

	getQuarterFromMonth: function(month) {
		// returns the quarter of the given month

		if (month < 3) {
			return 1;
		}
		else if (month < 6) {
			return 2; 
		}
		else if (month < 9) {
			return 3;
		}
		else {
			return 4;
		}
	},

	getNumberOfDaysInMonth: function(year, month) {

		// set required month to day 1, 00:00
		var theMonth = new Date(year, month, 1, 0);
		var nextMonth;
		// set next month to day 1, 06:00, the 6 hours compensate summertime shift
		if (month < 11) { // Jannuary - November
			nextMonth = new Date(year, (month + 1), 1, 6); // first day of next month, 06:00
		}
		else { // December
			nextMonth = new Date((year + 1), 0, 1, 6); // first month and day of next year, 06:00
		}
		var numberOfDaysInMonth = nextMonth.getTime() - theMonth.getTime();
		numberOfDaysInMonth = Math.floor(numberOfDaysInMonth / (1000 * 60 * 60 * 24));

		// alert('getNumberOfDaysInMonth\nyear: ' + year + '\nmonth: ' + month + '\nnumberOfDays: ' + numberOfDaysInMonth);

		return numberOfDaysInMonth;
	},

	convertDpObjectToDf: function(dpName) {

		var dp = datePicker[dpName];
		var dateUnit = dp.dateUnit;
		var df;

		// util.showObject(dp);

		var year = dp.year;

		if (dateUnit == 'year') {

			df = year;
		}
		else if (dateUnit == 'quarter') {
			df = 'q' + dp.quarter + '/' + year;
		}
		else {

			var month = datePicker.dfMonths[dp.month];

			df = month + year;

			if (dateUnit == 'day') {

				df = dp.day + df;
			}
		}

		return df;
	},


	convertDpObjectToTime: function(dpName) {

		var dp = datePicker[dpName];
		var isStartDate = (dpName == 'dp1') ? true : false;
		var dateUnit = dp.dateUnit;
		var year = dp.year;
		var month = 0;
		var day = 1;

		switch (dateUnit) {

			case 'y':
				month = isStartDate ? 0 : 11;
				day = isStartDate ? 1 : 31;
				break;

			case 'q':
				var quarter = dp.quarter;
				if (quarter == 1) {
					month = isStartDate ? 0 : 2;
					day = isStartDate ? 1 : 31;
				}
				else if (quarter == 2) {
					month = isStartDate ? 3 : 5;
					day = isStartDate ? 1 : 30;
				}
				else if (quarter == 3) {
					month = isStartDate ? 6 : 8;
					day = isStartDate ? 1 : 30;
				}
				else {
					month = isStartDate ? 9 : 11;
					day = isStartDate ? 1 : 31;
				}
				break;

			case 'm':
				month = dp.month;
				day = isStartDate ? 1 : datePickerUtil.getNumberOfDaysInMonth(year, month);
				break;

			case 'd':
				month = dp.month;
				day = dp.day;
				break;
		}

		var theDate = new Date(year, month, day);
		var theTime = theDate.getTime();

		return theTime;
	},

	convertMonthTextToInt: function(monthText) {

		var dfMonths = datePicker.dfMonths;
		for (var i = 0; i < dfMonths.length; i++) {
			if (monthText == dfMonths[i]) {
				break;
			}
		}

		return i;
	},


	fixInternetExplorerAnchors: function(elementId) {

		// Fix Internet Explorer 6 text-hover problem by adding a height to all anchor elements
		// Don't use this for IE7 and above because it would actually apply the height and shrink the date picker.
		var table = util.getE(elementId);

		if (util.userAgent.isIE6) {

			var anchors = table.getElementsByTagName('a');
			for (var i = 0; i < anchors.length; i++) {
				var a = anchors[i];
				a.style.height = '0.1em';
			}
		}
	},

	clearDateSelection: function(elementId) {

		// Deselect all date items of the given picker table element id,
		// though don't change the elements className where the current
		// className is "no-log-data"

		var table = util.getE(elementId);
		var anchors = table.getElementsByTagName('a');

		for (var i = 0; i < anchors.length; i++) {
			var a = anchors[i];
			if (a.className != 'no-log-data') {
				a.className = '';
			}
		}
	},

	createSpacerColumn: function() {
		// Creates a column which keeps a constant vertical space,
		// it is required due the variable number of months in the months column
		// and due the different height in the relative picker. It is used
		// in calendar and relative picker.

		var td = util.createE('td', {className:'vertical-spacer'});
		var div = util.createE('div', {visibility:'hidden'});

		for (var i = 0; i < 13; i++) {

			var a = util.createE('a', {href:'javascript:;', paddingLeft:'0px', paddingRight:'0px'});
			var text = util.createT('.');
			util.chainE(div, a, text);
		}

		util.chainE(td, div);

		/*
		var td = document.createElement('td');
		td.className = 'vertical-spacer';

		var div = document.createElement('div');
		div.style.visibility = 'hidden';

		for (var i = 0; i < 13; i++) {

			var a = document.createElement('a');
			a.href = 'javascript:;';
			a.style.paddingLeft = '0px';
			a.style.paddingRight = '0px';
			var txt = document.createTextNode('-');
			a.appendChild(txt);
			div.appendChild(a);
		}

		td.appendChild(div);
		*/

		return td;
	}
};

//
//
//
// datePicker Calendar
//
//
//

datePickerUtil.Calendar = function(earliest, latest, containerElementId) {

	this.earliest = earliest; // A reference to datePicker.earliest
	this.latest = latest; // A reference to datePicker.latest

	this.dp = null; // A reference to the active dp object (dp1 or dp2)

	// Active dp data (dp1 or dp2), they are provided upon picker init
	/*
	this.picker = ''; // dp1 || dp2
	this.dayOffset = 0; // the dayOffset of the active month
	this.dateUnit = ''; // // the active dateUnit: y | q | m | d | ''; empty is equal entire date range
	this.year = -1;
	this.quarter = -1;
	this.month = -1;
	this.day = -1;
	*/

	this.build(containerElementId);
}

datePickerUtil.Calendar.prototype = {


	init: function(dp) {

		// Init is invoked when switching between start and end date or upon first time initialization.

		this.dp = dp // dp is an active reference to dp1 or dp2

		// Override the active picker data of this with the provided dp

		// for (var prop in dp) {
		//	this[prop] = dp[prop];
		// }

		// Initializes the picker (dp1 or dp2) according the dp1 or dp2 object
		// Update the quarter, months and days column for the given dp object

		// util.showObject(dp);

		datePickerUtil.clearDateSelection('dp:calendar_picker');

		this.updateQuartersDisplay(dp.year);
		this.updateMonthsDisplay(dp.year);
		this.updateDaysDisplay(dp.year, dp.month);

		this.applyDateSelection();
	},

	itemActivated: function(dateUnit, dateNum) {

		// Invoked via datePicker.calendarPickerItemActivated()

		var dp = this.dp;

		// util.showObject(arguments);

		// Deselect all date items
		datePickerUtil.clearDateSelection('dp:calendar_picker');

		switch (dateUnit) {
			case 'year':
				this.setYear(dateNum, true);
				break;
			case 'quarter':
				this.setQuarter(dateNum, true);
				break;
			case 'month':
				this.setMonth(dateNum, true);
				break;
			case 'day':

				// Note, we ignore days where the className is 'no-log-data'!

				// alert('elementClassName: ' + elementClassName);

				// if (elementClassName != 'no-log-data') {

				// }
				var day = dateNum - dp.dayOffset + 1;
				// alert('selected day: ' + day);
				this.setDay(day);

				break;
		}

		this.applyDateSelection();
	},


	//
	// setYear, setQuarter, setMonth, setDay
	// 

	setYear: function(year, isActiveDateUnit) {

		// alert('setYear()');

		var dp = this.dp;

		//
		// Update global dp object (dp1 or dp2)
		//
		if (isActiveDateUnit) {
			dp.dateUnit = 'year';
			// Reset date items below year
			dp.quarter = -1;
			dp.month = -1;
			dp.day = -1;
		}
		dp.year = year;

		// util.showObject(dp);


		//
		//
		// If we set a year we must update the quarters and months display
		//
		//

		// We need to check how many months and quarters we display depending on earliest and latest date.

		// var lastMonthInYear = (year == dp.latest.year) ? dp.latest.month : 11;
		// var lastQuarterInYear = dp.getQuarterFromMonth(lastMonthInYear);

		//
		// Update the quartes display
		//

		this.updateQuartersDisplay(year);

		//
		// Set the quarter
		//

		var firstMonthInYear = (year == this.earliest.year) ? this.earliest.month : 0;
		var firstQuarterInYear = datePickerUtil.getQuarterFromMonth(firstMonthInYear);
		this.setQuarter(firstQuarterInYear, false);

		//
		// Update the months display
		//

		this.updateMonthsDisplay(year);
	},

	setQuarter: function(quarter, isActiveDateUnit) {

		// alert('setQuarter()');

		var dp = this.dp;

		//
		// Update global dp object (dp1 or dp2)
		//
		if (isActiveDateUnit) {
			dp.dateUnit = 'quarter';
			// Reset date items below quarter
			dp.month = -1;
			dp.day = -1;
		}

		dp.quarter = quarter;

		// As we set a quarter we need to select the first available month in this quarter.

		var firstMonthInYear = (dp.year == this.earliest.year) ? this.earliest.month : 0;

		var firstMonthInQuarter;

		if (quarter == 1) {
			firstMonthInQuarter = 0;
		}
		else if (quarter == 2) {
			firstMonthInQuarter = 3;
		}
		else if (quarter == 3) {
			firstMonthInQuarter = 6;
		}
		else {
			firstMonthInQuarter = 9;
		}

		// In case that the year does not cover all months we must increase the month number for maximum two increments

		if (firstMonthInQuarter < firstMonthInYear) {
			firstMonthInQuarter++;

			if (firstMonthInQuarter < firstMonthInYear) {
				firstMonthInQuarter++;
			}
		}

		this.setMonth(firstMonthInQuarter, false);

	},

	setMonth: function(month, isActiveDateUnit) {

		// alert('setMonth()');

		var dp = this.dp;

		//
		// Update global dp object (dp1 or dp2)
		//
		if (isActiveDateUnit) {
			dp.dateUnit = 'month';
			// Reset date items below month
			dp.day = -1;

			// We need to update the quarter
			dp.quarter = datePickerUtil.getQuarterFromMonth(month);
		}

		dp.month = month;

		// util.showObject(dp);

		// update the days column for the given month
		this.updateDaysDisplay(dp.year, month);
	},


	setDay: function(day) {

		this.dp.dateUnit = 'day';
		this.dp.day = day;
	},


	updateQuartersDisplay: function(year) {

		var firstMonthInYear = (year == this.earliest.year) ? this.earliest.month : 0;
		var lastMonthInYear = (year == this.latest.year) ? this.latest.month : 11;

		var firstQuarterInYear = datePickerUtil.getQuarterFromMonth(firstMonthInYear);
		var lastQuarterInYear = datePickerUtil.getQuarterFromMonth(lastMonthInYear);

		//
		// Update the quartes display
		//
		for (var i = 1; i < 5; i++) {
			anchorId = 'dp:cal:quarter:' + i;
			var showQuarter = (i >= firstQuarterInYear && i <= lastQuarterInYear) ? true : false;
			util.showE(anchorId, showQuarter);
		}
	},

	updateMonthsDisplay: function(year) {

		var firstMonthInYear = (year == this.earliest.year) ? this.earliest.month : 0;
		var lastMonthInYear = (year == this.latest.year) ? this.latest.month : 11;

		for (var i = 0; i < 12; i++) {
			anchorId = 'dp:cal:month:' + i;
			var showMonth = (i >= firstMonthInYear && i <= lastMonthInYear) ? true : false;
			util.showE(anchorId, showMonth);
		}
	},

	applyDateSelection: function() {

		// Selects each date item (year, quarter, month, day) according the active dp data
		// Note, a year, quarter and month is always and minimum semi-active!

		var dp = this.dp;

		var dateUnit = dp.dateUnit;

		// util.showObject(dp);

		var anchor;
		var className;

		// year
		anchor = util.getE('dp:cal:year:' + dp.year);
		className = (dateUnit == 'year') ? 'active' : 'semi-active';
		anchor.className = className;

		// quarter
		anchor = util.getE('dp:cal:quarter:' + dp.quarter);
		className = (dateUnit == 'quarter') ? 'active' : 'semi-active';
		anchor.className = className;
		// anchor.className = 'active';

		// month
		anchor = util.getE('dp:cal:month:' + dp.month);
		className = (dateUnit == 'month') ? 'active' : 'semi-active';
		anchor.className = className;


		// day
		if (dateUnit == 'day') {

			var dayCellNumber = dp.day + dp.dayOffset - 1;
			anchor = util.getE('dp:cal:day:' + dayCellNumber);
			anchor.className = 'active';
		}
	},


	//
	//
	// Date picker - days update utilities (updates the month label and days)
	//
	//

	updateDaysDisplay: function(year, month) {

		// alert('updateDaysDisplay()');

		// Update month label
		var dateLabel = lang.monthsShort[month] + ' ' + year;
		util.updateT('dp:month_year_display', dateLabel);

		// Update days
		var firstWeekdayInPicker = reportInfo.firstWeekday;

		// Create a date for the first of the month
		var theDate = new Date(year, month, 1);
		var firstWeekdayInMonth = theDate.getDay();

		// firstWeekdayZeroOffset is the position of the 0 weekday in the picker grid
		var firstWeekdayZeroOffset = (firstWeekdayInPicker > 0) ? 7 - firstWeekdayInPicker : 0;

		var dayOffset = firstWeekdayInMonth + firstWeekdayZeroOffset;
		if (dayOffset > 6) {dayOffset = dayOffset - 7;}

		var numberOfDays = datePickerUtil.getNumberOfDaysInMonth(year, month);

		// alert('numberOfDays: ' + numberOfDays);

		var dayCount = 1 - dayOffset;

		// Note, if a day is not within the earliest and latest date
		// then we apply the className 'no-log-data'.
		// We check this by earliestActiveDay and latestActiveDay
		var earliestActiveDay = 1;
		var latestActiveDay = numberOfDays;

		if (month == this.earliest.month && year == this.earliest.year) {
			earliestActiveDay = this.earliest.day;
		}

		if (month == this.latest.month && year == this.latest.year) {
			latestActiveDay = this.latest.day;
		}

		for (var i = 0; i < 37; i++) {

			var anchorId = 'dp:cal:day:' + i;
			var anchor;
			var className;

			if (dayCount > 0 && dayCount <= numberOfDays) {

				util.updateT(anchorId, dayCount);
				util.showE(anchorId);

				anchor = util.getE(anchorId);
				className = (dayCount >= earliestActiveDay && dayCount <= latestActiveDay) ? '' : 'no-log-data';
				anchor.className = className;
			}
			else {

				util.hideE(anchorId);
			}

			dayCount++;
		}

		// Set dayOffset
		this.dp.dayOffset = dayOffset;
	},


	//
	//
	//
	// build calendar HTML
	//
	//
	//

	// Note, for date selection we apply a single event. In order to check which items should actually proceed with
	// an event we use ':cal:' within the id

	build: function(containerElementId) {

		var container = util.getE(containerElementId);

		var table = util.createE('table', {id:'dp:calendar_picker', className:'date-picker', cellSpacing:'0'});
		var tbody = util.createE('tbody');
		var tr = document.createElement('tr');

		// table.cellSpacing = '0';

		var tdYear = this.createYearColumn();
		var tdQuarter = this.createQuarterColumn();
		var tdMonth = this.createMonthColumn();
		var tdDays= this.createDaysColumn();
		var tdSpacer = datePickerUtil.createSpacerColumn();
		var tdMonthYearDisplay = this.createMonthYearDisplayColumn();

		util.chainE(container, table, tbody, [tr, tdYear, tdQuarter, tdMonth, tdDays, tdSpacer, tdMonthYearDisplay]);

		datePickerUtil.fixInternetExplorerAnchors('dp:calendar_picker');
	},

	addColumnSpacerItem: function(td, regularCharLength) {

		var span = util.createE('span', {padding:'0px', display:'block', visibility:'hidden'});
		var spacerText = '';
		for (var i = -5; i < regularCharLength; i++) {
			spacerText += '_';
		}

		var text = util.createT(spacerText);
		util.chainE(td, span, text);
	},

	createYearColumn: function() {

		var td = document.createElement('td');
		td.className = 'year';
		td.id = 'dp:year:column';

		var theYear = this.earliest.year;
		var columnDone = false;

		while (!columnDone) {

			var a = document.createElement('a');
			a.id = 'dp:cal:year:' + theYear;
			a.href = 'javascript:;';
			var txt = document.createTextNode(theYear);

			a.appendChild(txt);
			td.appendChild(a);

			// YAHOO.util.Event.addListener(a, 'click', datePicker.setDateByEvent);

			if (theYear != this.latest.year) {
				theYear++;
			}
			else {
				columnDone = true;
			}
		}

		this.addColumnSpacerItem(td, 3);

		return td;
	},

	createQuarterColumn: function() {

		var td = document.createElement('td');
		td.id = 'dp:quarter:column';

		var q = lang.quarterShort;

		for (var i = 1; i < 5; i++) {

			var a = document.createElement('a');
			a.id = 'dp:cal:quarter:' + i;
			a.href = 'javascript:;';
			var txt = document.createTextNode(q + i);

			a.appendChild(txt);
			td.appendChild(a);

			// YAHOO.util.Event.addListener(a, 'click', datePicker.setDateByEvent);
		}

		this.addColumnSpacerItem(td, 2);

		return td;
	},

	createMonthColumn: function() {

		var td = document.createElement('td');
		td.id = 'dp:month:column';

		var monthLabels = lang.monthsShort;

		for (var i = 0; i < 12; i++) {

			var a = document.createElement('a');
			a.id = 'dp:cal:month:' + i;
			a.href = 'javascript:;';
			var txt = document.createTextNode(monthLabels[i]);

			a.appendChild(txt);
			td.appendChild(a);

			// YAHOO.util.Event.addListener(a, 'click', datePicker.setDateByEvent);
		}

		this.addColumnSpacerItem(td, 3);

		return td;
	},

	createDaysColumn: function() {

		// Note, the day column Ids do not refer to the day but to a
		// day anchor / cell, there are 37 day cells (dp1:d:0 - dp1:d:36).
		// Upon setting/selecting a day we calculate the actual day by (cellNumber - dayOffset + 1)		

		var td = document.createElement('td');
		td.id = 'dp:day:column';
		td.className = 'days';

		//
		// Create the days table
		//

		var table = document.createElement('table');
		table.cellSpacing = 0;
		table.className = 'date-picker-days';
		var tbody = document.createElement('tbody');
		table.appendChild(tbody);
		td.appendChild(table);


		// Create the weekdays row
		// var firstWeekday = reportInfo.firstWeekday;
		var markedWeekday = reportInfo.markedWeekday
		var theWeekdayNum = reportInfo.firstWeekday
		var trWeekday = document.createElement('tr');
		for (var i = 0; i < 7; i++) {
			var thWeekday = document.createElement('th');
			if (theWeekdayNum == markedWeekday) {thWeekday.className = 'marked';}
			var weekdayLabel = lang.weekdaysShort[theWeekdayNum];
			var txtWeekday = document.createTextNode(weekdayLabel);
			thWeekday.appendChild(txtWeekday);
			trWeekday.appendChild(thWeekday);

			theWeekdayNum++;
			if (theWeekdayNum > 6) {
				// reset weekday to 0
				theWeekdayNum = 0;
			}
		}

		tbody.appendChild(trWeekday);

		// Create the day cells

		var cellCount = 0;

		for (var i = 0; i < 6; i++) {

			var trDays = document.createElement('tr');

			for (var j = 0; j < 7; j++) {

				if (cellCount < 37) {

					var tdDay = document.createElement('td');
					var aDay = document.createElement('a');
					aDay.id = 'dp:cal:day:' + cellCount;
					aDay.href = 'javascript:;';
					var txtDay = document.createTextNode('-');

					aDay.appendChild(txtDay);
					tdDay.appendChild(aDay);
					trDays.appendChild(tdDay);

					// YAHOO.util.Event.addListener(aDay, 'click', datePicker.setDateByEvent);

					cellCount++;
				}
				else {
					break;
				}
			}

			tbody.appendChild(trDays);
		}

		return td;
	},


	createMonthYearDisplayColumn: function() {

		//
		// Create the month label column
		//

		var td = util.createE('td', {className:'month-year-display'});
		var div = util.createE('div', {id:'dp:month_year_display', className:'month-year-display'});
		var text = util.createT('--- ----');

		var spacerDiv = util.createE('div', {className:'month-year-display', visibility:'hidden'});
		var spacerText = util.createT('AAA 0000');

		util.chainE([td, [div, text], [spacerDiv, spacerText]]);

		return td;
	}
}

//
//
//
// datePicker Relative
//
//
//

datePickerUtil.Relative = function(containerElementId, dp) {

	this.dp = dp; // A reference to the active dp object (dp3)
	this.build(containerElementId);
}



datePickerUtil.Relative.prototype = {

	applyDateSelection: function() {

		// Selects each relative date item according the active dp3 data

		datePickerUtil.clearDateSelection('dp:relative_picker');

		var dp = this.dp;

		// util.showObject(dp);

		// dp.isActive
		// if true we select each item (1 per column) in full color (we apply the relative date filter, i.e. df="last3M".
		// if false we select each item (1 per column) in semi-active color (we use the entire date range, respectively df="").
		// That means the dp object is always set to a valid relative date filter
		// and the isActive property determines if it is used or not.

		var isActive = dp.isActive;
		var className = isActive ? 'active' : 'semi-active';

		//
		// Select type
		//

		var typeElement = util.getE('dp:relative:type:' + dp.type);
		typeElement.className = className;

		//
		// Select dateUnitCount
		//

		var dateUnitCount = dp.dateUnitCount;

		if (dateUnitCount < 11) {

			var dateUnitCountElement = util.getE('dp:relative:date_unit_count:' + dateUnitCount);
			dateUnitCountElement.className = className;
		}		

		//
		// Select date unit
		//

		var dateUnitElement = util.getE('dp:relative:date_unit:' + dp.dateUnit);
		dateUnitElement.className = className;
	},



	//
	//
	//
	// build relative HTML
	//
	//
	//

	// Note, for date selection we apply a single event. In order to check which items should actually proceed with
	// an event we use ':cal:' within the id

	// Element ID schema
	// Anchor elements in dp:relative_picker which should fire a select event
	// should consist of 4 ID part items, i.e.:
	// dp:relative:type:last


	build: function(containerElementId) {

		var container = util.getE(containerElementId);

		var table = util.createE('table', {id:'dp:relative_picker', className:'date-picker', cellSpacing:'0'});
		var tbody = util.createE('tbody');
		var tr = document.createElement('tr');

		var tdType = this.createTypeColumn();
		var tdDateUnitCount = this.createDateUnitCountColumn();
		var tdDateUnit = this.createDateUnitColumn();
		var tdSpacer = datePickerUtil.createSpacerColumn();

		util.chainE(container, table, tbody, [tr, tdType, tdDateUnitCount, tdDateUnit, tdSpacer]);

		datePickerUtil.fixInternetExplorerAnchors('dp:relative_picker');
	},

	createTypeColumn: function() {

		var td = util.createE('td');
		var aRecent = util.createE('a', {id:'dp:relative:type:recent', href:'javascript:;', paddingRight:'18px'});
		var aLast = util.createE('a', {id:'dp:relative:type:last', className:'year', href:'javascript:;', paddingRight:'18px'});
		var txtRecent = util.createT(langVar('lang_stats.date_picker.recent'));
		var txtLast = util.createT(langVar('lang_stats.date_picker.last'));

		util.chainE([td, [aRecent, txtRecent], [aLast, txtLast]]);

		return td;
	},

	createDateUnitCountColumn: function() {

		// Number sequence (KHP-RC, check if date unit count sequence should be customizable per user)
		// n = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12];

		var td = util.createE('td');

		for (var i = 1; i < 11; i++) {

			var id = 'dp:relative:date_unit_count:' + i;
			var a = util.createE('a', {id:id, href:'javascript:;', paddingRight:'18px'});
			var text = util.createT(i);

			util.chainE(td, a, text);
		}

		return td;
	},

	createDateUnitColumn: function() {

		var td = util.createE('td');
		var dateUnits = datePicker.relativeDateUnits;
		var dateUnitLabelsPlural = datePicker.dateUnitLabelsPlural;

		for (var i = 0; i < dateUnits.length; i++) {

			var name = dateUnits[i];
			var id = 'dp:relative:date_unit:' + name;
			var a = util.createE('a', {id:id, href:'javascript:;', paddingRight:'18px'});
			var label = dateUnitLabelsPlural[name];
			var text = util.createT(label);

			util.chainE(td, a, text);
		}

		return td;
	}
}
