//
// databaseUtil - a collection of database server validation utilities
//

var databaseUtil = function() {

	function getIsValidMysqlDatabaseName(databaseName) {

		var isValid = false,
			errorMessage = '';

		if (databaseName.length > 64) {
			errorMessage = langVar('lang_admin.database.mysql_max_length_msg');
		}
		else if (/[\.\/\\]/.test(databaseName)) {
			errorMessage = langVar('lang_admin.database.mysql_invalid_char_msg');
		}
		else {
			isValid = true;
		}

		return {isValid:isValid, errorMessage:errorMessage};
	}

	function getIsValidOdbcMssqlDatabaseName(databaseName) {

		// odbc_mssql (MS SQL Server)
		// must be <= 128 characters
		// and must match the following rule (added on 09/Jan/2013 by KHP):
		// the name must start with a Unicode letter, _, @, or #; followed by one
		// or more letters, numbers, @, $, #, or _. The documentation clearly
		// states that embedded spaces or special characters are not allowed.

		// KHP 04/June/2013 - Added normal regex pattern so that unicode_hack() is only
		// used when normal pattern fails. This should solve problems with unicode_hack()
		// under specific circumstances.

		var isValid = false,
			errorMessage = '',
			pattern = /^[a-zA-Z_@#]{1}[0-9a-zA-Z_@#$]+$/,
			isValidPattern = pattern.test(databaseName),
			unicodePattern = unicode_hack(/^(\p{L}|[_@#]){1}(\p{L}|[0-9_@$#])+$/), // \p{L} matches any unicode letter, this requires the unicode_hack utility
			isValidUnicodePattern = unicodePattern.test(databaseName);

//		util.showObject({isValidPattern:isValidPattern, isValidUnicodePattern:isValidUnicodePattern});

		if (databaseName.length > 128) {

			errorMessage = langVar('lang_admin.database.mssql_max_length_msg');
		}
		else if (!isValidPattern && !isValidUnicodePattern) {

			// No error if one of the two pattern validates. We added isValidPattern because
			// isValidUnicodePattern may fail under certain circumstances with ASCII characters.
			errorMessage = langVar('lang_admin.database.mssql_name_definition_msg');
		}

		isValid = (errorMessage == '');

		return {isValid:isValid, errorMessage:errorMessage};
	}

	function getIsValidDatabaseName(serverType, databaseName) {

		// This validates the databaseName. We assume that the server side
		// uses quoted database names, which means almost any string is allowed.
		// So this covers basic validation required for quoted and unquoted
		// database names.

		if (serverType == 'mysql') {

			return getIsValidMysqlDatabaseName(databaseName);
		}
		else {

			// serverType is odbc_mssql
			return getIsValidOdbcMssqlDatabaseName(databaseName);
		}
	}

	// Return global properties and methods
	return {
		getIsValidDatabaseName: getIsValidDatabaseName
	}

}();
	
