//
// scrollUtil.js
//

var scrollUtil = {};

scrollUtil.Scroller = (function() {

	'use strict';
		
	// dependencies
	var YE = YAHOO.util.Event,
		YD = YAHOO.util.Dom,
		Constr;
	
	// constructor
	Constr = function(frameElementId, scrolledElementId, scrollHandler) {

		// If frameElementId and scrolledElementId is empty then
		// calculate regions from viewport and body element.

		this.hasFrameElement = (frameElementId !== '');

        this.frameElementId = frameElementId;
		this.frameElement = null;

        this.frameElemementTop = 0;
        this.frameElementBottom = 0;
        this.frameElementHeight = 0;

        this.scrolledElementId = scrolledElementId;
        this.scrolledElement = null;
		this.scrolledElementHeight = 0;
		this.scrollHandler = scrollHandler;

        // If isStandby is true we record scroll event properties
        // but don't fire the scrollHandler.
        // The system goes automatically in isStandby when the
        // scrollHandler is fired. The callee must invoke
        // restartListening() to reset isStandby to false.

        this.isStandby = false;

        // Up to date scroll properties
        this.verticalScrollLevel = 0.0; // float in range 0-100, 0 = scrolled to top, 100 = scrolled to bottom
        this.isScrollingDown = false;

        YE.addListener(window, 'resize', this._sizeChanged, this);
	};
	
	Constr.prototype = {

        _setBasicProperties: function() {

//          alert('_setBasicProperties()');

			// Note, frame may be a dedicated element container or the viewport.

			var frameBottom = 0;
			var frameHeight = 0;

			if (this.hasFrameElement) {

				if (this.frameElement === null) {
					this.frameElement = util.getE(this.frameElementId);
					this.scrolledElement = util.getE(this.scrolledElementId);
				}

				var region = YD.getRegion(this.frameElement);
				frameBottom = region.bottom;
				frameHeight = region.height;
			}
			else {

				if (this.scrolledElement === null) {

					// The scrolled element is the body element
					var bodyElements = document.getElementsByTagName('body');
					this.scrolledElement = bodyElements[0];
				}

				frameHeight = YD.getViewportHeight();
				frameBottom = frameHeight;
			}

            this.frameElementBottom = frameBottom;
            this.frameElementHeight = frameHeight;

//            console.log('_setBasicProperties - frameElementBottom: ' + this.frameElementBottom);
//            console.log('_setBasicProperties - frameElementHeight: ' + this.frameElementHeight);
        },
		
		activate: function() {

			var element = this.hasFrameElement ? this.frameElement : window;
			YE.addListener(element, 'scroll', this._scrollActivated, this);

            // Reset scroll properties
            this.verticalScrollLevel = 0;
            this.isScrollingDown = false;

            this.isStandby = false;
		},

		deactivate: function() {

             YE.removeListener(this.frameElementId, 'scroll', this._scrollActivated);
		},

        restartListening: function() {

            this.isStandby = false;
        },

        _getVerticalScrollLevel: function() {

//			console.log('_getVerticalScrollLevel - this.frameElementHeight: ' + this.frameElementHeight);
//			console.log('_getVerticalScrollLevel - this.frameElementBottom: ' + this.frameElementBottom);

			var region = YD.getRegion(this.scrolledElement),
                // scrolledTop = region.top,
                scrolledElementBottom = region.bottom,
                scrolledElementHeight = region.height,
                scrolledElementHiddenHeight = scrolledElementHeight - this.frameElementHeight, // hidden parts on top and bottom
                scrollBottom = 0,
                scrollTop = 0,
				verticalScrollLevel = 0;

			if (this.hasFrameElement) {

                scrollBottom = scrolledElementBottom - this.frameElementBottom;
                scrollTop = scrolledElementHiddenHeight - scrollBottom;
			}
			else {

				// Viewport with body element
				scrollTop = YD.getDocumentScrollTop();
			}

			verticalScrollLevel = 100 * (scrollTop / scrolledElementHiddenHeight);

//			util.showObject(region);

//			console.log('_getVerticalScrollLevel - YD.getDocumentScrollTop: ' + YD.getDocumentScrollTop());
//			console.log('_getVerticalScrollLevel - scrolledElementBottom: ' + region.bottom);
//			console.log('_getVerticalScrollLevel - scrolledElementHeight: ' + region.height);
//			console.log('_getVerticalScrollLevel - verticalScrollLevel: ' + verticalScrollLevel);

            return  {scrolledElementHeight: scrolledElementHeight, verticalScrollLevel: verticalScrollLevel};
        },
		
		_scrollActivated: function(evt, self) {

            // Starts scrolling
          
            // Set basic properties if not yet set
            if (self.frameElementHeight === 0) {
                self._setBasicProperties();
            }

			var obj = self._getVerticalScrollLevel();
            var verticalScrollLevel = obj.verticalScrollLevel;

//			console.log('_scrollActivated - self.verticalScrollLevel: ' + self.verticalScrollLevel);
//            console.log('_scrollActivated - verticalScrollLevel: ' + verticalScrollLevel);

            // Set up to date scroll properties
            self.isScrollingDown = (verticalScrollLevel > self.verticalScrollLevel);
            self.verticalScrollLevel = verticalScrollLevel;

//            console.log('_scrollActivated - self.isStandby: ' + self.isStandby);
//            console.log('_scrollActivated - isScrollingDown: ' + self.isScrollingDown);

            // Fire scroll handler if not in isStandby
            // and if verticalScrollLevel is greater than 20
            // and if scrolling down
            if (!self.isStandby &&
                verticalScrollLevel > 20 &&
                self.isScrollingDown) {

                // Set to standby
                self.isStandby = true;

                // Fire scrollHandler
                self.scrollHandler(verticalScrollLevel);
            }
		},

        _sizeChanged: function(evt, self) {
            // Element size changed, reset basic properties
            self._setBasicProperties();
        },
		
		getScrollInfo: function() {

            // Get a fresh verticalScrollLevel because it
            // could have changed due adding data by the
            // scrollHandler.
            // The isScrollingDown value is up to date
            // and should not be changed here.

			 // Set basic properties if not yet set
            if (this.frameElementHeight === 0) {
                this._setBasicProperties();
            }

			var obj = this._getVerticalScrollLevel();
			this.verticalScrollLevel = obj.verticalScrollLevel;
			
			var scrollInfo = {
                isScrollingDown: this.isScrollingDown,
				frameElementHeight: this.frameElementHeight,
				scrolledElementHeight: obj.scrolledElementHeight,
                verticalScrollLevel: obj.verticalScrollLevel
            };

            return scrollInfo;
		}
	};
	
	// return Constr
	return Constr;
	
}());