#!/usr/bin/perl
#
# dbiauth.pl - Perl script to authenticate users.
#
# DBI Authentication script using DBD::mysql database.
# Other databases could be substituted as long as they have 
# a DBD interface.
#
# This authenticates a username/password using Perl DBI, and provides
# access to configurations for that user.
#
# Detailed instructions are available in POD format. View them from 
# the command line with:
# perldoc dbiauth.pl
#
# If the authentication fails, it prints *FAILED*.  If the user is in the
# group specified by $admin_group below, it prints *ADMIN*. Otherwise, it
# prints the names of the configurations the user has, one per line.  So
# to use DBI authentication, just create two tables, users and sites, and
# configure the script to use those tables. Following are example table
# definitions.
# 
# CREATE TABLE users (
#   user_key int(10) unsigned DEFAULT '0' NOT NULL auto_increment,
#   user_code char(50) DEFAULT '' NOT NULL,
#   groups char(255) DEFAULT '' NOT NULL,
#   password char(40) DEFAULT '' NOT NULL,
#   email char(50) DEFAULT '' NOT NULL,
#   last_mod timestamp(14),
#   PRIMARY KEY (user_key),
#   UNIQUE user_code (user_code)
# );
#
# CREATE TABLE sites (
#   site_key int(10) unsigned DEFAULT '0' NOT NULL auto_increment,
#   user_key int(10) unsigned DEFAULT '0' NOT NULL,
#   site_code char(24) DEFAULT '' NOT NULL,
#   site_name varchar(80) DEFAULT '' NOT NULL,
#   last_mod timestamp(14),
#   PRIMARY KEY (site_key),
#   KEY user_key (user_key),
#   KEY site_name (site_name),
#   UNIQUE site_code (site_code)
# );
# 
# The groups field in the users table is a comma separated list of all
# groups the user belongs to. The password field is a value encrypted 
# with the perl crypt command. The user_code is the value entered for the 
# user name when logging in.
# The sites table is linked via the user_key. The site_code field
# is the name of the configuration. More than one site record 
# can be linked to a user record.
#
# Some example records for the table definition above:
# 
# INSERT INTO users VALUES (1,'admin','site,admin','enc_pass','',NULL);
# INSERT INTO users VALUES (2,'site','site','enc_pass','',NULL);
# INSERT INTO users VALUES (3,'odns','site','enc_pass','',NULL);
# 
# INSERT INTO sites VALUES (1,2,'stats','stats.make-tracks.com',NULL);
# INSERT INTO sites VALUES (2,2,'www','www.make-tracks.com',NULL);
# INSERT INTO sites VALUES (3,3,'odns','odns.make-tracks.com',NULL);
# 
# Logging in as user 'admin' would give Administrator access (user is in 
# admin group). Logging in as user site would list configurations named
# 'stats' and 'www'. Logging in as user 'odns' would list the 
# configuration named 'odns'.
#
# This script has only been tested using Linux & mySQL, but should work 
# with any OS and database that supports Perl, DBI, DBD and mySQL. 
#
# The location of this script should be entered in the Security tab of
# the Preferences, in the "Command-line authentication" field.
#

########################################################################
# You will need to modify these values in order to use this script:

# The base directory is where the script create and write to a log file
# and where it will look for a file containing the database password.
my $base_dir = '/home/stats/bin/Extras';

# The auth_log is the location of the file to log errors to. Logging
# occurs only if do_log is a true value.
my $auth_log = "$base_dir/auth.log";
my $do_log = 0; # change to 1 to turn on logging

# The database server and port to use-- change sql_server to the hostname
# of your database server.  You will probably want to leave sql_port set
# to 3306.
# my $sql_server = 'localhost';
# my $sql_port = '3306';
my $sql_server = '';
my $sql_port = '';

# The database name-- change to the name of your database
my $sql_db = "db_name";

# The database user-- change to the name of user to connect as
my $sql_db_user =	"db_user";


# The database password can either be entered into this script
#my $sql_db_pass =	"db_pass";
#
# Or it can be read from a protected file. If the password is being 
# read from a file, it must have read permissions for the user or 
# group that the stats program is running under.
my $sql_db_pass =	dbi_pwd_fetch("$base_dir/dbpasswd");

# Enter the names of the 'users' and 'sites' tables
my $users_tbl = 'users';			# users table
my $sites_tbl = 'sites';			# sites table

# Enter the field names used in the 'users' and 'sites' tables
my $passwd_fld = 'password';		# password field
my $groups_fld = 'groups';			# groups field, comma separated
my $userkey_fld = 'user_key';		# user_key from both tables
my $usercode_fld = 'user_code';		# user_code - same as login user
my $sitecode_fld = 'site_code';		# site_code - name of configs

# This is the value of the stats administrators group
my $admin_group = "stats_admin";

# Should I encrypt the password from the user before comparing to stored value
my $encrypt_passwd = 1;

# You will probably not need to make any modifications below this line
########################################################################

use strict;
use DBI;
use DBD::mysql;
my  $VERSION = '0.9';

# Open file for logging if do_log is set
open (LOG, ">>$auth_log") or die "Could not open file: $auth_log, $!\n" if $do_log;

# Verify command-line arguments
die "Usage: $0 <name> <password>\n" unless $#ARGV eq 1;

# Get the command-line arguments
my $username = $ARGV[0];
my $password = $ARGV[1];

# Connect to the DBI server
my $dbd_str = "DBI:mysql:$sql_db";
$dbd_str .= ";host=$sql_server" if $sql_server ne '';
$dbd_str .= ";port=$sql_port" if $sql_port ne '';
my $dbh = DBI->connect($dbd_str, $sql_db_user, $sql_db_pass, { RaiseError=>1, PrintError=>1, AutoCommit=>1 }) 
		|| die "DBI Connect Error: $DBI::err - $DBI::errstr";
# $dbh->trace(1, "$base_dir/dbi_trace.log");

# Create SQL select on users table (password, groups, user_key)
my $stmt = qq{
	SELECT $passwd_fld, $groups_fld, $userkey_fld
	FROM $users_tbl
	WHERE $usercode_fld = '$username'
};

my $row = $dbh->selectrow_arrayref($stmt) || ['', '', 0];
my $ret_pass = $row->[0];
my $group_str = $row->[1];
my $comp_key = $row->[2];
# print LOG "Row: " . join(";", @$row) . "\n" if $do_log;

print LOG "Checking: $username; $group_str\n" if $do_log;

if ($ret_pass eq '') {	 # If there is no such username, or the password is blank, fail.
    print "*FAILED*\n";

} else {	# If username exists, check if password matches
	# encrypt password
	if ($encrypt_passwd) {
		my $salt = substr($ret_pass, 0, 2);
		$password = crypt_password({password=>$password, salt=>$salt});
		# print LOG "Getting enc_pass: $password\n" if $do_log;
	}

	if ($ret_pass eq $password) {	## check passwords
		print LOG "Passwords matched\n" if $do_log;
		# If this user is an administrator, print *ADMIN*
		if ($group_str =~ m/$admin_group/) {
			print LOG "Admin user\n" if $do_log;
			print "*ADMIN*\n";
		} else {	# Otherwise, print the list of configurations this user can access
			# Create SQL select on sites table (site_code) - returns multiple rows/values
			my $site_stmt = qq{
				SELECT $sitecode_fld
				FROM $sites_tbl
				WHERE $userkey_fld = '$comp_key'
			};
			my $site_codes = $dbh->selectcol_arrayref($site_stmt) || [];
			# print LOG "Sites: " . join(",", @$site_codes) . "\n" if $do_log;
			my $configsList = join("\n", @$site_codes);
			# print LOG "configsList: $configsList\n" if $do_log;
			if ($configsList ne '') {
				print "$configsList\n";
			} else {
				print LOG "No Configurations found\n" if $do_log;
				print "*FAILED*\n";
			}
		}
	} else {
		print LOG "Passwords NOT matched\n" if $do_log;
		print "*FAILED*\n";
	}

} # if authenticated

close LOG if $do_log;
$dbh->disconnect();

# read password from protected file.
sub dbi_pwd_fetch {
	my (@parms) = @_;
	my $file = $parms[0] || "dbpasswd";
	my $passwd = '';
	my $pass_open = open(PASS, "<$file");
	if ($pass_open) {
		chomp($passwd = <PASS>);
		close(PASS);
	} else {
		die "Can't open password file - $file: $!\n";
	}
	return $passwd;
}

sub compute_salt {
  my $count = 2;
  my $str = "";
  my $enc = "./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";   # 64 = length($enc) in call to rand() below
  while ($count--) { 
  	$str .= substr($enc,int(rand(64)),1);
  }
  return $str;
}

sub crypt_password {
	my ($parms) = @_;
	my $pass = '';
	my $salt = '';
	if (ref($parms)) {
		$pass = $parms->{'password'} || '';
		$salt = $parms->{'salt'} || '';
	} elsif ($parms) {
		$pass = $parms;
	}
	if ($salt eq '') {
		$salt = compute_salt();
	}
	my $enc_pass = crypt($pass, $salt);
	return wantarray ? ($enc_pass,$pass) : $enc_pass;
}

exit 0;

1;

__DATA__


=head1 NAME

dbiauth.pl - Perl script to authenticate users for statistics reporting

=head1 SYNOPSIS

DBI Authentication script using DBD::mysql database.
Other databases could be substituted as long as they have 
a DBD interface.

This script authenticates a username/password for access to the
statistics reporter using Perl DBI, and provides access to
configurations for that user.

=head1 DESCRIPTION

If the authentication fails, it prints *FAILED*.  If the user is in the
group specified by $admin_group below, it prints *ADMIN*. Otherwise, it
prints the names of the configurations the user has, one per line.  So
to use DBI authentication, just create two tables, users and sites, and
configure the script to use those tables. Following are example table
definitions.

  CREATE TABLE users (
	user_key int(10) unsigned DEFAULT '0' NOT NULL auto_increment,
	user_code char(50) DEFAULT '' NOT NULL,
	groups char(255) DEFAULT '' NOT NULL,
	password char(40) DEFAULT '' NOT NULL,
	email char(50) DEFAULT '' NOT NULL,
	last_mod timestamp(14),
	PRIMARY KEY (user_key),
	UNIQUE user_code (user_code)
  );
  
  CREATE TABLE sites (
	site_key int(10) unsigned DEFAULT '0' NOT NULL auto_increment,
	user_key int(10) unsigned DEFAULT '0' NOT NULL,
	site_code char(24) DEFAULT '' NOT NULL,
	site_name varchar(80) DEFAULT '' NOT NULL,
	last_mod timestamp(14),
	PRIMARY KEY (site_key),
	KEY user_key (user_key),
	KEY site_name (site_name),
	UNIQUE site_code (site_code)
  );

The groups field in the users table is a comma separated list of all
groups the user belongs to. The password field is a value encrypted 
with the perl crypt command. The user_code is the value entered for the 
user name when logging in.
The sites table is linked via the user_key. The site_code field
is the name of the configuration. More than one site record 
can be linked to a user record.

Some example records for for the table definition above:

  INSERT INTO users VALUES (1,'admin','site,admin','enc_pass','',NULL);
  INSERT INTO users VALUES (2,'site','site','enc_pass','',NULL);
  INSERT INTO users VALUES (3,'odns','site','enc_pass','',NULL);
  
  INSERT INTO sites VALUES (1,2,'stats','stats.make-tracks.com',NULL);
  INSERT INTO sites VALUES (2,2,'www','www.make-tracks.com',NULL);
  INSERT INTO sites VALUES (3,3,'odns','odns.make-tracks.com',NULL);

Logging in as user 'admin' would give Administrator access (user is in 
admin group). Logging in as user 'site' would list configurations named
'stats' and 'www'. Logging in as user 'odns' would list the 
configuration named 'odns'.

This script has only been tested using Linux & mySQL, but should work 
with any OS and database that supports Perl, DBI, DBD and mySQL. 

The location of this script should be entered in the Security tab of
the Preferences, in the "Command-line authentication" field.

=head1 CONFIGURATION

You will need to configure the script with details of your database and
tables.

The base directory is where the script will create and write to a log
file and where it will look for a file containing the database password.

  my $base_dir = '/home/stats/bin/Extras';

The auth_log is the location of the file to log errors to. Logging
occurs only if do_log is a true value.

  my $auth_log = "$base_dir/auth.log";
  my $do_log = 0; # change to 1 to turn on logging

The admin_group is the value of the stats administrators group. Any
users belonging to this group will get full administrator access in
the statistics program.

  my $admin_group = "stats_admin";

The password passed in on the command line (received from user) will be
encrypted before comparing it with the value stored in the database. If
you are not storing encrypted passwords, you can turn this feature off.

  my $encrypt_passwd = 1; # change to 0 to turn off encrypting password


=head2 DATABASE

You will need to configure the database server and port to connect to.
Change the server to the hostname of your DBI server. Leave the value
for server blank to use the default server (usually localhost). You will
probably want to leave the port blank which will use the default value
(port 3306).

  my $sql_server = '';
  my $sql_port = '';
OR
  my $sql_server = '10.8.0.1';
  my $sql_port = '3306';

You also need to configure the name of your database.

  my $sql_db = "db_name";

Enter the user id used to connect to the database

  my $sql_db_user = "db_user";

The database password can either be entered into this script directly:

  my $sql_db_pass = "db_pass";

Or it can be read from a protected file. If the password is to 
be read from a file, it must have read permissions for the user 
or group that the statistics program is running under.

  my $sql_db_pass = dbi_pwd_fetch("$base_dir/dbpasswd");

=head2 TABLES

You will need to configure the names of the 'users' and 'sites' tables
and the fields used in each.

  my $users_tbl = 'users';          # users table
  my $sites_tbl = 'sites';          # sites table
  my $passwd_fld = 'password';      # password field
  my $groups_fld = 'groups';        # groups field, comma separated
  my $userkey_fld = 'user_key';     # user_key from both tables
  my $usercode_fld = 'user_code';   # user_code - same as login user
  my $sitecode_fld = 'site_code';   # site_code - name of configs

=head1 BUGS

I am aware of no bugs - if you find one, please send an email to
info@make-tracks.com.

When submitting bug reports, be sure to include full details, including
the VERSION of the module, and a sample of your table definitions.

=head1 CREDITS

This module was developed out of necessity to have better control over
user's access to configurations. I already had all the user and
site data in mySQL and I just needed a script to access it.

The statistics program already provided the needed hooks and an example script.
This script grew from there.

=head1 AUTHOR

  Charlie Garrison
  Make-Tracks, info@make-tracks.com
  http://www.make-tracks.com/

=cut
