#!/usr/bin/perl
#
# LDAP Authentication script
#
# This authenticates a username/password using LDAP, and provides
# access to profiles using LDAP groups.
#
# If the authentication fails, it prints *FAILED*.  If the user is in
# group specified by $admingroup below, it prints *ADMIN*.
# Otherwise, it prints the names of the groups the user is in,
# one per line.  So to use LDAP authentication, just create a
# "statsadmin" group, add all your administrators to it, create
# a group per profile (with the same name as the profile)
# and add the users authorized to view each profile to that
# profile's group.
#
# The location of this script should be entered in the Security tab of
# the Preferences, in the "Command-line authentication" field.
#
# NOTE: Apparently perl does not support encrypted password for LDAP,
# so you may need to turn off that feature on your LDAP server if you
# need to use this script.
#

########################################################################
# You will need to modify these values in order to use this script:


# The LDAP server and port to use-- change the server to the hostname of
# your LDAP server.  You will probably want to leave the port set to 389.
my $server = 'localhost';
my $port = 389;

# This is the base group we're looking for users in.  Set this
# to the LDAP group where your users are stored.
my $base = 'dc=flowerfire,dc=com';

# This specifies the label used to identify usernames.  You may need yo
# change this if you use a different label, e.g. "uid="
my $usernamelabel = "cn";

# This is the cn of the stats administrators group
my $admingroupcn = "statsadmin";

# You will probably not need to make any modifications below this line
########################################################################


use strict;
use Net::LDAP;

# Verify command-line arguments
die "Usage: $0 <name> <password>\n" if not $#ARGV eq 1;

# Get the command-line arguments
my $username = $ARGV[0];
my $password = $ARGV[1];

# Compute the DN of this user
my $userdn = "$usernamelabel=$username,$base";

# Connect to the LDAP server
my $ldap = Net::LDAP->new($server, port => $port) or
    die "Could not contact LDAP server $server:$port";

# Bind to the LDAP server, verifying username/password
my $mesg = $ldap->bind( $userdn, password => $password);

# If there is no such username, or the password is incorrect, fail.
if ($mesg->code()) {
    print "*FAILED*\n";
}

# If the username authenticated correctly, check its permissions
else {

    # Find the node with this username
    $mesg = $ldap->search(base   => $userdn,
			  scope  => 'base',
			  filter => "$usernamelabel=$username");
    my $entry = $mesg->pop_entry();
    my $userDN = $entry->dn();

    # Find all groups
    $mesg = $ldap->search(base => $base,
			  filter => "(objectclass=groupOfUniqueNames)");

    # Look through all groups, to see which ones this user belongs to
    my $userIsAdmin = 0;
    my $configsList = "";
    foreach my $groupentry ($mesg->all_entries) {

	# The DN for the group
	my $groupdn = $groupentry->dn();

	# Check if the user is in the group
	my $mesg = $ldap->compare($groupdn, attr=>"uniquemember", value=>$userdn);

	# Code 6 means this user is a member of this group
	if ($mesg->code() == 6) {

	    # Get the name of the group (cn), and add it to the list
	    my $cn = $groupentry->get_value('cn');
	    $configsList .= "$cn\n";

	    # If this is the admin group, it means this user is a stats administrator
	    if ($cn eq $admingroupcn) {
		$userIsAdmin = 1;
	    }

	} # if member

    } # for groups

    # If this user is an administrator, print *ADMIN*
    if ($userIsAdmin) {
	print "*ADMIN*\n";
    }

    # Otherwise, print the list of profiles this user can access
    else {
	print $configsList;
    }

} # if authenticated

# Unbind from the LDAP server
$ldap->unbind;

exit 0;
